;;; -*- Mode: LISP; Syntax: Common-lisp; Package: SILICA; Base: 10 -*-

(in-package "SILICA")

"Copyright (c) 1990 by International Lisp Associates"

(define-graphics-function draw-point (medium point &drawing-options)
  :spread-arguments ((point (point point-x point-y)))
  :type :point)

(define-graphics-function draw-points (medium point-sequence &drawing-options)
  :spread-arguments ((point-sequence (point-sequence coordinate-sequence)))
  :type :point)

(define-graphics-function draw-line (medium from-point to-point &drawing-options)
  :spread-arguments ((point (from-point from-x from-y)
			    (to-point to-x to-y)))
  :type :line)

(define-graphics-function draw-lines (medium point-sequence &drawing-options)
  :spread-arguments ((point-sequence (point-sequence coordinate-sequence)))
  :type :line
  :default-method* (((medium basic-medium) coordinate-sequence)
		    (loop (when (endp coordinate-sequence) (return))
			  (macrolet ((pop-coordinate ()
				       `(if (endp coordinate-sequence)
					    (error "Bad number of coordinates ~
						    passed to ~S" 'draw-lines)
					    (pop coordinate-sequence))))
			    (let ((from-x (pop-coordinate))
				  (from-y (pop-coordinate))
				  (to-x (pop-coordinate))
				  (to-y (pop-coordinate)))
			      (funcall #'(:graphics-internal draw-line*)
				       medium from-x from-y to-x to-y))))))

(define-graphics-function draw-polygon (medium point-sequence &key (closed t) (filled t)
					       &drawing-options)
  :spread-arguments ((point-sequence (point-sequence coordinate-sequence)))
  :type :area)

(define-graphics-function draw-rectangle (medium point1 point2 &key (filled t)
						 &drawing-options)
  :spread-arguments (;; This used to take a rectangle as an argument,
		     ;; but this is at odds with the documentation, so
		     ;; we took it out for 0.9 patch #4. --- rsl
		     (point (point1 x1 y1)
			    (point2 x2 y2)))
  :type :area
  ;; Implement rectangle drawing in terms of polygons by default.  

  ;; Note that this means that EVERY medium which does output-recording MUST
  ;; have a native implementation for the DRAW-RECTANGLE* internal method;
  ;; otherwise, the output record will contain two elements, one for the
  ;; rectangle and one for the polygon.  In CLIM, the method for
  ;; DRAW-RECTANGLE*-INTERNAL will be the one which invokes the same method
  ;; on the drawing medium (which does not do output recording).  The one on
  ;; the drawing medium may or may not be the method below...

  :default-method* (((medium basic-medium) x1 y1 x2 y2 &key (filled t))
		    
		    (with-stack-list (coordinate-sequence x1 y1 x1 y2 x2 y2 x2 y1)
		      ;; The macro actually permits ((:GRAPHICS-INTERNAL ...) ...),
		      ;; but the following is slightly more perspicuous (and should
		      ;; generate the same code with any reasonable compiler).
		      (funcall #'(:graphics-internal draw-polygon*) medium
			       coordinate-sequence :closed t :filled filled))))

;;; Intended as a crutch for existing clients that want a draw-rectangle
;;; that really takes a rectangle.  I don't know what the real solution is.
;;; -York
(defun draw-rectangle-with-rectangle (medium rectangle &rest drawing-options)
  (declare (dynamic-extent drawing-options))
  (with-rectangle-spread (rectangle x1 y1 x2 y2)
    (apply #'draw-rectangle* medium x1 y1 x2 y2 drawing-options)))

;;; Take advantage of compiler macro on DRAW-RECTANGLE* if possible.
;;; Too bad that just proclaimig DRAW-RECTANGLE-WITH-RECTANGLE as INLINE doesn't 
;;; do what you want in this case, at least under Genera.
(define-compiler-macro draw-rectangle-with-rectangle (medium rectangle &rest drawing-options)
  (let ((x1 (gensymbol 'x1))
        (y1 (gensymbol 'y1))
        (x2 (gensymbol 'x2))
        (y2 (gensymbol 'y2))
        (medium-var (gensymbol 'medium)))
   `(let ((,medium-var ,medium))
      (with-rectangle-spread (,rectangle ,x1 ,y1 ,x2 ,y2)
        (draw-rectangle* ,medium-var ,x1 ,y1 ,x2 ,y2 ,@drawing-options)))))

(define-graphics-function draw-ellipse
	(medium center-point radius-1-dx radius-1-dy radius-2-dx radius-2-dy
		&key (filled t) (start-angle 0) (end-angle 2pi)
		&drawing-options)
  :spread-arguments ((point (center-point center-x center-y)))
  :type :area)

(define-graphics-function draw-circle (medium center-point radius
					      &key (filled t) (start-angle 0) (end-angle 2pi)
					      &drawing-options)
  :spread-arguments ((point (center-point center-x center-y)))
  :type :area
  ;; See the definition of DRAW-RECTANGLE above about why ALL
  ;; output-recording media must supply DRAW-CIRCLE*-INTERNAL methods.
  :default-method* (((medium basic-medium) center-x center-y radius &rest args)
                    (declare (dynamic-extent args))
		    (apply #'(:graphics-internal draw-ellipse*) medium
			   center-x center-y radius 0 0 radius args)))

(define-graphics-function draw-text (medium string-or-char point &key (start 0) end
					    (align-x :left) (align-y :baseline) toward-point
					    transform-glyphs &drawing-options)
  :spread-arguments ((point (point point-x point-y)))
  :type :text)


;;;
;;;
;;;

(defoperation medium-force-output clg-graphics ((medium clg-medium)))

(defoperation medium-finish-output clg-graphics ((medium clg-medium)))

(defmethod with-text-style-internal ((medium basic-medium) style continuation original-stream)
  ;; Quickly handle a simple case.
  (if (or (null style) (eql style *null-text-style*))
      (funcall continuation original-stream)
      ;;; Bind MEDIUM-TEXT-STYLE if style changes.
      (with-slots (medium-text-style) medium
	(let* ((old-text-style medium-text-style)
	       (new-text-style (merge-text-styles style old-text-style)))
	  (letf-globally ((medium-text-style new-text-style))
	    (funcall continuation original-stream))))))

(defoperation with-text-style-internal clg-graphics ((medium clg-medium)
						     style continuation original-stream))
