;;; -*- Mode: LISP; Syntax: Common-lisp; Package: SILICA; Base: 10 -*-

(in-package "SILICA")

"Copyright (c) 1990, 1991 by International Lisp Associates"

;;; Prepare for the forward references to these SETF methods in this file.
(defgeneric (setf medium-transformation) (new-transformation medium))
(defgeneric (setf medium-clipping-region) (new-clipping-region medium))

;;; This is probably not general-purpose enough to put in LISP-UTILITIES or UTILITIES.
;;; This macro CDDRs down a property list which is really a &REST keyword list.  It
;;; uses GETF to make sure it retains the &KEY semantics.  It returns a list of all
;;; the unhandled keyword/value pairs.
(defmacro option-keyword-case ((options option-value-var &optional 
			       (option-keyword-var (make-symbol "KEY"))) &body cases)
  (let* ((remaining-var (make-symbol "REMAINING"))
	 (none-handled-var (make-symbol "NONE-HANDLED"))
	 (count-var (make-symbol "COUNT"))
	 (options-var (make-symbol "REMAINING"))
	 (original-options-var (make-symbol "ORIGINAL-OPTIONS"))
	 (copy-unhandled 
	   `(when ,none-handled-var
	      (let ((,original-options-var ,original-options-var))
		(dotimes (i ,count-var (setf ,none-handled-var nil))
		  #+Genera-release-8 (declare (ignore i))
		  (push (pop ,original-options-var) ,remaining-var))))))
    `(let* ((,remaining-var nil)
	    (,none-handled-var t)
	    (,count-var 0)
	    (,options-var ,options)
	    (,original-options-var ,options-var)
	    ,option-keyword-var ,option-value-var)
       (declare (fixnum ,count-var))
       (loop (when (null ,options-var)
	       (when ,remaining-var (setf ,remaining-var (nreverse ,remaining-var)))
	       (return))
	     (setf ,option-keyword-var (pop ,options-var))
	     (when (endp ,options-var)
	       (error "Incorrect argument list; odd number of keywords."))
	     ;; GETF to preserve &KEY semantics
	     (setf ,option-value-var (prog1 (getf ,original-options-var ,option-keyword-var)
					    (pop ,options-var)))
	     (case ,option-keyword-var
	       ,@(mapcar #'(lambda (case) (list* (car case) copy-unhandled (cdr case)))
			 cases)
	       (otherwise
		 ,copy-unhandled
		 ;; This might cause subtle bugs later.  For example, think about
		 ;; (:alignment :X :X 42).  For now, there are no cases I know of
		 ;; where the value of some keyword is the same as the name of a
		 ;; different keyword.  This implementation cares more about CONSing
		 ;; and correctness than about searching time.
		 (unless (member ,option-keyword-var ,remaining-var)
		   (push ,option-keyword-var ,remaining-var)
		   (push ,option-value-var ,remaining-var)))))
       (when ,none-handled-var (setf ,remaining-var ,original-options-var))
       ,remaining-var)))



(eval-when (compile load eval)
  (defparameter *drawing-options-keywords*
		'(:transformation :ink :clipping-region
		  :text-style :text-face :text-family :text-size
		  :line-style :line-units :line-thickness :line-scale-lines :line-joint-shape
		  :line-cap-shape :line-dashed :line-dash-pattern :line-scale-dashes
		  :line-initial-dash-phase))
  (defparameter *drawing-options-keyword-arguments*
		'((:default . #1= (transformation ink clipping-region))
		  (:point .   #2= (line-thickness line-scale-lines line-style . #1#))
		  (:line .    #3= (line-units line-joint-shape line-cap-shape line-dashed
				   line-dash-pattern line-scale-dashes . #2#))
		  (:area .    #3#)
		  (:text .    (text-face text-family text-size text-style . #1#)))))

;;; A few macro definitions:
(defmacro with-drawing-options ((medium &rest medium-options) &body body)
  (let (med-bindings ls-bindings medium-var line-style-var text-style)
    (when (listp medium)
      (setf medium-var (second medium) medium (first medium)))
    (macrolet ((generate-medium () `(unless medium-var
				      (setf medium-var
					    (if (symbolp medium) medium (gensymbol 'medium)))))
	       (generate-line-style ()
		 `(unless line-style-var
		    (generate-medium)
		    (setf line-style-var (gensymbol 'line-style)))))
      (let* (ink transformation clipping-region line-style
	     text-family text-face text-size
	     (line-style-options
	       (option-keyword-case (medium-options value)
		 (:ink (setf ink value))
		 (:transformation (setf transformation value))
		 (:clipping-region (setf clipping-region value))
		 (:line-style (setf line-style value))
		 (:text-style (setf text-style value))
		 (:text-family (setf text-family value))
		 (:text-face (setf text-face value))
		 (:text-size (setf text-size value)))))
	(when (or text-family text-face text-size)
	  (let ((merged-text-form `(make-text-style ,text-family ,text-face ,text-size)))
	    (if text-style
		(setf text-style `(merge-text-styles ,merged-text-form ,text-style))
		(setf text-style merged-text-form))))
	(when text-style
	  (generate-medium))					;Will add WITH-TEXT-STYLE below
	(when transformation
	  (generate-medium)
	  (push `((medium-transformation ,medium-var)
		  (compose-transformations ,transformation 
					   (medium-transformation ,medium-var)))
		med-bindings))
	(when clipping-region
	  (generate-medium)
	  (push `((medium-clipping-region ,medium-var)
		  (region-intersection ,clipping-region 
				       (medium-clipping-region ,medium-var)))
		med-bindings))
	(when ink
	  (generate-medium)
	  (push `((medium-ink ,medium-var) ,ink) med-bindings))
	(when line-style
	  (generate-medium)
	  (push `((medium-line-style ,medium-var)
		  (merge-line-styles ,line-style (medium-line-style ,medium-var)))
		med-bindings))
	(when line-style-options
	  (generate-line-style)
	  (macrolet ((bind-ls (thing value)
		       (when (string= thing 'line-style- :end1 (length "line-style-"))
			 (setf thing (fintern "~A-~A" '%ls
					      (subseq (string thing) (length "line-style-")))))
		       `(push `((,',thing ,line-style-var) ,,value) ls-bindings)))
	    (let ((leftover-keyword-options
		    (option-keyword-case (line-style-options value)
		      (:line-units (bind-ls line-style-units value))
		      (:line-thickness (bind-ls line-style-thickness value))
		      (:line-scale-lines (bind-ls line-style-scale-lines value))
		      (:line-joint-shape (bind-ls line-style-joint-shape value))
		      (:line-cap-shape (bind-ls line-style-cap-shape value))
		      (:line-dashed (bind-ls line-style-dashed value))
		      (:line-dash-pattern (bind-ls line-style-dash-pattern value))
		      (:line-scale-dashes (bind-ls line-style-scale-dashes value))
		      (:line-initial-dash-phase
			(bind-ls line-style-initial-dash-phase value)))))
	      (when leftover-keyword-options
		(warn "~S passed extraneous keyword options ~S"
		      'with-drawing-options leftover-keyword-options)))))))
    (when line-style-var
      (setf body `((let ((,line-style-var (medium-line-style ,medium-var)))
		     (letf-globally (((%ls-copy-cache ,line-style-var) nil)
				     ,@ls-bindings)
		       ,@body)))))
    (when text-style
      (setf body `((with-text-style (,text-style ,medium-var)
		     ,@body))))
    (when medium-var
      (setf body `((let (,@(if (not (eql medium-var medium))
			       `((,medium-var ,medium))))
		     (letf-globally ,med-bindings
		       ,@body))))))
  `(progn ,@body))

(defmacro with-merged-medium-from-drawing-options
	  ((type remaining-options medium drawing-options) &body body)
  #+Genera (declare (zwei:indentation 0 3 1 1))
  `(flet ((merged-drawing-body (,remaining-options) ,@body))
     (declare (dynamic-extent #'merged-drawing-body))
     (with-merged-medium-from-drawing-options-internal
       ',type #'merged-drawing-body ,medium ,drawing-options)))

(defmacro with-merged-line-style-options
	  ((medium line-style-options new-line-style) &body body)
  #+Genera (declare (zwei:indentation 0 3 1 1))
  `(flet ((line-style-body () ,@body))
     (declare (dynamic-extent #'line-style-body))
     (with-merged-line-style-options-internal
       ,medium #'line-style-body ,new-line-style ,line-style-options)))

(defmacro with-identity-transformation ((medium) &body body)
  `(with-drawing-options (,medium :transformation +identity-transformation+)
     ,@body))

(defmacro with-translation ((medium dx dy) &body body)
  `(with-drawing-options (,medium :transformation
				  (make-translation-transformation ,dx ,dy))
     ,@body))
 
(defmacro with-scaling ((medium sx &optional (sy nil sy-p)) &body body)
  `(with-drawing-options 
       (,medium :transformation (make-scaling-transformation
				 ,sx ,(if sy-p sy sx)))
     ,@body))
 
(defmacro with-rotation ((medium angle) &body body)
  `(with-drawing-options (,medium :transformation
				  (make-rotation-transformation ,angle))
     ,@body))

(defmacro with-text-style ((style &optional medium) &body body)
  (default-output-stream medium)
  `(flet ((with-text-style (,medium)
	    ;; Don't warn about non-use of newly-bound medium:
	    (progn ,medium nil)
	    ,@body)) 
     (declare (dynamic-extent #'with-text-style))
     (with-text-style-internal ,medium ,style #'with-text-style ,medium)))

(defmacro with-text-family ((family &optional medium) &body body)
  `(with-text-style ((make-text-style ,family nil nil) ,medium) ,@body))

(defmacro with-text-face ((face &optional medium) &body body)
  `(with-text-style ((make-text-style nil ,face nil) ,medium) ,@body))

(defmacro with-text-size ((size &optional medium) &body body)
  `(with-text-style ((make-text-style nil nil ,size) ,medium) ,@body))

;;; Some implementation notes:
;;; 
;;; 1.  There is no particular power to making LINE-STYLE be a class instead of a structure.
;;;	This is probably the first place to look for performance improvements in this stuff.
;;;
;;; 2.  A possible performance improvement might be found by not binding the copy-cache
;;;	variable but merely using EQUALP to determine if it's still correct, and making
;;;	a new one if not.  I don't know whether we're going to see a lot more munging of 
;;;	the state back to the same state (e.g., drawing two objects in a row with the
;;;	same ink) or a lot more of mostly using the same state but occasionally changing
;;;	the state to something else.
;;;

(defclass line-style ()
     ((units :initarg :units :initform 'nil :reader line-style-units
	     :accessor %ls-units)
      (thickness :initarg :thickness :initform 'nil :reader line-style-thickness
		 :accessor %ls-thickness)
      (scale-lines :initarg :scale-lines :initform 'nil :reader line-style-scale-lines
		   :accessor %ls-scale-lines)
      (joint-shape :initarg :joint-shape :initform 'nil :reader line-style-joint-shape
		   :accessor %ls-joint-shape)
      (cap-shape :initarg :cap-shape :initform 'nil :reader line-style-cap-shape
		 :accessor %ls-cap-shape)
      (dashed :initarg :dashed :initform 'nil :reader line-style-dashed
	      :accessor %ls-dashed)
      (dash-pattern :initarg :dash-pattern :initform 'nil :reader line-style-dash-pattern
		    :accessor %ls-dash-pattern)
      (scale-dashes :initarg :scale-dashes :initform 'nil :reader line-style-scale-dashes
		    :accessor %ls-scale-dashes)
      (initial-dash-phase :initarg :initial-dash-phase :initform 'nil
			  :reader line-style-initial-dash-phase
			  :accessor %ls-initial-dash-phase)
      (copy-cache :initform nil :accessor %ls-copy-cache)))

(define-constructor make-line-style line-style
  (&key units thickness scale-lines joint-shape
	cap-shape dashed dash-pattern scale-dashes
	initial-dash-phase)
  :units units :thickness thickness :scale-lines scale-lines
  :joint-shape joint-shape :cap-shape cap-shape :dashed dashed
  :dash-pattern dash-pattern :scale-dashes scale-dashes
  :initial-dash-phase initial-dash-phase)

(defmethod copy-line-style ((line-style line-style))
  (with-slots (units thickness scale-lines joint-shape cap-shape dashed
		     dash-pattern scale-dashes initial-dash-phase copy-cache)
	      line-style
    (or copy-cache
	(setf copy-cache
		(make-line-style :units units :thickness thickness :scale-lines scale-lines
				 :joint-shape joint-shape :cap-shape cap-shape :dashed dashed
				 :dash-pattern dash-pattern :scale-dashes scale-dashes
				 :initial-dash-phase initial-dash-phase)))))

(defmethod print-object ((line-style line-style) stream)
  (let ((default t))
    (print-unreadable-object (line-style stream)
      (write-string "Line style:" stream)
      (macrolet ((print-thing (name)
		   `(when (slot-value line-style ',name)
		      (setf default nil)
		      (format stream " ~(~A~) ~S" ',(intern (string name) *keyword-package*)
			      (slot-value line-style ',name))))
		 (print-things (&rest names)
		   `(progn ,@(mapcar #'(lambda (name) `(print-thing ,name)) names))))
	(print-things units thickness scale-lines joint-shape
		      cap-shape dashed dash-pattern scale-dashes initial-dash-phase))
      (when default (write-string " default" stream)))))

(defparameter *null-line-style* (make-line-style))


;;; Handle foreground, background, etc.
;;; --- Maybe RSL knows where this should really go...
;;; --- RSL thinks this should be a generic function on the class of ink.
(defun copy-ink (ink)
  (etypecase ink
    (symbol ink)
    (number ink)
    (color (copy-color ink))))

;;; The following gets called when DRAW-xxx(*) is invoked without the compiler
;;; optimizer, either because it is being called interpreted, called with
;;; APPLY, or because the compiler optimizer didn't work for some reason (e.g.,
;;; the compiler doesn't support compiler optimizers.).

;;; This is the version which merges in place.  PARSE-DRAWING-OPTIONS actually
;;; parses everything except line-style options, which is done in the function
;;; WITH-MERGED-LINE-STYLE-OPTIONS-INTERNAL.
(defun with-merged-medium-from-drawing-options-internal (type continuation medium options)
  (declare (dynamic-extent continuation))
  (when (null options)						; Cheap and easy optimization.
    (return-from with-merged-medium-from-drawing-options-internal
      (funcall continuation nil)))
  (multiple-value-bind (new-ink new-transform new-text-style new-line-style
			new-clipping-region line-style-options
			remaining-options)
      (parse-drawing-options type options)
    (letf-globally-if new-ink (((medium-ink medium) new-ink))
      (letf-globally-if new-transform
	   (((medium-transformation medium)
	     (compose-transformations new-transform (medium-transformation medium))))
	(letf-globally-if new-clipping-region
	     (((medium-clipping-region medium)
	       (region-intersection new-clipping-region (medium-clipping-region medium))))
	  (macrolet ((do-it ()
		       `(if new-text-style
			    (with-text-style (new-text-style medium)
			      (funcall continuation remaining-options))
			    (funcall continuation remaining-options))))
	    (if (or line-style-options new-line-style)
		(with-merged-line-style-options (medium line-style-options new-line-style)
		  (do-it))
		(do-it))))))))
     
;;; This function handles :INK, :TRANSFORMATION, :CLIPPING-REGION, :TEXT-STYLE,
;;; :TEXT-FACE/FAMILY/SIZE, and :LINE-STYLE.  It also recognizes line-style
;;; option keywords, returning them as a separate list.  It returns a list 
;;; of all options not recognized.
(defun parse-drawing-options (type drawing-options)
  #+Genera (declare (values new-ink new-transform new-text-style new-line-style 
			    new-clipping-region line-style-keywords
			    remaining-options))
  (macrolet ((ensure-type (type)
	       (let ((predicate
		       (ecase type
			 ((:text :point) `(eql ',type type))
			 (:line `(or (eql ':line type) (eql ':point type)))
			 (:area `(or (eql ':area type)
				     (eql ':line type)
				     (eql ':point type))))))
		 `(unless ,predicate
		    (error "~S is not a valid keyword" keyword)))))
    (let* (new-ink
	   new-transform
	   new-text-style
	   new-line-style
	   new-clipping-region
	   pre-merged-text-style
	   line-style-keywords
	   (remaining-options
	     (option-keyword-case (drawing-options value keyword)
	       (:ink (setf new-ink value))
	       (:transformation (setf new-transform value))
	       (:text-style (ensure-type :text) (setf new-text-style value))
	       (:clipping-region (setf new-clipping-region value))
	       (:line-style  (ensure-type :area) (setf new-line-style value))
	       (:text-family
		 (ensure-type :text)
		 (when value (setf pre-merged-text-style
				     (merge-text-styles (make-text-style value nil nil)
							pre-merged-text-style))))
	       (:text-face
		 (ensure-type :text)
		 (when value (setf pre-merged-text-style
				     (merge-text-styles (make-text-style nil value nil)
							pre-merged-text-style))))
	       (:text-size
		 (ensure-type :text)
		 (when value (setf pre-merged-text-style
				     (merge-text-styles (make-text-style nil nil value)
							pre-merged-text-style))))
	       ((:line-units :line-thickness :line-scale-lines :line-joint-shape
			     :line-cap-shape :line-dashed :line-dash-pattern :line-scale-dashes
			     :line-initial-dash-phase)
		(ensure-type :area)
		;; See comment above in the definition of OPTION-KEYWORD-CASE concerning
		;; the danger of the following construction.
		(unless (member keyword line-style-keywords)
		  (push keyword line-style-keywords)
		  (push value line-style-keywords))))))
      (when pre-merged-text-style
	(if new-text-style
	    (setf new-text-style (merge-text-styles pre-merged-text-style new-text-style))
	    (setf new-text-style pre-merged-text-style)))
      (when (eql new-text-style *null-text-style*) (setf new-text-style nil))
      (values new-ink new-transform new-text-style new-line-style 
	      new-clipping-region
	      (nreverse line-style-keywords) remaining-options))))

(defun with-merged-line-style-options-internal (medium continuation new-line-style
						       line-style-options)
  (declare (dynamic-extent continuation))
  (when (and (null line-style-options) (null new-line-style))
    (return-from with-merged-line-style-options-internal
      (funcall continuation)))
  (letf-globally-if new-line-style
       (((medium-line-style medium)
	 (merge-line-styles (medium-line-style medium) new-line-style)))
    ;; Although it has a name like that of a macro, the following is really a function.
    (with-line-style-options-bound (medium-line-style medium) continuation
				   line-style-options)))

(defmethod with-line-style-options-bound ((line-style line-style) continuation
					  line-style-options)
  (declare (dynamic-extent continuation))
  (when (null line-style-options)
    (return-from with-line-style-options-bound (funcall continuation)))
  (with-slots (units thickness scale-lines joint-shape cap-shape dashed dash-pattern
		     scale-dashes initial-dash-phase copy-cache) line-style
    (letf-globally ((units)
		    (thickness)
		    (scale-lines)
		    (joint-shape)
		    (cap-shape)
		    (dashed)
		    (dash-pattern)
		    (scale-dashes)
		    (initial-dash-phase)
		    (copy-cache))
      (let ((unhandled-line-styles
	      (option-keyword-case (line-style-options value)
		(:line-units (setf units value copy-cache nil))
		(:line-thickness (setf thickness value copy-cache nil))
		(:line-scale-lines (setf scale-lines value copy-cache nil))
		(:line-joint-shape (setf joint-shape value copy-cache nil))
		(:line-cap-shape (setf cap-shape value copy-cache nil))
		(:line-dashed (setf dashed value copy-cache nil))
		(:line-dash-pattern (setf dash-pattern value copy-cache nil))
		(:line-scale-dashes (setf scale-dashes value copy-cache nil))
		(:line-initial-dash-phase (setf initial-dash-phase value copy-cache nil)))))
	(when unhandled-line-styles
	  (error "Internal inconsistency: ~S called with unhandled line style keywords ~S"
		 'with-line-style-options-bound unhandled-line-styles)))
      (funcall continuation))))

(defmethod merge-line-styles ((line-style line-style) (new-line-style line-style))
  (let ((copy-copy-cache (slot-value line-style 'copy-cache)))
    (macrolet ((copy-it (thing)
		 (let ((new-thing (fintern "~A-~A" 'new thing)))
		   `(with-slots (,thing) line-style
		      (let ((,new-thing (slot-value new-line-style ',thing)))
			(when ,new-thing
			  ;; Maintain cache consistency as cheaply as possible.
			  (when (and copy-copy-cache
				     (not (eql ,new-thing ,thing)))
			    (setf copy-copy-cache nil))
			  (setf ,thing ,new-thing))))))
	       (copy-everything ()
		 `(progn ,@(with-collection
			     (dolist (thing '(units thickness scale-lines joint-shape
						    cap-shape dashed dash-pattern scale-dashes
						    initial-dash-phase copy-cache) )
			       (collect `(copy-it ,thing)))))))
      (copy-everything))
    (setf (slot-value line-style 'copy-cache) copy-copy-cache))
  line-style)


#||
;;; Moving DEFPROTOCOL of CLG-GRAPHICS
;;; Moving DEFROLE of CLG-MEDIUM to new file graphics-protocol. 
(defprotocol clg-graphics ()
  (:roles clg-medium)
  (:documentation "The Common Lisp Graphics package."))

(defrole clg-medium ()
  ((foreground :accessor foreground)
   (background :accessor background)
   (ink :accessor medium-ink)
   (medium-text-style :accessor medium-text-style)
   (default-text-style :accessor medium-default-text-style)
   (line-size :accessor line-size)
   (line-style :accessor medium-line-style)

   (clipping-region :accessor medium-clipping-region)
   (transformation :accessor medium-transformation)))
||#



#+Genera 
(scl:defprop define-graphics-function "Graphics function" si:definition-type-name)

;;; See the file GRAPHICS-FUNCTIONS for examples of the uses of this macro.
(defmacro define-graphics-function (unspread-function-name unspread-lambda-list
				    &key spread-arguments type default-method*)
  #+Genera (declare (zwei:indentation 1 7 2 1))
  (multiple-value-bind
    (unspread-function-lambda-list unspread-function-body
     spread-function-name spread-function-lambda-list spread-function-body
     method-function-name method-lambda-list spread-arguments-descriptors
     unspread-argument-list spread-argument-list method-argument-list
     pretty-unspread-lambda-list pretty-spread-lambda-list
     #+PCL bogus-generic-function-definition)
      ;; These two bindings are so the WARNings in COMPUTE-SPREAD-ARGUMENTS
      ;; will give good information about where the error occurred.
      (let (#+Genera (compiler:default-warning-function unspread-function-name)
	    #+Genera (compiler:default-warning-definition-type 'define-graphics-function))
	(compute-spread-arguments type unspread-function-name
				  unspread-lambda-list spread-arguments))
    (declare (ignore unspread-argument-list spread-argument-list
                     #-Genera pretty-unspread-lambda-list
                     #-Genera pretty-spread-lambda-list))
    (unless default-method*
      (setf default-method*
	      `(((medium basic-medium) ,@method-lambda-list)
		(declare (ignore ,@method-argument-list))
		(error "The drawing method for ~S has not been defined for medium class ~S"
		       ',unspread-function-name (class-name (class-of medium))))))
    (let ((default-method-arglist (first default-method*)))
      (unless (member '&key default-method-arglist)
	(nconc default-method-arglist (list '&key '&allow-other-keys))))
    `(progn
       ;; Build database of graphics operations.  This has to go outside of the
       ;; DEFINE-GROUP due to a bug in Genera.  Very sorry.
       (eval-when (compile load eval)
	 (define-graphics-function-compile-time
	   ',unspread-function-name
	   :spread-function-name ',spread-function-name
	   :method-function-name ',method-function-name
	   :unspread-lambda-list ',unspread-function-lambda-list
	   :spread-lambda-list ',spread-function-lambda-list
	   :method-lambda-list ',method-lambda-list
	   :spread-arguments ',spread-arguments-descriptors
	   :method-argument-list ',method-argument-list
	   :type ',type))

       (define-group ,unspread-function-name define-graphics-function
	 ;; Define DRAW-FOO
	 (defun ,unspread-function-name ,unspread-function-lambda-list
           #+Genera (declare (arglist ,@pretty-unspread-lambda-list))
	   (declare (dynamic-extent drawing-options))
	   ,unspread-function-body)
	 ;; Define DRAW-FOO*
	 (defun ,spread-function-name ,spread-function-lambda-list
	   ;; #-Allegro
	   ;; Compiler Bug with interaction between dynamic extent and closures
	   ;; and large number of arguments prevents this from working.
	   ;; Fixed if using patch28
	   ;; -- RR -- 
	   (declare (dynamic-extent drawing-options))
           #+Genera (declare (arglist ,@pretty-spread-lambda-list))
	   ,spread-function-body)
	 
	 ;; Define Operations on CLG graphics
	 (defoperation ,method-function-name clg-graphics
	   ((medium clg-medium) ,@method-lambda-list))

	 ;; Define generic function for DRAW-FOO*-INTERNAL
	 #+Ignore ;; This is also done by DEFOPERATION...
	 (define-graphics-function-generic ,spread-function-name
					   (medium ,@method-lambda-list))
	 ;; Define default method for DRAW-FOO*-INTERNAL
	 (define-graphics-function-method ,spread-function-name ,@default-method*)
	 ;; Do compiler optimizers.
	 (define-graphics-function-compiler-optimizers ,unspread-function-name)
	   
	 ;; PCL doesn't check for keyword arguments correctly.
	 #+PCL ,bogus-generic-function-definition
	 ;; Return the name of the group.
	 ',unspread-function-name))))

;;; Define a macro named WITH-type-SPREAD, which explodes an object of type TYPE.
;;; The default macro generates (MULTIPLE-VALUE-BIND ,SPREAD-ARGS (,FUNCTION ,ARG) ,@BODY)
;;; which is right for most kinds of bindings.
(defmacro define-spreading-function (spread-type function
				     &optional macro-arglist &body macro-body)
  #+Genera (declare (zwei:indentation 2 7 3 1))
  (unless macro-arglist
    (setf macro-arglist '((value &rest spread-vars) &body body)
	  macro-body `(`(multiple-value-bind ,spread-vars (,',function ,value)
			  ,@body))))
  (let ((with-spread-type-macro-name (fintern "~A-~A-~A" 'with spread-type 'spread)))
    `(define-group ,spread-type define-spreading-function
       ;; Make sure function is defined.  Issue warning for right definition under Genera.
       (let (#+Genera (compiler:default-warning-function ',spread-type)
	     #+Genera (compiler:default-warning-definition-type 'define-spreading-function))
	 (unless (fboundp ',function)
	   (warn "Function ~S to be used for spreading ~S arguments is not defined"
		 ',function ',spread-type)))
       (defmacro ,with-spread-type-macro-name ,macro-arglist ,@macro-body)
       (setf (get ',spread-type 'spreading-macro) ',with-spread-type-macro-name)
       ',spread-type)))

#+Genera 
(scl:defprop define-spreading-function "Point spread function" si:definition-type-name)

(define-spreading-function point point-position*)
(define-spreading-function rectangle bounding-rectangle*)

;;; Probably won't be used by anything internal, (the macro uses the
;;; on-the-stack version below), but easy to define, so...
(defun point-sequence-coordinates (point-sequence)
  (with-collection
    (dolist (point point-sequence)
      (with-point-spread (point point-x point-y)	;Macro defined in the D-S-F POINT form
	(collect point-x) (collect point-y)))))

(define-spreading-function point-sequence point-sequence-coordinates
	((point-sequence coordinate-sequence) &body body)
  `(flet ((with-point-sequence-spread
	    (,coordinate-sequence)
	    ,@body))
     (declare (dynamic-extent #'with-point-sequence-spread))
     (with-point-sequence-internal #'with-point-sequence-spread ,point-sequence)))

;;; The variable *coord-sequence-cache* is a list of coordinates, although
;;; that doesn't matter.  We are actually storing the conses in the list.
;;; When we go to convert a point, we take the first two conses off the list.
;;; We might be able to do this somewhat better by counting off the conses we
;;; need first (thereby removing a lot of the RPLACD activity), but that's for
;;; later if we decide this is a problem.  Cleaned up from DCPL's version in
;;; February, 1991 by rsl.

(defvar *coord-sequence-cache*
	;; Start the user off with a minimal cache.  We can make this smaller
	;; if necessary.  Maybe the length should be a parameter?
	(let ((result nil))
	  (dotimes (i 16 result)
	    (progn i) ;; (declare (ignore i))
	    (push :unused-y result)
	    (push :unused-x result))))

(defun with-point-sequence-internal (continuation points)
  (let ((coords '())
	(tail nil))
    (dolist (point points)
      (with-point-spread (point x y)
	(let ((coord-piece (without-scheduling	;; One user at a time, please
			     (let ((cache *coord-sequence-cache*))
			       (when cache
				 (setf *coord-sequence-cache* (cddr cache)))
			       cache))))
	  (if coord-piece
	      (setf (first coord-piece) x
		    (second coord-piece) y
		    (cddr coord-piece) nil)
	      (setq coord-piece (list* x y nil)))
	  (if tail
	      (setf (cdr tail) coord-piece)
	      (setf coords coord-piece))
	  (setf tail (cdr coord-piece)))))
    (multiple-value-prog1
      (funcall continuation coords)
      (when coords
	(let ((last (last coords)))
	  (without-scheduling (setf (cdr last) *coord-sequence-cache*
				    *coord-sequence-cache* coords)))))))

;;; Supposed to prevent consing of point sequences except on the stack.
#+Ignore ;;; I don't really think this is worth it any more.  --- rsl
(defun with-point-sequence-internal (continuation point-sequence)
  (labels ((with-coordinates (point-sequence &rest coordinates)
	     (declare (dynamic-extent coordinates))
	     (when (null point-sequence)
	       (return-from with-coordinates (funcall continuation (nreverse coordinates))))
	     (with-point-spread ((first point-sequence) x y)
	       (apply #'with-coordinates (rest point-sequence) y x coordinates))))
    (with-coordinates point-sequence)))

;;; Other version doesn't work on I architecture (can't NREVERSE a stack list)
;;; so cons in order on stack.
#+Ignore ;; #+(and Genera imach)
;;; Does this work??  It doesn't on the 36xx.
(defun with-point-sequence-internal (continuation point-sequence)
  (let ((results (make-list (* 2 (length point-sequence)) :area sys:stack-area))
	(index 0))
    (dolist (point point-sequence)
      (with-point-spread (point x y)
	(setf (elt results index) x)
	(setf (elt results (1+ index)) y))
      (incf index 2))
    (funcall continuation results)))

(defun compute-spread-arguments (type unspread-function-name simple-lambda-list
				      spread-arguments &aux temp)
  #+Genera (declare 
	     (values unspread-function-lambda-list unspread-function-body
		     spread-function-name spread-function-lambda-list spread-function-body
		     method-function-name method-lambda-list
		     spread-arguments-descriptors
		     unspread-arguments spread-arguments method-arguments
		     pretty-unspread-lambda-list pretty-spread-lambda-list
		     #+PCL
		     bogus-generic-function-definition))
  ;; A little syntax checking; these arguments are otherwise ignored.
  (if (eql (first simple-lambda-list) 'medium)
      (pop simple-lambda-list)
      (warn "~S argument omitted from graphics function definition for ~S"
	    'medium unspread-function-name))
  (let ((the-end (last simple-lambda-list)))
    (if (eql (first the-end) '&drawing-options)
	(setf simple-lambda-list (butlast simple-lambda-list))
	(warn "~S omitted from graphics function definition for ~S"
	      '&drawing-options unspread-function-name)))
  (dolist (sa spread-arguments)
    (dolist (sa-desc (cdr sa))
      (unless (member (first sa-desc) simple-lambda-list)
	(warn "Spread argument ~S is missing from lambda-list for graphics function ~S"
	      (first sa-desc) unspread-function-name))))
  (let* ((spread-function-name (fintern "~A*" unspread-function-name))
	 (method-function-name (fintern "~A*-~A" unspread-function-name 'internal))
	 #+PCL  ;; cover up a bug in PCL
	 (bogus-generic-function-name
	   (fintern "~A*-~A" unspread-function-name 'generic-for-PCL-error-checking))
	 (spread-arguments-descriptors
	   (with-collection
	     (dolist (sa spread-arguments)
	       (let ((type (first sa)))
		 (unless (get type 'spreading-macro)
		   (warn "Unknown argument type ~S for arguments ~S in graphics function ~S"
			 type (mapcar #'car (cdr sa)) unspread-function-name))
		 (dolist (sa-desc (cdr sa))
		   (let ((unspread-arg (car sa-desc))
			 (spread-args (cdr sa-desc)))
		     (collect `(,unspread-arg ,type ,@spread-args))))))))
	 (unspread-function-lambda-list
	   (with-collection
	     (collect 'medium)			;Put it back.
	     (dolist (llv simple-lambda-list)
	       (when (member llv '(&rest &key &allow-other-keys)) (return))
	       (collect llv))
	     (collect '&rest)
	     (collect 'drawing-options)))
	 (spread-function-lambda-list
	   (with-collection
	     (collect 'medium)			;Ditto.
	     (dolist (llv (cdr unspread-function-lambda-list))
	       (if (setf temp (assoc llv spread-arguments-descriptors))
		   (dolist (spr-arg (cddr temp)) (collect spr-arg))
		   (collect llv)))))
	 (method-lambda-list
	   (with-collection
	     (dolist (llv simple-lambda-list)
	       (if (setf temp (assoc llv spread-arguments-descriptors))
		   (dolist (spr-arg (cddr temp)) (collect spr-arg))
		   (collect llv)))
	     ;; Ensure checking for undefined keywords.
	     ;; Warning!  PCL doesn't check for invalid keywords.  Sux.
	     (unless (member '&key simple-lambda-list) (collect '&key))))
	 (unspread-arguments
	   (with-collection
	     (dolist (llv unspread-function-lambda-list)
	       (unless (member llv '(&optional &rest))
		 (when (consp llv)
		   (setf llv (car llv)))
		 (collect llv)))))
	 (spread-arguments
	   (with-collection
	     (dolist (arg unspread-arguments)
	       (if (setf temp (assoc arg spread-arguments-descriptors))
		   (dolist (sa (cddr temp)) (collect sa))
		   (collect arg)))))
	 (method-arguments
	   (with-collection
	     (dolist (llv method-lambda-list)
	       (unless (member llv lambda-list-keywords)
		 (collect
		   (if (symbolp llv) llv
		       (if (listp llv)
			   (if (listp (car llv)) (cadar llv) (car llv))
			   (warn "Unknown format of argument ~S in argument list to ~S"
				 llv unspread-function-name))))))))
	 (unspread-function-body
	   ;; (apply #'draw-foo* medium draw-foo-spread-args drawing-options)
	   (let ((form 
		   `(apply #',spread-function-name
			   ,@(with-collection
			       (dolist (llv unspread-arguments)
				 (if (setf temp (assoc llv spread-arguments-descriptors))
				     (dolist (sa (cddr temp)) (collect sa))
				     (collect llv)))))))
	     (dolist (sa-desc spread-arguments-descriptors form)
	       ;; This is `(with-point-spread (point point-x point-y) ,form), for example:
	       (setf form
		       `(,(get (cadr sa-desc) 'spreading-macro)
			 (,(car sa-desc) ,@(cddr sa-desc))
			 ,form)))))
	 (spread-function-body
	   `(with-merged-medium-from-drawing-options
		(,type remaining-options medium drawing-options)
	      (apply #',#+PCL bogus-generic-function-name #-PCL method-function-name
		     ,(first spread-arguments)
		     ,@(cdr (butlast spread-arguments)) remaining-options)))
	 (pretty-unspread-lambda-list
	   `(medium ,@simple-lambda-list ,@(unless (member '&key simple-lambda-list) '(&key))
		    ,@(cdr (assoc type *drawing-options-keyword-arguments*))))
	 (pretty-spread-lambda-list
	   (with-collection
	     (dolist (llv pretty-unspread-lambda-list)
	       (if (setf temp (assoc llv spread-arguments-descriptors))
		   (dolist (spr-arg (cddr temp)) (collect spr-arg))
		   (collect llv)))))
	 #+PCL
	 (bogus-generic-function-definition
	   (let* ((foolish-lambda-list (copy-list method-lambda-list))
		  (rest-position (position '&rest foolish-lambda-list))
		  (rest-arg (and rest-position (nth (1+ rest-position) foolish-lambda-list)))
		  (key-position (and (null rest-arg) (position '&key foolish-lambda-list))))
	     ;; Check at compile time to see whether this is still needed.  Yuk.
	     (macrolet ((check-PCL-version ()
			  (unless (or (search "Rainy Day" pcl::*pcl-system-date*)
				      (search "May Day" pcl::*pcl-system-date*))
			    (warn "Check ~S to see if bogus generic function can be removed."
				  'compute-spread-arguments))))
	       (check-PCL-version))
	     (when (null rest-arg)
	       (setf rest-arg (gensymbol 'rest)
		     foolish-lambda-list
		       (if (null key-position)
			   (append foolish-lambda-list `(&rest ,rest-arg))
			   (prog1 
			     (append (subseq foolish-lambda-list 0 key-position)
				     `(&rest ,rest-arg)
				     (subseq foolish-lambda-list key-position))
			     (incf key-position 2)))))
	     `(defun ,bogus-generic-function-name (medium ,@foolish-lambda-list)
		(declare (dynamic-extent ,rest-arg))
		,(when key-position
		   `(declare (ignore ,@(with-collection
					 (dolist (key-arg (nthcdr key-position
								  foolish-lambda-list))
					   (unless (member key-arg lambda-list-keywords)
					     (collect (if (atom key-arg) key-arg
							  (if (atom (car key-arg))
							      (car key-arg)
							      (cadar key-arg))))))))))
		(apply #',method-function-name
		       medium ,@(cdr (butlast spread-arguments)) ,rest-arg)))))
    (values unspread-function-lambda-list unspread-function-body
	    spread-function-name spread-function-lambda-list spread-function-body
	    method-function-name method-lambda-list 
	    spread-arguments-descriptors
	    unspread-arguments spread-arguments method-arguments
	    pretty-unspread-lambda-list pretty-spread-lambda-list
	    #+PCL bogus-generic-function-definition)))

(defvar *graphics-generic-functions* nil)

(defstruct (graphics-function (:conc-name gf-)
			      (:print-function (lambda (gf stream ignore)
						 (declare (ignore ignore))
						 (print-unreadable-object (gf stream)
						   (prin1 'graphics-function stream)
						   (princ #\space stream)
						   (prin1 (gf-unspread-function-name gf)
							  stream)))))
  unspread-function-name
  spread-function-name
  type
  method-function-name
  unspread-lambda-list
  spread-lambda-list
  method-lambda-list
  spread-arguments
  method-argument-list)

(defun find-graphics-function (name &key (if-does-not-exist :error))
  (or (find name *graphics-generic-functions* :key #'gf-unspread-function-name)
      (ecase if-does-not-exist
	((nil) nil)
	(:error (error "Can't find graphics function ~S" name))
	(:create (let ((gf (make-graphics-function :unspread-function-name name)))
		   (push gf *graphics-generic-functions*)
		   gf)))))

(defun find-graphics-spread-function (name)
  (or (find name *graphics-generic-functions*
	    :key #'gf-spread-function-name :test #'string-equal)
      (error "Can't find graphics function corresponding to ~S" name)))

(defun define-graphics-function-compile-time (name &key
						   spread-function-name
						   type method-function-name
						   unspread-lambda-list spread-lambda-list
						   method-lambda-list spread-arguments
						   method-argument-list)
  (let ((gf (find-graphics-function name :if-does-not-exist :create)))
    (setf (gf-type gf) type
	  (gf-spread-function-name gf) spread-function-name
	  (gf-method-function-name gf) method-function-name
	  (gf-unspread-lambda-list gf) unspread-lambda-list
	  (gf-spread-lambda-list gf) spread-lambda-list
	  (gf-method-lambda-list gf) method-lambda-list
	  (gf-spread-arguments gf) spread-arguments
	  (gf-method-argument-list gf) method-argument-list)
    gf))

;;; This compiler macro will turn
;;;
;;;  (DRAW-FOO MEDIUM REQUIRED-ARGS . DRAWING-AND-OTHER-KEYWORD-ARGS) 
;;;
;;; into
;;;
;;;  (WITH-DRAWING-OPTIONS (MEDIUM DRAWING-KEYWORD-ARGS)
;;;    (DRAW-FOO*-INTERNAL MEDIUM SPREAD-REQUIRED-ARGS . OTHER-KEYWORD-ARGS))
;;;
;;; at compile time.  Depends on DEFINE-COMPILER-MACRO, the ANSI way to write portable
;;; compiler optimizers.

;;; We call the spread function, not the internal generic function, unless we know the
;;; values of all keywords at compile time.  Consider, for example,
;;;
;;; (DRAW-POLYGON* ... FILLED-OR-CLOSED T)
;;;
;;; where FILLED-OR-CLOSED might be :FILLED or :CLOSED at runtime.  If we know all the
;;; keyword arguments at compile time, we can call the generic directly instead, saving
;;; a call to WITH-MERGED-MEDIUM-FROM-DRAWING-OPTIONS.


;;; --- This macro has a bug: If you lexically use one of the spread arguments' names in
;;; the drawing options passed to an unspread function, you will get the wrong value.
;;; For example,
;;;
;;;   (LET ((POINT-X +RED+))
;;;     (DRAW-POINT POINT :INK POINT-X))
;;;
;;; turns into the equivalent of
;;;
;;;   (LET ((POINT-X +RED+))
;;;     (MULTIPLE-VALUE-BIND (POINT-X POINT-Y) (POINT-POSITION POINT)
;;;	  (DRAW-POINT* POINT-X POINT-X :INK POINT-X)
;;;
;;; This is too hard to fix right now, and is unlikely to come up in practice because
;;; the variable names are in the CLIM-SHARED package and are not exposed to users.

(defmacro define-graphics-function-compiler-optimizers (function-name)
  (let* ((gf (find-graphics-function function-name))
	 (unspread-args (gf-unspread-lambda-list gf))
	 (spread-function-name (gf-spread-function-name gf))
	 (method-name (gf-method-function-name gf))
	 (spread-args (gf-spread-lambda-list gf))
	 (positional-args (subseq spread-args 0 (position '&rest spread-args)))
	 (spread-arguments (gf-spread-arguments gf))
	 (whole-form-var (gensymbol 'form))
	 (medium-name (gensymbol 'medium))
         #-ccl (spread-body ``(,',spread-function-name ,',medium-name
                                                       ,,@(cdr positional-args) ,@other-keywords))
         #-ccl (unspread-body ``(,',spread-function-name ,',medium-name
                                                         ,,@(mapcar
                                                             #'(lambda (var) (if (member var unspread-args) var `',var))
                                                             (cdr positional-args))
                                                         ,@other-keywords))
         #+ccl ;; Workaround for bug in CCL 2.0a3  --RWK
         (spread-body `(list* ',spread-function-name ',medium-name ,@(cdr positional-args) other-keywords))
         #+ccl ;; Workaround for bug in CCL 2.0a3  --RWK
         (unspread-pos (mapcar #'(lambda (var) (if (member var unspread-args) var `',var))
                               (cdr positional-args)))
         #+ccl ;; Workaround for bug in CCL 2.0a3  --RWK
         (unspread-body `(list* ',spread-function-name ',medium-name ,@unspread-pos other-keywords)))
  (flet ((finalize-body (body)
            ;; (with-drawing-options ((medium-exp #:medium) {drawing-args*}) ,@body)
           ``(with-drawing-options ((,,(first spread-args) ,',medium-name)
                                    ,@drawing-keywords)
               ;; The ALL-KEYWORDS-CONSTANT-P is evaluated at COMPILER-MACROEXPAND time!
	       ;; Yes, this is (a little) hard to understand.  What did you expect?  
	       ;; This is a macro-writing macro.
	       ,(if all-keywords-constant-p
		    ,(subst method-name spread-function-name body)
		    ,body))))
      (dolist (sa-desc (reverse spread-arguments))	;Preserve lexical order of evaluation
	;; (with-{type}-spread (unspread-positional-arg {spread-positional-arg}*) ...)
	(setf unspread-body
		``(,',(get (cadr sa-desc) 'spreading-macro)
		   (,,(car sa-desc) ,@',(cddr sa-desc))  ;; ,@', -- not ,',@  !! -RWK  Quote only takes 1 arg!
		   ,,unspread-body)))
      `(define-group ,function-name define-graphics-function-compiler-optimizers
	 (define-compiler-macro ,function-name (,@unspread-args)
	   (with-drawing-options-parsed ,function-name
					(drawing-options drawing-keywords other-keywords
							 all-keywords-constant-p)
	     ,(finalize-body unspread-body)))
	 (define-compiler-macro ,spread-function-name (&whole ,whole-form-var ,@spread-args)
	   (with-drawing-options-parsed ,spread-function-name
					(drawing-options drawing-keywords other-keywords
							 all-keywords-constant-p)
	     (if (and (null drawing-keywords) (not all-keywords-constant-p))
		 ,whole-form-var ;Nothing to optimize, return original
		 ,(finalize-body spread-body))))))))

;;; (define-graphics-function-compiler-optimizers draw-circle)

;;; Macro used at COMPILER-MACRO expansion time for above macros.
;;; Checks to make sure argument list has a reasonable number of arguments; will sometimes
;;; catch problems like forgetting to pass the medium into the drawing function.
(defmacro with-drawing-options-parsed (function-name
				       (drawing-options drawing-keywords other-keywords
							&optional all-keywords-constant-p)
				       &body body)
  (let ((drawing-options-copy (gensymbol 'drawing-options)))
    `(let ((,drawing-options-copy ,drawing-options)
	   (,drawing-keywords nil)
	   (,other-keywords nil)
	   ,@(when all-keywords-constant-p
	       `((,all-keywords-constant-p t))))
       (cond ((evenp (length ,drawing-options-copy))
	      (loop (when (null ,drawing-options-copy) (return))
		    (let* ((key (pop ,drawing-options-copy))
			   (key-constant-p (constantp key))
			   (evaluated-key (if key-constant-p (eval key) key))
			   (val (pop ,drawing-options-copy)))
		      (macrolet ((push-em (list)
				   `(progn (push evaluated-key ,list) (push val ,list))))
			(if (member key *drawing-options-keywords*)
			    (push-em ,drawing-keywords)
			    (progn (push-em ,other-keywords)
				   ,(when all-keywords-constant-p
				      `(unless key-constant-p
					 (setf ,all-keywords-constant-p nil))))))))
	      (setf ,drawing-keywords (nreverse ,drawing-keywords)
		    ,other-keywords (nreverse ,other-keywords)))
	     (t
	      (progn (warn "Odd number of keyword arguments to graphics function ~S"
			   ',function-name)
		     (setf ,drawing-keywords nil
			   ,other-keywords ,drawing-options-copy))))
       ,@body)))

(defmacro define-graphics-function-generic (spread-name lambda-list)
  (let ((method-name (gf-method-function-name (find-graphics-spread-function spread-name))))
    `(defgeneric ,method-name ,lambda-list)))

(defmacro define-graphics-function-method (spread-name &body quals-ll-and-method-body)
  #+Genera (declare (zwei:indentation .
				      #-PCL zwei:indent-for-clos-defmethod
				      #+PCL zwei:indent-clos-defmethod))
  (let ((method-name (gf-method-function-name (find-graphics-spread-function spread-name)))
	(quals nil)
	(lambda-list nil)
	(body nil))
    ;; Same old method body parser...
    (loop (when (listp (first quals-ll-and-method-body))
	    (setf quals (nreverse quals)
		  lambda-list (pop quals-ll-and-method-body)
		  body quals-ll-and-method-body)
	    (return))
	  (push (pop quals-ll-and-method-body) quals))
    (labels ((fix-up-body (body)
	       (when (atom body) (return-from fix-up-body body))
	       (macrolet ((maybe-fix-function-reference (reference)
			    `(when (and (listp ,reference)
					(eql (first ,reference) ':graphics-internal))
			       (setf ,reference (gf-method-function-name
						  (find-graphics-spread-function
						    (second ,reference)))))))
		 (maybe-fix-function-reference (first body))
		 (dolist (arg (rest body) body)
		   (when (listp arg)
		     (when (eql (first arg) 
				;;--- Kludge because of Genera 8.1/Genera 8.2 split
				#+Genera (first '#'car) #-Genera 'function)
		       (maybe-fix-function-reference (second arg)))
		     (fix-up-body arg))))))
      `(defmethod ,method-name ,@quals ,lambda-list ,@(mapcar #'fix-up-body body)))))

#||

;;; Gee -- I wonder why he needed this?  [Hint: PCL]
(defun remove-all-graphics-generics ()
  (mapcar (lambda (x) (fmakunbound (gf-method-function-name x)))
	  *graphics-generic-functions*))
||#
