;;; -*- Mode: Lisp; Package: ON-GENERA; Base: 10.; Syntax: Common-Lisp -*-
;;;
;;; Copyright (c) 1989 by Xerox Corporations.  All rights reserved.
;;;

(in-package "ON-GENERA")

;;;
;;; Pixmap Support
;;;

(defmethod realize-pixmap ((port genera-port) (pixmap pixmap))
  (let* ((n-rows (pixmap-height pixmap))
	 (n-cols (pixmap-width pixmap))
	 (data (pixmap-data pixmap))
	 (raster (tv:make-sheet-bit-array (genera-screen port) n-cols n-rows)))
    (when data
      ;; Deal with the other formats, presumably by dispatching to a sub-method
      ;; that can be specialized.
      (assert (eq (pixmap-format pixmap) ':bitmap) ())
      (assert (= n-rows (length data)) ())
      ;; Have Richard write something fast
      (dotimes (row n-rows)
	(let* ((row-data (elt data row))
	       (n (parse-integer row-data :radix 16.)))
	  (dotimes (col n-cols)
	    (let ((bit (ldb (byte 1 col) n)))
	      (setf (tv:raster-aref raster col row) bit))))))
    raster))

;;; This doesn't do anything.  I understand why it doesn't get called
;;; with the PIXMAP object itself, but don't like it; not that any of
;;; that matters.
(defmethod unrealize-pixmap ((port genera-port) pixmap-data)
  (declare (ignore pixmap-data))
  nil)

#||

;;;
;;; Cursors 
;;;

(defvar *cursor->clx-mapping*
  '((:vertical-scroll . 116)
    (:scroll-up . 114)
    (:scroll-down . 106)
    (:vertical-thumb . 112)
    (:horizontal-scroll . 108)
    (:scroll-left . 110)
    (:scroll-right . 112)
    (:horizontal-thumb . 114)
    (:default . 132)
    (:button . 132)			;Or 38?
    (:busy  . 150)
    (:prompt . 92)
    ;; For rubber banding
    (:upper-left . 134)
    (:upper-right . 136)
    (:lower-left . 12)
    (:lower-right . 14)
    (:move . 52)
    ;; Prompting for a position
    (:position . 34)))

(defmethod realize-cursor :around ((port genera-port) cursor)
  (with-slots (cursor-cache) port
    (or (getf cursor-cache cursor)
	(setf (getf cursor-cache cursor)
	      (call-next-method)))))

(defmethod realize-cursor ((port genera-port) cursor)
  (with-slots (cursor-font x-screen) port
    (let ((char-number 
	   (or (cdr (assoc cursor *cursor->clx-mapping*))
	       1)))
      (xlib:create-glyph-cursor
	:source-font cursor-font
	:source-char char-number
	:mask-font cursor-font
	:mask-char (1+ char-number)
	;;; ??? Should query for black and white or use the b/w from screen
	:foreground (xlib:make-color :red   0.0 :green 0.0 :blue  0.0)
	:background (xlib:make-color :red   1.0 :green 1.0 :blue  1.0)))))

(defmethod realize-cursor ((port genera-port) (cursor cursor))
  (with-slots (x-screen) port
    (let* ((colors (xlib:query-colors 
		     (xlib:screen-default-colormap x-screen)
		     (list (xlib:screen-black-pixel x-screen)
			   (xlib:screen-white-pixel x-screen)))))
      (xlib:create-cursor :source (cursor cursor)
			  :mask   (mask cursor)
			  :x      (x-offset cursor)
			  :y      (y-offset cursor)
			  ;; black & white respectively
			  :foreground (first colors)
			  :background (second  colors)))))
||#

;;; Cursors I didn't do anything about yet.
;    (:vertical-thumb . 112)
;    (:horizontal-thumb . 114)
;    (:button . 132)			;Or 38?
;    (:prompt . 92)
;    ;; For rubber banding
;    (:upper-right . 136)
;    (:lower-left . 12)
;    (:move . 52)
;    ;; Prompting for a position
;    (:position . 34)))

(defmethod genera-mouse-char-for-cursor ((cursor (eql ':vertical-scroll)))
  #\mouse:vertical-double-arrow)

(defmethod genera-mouse-char-for-cursor ((cursor (eql ':scroll-up)))
  #\mouse:up-arrow)

(defmethod genera-mouse-char-for-cursor ((cursor (eql ':scroll-down)))
  #\mouse:down-arrow)

(defmethod genera-mouse-char-for-cursor ((cursor (eql ':horizontal-scroll)))
  #\mouse:horizontal-double-arrow)

(defmethod genera-mouse-char-for-cursor ((cursor (eql ':scroll-left)))
  #\mouse:left-arrow)

(defmethod genera-mouse-char-for-cursor ((cursor (eql ':scroll-right)))
  #\mouse:right-arrow)

(defmethod genera-mouse-char-for-cursor ((cursor (eql ':default)))
  #\mouse:nw-arrow)

(defmethod genera-mouse-char-for-cursor ((cursor (eql ':busy)))
  #\mouse:hourglass)

(defmethod genera-mouse-char-for-cursor ((cursor (eql ':upper-left)))
  #\mouse:nw-corner)

(defmethod genera-mouse-char-for-cursor ((cursor (eql ':lower-right)))
  #\mouse:se-corner)

(defmethod genera-mouse-char-for-cursor ((cursor t))
  ;; any other cursor we don't know about yet
  #\mouse:ne-arrow)

(defmethod install-port-cursor ((port genera-port) sheet cursor)
  (declare (ignore sheet))
  (let ((char (genera-mouse-char-for-cursor cursor)))
    ;; note that this means that CLIM requires DW for the nonce...
    ;; It would be trivial to encode this information in the above methods, though
    (let ((entry (assoc char dw::*mouse-blinker-characters*))
	  (x 0) (y 0))
      (when entry
	(scl:destructuring-bind (nil xx yy &optional nil) entry
	  (setq x xx y yy)))
      (tv:mouse-set-blinker-definition ':character x y ':on ':set-character char))))

(defmethod set-cursor-location ((port genera-port) sheet location)
  ;; we don't do anything about this yet.
  (declare (ignore sheet location))
  )

(scl:defmethod (ensure-blinker-for-cursor silica-window) (cursor)
  (unless blinker-table (setq blinker-table (make-hash-table)))
  (or (gethash cursor blinker-table)
      (let ((blinker (tv:make-blinker scl:self 'tv:rectangular-blinker :follow-p T)))
	(setf (gethash cursor blinker-table) blinker)
	(scl:send blinker :set-visibility nil)
	blinker)))

(defun ensure-blinker-matches-cursor (cursor stream)
  (let ((active (cursor-active cursor))
	(state (cursor-state cursor))
	(focus (cursor-focus cursor)))
    (let* ((mirror (sheet-mirror! stream))
	   (blinker (ensure-blinker-for-cursor mirror cursor)))
      (when blinker
	(let ((transformation (fetch-native-transformation stream)))
	  (multiple-value-bind (x y) (bounding-rectangle* cursor)
	    (multiple-value-setq (x y)
	      (transform-point* 
		transformation x y))
	    (cond ((and active state focus)
		   (scl:send mirror :set-cursorpos x y)
		   (scl:send blinker :set-visibility :blink))
		  ((and active state)
		   (scl:send mirror :set-cursorpos x y)
		   (scl:send blinker :set-visibility T))
		  (t (scl:send blinker :set-visibility nil)))))))))

(defmethod port-note-cursor-change ((port genera-port) cursor stream type old new)
  (declare (ignore type old new))
  (ensure-blinker-matches-cursor cursor stream))

#||
;;;
;;; COLORS sharked from Janssen
;;;

(defmethod realize-color ((port genera-port) color)
  "get a new color-id for this color and declare it to the connection "
  (with-slots (x-screen) port
    (cond
      ((> (xlib:screen-root-depth x-screen) 1) 
					;otherwise no point in trying
       (xlib:alloc-color (xlib:screen-default-colormap x-screen)
			 (xlib:make-color
			   :red (color-red-value color)
			   :green (color-green-value color)
			   :blue (color-blue-value color))))
      (t				; monochrome case
       ;; Janssen says: this is just a hack now ...
       (if (almost-white-p COLOR)
	   (xlib:screen-white-pixel x-screen)
	   (xlib:screen-black-pixel x-screen))))))


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; Are these really needed by DWS??
;;;
;;; return a list of all available color names for this canvas.
;;; for x, all colors are available.
(defmethod available-colors ((port genera-port) canvas)
  (declare (ignore canvas))
  (cond
    ((> (xlib:screen-root-depth (x-screen port)) 1)  (color-names))
    (t  '(:white :black))))

(defmethod screen-color ((port genera-port) canvas color)
  "return the actual color put on the screen for the given color"
  (declare (ignore canvas))
  (let ((colors
	 (xlib:query-colors (xlib:screen-default-colormap (x-screen port))
			    (list (realize-color port color))))
	red-value green-value blue-value)
    (setf red-value (xlib:color-red (car colors))
	  green-value (xlib:color-green (car colors))
	  blue-value (xlib:color-blue (car colors)))
    (if (and 
	  (eql (color-red-value color) red-value)
	  (eql (color-green-value color) green-value)
	  (eql (color-blue-value color) blue-value))
	;; if the screen colors are the same as the wsii-color, return it
	color
	;; otherwise construct a new color which is what is on the screen
	(make-color :red red-value :green green-value :blue blue-value))))

||#

;;; Font/text-style

(defmethod realize-text-style ((port genera-port) text-style)
  (let ((font-id (text-style-mapping port *standard-character-set* text-style)))
    (etypecase font-id
      (symbol (symbol-value font-id))
      (sys:font font-id))))

;;;
;;; Text-Style Operations
;;;

(defmethod text-style-ascent ((text-style text-style) (medium basic-genera-medium))
  (sys:font-baseline (realize-text-style (port medium) text-style)))
					
(defmethod text-style-descent ((text-style text-style) (medium basic-genera-medium))
  (let ((font (realize-text-style (port medium) text-style)))
    (- (sys:font-char-height font) (sys:font-baseline font))))
					
(defmethod text-style-height ((text-style text-style) (medium basic-genera-medium))
  (sys::font-char-height (realize-text-style (port medium) text-style)))

(defmethod string-width (string (text-style text-style) 
				(medium basic-genera-medium)
				&key (start 0) end)
  (let ((font (realize-text-style (port medium) text-style))
	(genera-window (sheet-mirror! (medium-sheet medium))))
    (scl:send genera-window :string-length string start end nil
	      (si:parse-character-style 
		(list ':device-font (sys:font-name font) :normal)))))

(defmethod string-height (string (text-style text-style) 
				 (medium basic-genera-medium))
  (declare (ignore string))
  (text-style-height text-style medium))

#||
(defmethod char-width (char (text-style text-style) (medium basic-genera-medium))
  (xlib:text-width (realize-text-style (port medium) text-style) 
		   (string char)))

||#
