;;; -*- Mode: Lisp; Package: ON-GENERA; Base: 10.; Syntax: Common-Lisp -*-

"Genera port implementation Copyright (c) 1990 by International Lisp Associates, Inc.
 All rights reserved."

;;;
;;; Copyright (c) 1989 by Xerox Corporations.  All rights reserved.
;;;


;;;
;;; ON-GENERA:  Genera Port Implementation
;;;





(defvar *genera-default-server-path* `(:genera :host ,net:*local-host*
					       :screen ,tv:main-screen))

;;; Should the last port or the first port to set this variable win?
(unless *default-server-path*
  (setq *default-server-path* *genera-default-server-path*))

(pushnew *genera-default-server-path*
	 *server-path-defaults*
	 :key #'car :test #'eq)

(scl:add-initialization "Fix default server path when changing hosts"
			'(setf (getf (cdr *genera-default-server-path*) :host)
				 net:*local-host*
			       (getf (cdr *genera-default-server-path*) :screen)
				 tv:main-screen)
			()
			'neti:local-name-initialization-list)

(scl:add-initialization "Fix default server path at boot time"
			'(setf (getf (cdr *genera-default-server-path*) :screen)
				 tv:main-screen)
			'(:warm))

;;; 
;;; Genera Port
;;;

;;; In general, "sheet" is used to mean Silica sheet.  The exceptions
;;; occur in the names of Genera window system functions.  These
;;; functions are in the "TV" package.
  
(defclass genera-port (port)
  ((height-pixels)
   ;; Genera Specific stuff
   (genera-screen :initform nil :accessor genera-screen)
   (genera-console :initform nil :accessor genera-console)
     
   ;; The slot CURSOR-CACHE is used to map between silica and x cursors.
   ;; The CURSOR-FONT holds the X cursor font.
   (cursor-font)
   (cursor-cache :initform nil :type nil)
   (type :allocation :class 
	 :initform :genera
	 :reader port-type)
   (pointer-documentation-string
     :initform (make-array 200 :element-type 'character :fill-pointer 0)
     :accessor port-pointer-documentation-string)
   )
  (:default-initargs
    :font-for-undefined-style 'fonts:boxfont)
  )

(defmethod find-port-type ((type (eql :genera))) 
  (find-class 'genera-port))

(defmethod initialize-instance :after
	   ((port genera-port) &key server-path &allow-other-keys)
  ;; This is assuming we are only porting to the root window of a screen,
  ;; which need not be the case.
  ;; And also the zeroth screen.
  (let ((host   (getf (cdr server-path) :host net:*local-host*))
	(screen (getf (cdr server-path) :screen tv:main-screen)))
	  
    (assert (eql (net:parse-host host) net:*local-host*)
	    (host)
	    "Can only open a port to the local host, not ~S" host)
    (with-slots (genera-screen
		 genera-console
		 cursor-font
		 height-pixels) 
	port
      (setf genera-screen   screen 
	    genera-console  (tv:sheet-console genera-screen)
	    cursor-font     fonts:mouse
	    height-pixels   (tv:sheet-inside-height genera-screen)))
    (initialize-genera-display-device port)))

(defmethod port-match ((port genera-port) server-path)
  (let ((pkeys (cdr (port-server-path port)))
	(akeys (cdr server-path)))
    (and (eq (car server-path) :genera)
	 ;; Well, this makes sure that the value of :HOST is one
	 ;; of the names of the local host, but maybe the actual
	 ;; host object should get put into the server-path list?
	 ;; Or maybe the genera server path spec just shouldn't have
	 ;; a :HOST field at all.
	 ;; After all, this could cause a namespace update! -York
	 ;; Don't bother to check against the name in the port's server path.
	 (eql (net:parse-host (getf pkeys :host net:*local-host*))
	      (net:parse-host (getf akeys :host net:*local-host*)))
	 (eq  (getf akeys :screen)
	      (getf pkeys :screen)))))
					;
stuff for the graft initialization
;	  width-pixels    (tv:sheet-inside-width genera-screen)
;	  ;; an approximation at best
;	  width-mm        356.0			; 14" at 75 pixels/inch
;	  height-mm       254.0)		; 10"


(defmethod destroy-port :after ((port genera-port))
  (with-slots (genera-screen) port
    ;;--- Needs to be a defined accessor (or mapping function)
    (do* ((tail (slot-value port 'mirror->sheet-table)
		;; Really a PLIST
		(cddr tail))
	  (genera-window (first tail) (first tail)))
	 ((null tail) nil)
      (scl:send genera-window :kill))))

;; Use the default methods for PORT-FORCE-OUTPUT and PORT-FINISH-OUTPUT.
;; Perhaps someday these could frob the screen manager or something.

(defmethod do-with-port ((port genera-port) continuation)
  (declare (dynamic-extent continuation))
  (funcall continuation))			;--- ???

(defmethod do-with-pointer ((port genera-port) continuation)
  (declare (dynamic-extent continuation))
  #-Ignore
  (funcall continuation)
  #+Ignore
  (with-slots (genera-console) port
    (let ((mouse (tv:console-mouse genera-console)))
      (tv:with-this-mouse-grabbed (mouse)
	(funcall continuation)))))

;;; Use default methods for do-with-port, port-match

(defmethod port-default-pointer-documentation-handler ((port genera-port))
  (let ((screen (slot-value port 'genera-screen)))
    (tv:screen-who-line-screen screen)))

(defmethod set-pointer-documentation ((handler genera-port) pointer-documentation)
  (let ((string (slot-value handler 'pointer-documentation-string)))
    (setf (fill-pointer string) (min (array-dimension string 0)
				     (length pointer-documentation)))
    (replace string pointer-documentation)))

;;;
;;; Genera Window Allocations
;;;


;; Replaces the old with-new-native-region.  This version is functional
;; as I (DCPL) think it should be.  Most of this is computation (the
;; realm of functions), not translation (the realm of macros).  (The X
;; version binds four magic variables, which is stylisticall poor, CF
;; comments above sheet-native-native-region* in this file.)
;;
;; --- I don't know why this isn't portable.  Until I do, it still
;; specializes on genera-port.
(defmethod sheet-native-region* ((port genera-port) sheet)
  (declare (values min-x min-y max-x max-y))
  ;; This returns the sheet's region transformed via transformations, as
  ;; opposed to -native-native- which really queries the mirror.
  (let* ((parent (sheet-parent sheet))
	 (region (sheet-region sheet))
	 (sheet-to-parent (sheet-transformation sheet))
	 (parent-to-native (fetch-native-transformation parent)))
    (multiple-value-bind (parent-min-x parent-min-y parent-max-x parent-max-y)
	(with-bounding-rectangle* (region-min-x region-min-y region-max-x region-max-y)
				  region
	  (careful-transform-rectangle* sheet-to-parent
					region-min-x region-min-y region-max-x region-max-y))
      (multiple-value-bind (native-min-x native-min-y native-max-x native-max-y)
	  (careful-transform-rectangle* parent-to-native
					parent-min-x parent-min-y parent-max-x parent-max-y)
	(values (round native-min-x)
		(round native-min-y)
		(round native-max-x)
		(round native-max-y))))))

(defmacro with-new-native-region ((sheet min-x-var min-y-var max-x-var max-y-var)
				  &body body)
  `(with-slots (region transformation) ,sheet
     (multiple-value-bind (nx ny nx2 ny2)
	 (with-bounding-rectangle* (min-x min-y max-x max-y) region
	   (transform-rectangle* transformation min-x min-y max-x max-y))
       (multiple-value-setq (nx ny nx2 ny2)
	 (transform-rectangle* 
	  (fetch-native-transformation (sheet-parent ,sheet))
	  nx ny nx2 ny2))
       (let ((,min-x-var (round nx))
	     (,min-y-var (round ny))
	     (,max-x-var (round nx2))
	     (,max-y-var (round ny2)))
	 ,@body))))

#+Rao-version
(defmacro with-new-native-region ((sheet min-x-var min-y-var max-x-var max-y-var)
				  &body body)
  (let ((width-var (gensym))
	(height-var (gensym)))
    `(let (,max-x-var ,max-y-var)
       (with-slots (region transformation) ,sheet
	 (multiple-value-bind (,min-x-var ,min-y-var ,width-var ,height-var)
	     (with-rectangle-wh region
				(transform-rectangle* transformation
						      min-x min-y width height))
	   (multiple-value-setq (,min-x-var ,min-y-var ,width-var ,height-var)
	     (transform-rectangle* 
	       (fetch-native-transformation (sheet-parent ,sheet))
	       ,min-x-var ,min-y-var ,width-var ,height-var))
	   (setq ,min-x-var (round ,min-x-var))
	   (setq ,min-y-var (round ,min-y-var))
	   (setq ,max-x-var (round (+ ,min-x-var ,width-var)))
	   (setq ,max-y-var (round (+ ,min-y-var ,height-var)))
	   ,@body)))))


;; See general comments about sheet-native-native-region* and
;; sheet-native-region* about functions vs macros, etc.
(defun genera-window-margins (mirror)
  (declare (values left top inside-width inside-height))
  (multiple-value-bind (left top)
      (scl:send mirror :margins)
    (multiple-value-bind (width height)
	(scl:send mirror :inside-size)
      (values left top width height))))

(defmacro with-genera-window-margins ((window &key (left-margin (gensymbol 'left))
						   (top-margin (gensymbol 'top))
						   inside-width inside-height)
				      &body body)
  (let* ((window-var (if (symbolp window) window (gensymbol 'window)))
	 (resulting-body
	   `(multiple-value-bind (,left-margin ,top-margin)
		(scl:send ,window-var :margins)
	      (progn ,left-margin ,top-margin nil)	;Make sure they're used
	      ,@body)))
    (setf resulting-body
	    (cond ((and inside-width inside-height)
		   `(multiple-value-bind (,inside-width ,inside-height)
			(scl:send ,window-var :inside-size)
		      ,resulting-body))
		  (inside-width
		   `(let ((,inside-width (scl:send ,window-var :inside-width)))
		      ,resulting-body))
		  (inside-height
		   `(let ((,inside-height (scl:send ,window-var :inside-height)))
		      ,resulting-body))
		  (t resulting-body)))
    (unless (eql window window-var)
      (setf resulting-body
	      `(let ((,window-var ,window)) ,resulting-body)))
    resulting-body))


;; Replaces the old with-current-native-pos.  This version is
;; functional as I (DCPL) think it should be.  Specifically, the old
;; macro bound four magically named variables, which is stylistically
;; poor.  Furthermore, most of this is computation (the realm of
;; functions), not translation (the realm of macros).
(defmethod sheet-native-native-region* ((port genera-port) sheet &optional mirror)
  (declare (values min-x min-y max-x max-y))
  ;; --- Two things to note.  (1) This is called -native-native- because
  ;; it returns things relative to the top level screen, not the
  ;; containing mirror, as -native- usually does.  (2) The region it
  ;; returns is inclusive/exclusive, instead of inclusive/inclusive.
  ;; That's how the old code worked and the callers want it for now.
  ;; Someday, in a big cleanup, this can be revamped.
  (let ((mirror (or mirror (sheet-mirror sheet))))
    (multiple-value-bind (width height)
	(scl:send mirror :size)
      (multiple-value-bind (xoff yoff)
	  (tv:sheet-calculate-offsets mirror (tv:sheet-screen mirror))
	(values xoff
		yoff
		(+ xoff width)
		(+ yoff height))))))

;;;--- binds four magically named variables, genera-min-x genera-min-y genera-max-x genera-max-y
;;; Need the edges of the window in the root coordinate system.
;;; There must be a better way.
(defmacro with-current-native-pos ((sheet &optional genera-window) &body body)
  `(let* ((genera-window ,(or genera-window `(sheet-mirror ,sheet)))
	  ;; unlike X, we can't have an extra layer in the tree due to window managers
	  )
      (multiple-value-bind (width height)
	  (scl:send genera-window :size)
	(multiple-value-bind (xoff yoff)
	    (tv:sheet-calculate-offsets genera-window (scl:send genera-window :screen))
	  (let ((genera-min-x xoff)
		(genera-min-y yoff)
		(genera-max-x (+ xoff width))
		(genera-max-y (+ yoff height)))
	    (ignore genera-min-x genera-min-y genera-max-x genera-max-y)
	    ,@body)))))

#+testing
(defun foo (s)
  (with-new-native-region (s rl rt rr rb)
    (list rl rt rr rb)))

;;; --- shouldn't some of this be portable?
(defmethod realize-graft ((port genera-port) graft)
  (with-slots (units 
		width-pixel height-pixel width-mm height-mm 
		region
		pixels-per-point)
	      graft
    (let ((genera-screen (genera-screen port)))
      (setf width-pixel (scl:send genera-screen :inside-width)
	    height-pixel (scl:send genera-screen :inside-height)
	    ;; --- bogus numbers copied from x-port
	    width-mm        360.0
	    height-mm	  280.0
	    pixels-per-point 1)
      (setf region
	    (ecase units
	      (:pixel
		(make-rectangle* 0 0 width-pixel height-pixel))))
      (setf (sheet-native-transformation graft) +identity-transformation+)
      (setf (sheet-mirror graft) genera-screen)
      (update-native-transformation port graft))))

(defvar *silica-window-count* 0)

(defvar *temporary-genera-mirror-desired* nil)

(defmethod realize-mirror ((port genera-port) sheet)
  (with-slots (genera-screen height-pixels) port
    (let* ((genera-parent (sheet-mirror! sheet))
	   (name (format nil "Silica Window ~D" (incf *silica-window-count*)))
	   (genera-window
	     (multiple-value-bind (min-x min-y max-x max-y)
		 (sheet-native-region* port sheet)
	       (tv:make-window
		 (if *temporary-genera-mirror-desired* 'temporary-silica-window 'silica-window)
		 ;; probably the screen
		 :superior genera-parent
		 :name name
		 :sheet sheet
		 :save-bits t
		 :deexposed-typeout-action :permit
		 :label nil  ;;name
		 ;; Want inside/outside coord system to match if possible
		 :borders nil
		 :x min-x :y min-y 
		 :width (- max-x min-x)
		 :height (- max-y min-y)))))
      (setf (sheet-mirror sheet) genera-window)
      (setf (sheet-native-transformation sheet) +identity-transformation+)
      (update-native-transformation port sheet)
	
      genera-window)))

#+obsolete?
(defmethod install-sheet-settings ((sheet genera-port-child-part) &rest settings)
  (let* ((notify-region? nil)
	 (notify-transformation? nil))
    (setf (sheet-settings sheet) settings)
    (when (getf settings :left)
      (setf (transformation-dx (sheet-transformation sheet))
	    (getf settings :left))
      (setq notify-transformation? t))
    
    (when (getf settings :bottom)
      (setf (transformation-dy (sheet-transformation sheet))
	    (getf settings :bottom))
      (setq notify-transformation? t))
    
    (when (getf settings :width) 
      (setf (rectangle-width (sheet-region sheet))
	    (getf settings :width))
      (setq notify-region? t))
    (when (getf settings :height) 
      (setf (rectangle-height (sheet-region sheet)) 
	    (getf settings :height))
      (setq notify-region? t))
    (when notify-region?
      (notify-region-changed sheet))))

(defmethod destroy-mirror ((port genera-port) sheet)
  (scl:send (sheet-mirror sheet) :kill))

(defmethod enable-mirror ((port genera-port) sheet)
  (let ((mirror (sheet-mirror sheet)))
    ;;--- what is the Genera interpretation of this?
    #+ignore
    (with-slots (settings) sheet
      (unless (getf settings :plain)
	(setf (xlib::window-override-redirect mirror) :off)))
    (scl:send mirror :expose)))

(defmethod disable-mirror ((port genera-port) sheet)
  (let ((mirror  (sheet-mirror sheet)))
        (scl:send mirror :deactivate)))

(defmethod raise-mirror ((port genera-port) (sheet mirrored-sheet-mixin))
  (scl:send (sheet-mirror sheet) :expose))

(defmethod bury-mirror ((port genera-port) (sheet mirrored-sheet-mixin))
  (scl:send (sheet-mirror sheet) :bury))

(defmethod mirror-origin ((port genera-port) (sheet mirrored-sheet-mixin))
  port sheet
  :NW)

(defmethod mirror-inside-region* ((port genera-port) (sheet mirrored-sheet-mixin))
  (multiple-value-bind (lft top rgt bot)
      (scl:send (sheet-mirror sheet) :inside-edges)
    (values lft top rgt bot)))

(defmethod sheet-actual-native-edges* ((port genera-port)
				       (sheet mirrored-sheet-mixin))
  (let* ((mirror (sheet-mirror sheet))
	 (parent-mirror (sheet-mirror! (sheet-parent sheet))))
    (multiple-value-bind (width height) (scl:send mirror :inside-size)
      (multiple-value-bind (x y) (tv:sheet-calculate-offsets mirror parent-mirror)
	(values x
		y
		(+ x width)
		(+ y height))))))

;; How much of this want's to be generic?  For example, some systems may
;; not need the margin hacking.  Some systems may allow the user to
;; place the window outside the 'superior' without complaining.
(defmethod set-sheet-actual-native-edges* ((port genera-port)
					   (sheet mirrored-sheet-mixin)
					   min-x min-y lim-x lim-y)
  (let ((mirror (sheet-mirror sheet)))
    (multiple-value-bind (more-min-x more-min-y more-lim-x more-lim-y)
	(scl:send mirror :margins)		;--- mirror-margins ???
      (decf min-x more-min-x)
      (decf min-y more-min-y)
      (incf lim-x more-lim-x)
      (incf lim-y more-lim-y))
    ;; do clipping, since Genera won't.  Try to keep size the same,
    ;; repositioning if necessary.
    (multiple-value-bind (clipping-min-x clipping-min-y clipping-lim-x clipping-lim-y)
	(scl:send (tv:sheet-superior mirror) :inside-edges)	;--- mirror-inside-edges ???
      (multiple-value-bind (min-x min-y lim-x lim-y adjusted-p)
	  (fit-region*-in-region*
	    min-x min-y lim-x lim-y
	    clipping-min-x clipping-min-y clipping-lim-x clipping-lim-y)
	(scl:send mirror :set-edges min-x min-y lim-x lim-y)
	(when adjusted-p
	  (mirror-region-updated port sheet))))))

(defmethod mirror-region ((port genera-port) (sheet sheet))
  (let ((genera-window (sheet-mirror sheet)))
    (when genera-window
      (multiple-value-bind (genera-min-x genera-min-y genera-max-x genera-max-y)
	  (sheet-native-native-region* port sheet genera-window)
	(make-rectangle* 
	  genera-min-x genera-min-y
	  genera-max-x genera-max-y)))))

(defmethod mirror-region* ((port genera-port) (sheet sheet))
  (let ((genera-window (sheet-mirror sheet)))
    (when genera-window
      (multiple-value-bind (genera-min-x genera-min-y genera-max-x genera-max-y)
	  (sheet-native-native-region* port sheet genera-window)
	(values genera-min-x genera-min-y
		genera-max-x genera-max-y)))))

;;;
;;; Input Querying
;;;

(defmethod do-query-mouse ((port genera-port) sheet)
  (let* ((genera-window (sheet-mirror sheet))
	 (mouse (tv:sheet-mouse genera-window))
	 (screen (scl:send genera-window :screen)))
    (multiple-value-bind (ox oy)
	(tv:sheet-calculate-offsets genera-window screen)
      (values (- (tv:mouse-x mouse) ox) (- (tv:mouse-y mouse) oy)
	      ;;--- convert to X-style button mask
	      (tv:mouse-buttons nil mouse)))))


;;; Port trigger not nil means that the port is triggering the
;;; enabling/disabling.  This allows port enabled/disabled  methods to
;;; distinguish which way the flow of control is going.

(defvar *port-trigger* nil)

;;; --- what is this called now?
(defmethod ring-bell ((self genera-port))
  "ring the bell"
  ;; 0 percent from normal
  (scl:beep :beep (slot-value self 'genera-screen)))


;;;
;;; Input Processing
;;;

(defun make-state-from-buttons (buttons)
  (let ((state 0))
    ;; --- why isn't state-set a macro or subst??
    (when (ldb-test (byte 1 0) buttons)
      (setq state (state-set state :left)))
    (when (ldb-test (byte 1 1) buttons)
      (setq state (state-set state :middle)))
    (when (ldb-test (byte 1 2) buttons)
      (setq state (state-set state :right)))
    state))

(defun current-shift-state (&optional (state 0) (mouse tv:main-mouse))
  ;; Take only the upper bits of state, compute the lower bits from the current
  ;; shifts.
  (let ((shifts (tv:mouse-chord-shifts mouse))
	(state (logand state #xFF00)))
    (macrolet ((do-shift (shift key)
		 `(when (si:bit-test (si:name-bit ,shift) shifts)
		    (setf state (state-set state ,key)))))
      ;; why is SHIFT different from sys:%kbd-shifts-shift
      (when (ldb-test (byte 1 4) shifts)
	(setf state (state-set state +shift-key+)))
      (do-shift :control +control-key+)
      (do-shift :meta +meta-key+)
      (do-shift :super +super-key+)
      (do-shift :hyper +hyper-key+))
    state))

(defmethod do-poll-pointer ((port genera-port) sheet)
  (let* ((mouse (tv:sheet-mouse (sheet-mirror sheet))))
    (values (tv:mouse-x mouse)
	    (tv:mouse-y mouse)
	    (current-shift-state
	      (make-state-from-buttons (tv:mouse-last-buttons mouse))
	      mouse))))

(defmethod prompt-for-location ((port genera-port))
  (with-slots (genera-screen genera-console height-pixels) port
    (let ((mouse (sys:console-mouse genera-console)))
      (tv:with-this-mouse-and-buttons-grabbed (mouse)
	(tv:wait-for-mouse-button-down "Button" nil nil mouse)
	(tv:wait-for-mouse-button-up "Release Button" nil mouse)
	(values (sys:mouse-x mouse) (sys:mouse-y mouse))))))

#+ignore ; convert later
(defmethod prompt-for-region-location ((port x-port) width height)
  (with-slots (x-root x-display rband-gc height-pixels) port
    (multiple-value-bind (x y)
	(clx-utils::clx-prompt-for-screen-region-location 
	  x-root x-display 
	  (realize-cursor port :upper-left) rband-gc width height)
      (values x (- height-pixels y height)))))
    
#+ignore ; convert later
(defmethod prompt-for-region ((port x-port) left bottom width height)
  (with-slots (x-root x-display height-pixels rband-gc) port
    (multiple-value-setq 
	(left bottom width height)
      (clx-utils::clx-prompt-for-screen-region 
	x-root x-display rband-gc
	(realize-cursor port :upper-left) 
	(realize-cursor port :lower-left)
	(realize-cursor port :upper-right)
	(realize-cursor port :lower-right)
	left (- height-pixels bottom) width height))
    (values left (- height-pixels bottom height) width height)))

(defmacro genera-button-number->standard-button-name (code)
  `(aref '#(:left :middle :right) ,code))



(scl:defflavor silica-window
	((sheet nil)
	 (mouse-x nil)
	 (mouse-y nil)
	 (mouse-moved nil)
	 (blinker-table nil))
	(tv:changeable-name-mixin tv:window)
  (:initable-instance-variables sheet)
  (:writable-instance-variables sheet)
  (:default-init-plist :blinker-p nil))

(scl:defflavor temporary-silica-window
	((sheet nil)
	 (mouse-x nil)
	 (mouse-y nil)
	 (mouse-moved nil))
	(tv:temporary-window-mixin silica-window)
  (:initable-instance-variables sheet)
  (:writable-instance-variables sheet))

;;;--- This is a temporary method.  We will eventually make the silica-windows
;;; have :SAVE-BITS NIL and ensure that no actual output is attempted when the
;;; Silica sheet is not enabled.  For now, however, the stream text output stuff
;;; tries to draw on deexposed windows and insists that there be a bit array.
(scl:defmethod (:expose silica-window :before) (&rest ignore)
  ;;
  (scl:send scl:self :clear-window))

;;; When the window is first activated ("mapped") run the repaint handler
(scl:defmethod (:expose silica-window :after) (&rest ignore)
  (let ((*port-trigger* t))
    (enable-sheet sheet)))

(scl:defmethod (:refresh silica-window :after) (&optional (type ':complete-redisplay))
  (declare (ignore type))
  (let ((*port-trigger* t))
    (repaint-sheet sheet (sheet-region sheet))))

;;; --- problem: We disable the viewport that corresponds to the Genera window,
;;; but that leaves the window stream enabled.  Should en/disable on the viewport
;;; forward to the drawing-surface?
(scl:defmethod (:deexpose silica-window :after) (&rest ignore)
  (let ((*port-trigger* t))
    #+++ ;; want to call shrink-sheet, but that doesn't appear to be supported
    (disable-sheet sheet)))

;;; Called on position changes as well as size?
(scl:defmethod (:change-of-size-or-margins silica-window :after) (&rest options)
  (declare (ignore options))
  (let ((*port-trigger* t)
	(sheet (silica-window-sheet scl:self))
	(port (port sheet)))
    (mirror-region-updated port sheet)))

;;; Save this state (rather than using (mouse-x mouse)) 'cause the coords
;;; are in window coordinates already.
(defvar *mouse-moved* nil)
(defvar *mouse-x* nil)
(defvar *mouse-y* nil)
(defvar *mouse-buttons* 0)
(defvar *mouse-window* nil)
(defvar *mouse-button-released* nil)

;;; Really should be per console, as with all of these special variables.
(defvar *old-mouse-chord-shifts* 0)

(defun buttons-up (old-buttons new-buttons)
  #+Genera (declare (values buttons-up buttons-down))
  (values (boole boole-andc2 old-buttons new-buttons)
	  (boole boole-andc2 new-buttons old-buttons)))

(defmacro map-over-genera-shift-keysyms ((keysym-var genera-shift-mask) &body body)
  `(flet ((map-over-genera-shift-keysyms
	    (,keysym-var)
	    ,@body))
     (declare (dynamic-extent #'map-over-genera-shift-keysyms))
     (invoke-on-genera-shift-keysyms
       ,genera-shift-mask #'map-over-genera-shift-keysyms)))

(defun invoke-on-genera-shift-keysyms (genera-shift-mask continuation)
  (declare (dynamic-extent continuation))
  (macrolet ((do-shift (shift shift-name)
	       `(when (si:bit-test (si:name-bit ,shift) genera-shift-mask)
		  (funcall continuation ,shift-name))))
    ;; why is SHIFT different from sys:%kbd-shifts-shift
    (when (ldb-test (byte 1 4) genera-shift-mask)
      (funcall continuation :left-shift))
    (do-shift :control :left-control)
    (do-shift :meta :left-meta)
    (do-shift :super :left-super)
    (do-shift :hyper :left-hyper))
  nil)

;;; This is a horrible kludge for CLIM purposes.
(scl:advise tv:mouse-set-blinker-cursorpos-internal :after compute-buttons-released nil
  (kludge-buttons-released-when-mouse-over-no-window (first scl:arglist)))

;;; This is the guts of the horrible kludge for CLIM purposes.
(defun kludge-buttons-released-when-mouse-over-no-window (mouse)
  (multiple-value-bind (x y)
      (if (tv:mouse-warp-internal mouse)
	  (values (tv:mouse-last-x mouse)
		  (tv:mouse-last-y mouse))
	  (values (tv:mouse-x mouse)
		  (tv:mouse-y mouse)))
    (without-scheduling
      (let* ((old-buttons *mouse-buttons*)
	     (new-buttons tv:mouse-last-buttons)
	     (buttons-up (buttons-up old-buttons new-buttons))
	     ;; merge them all, for now.  This might drop a second up
	     ;; transition before the first one is noticed.  Even if we
	     ;; handled that here (by delaying updating *mouse-buttons*
	     ;; (actually, LOGIORing old and new) and
	     ;; *mouse-button-released*, it could require a mouse motion
	     ;; to actually notice the button up, which is gross!
	     (merged-up (logior buttons-up (or *mouse-button-released* 0)))
	     (new-released (and (not (zerop merged-up)) merged-up)))
	(setq *mouse-buttons* new-buttons
	      ;; doesn't yet handle multiple buttons released at once...
	      *mouse-button-released* new-released
	      *mouse-window* (let ((win
				     (tv:window-under-mouse-internal
				       mouse ':mouse-select ':active x y)))
			       (when (typep win 'silica-window) win))
	      *mouse-x* x
	      *mouse-y* y
	      *mouse-moved* ':pointer-motion)))))

(si:compile-advice 'tv:mouse-set-blinker-cursorpos-internal)

(scl:defmethod (:mouse-moves silica-window :after) (x y)
  (without-scheduling
    (let* ((old-buttons *mouse-buttons*)
	   (new-buttons tv:mouse-last-buttons)
	   (buttons-up (buttons-up old-buttons new-buttons))
	   ;; merge them all, for now.  This might drop a second up
	   ;; transition before the first one is noticed.  Even if we
	   ;; handled that here (by delaying updating *mouse-buttons*
	   ;; (actually, LOGIORing old and new) and
	   ;; *mouse-button-released*, it could require a mouse motion
	   ;; to actually notice the button up, which is gross!
	   (merged-up (logior buttons-up (or *mouse-button-released* 0)))
	   (new-released (and (not (zerop merged-up)) merged-up)))
      (setq *mouse-window* scl:self
	    *mouse-x* x
	    *mouse-y* y
	    *mouse-buttons* new-buttons
	    ;; doesn't yet handle multiple buttons released at once...
	    *mouse-button-released* new-released
	    *mouse-moved* ':pointer-motion))))

(scl:defmethod (:handle-mouse silica-window :before) ()
  (scl:send scl:self :select)			;!!
  (without-scheduling
    (setq *mouse-window* scl:self
	  *mouse-moved* ':enter-notify
	  *mouse-x* -1 *mouse-y* -1)))		;--- Unsure this is right.

(scl:defmethod (:handle-mouse silica-window :after) ()
  (ecase :give-up-kbd-focus			;!!
    (:deselect-too-big-a-hammer (scl:send scl:self :deselect))
    (:give-up-kbd-focus
      (let* ((exposed (scl:send (tv:sheet-screen scl:self) :exposed-inferiors))
	     (window (find-if #'(lambda (window) (member window exposed))
			      tv:previously-selected-windows)))
	(when window (scl:send window :select)))))
  (without-scheduling
    (setq *mouse-window* scl:self		;--- Which window is this notification for?
	  *mouse-moved* ':leave-notify
	  *mouse-x* -1 *mouse-y* -1)))		;--- ???

(scl:defmethod (:who-line-documentation-string silica-window) ()
  (port-pointer-documentation-string (port sheet)))

(defmacro let-state ((&rest let-clauses) &body body)
  `(multiple-value-bind ,(mapcar #'first let-clauses)
       (without-scheduling
	 (values ,@(mapcar #'second let-clauses)))
     ,@body))

(scl:compile-flavor-methods silica-window temporary-silica-window)

#+Debugging
(defvar *process-next-event-lock* (initial-lock-value "Genera PROCESS-NEXT-EVENT lock"))

#+Debugging
(defmethod process-next-event :around ((port genera-port) &key)
  (with-lockf (*process-next-event-lock* "Event lock")
    (call-next-method)))

;;; Take events out of the global queue and distribute them, and get key
;;; press "events" as characters out of the window's io-buffer.
;;;
;;; This method communicates with the dispatch-device-event method below through
;;; distribute-device-event.  It does this by passing in extra keywords, which
;;; are then passed back to dispatch.  I use the event-naming keywords used by
;;; the QUEUE-INPUT method.

;;; --- What should we do with non-:MOUSE-BUTTON blips?  What we
;;; --- currently do is just drop them on the floor...
(defmethod process-next-event ((port genera-port) &key (timeout nil) (wait-test nil)
			       ((:state whostate) "Genera Event"))
  (declare (special *port-trigger*)
	   (optimize (speed 3) (safety 0) (compilation-speed 0)))
  ;; Motion events are queued by the mouse process (in the :mouse-moves method above)
  ;; but the character input side isn't so well defined.  Rely on the fact that
  ;; only the selected window can be receiving input.
  (let ((genera-window nil)
	(shifts-up nil)
	(shifts-down nil))
    (flet ((await-Genera-event ()
	     (or *mouse-moved*
		 ;; which mouse to pass off to MOUSE-CHORD-SHIFTS?
		 (let ((shifts (tv:mouse-chord-shifts))
		       (old *old-mouse-chord-shifts*))
		   (when (/= old shifts)
		     (multiple-value-setq (shifts-up shifts-down)
		       (buttons-up old shifts))
		     (setq *old-mouse-chord-shifts* shifts)
		     t))
		 (when (and (typep tv:selected-window 'silica-window)
			    (scl:send tv:selected-window :listen))
		   (setq genera-window tv:selected-window)
		   t)
		 (when wait-test
		   (funcall wait-test)))))
      (declare (dynamic-extent #'await-Genera-event))
      (scl:process-wait-with-timeout whostate timeout #'await-Genera-event))
    (with-slots (distributor) port
      ;; don't distribute any events unless there's a distributor.
      ;; I have absolutely no idea how this mechanism intends to deal with the
      ;; single threaded environment.  Ask Rao.
      (process-wait "Pending Distribution"
		    #'(lambda () 
			;; Wait for the distributor to say its ready.
			(and distributor
			     (distributor-enabled distributor))))
      (macrolet ((distribute (x y &rest keys)
		   (let ((body `(distribute-device-event distributor port sheet
							 :time (get-internal-real-time)
							 ,@keys)))
		     (when (and x y)
		       (setf body `(multiple-value-bind (left top)
				       (if *mouse-window*
					   (genera-window-margins *mouse-window*)
					   (values 0 0))
				     (,@body :native-x (- ,x left) 
					     :native-y (- ,y top)))))
		     ;; Flushed :IMPEACH-DISTRIBUTOR test. 1/3/90, York
		     `(progn ,body
			t))))
	(cond (*mouse-moved*
	       (let-state ((mouse-moved (shiftf *mouse-moved* nil))	;capture and reset
			   (mouse-window	*mouse-window*)
			   (mouse-x		*mouse-x*)
			   (mouse-y		*mouse-y*)
			   (mouse-buttons	*mouse-buttons*)
			   (mouse-button-released (shiftf *mouse-button-released* nil)))
		 (when mouse-moved		;check again
		   (let ((sheet (and mouse-window (silica-window-sheet mouse-window))))
		     (when sheet
		       (distribute mouse-x mouse-y
				   :event-key mouse-moved 
				   :state
				   (current-shift-state
				    (make-state-from-buttons mouse-buttons)
				    (tv:sheet-mouse mouse-window))
				   :moved-p t))))
		 (when mouse-button-released
		   (let ((sheet (and mouse-window (silica-window-sheet mouse-window))))
		     ;; hmm?
		     (when sheet

		       (distribute mouse-x mouse-y
				   :event-key ':button-release
				   :button (genera-button-number->standard-button-name
					     (ash mouse-button-released -1))
				   :state (current-shift-state
						  (make-state-from-buttons mouse-button-released)
						  (tv:sheet-mouse mouse-window))
				   ))))
		   ))
	      ;; handle shift press and release events
	      ((or shifts-up shifts-down)
	       (when (typep tv:selected-window 'silica-window)
		 (let* ((genera-window tv:selected-window)
			(sheet (silica-window-sheet genera-window)))
		   (when sheet
		     (let ((state (current-shift-state 0 (tv:sheet-mouse genera-window))))
		       (when shifts-up
			 (map-over-genera-shift-keysyms (shift-keysym shifts-up)
			   (distribute *mouse-x* *mouse-y*
				     :event-key ':key-release
				     :keysym shift-keysym
				     :char nil
				     :keyboard-p T
				     :state state)))
		       (when shifts-down
			 (map-over-genera-shift-keysyms (shift-keysym shifts-down)
			   (distribute *mouse-x* *mouse-y*
				       :event-key ':key-press
				       :keysym shift-keysym
				       :char nil
				       :keyboard-p T
				       :state state))))))))

	      ;; Make sure we read from the same window that :LISTEN return T on, even
	      ;; if the selected window state has changed.
	      ((and genera-window		; must be a silica-window
		    (scl:send genera-window :listen))
	       (let ((thing (let ((sys:kbd-intercepted-characters
				    (remove #\Abort sys:kbd-intercepted-characters)))
			      (scl:send genera-window :any-tyi-no-hang)))
		     (sheet (silica-window-sheet genera-window)))
		 (typecase thing
		   (character
		     (when sheet
		       ;;--- Not snapshotting simultaneous X and Y, but I don't
		       ;; think it matters here.
		       ;; remember that state is always the state before
		       ;; this event was processed.  Since we're not yet distributing
		       ;; the key presses for the shifts, no big deal.
		       (let ((keysym (genera-character->keysym thing))
			     (char thing))
			 (when keysym
			   (distribute *mouse-x* *mouse-y*
				       :keysym keysym
				       :char char
				       :keyboard-p t :event-key ':key-press
				       :state (current-shift-state 
						0 (tv:sheet-mouse genera-window)))
			   (distribute *mouse-x* *mouse-y*
				       :keysym keysym
				       :char char
				       :keyboard-p t :event-key ':key-release
				       :state (current-shift-state 
						0 (tv:sheet-mouse genera-window)))))))
		   ;; See if it is a button-click blip
		   (list
		     (when (eql (first thing) ':mouse-button)
		       ;; (:mouse-button #\mouse-left window x y)
		       (let* ((x (fourth thing))
			      (y (fifth thing))
			      (code (tv:char-mouse-button (second thing)))
			      (win (third thing))
			      (button (genera-button-number->standard-button-name code))
			      ;; ---handle shift keys
			      #+Ignore
			      (state (make-state-from-buttons (ash 1 code)))
			      )
			 (when sheet
			   (distribute x y :event-key ':button-press 
				       :button button
				       ;; --- until we really track button up/down transitions
				       :state (current-shift-state 0 (tv:sheet-mouse win)))
			   #+Ignore
			   ;; --- until we really track button up/down transitions
			   (distribute x y :event-key ':button-release
				       :button button
				       :state state
				       )))))))))))))

(defmethod get-port-canonical-gesture-spec (gesture-spec (port genera-port))
  (let* ((console (slot-value port 'genera-console))
	 (keyboard-table (si:keyboard-keyboard-table (si:console-keyboard console))))
    (let ((keysym (car gesture-spec))
	  (shifts (cdr gesture-spec)))
      (let (
	    (genera-char (keysym->genera-character keysym))
	    (genera-charcode nil)
	    (needed-shifts nil)
	    (genera-keycode nil))
	(unless genera-char (return-from get-port-canonical-gesture-spec nil))
	(setq genera-charcode (char-code genera-char))
	(multiple-value-setq (needed-shifts genera-keycode)
	  (block find-it
	    (dotimes (i 4)
	      (dotimes (j 128)
		(when (= (aref keyboard-table i j) genera-charcode)
		  (return-from find-it (values i j)))))))
	(unless genera-keycode (return-from get-port-canonical-gesture-spec nil))
	(if (and (not (alpha-char-p genera-char))
		 (ldb-test si:%%kbd-mapping-table-index-shift needed-shifts))
	    (setq needed-shifts (make-shift-mask :shift))
	    (setq needed-shifts 0))
	(setq shifts (logior shifts needed-shifts))
	;; ok, now we've got the genera keycode index into the keyboard table.
	;; We find all of the shifts which match SHIFTS and look them up in the table.
	(let* ((real-charcode (aref keyboard-table
				    ;; the only of our modifier keys which can
				    ;; actually change the keysym
				    ;; in this port is :SHIFT
				    (if (state-match-p shifts :shift)
					(make-shift-mask :shift)
					0)
				    genera-keycode))
	       (real-genera-char (and real-charcode (code-char real-charcode)))
	       (real-genera-keysym (and real-genera-char
					(genera-character->keysym real-genera-char))))
	  (cons real-genera-keysym shifts))))))

;;;
;;; Input Dispatching Support
;;;

(defmethod filter-port-event-keys ((port genera-port) contract delivery-thunk
				   &rest keys
				   &key event-key state code 
				   &allow-other-keys)

  (declare (ignore contract)
	   (dynamic-extent keys))

  (when (member event-key '(:motion-notify :enter-notify :leave-notify))
    (setq event-key :pointer-motion))

  (apply delivery-thunk :event-key event-key keys))




;;; Character appearances for Genera

(defmethod text-style-mapping :around
    ((port genera-port) character-set text-style)
  (let ((font-or-font-name (call-next-method)))
    (etypecase font-or-font-name
      (null nil)
      (sys:font font-or-font-name)
      (symbol
	(when (not (boundp font-or-font-name))
	  (handler-case (fed:find-and-load-font font-or-font-name)
	    (fed:font-not-found ()
	      (error "Mapping ~S for ~A characters in ~A character set could not be found."
		     font-or-font-name text-style (or character-set :default)))))
	(let ((font (symbol-value font-or-font-name)))
	  (unless (typep font 'sys:font)
	    (error "Mapping for ~A characters in ~A character set is not a font."
		   text-style (or character-set :default)))
	  ;; Simple cache: hope nobody changes the font later
	  (add-text-style-mapping
	    port character-set (parse-text-style text-style) font)
	  font)))))
#||
;;; to test:
(let ((port (find-port :server-path '(:genera))))
  (add-text-style-mapping port *standard-character-set* '(:hand :roman :normal) 'fonts:scrawl)
  (add-text-style-mapping port *standard-character-set* '(:hand :italic :normal) 'fonts:hand)
  (mapcar #'(lambda (face)
	      (text-style-mapping port *standard-character-set*
				  (make-text-style :hand face :normal)))
	  '(:roman :italic :bold)))

;; Should return (#<FONT SCRAWL ...> #<FONT HAND ...> #<FONT BOXFONT ...>)
||#

(defparameter *sheet-logical-size-alist* nil)

(defmethod initialize-genera-display-device ((display-device genera-port))
  ;;; This has been moved to the DEFCLASS above.
  #+Ignore
  (add-text-style-mapping display-device *standard-character-set*
			  *undefined-text-style* 'fonts:boxfont)
  (let ((screen (genera-screen display-device)))
    (let ((device (scl:send screen :display-device-type)))
      (cond
	#+Imach
	((and (find-package 'mtb)
	      (typep device (intern "SMALL-SCREEN-GENERA-FONTS-MAC-DISPLAY-DEVICE"
				    'mtb)))
	 (define-small-screen-text-style-mappings display-device))
	(t (define-normal-screen-text-style-mappings display-device))))))

(defmethod standardize-text-style ((display-device genera-port) character-set style)
  (standardize-text-style-1
    display-device style character-set *sheet-logical-size-alist*))

;;;;;;;;;;;;;;;;

(defun define-normal-screen-text-style-mappings (display-device)
  (define-text-style-mappings-load-time display-device *standard-character-set*
    '((:family :fix (:face :roman (:size 20 fonts:bigfnt
					 14 fonts:medfnt
					 12 fonts:cptfont
					 10 fonts:tvfont
					 8 fonts:einy7
					 6 fonts:tiny)
			   :italic (:size 20 fonts:bigfnti
					  14 fonts:medfnti
					  12 fonts:cptfonti
					  10 fonts:tvfonti
					  8 fonts:einy7
					  6 fonts:tiny)
			   :bold (:size 20 fonts:bigfntb
					14 fonts:medfntb
					12 fonts:cptfontcb
					10 fonts:tvfontb
					8 fonts:einy7
					6 fonts:tiny)
			   (:bold :italic) (:size 20 fonts:bigfntbi
						  14 fonts:medfntbi
						  12 fonts:cptfontbi
						  10 fonts:tvfontbi
						  8 fonts:einy7
						  6 fonts:tiny)
			   (:bold :extended) (:size 12 fonts:cptfontb
						    10 fonts:tvfontb)
			   :condensed (:size 12 fonts:cptfontc)
			   (:extra :condensed) (:size 12 fonts:cptfontcc
						      10 fonts:tvfont)))))

  (define-text-style-mappings-load-time display-device *standard-character-set*
    '((:family :serif (:face :roman (:size 20 fonts:dutch20
				       14 fonts:dutch14
				       12 fonts:tr12
				       10 fonts:tr10
				        8 fonts:tr8)
			 :italic (:size 20 fonts:dutch20I
					14 fonts:dutch14I
					12 fonts:tr12I
					10 fonts:tr10I
				 	 8 fonts:tr8I)
			 :bold (:size 20 fonts:dutch20B
				      14 fonts:dutch14B
				      12 fonts:tr12B
				      10 fonts:tr10B
				       8 fonts:tr8B)
			 (:bold :italic) (:size 20 fonts:dutch20BI
						14 fonts:dutch14BI
						12 fonts:tr12BI
						10 fonts:tr10BI
						 8 fonts:tr8BI)))))

  (define-text-style-mappings-load-time display-device *standard-character-set*
    '((:family :sans-serif (:face :roman (:size 20 fonts:swiss20
					    14	fonts:hl14
					    12	fonts:hl12
					    10	fonts:hl10
					     8 fonts:hl8)
			      :italic (:size 20 fonts:swiss20I
					     14 fonts:hl14I
					     12 fonts:hl12I
					     10 fonts:hl10I
					      8 fonts:hl8I)
			      :bold (:size 20 fonts:swiss20B
					   14 fonts:hl14B
					   12 fonts:hl12B
					   10 fonts:hl10B
					    8 fonts:hl8B)
			      (:bold :italic) (:size 20 fonts:swiss20BI
						     14 fonts:hl14BI
						     12 fonts:hl12BI
						     10 fonts:hl10BI
						      8 fonts:hl8BI)))))

  (setq *sheet-logical-size-alist*
	'((:tiny        6)
	  (:very-small  8)
	  (:small      10)
	  (:normal     12)
	  (:large      14)
	  (:very-large 20)
	  (:huge       24)))
  ) ;; end define-normal-screen-text-style-mappings

;;; For MacIvory screens.
(defun define-small-screen-text-style-mappings (display-device)
  (define-text-style-mappings-load-time display-device *standard-character-set*
    '((:family :fix (:face :roman (:size 20 fonts:medfnt
				     14 fonts:cptfont
				     12 fonts:tvfont
				     10 fonts:einy7
				      8 fonts:einy7
				      6 fonts:einy7)
		       :italic (:size 20 fonts:medfnti
				      14 fonts:cptfonti
				      12 fonts:tvfonti 
				      10 fonts:einy7
				       8 fonts:einy7
				       6 fonts:einy7)
		       :bold (:size 20 fonts:medfntb
				    14 fonts:cptfontcb
				    12 fonts:tvfontb 
				    10 fonts:einy7
				     8 fonts:einy7
				     6 fonts:einy7)
		       (:bold :italic) (:size 20 fonts:medfntbi
					      14 fonts:cptfontbi
					      12 fonts:tvfontbi
					      10 fonts:einy7
					       8 fonts:einy7
					       6 fonts:einy7)
		       (:bold :extended) (:size 12 fonts:tvfontb
						10 fonts:cptfontb)
		       :condensed (:size 12 fonts:cptfontc)
		       (:extra :condensed) (:size 12 fonts:cptfontcc
						  10 fonts:tvfont)))))

  (define-text-style-mappings-load-time display-device *standard-character-set*
    '((:family :serif (:face :roman (:size 20 fonts:dutch14
				       14 fonts:tr12
				       12 fonts:tr10
				       10 fonts:tr8
				        8 fonts:tr8)
			 :italic (:size 20 fonts:dutch14I
					14 fonts:tr12I
					12 fonts:tr10I
					10 fonts:tr8I
					 8 fonts:tr8I)
			 :bold (:size 20 fonts:dutch14B
				      14 fonts:tr12B
				      12 fonts:tr10B
				      10 fonts:tr8B
				       8 fonts:tr8B)
			 (:bold :italic) (:size 20 fonts:dutch14BI
						14 fonts:tr12BI
						12 fonts:tr10BI
						10 fonts:tr8BI
						 8 fonts:tr8BI)))))

  (define-text-style-mappings-load-time display-device *standard-character-set*
    '((:family :sans-serif (:face :roman (:size 20 fonts:hl14
					    14 fonts:hl12
					    12 fonts:hl10
					    10 fonts:hl8
					     8 fonts:hl8)
			      :italic (:size 20 fonts:hl14I
					     14 fonts:hl12I
					     12 fonts:hl10I
					     10 fonts:hl8I
					      8 fonts:hl8I)
			      :bold (:size 20 fonts:hl14B
					   14 fonts:hl12B
					   12 fonts:hl10B
					   10 fonts:hl8B
					    8 fonts:hl8B)
			      (:bold :italic) (:size 20 fonts:hl14BI
						     14 fonts:hl12BI
						     12 fonts:hl10BI
						     10 fonts:hl8BI
						      8 fonts:hl8BI)))))


  (setq *sheet-logical-size-alist*
	'((:tiny        6)
	  (:very-small  8)
	  (:small      10)
	  (:normal     12)
	  (:large      14)
	  (:very-large 20)
	  (:huge       24)))
  ) ;; end define-small-screen-text-style-mappings
