;;; -*- Mode: Lisp; Package: ON-GENERA; Base: 10.; Syntax: Common-Lisp -*-
;;;
;;; Copyright (c) 1989 by Xerox Corporations.  All rights reserved.
;;;

(in-package "ON-GENERA")

;;;
;;; BASIC-GENERA-DISPLAY-MEDIUM
;;;   A Foundation Class for Display Mediums on Genera
;;;

(defclass basic-genera-medium (medium)
    (drawable))

(defclass basic-genera-display-medium (basic-genera-medium display-medium)
    ((window-system-clipping-region :initform (list 0 0 0 0))))

(defclass basic-genera-pixmap-medium (basic-genera-medium pixmap-medium)
    ((associated-medium :initarg :associated-medium)))

;; MEDIUM-FORCE/FINISH-OUTPUT don't do anything extra in genera.
;; Perhaps someday they could frob the screen manager or something.

;;; Why do we have these unbound slot accessors at all?  There doesn't
;;; appear to be a way to specify the DRAWABLE except via this
;;; initialization; why not fold it into the object at MAKE-INSTANCE
;;; time?

(defmethod slot-unbound (class
			  (medium basic-genera-display-medium)
			  (slot-name (eql 'drawable)))
  #-PCL (declare (ignore class))
  (setf (slot-value medium 'drawable) 
	  (sheet-mirror (fetch-mirrored-sheet (medium-sheet medium)))))

;;; Eliminate many of the uses of the SLOT-UNBOUND method above.
(defmethod (setf medium-sheet) :after (new-sheet (medium basic-genera-display-medium)) 
  (if new-sheet
      (setf (slot-value medium 'drawable) 
	      (sheet-mirror (fetch-mirrored-sheet new-sheet)))
      (slot-makunbound  medium 'drawable)))

(defmethod slot-unbound (class
			  (medium basic-genera-pixmap-medium)
			  (slot-name (eql 'drawable)))
  #-PCL (declare (ignore class))
  (let* ((other (slot-value medium 'associated-medium))
	 (port (port other))
	 (sheet (medium-sheet other))
	 (mirror (sheet-mirror! sheet)))
    (multiple-value-bind (min-x min-y max-x max-y)
	(sheet-native-region* port sheet)
      (setf (slot-value medium 'drawable)
	    ;; --- should share with realize-mirror
	    (tv:make-window
	      'tv:window
	      ;; probably the screen
	      :superior (tv:sheet-screen mirror)
	      :name "CLIM off-screen drawing window"
	      :save-bits t
	      :activate-p t
	      :expose-p nil
	      :deexposed-typeout-action :permit
	      :blinker-p t
	      :label nil  ;;name
	      ;; Want inside/outside coord system to match if possible
	      :borders nil
	      :x min-x :y min-y 
	      :width (- max-x min-x)
	      :height (- max-y min-y))))))

(defmethod cleanup-display-medium :after 
	   ((display-medium basic-genera-display-medium))
  ;; gcontext should still be applicable since medium are only reused within
  ;; the scope of a graft.
  (slot-makunbound display-medium 'drawable))

(defmethod validate-medium :after ((medium basic-genera-display-medium))
  (with-slots (device-clipping-region window-system-clipping-region drawable)
      medium
    (typecase device-clipping-region
      (rectangle
	(with-bounding-rectangle* (x1 y1 x2 y2) device-clipping-region
	  (let* ((wscr window-system-clipping-region)
		 (temp wscr))
	    (setf (first temp) (integerize-coordinate x1) temp (cdr temp)
		  (first temp) (integerize-coordinate y1) temp (cdr temp)
		  (first temp) (integerize-coordinate x2) temp (cdr temp)
		  (first temp) (integerize-coordinate y2) temp (cdr temp))
	    (setf (tv:sheet-clipping-region drawable) wscr))))
      (nowhere
	(setf (tv:sheet-clipping-region drawable) '(0 0 0 0)))
      (everywhere
       ;; ???  This doesn't work for pixmap-medium
	(setf (tv:sheet-clipping-region drawable) 
	      (multiple-value-list (scl:send drawable :inside-edges)))
	)

      ;; For non-handled region types, a do-clipped operation will have to do
      ;; it piecewise at output time.
      (rectangle-set 
	(error "Genera port can't handle non-rectangular clipping region."))
      (otherwise (unimplemented "clipping on weird region?")))))

#||

(defmethod initialize-instance :after 
	   ((medium basic-x-pixmap-medium) &key pixmap &allow-other-keys)
  ;; Trigger the transformation setup
  (when pixmap (setf (medium-pixmap medium) pixmap)))

(defmethod (setf medium-pixmap) :after (pixmap (medium basic-x-pixmap-medium))
  (let ((xform (device-transformation medium)))
    (let ((xf (compose-with-translation +identity-transformation+ 
				   0 (pixmap-height pixmap)
				   :reuse xform)))
      (setf xform (compose-with-scaling xf 1 -1 :reuse xf)))
    (setf (device-transformation medium) xform)))

(defmethod slot-unbound (class
			  (medium basic-x-medium)
			  (slot-name (eql 'gcontext)))
  (declare (ignore class))
  ;; A GCONTEXT can be used on any drawable with same root and same depth as
  ;; the drawable it is created for.  All children of graft will necessarily be
  ;; of same root, but I'm also assuming that they will be of the same depth as
  ;; the root of their screen.   
  (setf (slot-value medium 'gcontext)
	(with-slots (x-screen x-root)
	    (graft medium)
	  (xlib:create-gcontext 
	    :drawable x-root
	    :foreground (xlib:screen-black-pixel x-screen)
	    :background (xlib:screen-white-pixel x-screen)
	    :fill-style :solid))))

;;;
;;; DM Operations.
;;;

(defmethod (setf device-clipping-region) :after
	   ((region region) (medium basic-x-medium))
  (with-slots (gcontext device-clipping-region device-transformation drawable) 
      medium
    drawable
    (typecase device-clipping-region
      (rectangle 
       (setf (xlib:gcontext-clip-mask gcontext)
	     (list (round (rectangle-min-x device-clipping-region))
		   (round (rectangle-min-y device-clipping-region))
		   (round (rectangle-width device-clipping-region))
		   (round (rectangle-height device-clipping-region)))))
      (nowhere
       (setf (xlib:gcontext-clip-mask gcontext)
	     (list 0 0 0 0)))
      (everywhere
       ;; ???  This doesn't work for pixmap-medium
       (setq device-clipping-region
	     (transform-region
	       device-transformation
	       (sheet-region (contract-sheet medium))))

       (setf (xlib:gcontext-clip-mask gcontext)
	     (list (round (rectangle-min-x device-clipping-region))
		   (round (rectangle-min-y device-clipping-region))
		   (round (rectangle-width device-clipping-region))
		   (round (rectangle-height device-clipping-region)))))

      ;; For non-handled region types, a do-clipped operation will have to do
      ;; it piecewise at output time.
      (rectangle-set 
       (setf (xlib:gcontext-clip-mask gcontext :unsorted)
	     (mapcan #'(lambda (rectangle)
			 (list (round (rectangle-min-x rectangle))
			       (round (rectangle-min-y rectangle))
			       (round (rectangle-width rectangle))
			       (round (rectangle-height rectangle))))
		     (rectangles device-clipping-region))))
      (otherwise (unimplemented "clipping on weird region?")))))

(defmethod (setf medium-pixmap) :after 
	   ((pixmap pixmap) (medium basic-x-pixmap-medium))
  (setf (drawable medium) (realize-pixmap (graft medium) pixmap)))

;;;
;;; Font Operations
;;;

(defmethod font-ascent ((font font) (medium basic-x-medium))
  (xlib:font-ascent (realize-font (graft medium) font)))
					
(defmethod font-descent ((font font) (medium basic-x-medium))
  (xlib:font-descent (realize-font (graft medium) font)))
					
(defmethod font-height ((font font) (medium basic-x-medium))
  (+ (font-ascent font medium)
     (font-descent font medium)))

(defmethod string-width (string (font font) (medium basic-x-medium))
  (xlib:text-width (realize-font (graft medium) font) string))

||#
