;;; -*- Mode: Lisp; Package: ON-GENERA; Base: 10.; Syntax: Common-Lisp -*-
;;;

"Copyright (c) 1990 by International Lisp Associates.  All rights reserved."

;;;
;;; Copyright (c) 1989 by Xerox Corporations.  All rights reserved.
;;;

(in-package "ON-GENERA")

;;;
;;; CLG Media
;;;

(defclass clg-medium (basic-clg-medium basic-genera-medium)
    ;; this hash table still isn't quite good enough, as
    ;; (1 0 0) and (1.0 0 0) aren't EQUAL.
    ((color-table :initform (make-hash-table :test #'equal))))

(defmethod genera-color-from-color ((medium clg-medium) color color-p)
  (genera-color-from-rgb 
    medium
    (color-red-value color) (color-green-value color) (color-blue-value color)
    color-p))

(defmethod genera-color-from-rgb ((medium clg-medium) red green blue color-p)
  (unless color-p				;On B/W, BLACK is WHITE...
    (setf red (- 1 red) green (- 1 green) blue (- 1 blue)))
  (with-slots (color-table) medium
    (with-stack-list (key red green blue)
      (or (gethash key color-table)
	  (setf (gethash (copy-list key) color-table)
		(color:make-color :red red
				  :green green
				  :blue blue))))))

(defclass clg-display-medium (clg-medium basic-clg-display-medium 
					 basic-genera-display-medium)
    ())

(defclass clg-pixmap-medium (clg-medium basic-clg-pixmap-medium
					basic-genera-pixmap-medium)
    ())

(defmethod display-medium-type ((graft genera-port) (type (eql :clim)))
  'clg-display-medium)
(defmethod pixmap-medium-type ((graft genera-port) (type (eql :clim)))
  'clg-pixmap-medium)

;; medium-force/finish-output inherit from basic-genera-medium


;;;
;;; Generation of Graphics Operation Code
;;;

;;; Questions: Do we want to do transformations here, or have GRAPHICS:DRAW-FOO do them?
;;; Have to figure out how to translate inks into Genera pattern/etc stuff.
;;; Can we do without GRAPHICS:DRAW-FOO?

(defmacro with-device-point ((transformation px py &optional (new-px px) (new-py py))
			     &body body)
  (expand-with-device 'transform-point* transformation (list px py) (list new-px new-py) body))

(defmacro with-device-distance ((transformation dx dy &optional (new-dx dx) (new-dy dy))
				&body body)
  (expand-with-device 'transform-distance transformation (list dx dy) (list new-dx new-dy)
		      body))

(defmacro with-device-rectangle ((transformation min-x min-y max-x max-y
				  &optional (new-min-x min-x) (new-min-y min-y)
					    (new-max-x max-x) (new-max-y max-y))
				 &body body)
  (expand-with-device 'transform-rectangle* transformation (list min-x min-y max-x max-y)
		      (list new-min-x new-min-y new-max-x new-max-y) body))

(defmacro with-two-device-distances ((transformation dx1 dy1 dx2 dy2
				      &optional (new-dx1 dx1) (new-dy1 dy1)
						(new-dx2 dx2) (new-dy2 dy2))
				     &body body)
  (expand-with-device
    'transform-distance transformation
    (list dx1 dy1) (list new-dx1 new-dy1)
    (list (expand-with-device 'transform-distance transformation
			      (list dx2 dy2) (list new-dx2 new-dy2)
			      body))))

(eval-when (compile load eval)
  (defun expand-with-device (transformer transformation from to body)
    `(multiple-value-bind ,to
	 (if (eql ,transformation +identity-transformation+)
	     (values ,@from)
	     (,transformer ,transformation ,@from))
       ,@(mapcar #'(lambda (to) `(setf ,to (integerize-coordinate ,to))) to)
       ,@body)))

(defmacro do-graphics-function-spread-arguments (((type arg args*) gf) &body body)
  `(dolist (spread-arg (gf-spread-arguments ,gf))
     (let ((,arg (pop spread-arg))
	   (,type (pop spread-arg))
	   (,args* spread-arg))
       ,@body)))


(defun genera-pattern-and-alu-from-ink (ink port medium drawable)
  (etypecase ink
    (symbol  ;;This is bogus, of course; these should be instances.
      (ecase ink
	(:foreground
	  (values
	    (genera-pattern-and-alu-from-ink (medium-foreground medium) port medium drawable)
	    :draw))
	(:background
	  (values
	    (genera-pattern-and-alu-from-ink (medium-background medium) port medium drawable)
	    :erase))
	(:flipping-ink
	  (multiple-value-bind (fpat falu)
	      (genera-pattern-and-alu-from-ink (medium-foreground medium) port medium drawable)
	    (multiple-value-bind (bpat balu)
		(genera-pattern-and-alu-from-ink
		  (medium-background medium) port medium drawable)
	      (declare (ignore bpat))
	      (values fpat
		      (scl:send (scl:send drawable :screen)
				:exchange-two-colors-aluf
				falu
				balu)))))))
    (color (values (genera-color-from-color medium ink (color:color-stream-p drawable))
		   :draw))

    ((integer 0 100)						;Raoian kludge?
     (let ((rgb-value (/ ink 100.0))
	   (color-p (color:color-stream-p drawable)))
       (setf rgb-value (- 1 rgb-value))
       (values (genera-color-from-rgb medium rgb-value rgb-value rgb-value color-p)
	       :draw)))

    (pixmap (values (realize-pixmap port ink)
		    :draw))))

;;; This whole file is #+Genera, so the following package kludge will work.
;;; We need these slot names and class names to be in the Silica package, and they are not
;;; exported from there, so just write these methods as if they were in the Silica package.

;;; These methods are not yet in use.  They will be used for ellipses, at least...

#||
;; --- commented out by doughty (10/1/91).  We'll see if they're not in use yet!

silica::(progn

(defmethod on-genera::genera-transformation ((x identity-transformation))
  '(1 0 0 1 0 0))

(defmethod on-genera::genera-transformation ((x translation))
  (with-slots (m20 m21) x
    (list 1 0 0 1 m20 m21)))

(defmethod on-genera::genera-transformation ((x st-transformation))
  (with-slots (m00 m11 m20 m21) x
    (list m00 0 0 m11 m20 m21)))

(defmethod on-genera::genera-transformation ((x srt-transformation))
  (with-slots (m00 m01 m10 m11 m20 m21) x
    (list m00 m01 m10 m11 m20 m21)))

)

||#

#||

Notes and commentary (DCPL 900925).

Historical note: the original problem that led to an investigration and
subsequently these notes is a rather obscure bug in Genera (7.2), such
that if you set the ALU to BOOLE-XOR in the graphics state, and then
called (optimized) draw-line, somewhere along the way the ALU is
forgotten.  A bug report was sent to this effect.  Look for any of
	     <19900809173500.3.DCPL@DIAMOND-HEAD.ILA.COM>,
             <19900810160930.6.DCPL@DIAMOND-HEAD.ILA.COM>,
             <19900810161847.7.DCPL@DIAMOND-HEAD.ILA.COM>,
             <19900810163307.8.DOUGHTY@BUGS-BUNNY.ILA.COM>,
             <19900810164628.8.DCPL@DIAMOND-HEAD.ILA.COM>,
             <19900810202230.5.CHF@FLUTTERING-SHEARWATER.SCRC.Symbolics.COM>,
             <19900811163844.3.DOUGHTY@BUGS-BUNNY.ILA.COM>,
	     <19900816153056.2.CHF@FLUTTERING-SHEARWATER.SCRC.Symbolics.COM>
In the archives.  So, we need to work around it.

The solution RSL came up with and installed was to get the ALU and pass
it in to the graphics:draw-xxx funtions, and that kept the optimizers
alive and made the ALU work.

The solution DCPL came up with was to pass the specifications as keyword
arguments to the graphics:draw-xxx functions, thereby bypassing the
graphics state and any other bugs it may have.  Another advantage is
that the graphics state is only setup once, when needed, instead of at
the beginning of a function (e.g., draw-rectangle) that may decline to
use it and need to call the next method (e.g., draw-rectangle with
non-rectilinear transform).  This multiple setup probably is not a
problem in practice, but it does lose some appeal.  A drawback is that
all the optimizers go away and perhaps some speed is lost.

Here are some of DCPL's thoughts on other things he was going to do.

The syntax if define-genera-graphics-methods could use some improvement.
Currently it is like defstruct where it really could be much more like
defun.  After a few iterations, the syntax I found was
    (define-genera-graphics-methods draw-rectangle (ink line-thickness &do-not-convert-points)
       . body)

Although ink and line-thickness aren't arguments per se, they are in
some sense.  They are the :USING arguments of today.  But from what I
can tell, every method has them.  There are also very few options (two
now), which is why I collapsed them into the argument list as & symbols.
RSL's inclusion of the alu would be something like &ALU ALU.  This
requires a little more parsing in the macro, but aesthetically it's not
bad (and it's the best I've come up with to date).

The style of pop-coordinate with an optional "return code" is a little
poor.  It is not hard to do and endp test before the first
pop-coordinate and take the appropriate action, instead of folding it
into an optional return-code.  In some sense, this is an
optimistic/pessimistic style difference, the current implementation
being optimistic, but it seems out of style.  The reason I see it out of
style is that a piece of flow-of-control code is passed to the macro,
and it is a little disturbing to see something as functional as
pop-coordinate do something as special-form-ish as taking optional code
fragments.  The best Lisp way I can think of to do this would be a
(global) macro that looked like
	(with-coordinate-popping (pop-name) coord-seq
	 . body)
that expanded into something like
	(flet ((#:pop-no-coord ()
		(error ...)))
	  (macrolet ((pop-name ()
		       `(if (endp coord-seq)
			    (#:pop-no-cond)
			    (pop coord-seq))))
	    . body))
[I was originally going to have setup a tagbody and a GO to an error
clause, but the GO would have lost any information about where the error
came from, plus more hair to arrange for the body to return a value
through the tagbody (not hard).]  The FLET could go away and its caller
be replaced with the error message.


with-stack-arg-if is a little (too) funky.  A better name would be
with-stack-keyword-pair-if, and it's syntax would much better be
something like
	(with-stack-keyword-pair-if cond (var keyword value var)
	  . body)
This has the following advantages of the macrolet'd with-stack-arg-if.
 -  The condition is in a place consistent with the condition of other
    -if forms.
 - The binding part looks like with-stack-list*.
 - It can be made global.
Also, if cond is (compile-time) :VALUE the (runtime) value becomes the
condition (preserving a feature of the old with-stack-arg-if).  The
working definition I had was
    (defmacro with-stack-keyword-pair-if (cond (var keyword value var-again) 
					  &body body)
      (assert (eq var var-again))
      (let ((test (gensymbol 'if))
	    (temp-var (gensymbol 'var))
	    (val-for-cond (eq cond ':VALUE)))
	`(let ((,test ,(if val-for-cond value cond)))
	   (with-stack-list* (,temp-var
			      (and ,test ,keyword)
			      ,(if val-for-cond test `(and ,test ,value))
			      ,var-again)
	     (let ((,var (if ,test ,temp-var ,var)))
	       ,@body)))))

The biggest wart in DCPL's scheme is calling the graphics:draw-xxx
functions.  Since they needed to be called with keywords, something was
needed to get ahold of them.  I chose to do it implicitly, since there
wasn't any reason the method writers needed to have control over it.
The solution was to MACROLET :apply-with-genera-keywords to expand into
the code to pull apart INK, LINE-THICKNESS, etc, and do the apply.  This
would have the advantage the pulling apart would only happen when it was
going to be used.  So a call might look like
	(:apply-with-genera-keywords
	  #'graphics:draw-rectangle x1 y1 x2 y2 :stream drawable :filled filled)
which would expand into
	(much-setup-binding-#:keywords
	  (apply #'graphics:draw-rectangle ... :filled filled #:keywords))

||#


(zwei:defindentation (with-stack-arg-if 1 1))	;Get indentation right.

(eval-when (compile load eval)
  (defparameter *line-style-accessor-data*
		'((:units line-style-units)
		  (:thickness line-style-thickness 1)
		  (:scale-lines line-style-scale-lines)
		  (:line-end-shape line-style-cap-shape :butt)
		  (:line-joint-shape line-style-joint-shape :miter)
		  (:dashed line-style-dashed)
		  (:dash-pattern line-style-dash-pattern)
		  (:scale-dashes line-style-scale-dashes)
		  (:initial-dash-phase line-style-initial-dash-phase))))

(defmacro with-genera-graphics-context ((drawable port medium &key using alu)
					&body continuation)
  (let* ((arguments (gensymbol 'arguments))
	 (body `(apply #'graphics::with-drawing-state-internal-0
		       #'continuation ,drawable ,arguments))
	 (alu-argument nil))
    (dolist (using using)
      (ecase using
	;; Also will need text style somewhere in here, but not yet.
	(ink
	  (let ((ink (gensymbol 'ink)))
	    (setf alu-argument alu)
	    (setf body
		    `(let ((,ink (medium-ink ,medium)))
		       (multiple-value-bind (.pat. .alu.)
			   (genera-pattern-and-alu-from-ink ,ink ,port ,medium ,drawable)
			 (setf ,alu-argument .alu.)
			 (with-stack-arg-if (.pat. :pattern .pat.)
			   (with-stack-arg-if (t :alu .alu.)
			     ,body)))))))
	(line-style
	  (let ((line-style (gensymbol 'line-style)))
	    (dolist (ls-part *line-style-accessor-data*)
	      (let* ((ls-genera-keyword (pop ls-part))
		     (ls-accessor (pop ls-part))
		     (ls-default-p (not (null ls-part)))
		     (ls-default (pop ls-part))
		     (ls-default-test
		       (if ls-default-p
			   `(let ((,ls-accessor (,ls-accessor ,line-style)))
			      (and (not (eql ,ls-accessor ',ls-default)) ,ls-accessor))
			   `(,ls-accessor ,line-style))))
		(setf body
			`(with-stack-arg-if (,ls-default-test ,ls-genera-keyword)
			   ,body))))
	    (setf body `(let ((,line-style (medium-line-style ,medium)))
			  ,body))))))
    (setf body `(let ((,arguments nil)) ,body))
    `(let (,@(when alu-argument `((,alu-argument nil))))
       (flet ((continuation (,drawable)
		,@continuation))
	 (declare (dynamic-extent #'continuation))
	 (macrolet ((with-stack-arg-if ((condition keyword &optional value)
					&body body)
		      (let* ((condition-value (gensymbol 'if))
			     (evaluated-value (if value (gensymbol keyword) condition-value)))
			`(let* ((,condition-value ,condition)
				,@(when value
				    `((,evaluated-value (when ,condition-value ,value)))))
			   (with-stack-list* (,',arguments ,keyword ,evaluated-value
					      ,',arguments)
			     (unless ,condition-value (setf ,',arguments (cddr ,',arguments)))
			     ,@body)))))
	   ,body)))))

#||
;;; Gross kludge to put into the body of the continuation:

	      (let* ((drawing-state (graphics::get-drawing-state ,drawable)))
		(graphics::drawing-state-source-and-alu drawing-state ,drawable)
		(setf (ldb graphics::%%drawing-state-parameters-modified
			   (graphics::drawing-state-flags drawing-state)) 0))
||#

(defmacro with-appropriate-scan-conversion-mode ((stream-variable raster) &body body)
  `(flet ((with-appropriate-scan-conversion-mode (,stream-variable) ,@body))
     (declare (dynamic-extent #'with-appropriate-scan-conversion-mode))
     (with-appropriate-scan-conversion-mode-internal 
       ,stream-variable ,raster
       #'with-appropriate-scan-conversion-mode)))

(defun with-appropriate-scan-conversion-mode-internal (stream raster continuation)
  (declare (dynamic-extent continuation))
  #-Ignore-for-now (declare (ignore raster))
  #+Imach
  (progn
    (graphics:with-scan-conversion-mode (stream :host-allowed t)
      (funcall continuation stream))
    #+Ignore-for-now
    (cond (raster
	   (sys:system-case 
	     (:macivory
	       (scl:destructuring-bind (width height) (array-dimensions raster)
		 (graphics:with-scan-conversion-mode (stream :host-allowed
							     (and (<= width 8)
								  (<= height 8)))
		   (funcall continuation stream))))
	     (t (funcall continuation stream))))
	  (t (funcall continuation stream))))
  #-Imach
  (funcall continuation stream))

;;; Give m-. a little help.  This whole file is #+Genera.
(eval-when (compile load eval)
  (setf (get 'define-genera-graphics-methods 'zwei:definition-function-spec-finder)
	  (get 'defstruct 'zwei:definition-function-spec-finder)))

(defmacro define-genera-graphics-methods ((name &key using alu (convert-points-p t))
					  &body body)
  (let* ((gf (find-graphics-function name))
	 (name* (gf-spread-function-name gf)))
    `(define-group ,name define-genera-graphics-methods
       (define-graphics-function-method ,name* ((medium clg-medium)
						,@(gf-method-lambda-list gf))
	 ,(let ((body `((with-genera-graphics-context
			  (drawable port medium :using ,using :alu ,alu)
			  (block ,name
			    ,@body)))))
	    (when convert-points-p
	      (do-graphics-function-spread-arguments ((type arg args*) gf)
		(declare (ignore arg))
		(ecase type
		  (point (setf body `((with-device-point
					(device-transformation ,@args*)
					,@body))))
		  (rectangle (setf body `((with-device-rectangle
					    (device-transformation ,@args*)
					    ,@body)))))))
	    `(let ((device-transformation (slot-value medium 'device-transformation))
		   (drawable (slot-value medium 'drawable))
		   (port (slot-value medium 'port)))
	       (progn device-transformation drawable port nil)
	       (with-appropriate-scan-conversion-mode (drawable 'foo)
		 ,@body))))
       (define-graphics-function-method ,name* :around
	   ((medium clg-display-medium) ,@(gf-method-lambda-list gf))
	 ;; Disgusting.  8.2 CLOS doesn't manage to handle IGNORE declarations
	 ;; for ignored keyword args to methods!
	 #+Ignore
	 ;; Disgusting.  PCL uses these arguments, and doesn't fix the declaration.
	 (#-PCL declare #+PCL progn ;; Depends on the fact that IGNORE is also a function
	  (ignore ,@(gf-method-argument-list gf)))
	 #-Ignore
	 (progn ,@(gf-method-argument-list gf))
	 (with-output-protection medium (call-next-method))))))

(scl:defprop define-genera-graphics-methods "Genera graphics implementation for"
	     si:definition-type-name)

;;;
;;; Graphics Operations
;;;

(define-genera-graphics-methods (draw-point :using (ink line-style) :alu alu)
  (graphics:draw-point point-x point-y :stream drawable :alu alu))

(define-genera-graphics-methods (draw-points :using (ink line-style)
					     :convert-points-p nil
					     :alu alu)
  (macrolet ((pop-coordinate (&optional (return-code '(error "Bad number of coordinates ~
							      passed to ~S" 'draw-points)))
	       `(if (endp coordinate-sequence)
		    ,return-code
		    (pop coordinate-sequence))))
    (loop
      (let ((x (pop-coordinate (return)))
	    (y (pop-coordinate)))
	(with-device-point (device-transformation x y)
	  (graphics:draw-point x y :stream drawable :alu alu))))))

(define-genera-graphics-methods (draw-line :using (ink line-style) :alu alu)
  (graphics:draw-line from-x from-y to-x to-y :stream drawable :alu alu))

(define-genera-graphics-methods (draw-lines :using (ink line-style)
					    :convert-points-p nil
					    :alu alu)
  (macrolet ((pop-coordinate (&optional (return-code '(error "Bad number of coordinates ~
							      passed to ~S" 'draw-lines)))
	       `(if (endp coordinate-sequence)
		    ,return-code
		    (pop coordinate-sequence))))
    (loop (let ((from-x (pop-coordinate (return)))
		(from-y (pop-coordinate))
		(to-x (pop-coordinate))
		(to-y (pop-coordinate)))
	    (with-device-point (device-transformation from-x from-y)
	      (with-device-point (device-transformation to-x to-y)
		(graphics:draw-line from-x from-y to-x to-y :stream drawable :alu alu)))))))

(define-genera-graphics-methods (draw-rectangle :using (ink line-style)
						:convert-points-p nil
						:alu alu)
  (if (rectilinear-transformation-p device-transformation)
      (with-device-rectangle (device-transformation x1 y1 x2 y2)
	(when (and filled
		   (reflection-transformation-p device-transformation))
	  ;; --- See comment in x-clg.lisp for what this is all about
	  (incf y1)
	  (incf y2))
	(graphics:draw-rectangle x1 y1 x2 y2 :stream drawable :filled filled) :alu alu)
      (with-stack-list (list x1 y1 x1 y2 x2 y2 x2 y1) 
	(draw-polygon* medium list :closed t :filled filled))))

(define-genera-graphics-methods (draw-polygon :using (ink line-style)
					      :convert-points-p nil :alu alu)
  (macrolet ((pop-coordinate (&optional (return-code
					  '(error "Bad number of coordinates ~
						     passed to ~S" 'draw-polygon)))
	       `(if (endp coordinate-sequence)
		    ,return-code
		    (pop coordinate-sequence))))
    (if closed
	(let ((new-points nil))
	  (loop (let ((x (pop-coordinate
			   (progn 
			     (graphics:draw-polygon (nreverse new-points)
						    :stream drawable :filled filled)
			     (return))))			       
		      (y (pop-coordinate)))
		  (with-device-point (device-transformation x y)
		    (push x new-points)
		    (push y new-points)))))
	(let ((this-x (pop-coordinate (return-from draw-polygon)))
	      (this-y (pop-coordinate)))
	  (with-device-point (device-transformation this-x this-y)
	    (loop (let ((next-x (pop-coordinate (return-from draw-polygon)))
			(next-y (pop-coordinate)))
		    (with-device-point (device-transformation next-x next-y)
		      (graphics:draw-line this-x this-y next-x next-y
					  :stream drawable :alu alu)
		      (setf this-x next-x this-y next-y)))))))))

#+Ignore
(define-genera-graphics-methods (draw-polygon :using (ink line-style)
					      :convert-points-p nil)
  (macrolet ((pop-coordinate (&optional (return-code '(error "Bad number of coordinates ~
							      passed to ~S" 'draw-polygon)))
	       `(if (endp coordinate-sequence)
		    ,return-code
		    (pop coordinate-sequence))))
    (flet ((path-function (drawable)
	     (let ((first-x (pop-coordinate))
		   (first-y (pop-coordinate)))
	       (with-device-point (device-transformation first-x first-y)
		 (graphics:set-current-position first-x first-y :stream drawable)))
	     (loop (let ((x (pop-coordinate (return)))
			 (y (pop-coordinate)))
		     (with-device-point (device-transformation x y)
		       (graphics:draw-line-to x y :stream drawable))))
	     (when closed (graphics:close-path :stream drawable))))
      (graphics:draw-path #'path-function :stream drawable :filled filled))))

(define-genera-graphics-methods (draw-text :using (ink) :alu alu)
  ;; --- still need to handle START and END efficiently
  (let ((genera-character-style
	  ;; what do we do about the weird lexical references to PORT.
	  ;; Also, how do we justify simply assuming *standard-character-set*?
	  (si:backtranslate-font
	    (text-style-mapping port *standard-character-set*
				(medium-text-style medium))
	    (scl:send drawable :display-device-type))))
    (flet ((doit (stuff)
	     (declare (sys:downward-function))
	     (graphics:draw-string stuff point-x point-y :stream drawable
				   :character-style genera-character-style
				   :attachment-y align-y
				   :attachment-x align-x
				   :alu alu)))
      (cond ((characterp string-or-char)
	     (sys:with-stack-array 
	       (string 1 :initial-element string-or-char :element-type 'character)
	       (doit string)))
	    (t
	     (unless end (setq end (length string-or-char)))
	     (sys:with-stack-array (string (- end start)
					   :displaced-to string-or-char
					   :displaced-index-offset start
					   :element-type (array-element-type string-or-char))
	       (doit string)))))))

(define-genera-graphics-methods (draw-ellipse :using (ink line-style) :alu alu)
  (with-two-device-distances (device-transformation
			       radius-1-dx radius-1-dy radius-2-dx radius-2-dy
			       r1dx r1dy r2dx r2dy)
    ;; --- What do we do about rotated ellipses?
    ;; --- I guess we pass in the :rotation, :scale-x, and :scale-y keywords.
    ;; --- Actually, what we really do is to make a three-point transformation
    ;; --- (0,0)->(cx,cy), (0,1)->(cx+dx1,cy+dy1), (1,0)->(cx+dx2,cy+dy2), and
    ;; --- draw the unit circle centered at (0,0), just like the spec says.
    ;; --- Unfortunately, this is broken for some transformations at the moment, so use this
    ;; --- ancient technology for now (no worse than X, anyway). -- rsl
    (unless (and (zerop r2dx) (zerop r1dy))
      ;; Handle rotated circles as a special case
      (let ((r1^2 (+ (* r1dx r1dx) (* r1dy r1dy)))
	    (r2^2 (+ (* r2dx r2dx) (* r2dy r2dy))))
	(if (= r1^2 r2^2)
	    (let ((new-radius (sqrt r1^2)))
	      (unless (integerp new-radius)
		(setf new-radius (floor (+ new-radius 0.5))))
	      (setf r1dx new-radius r1dy 0
		    r2dx 0	  r2dy new-radius))
	    (assert (= r2dx r1dy 0)))))		;Same old failure mode.
    (when (= end-angle 2pi)
      ;; get around Genera bug.
      (setq end-angle graphics:2pi))
    (graphics:draw-ellipse center-x center-y
			   r1dx r2dy
			   :start-angle start-angle :end-angle end-angle
			   :filled filled
			   :stream drawable :alu alu)))


;;; COPY-AREA: have to worry about two DISPLAY-MEDIUM objects, protecting them individually.

;;; Protect the destination, if needed.
(defmethod copy-area :around ((medium clg-display-medium) x y 
			      source left bottom
			      width height
			      &optional (boole boole-1))
  (declare (ignore x y source left bottom width height boole))
  (with-output-protection medium
    (call-next-method)))

;;; Protect the source, if needed.
(defmethod copy-area :around (medium x y 
			      (source clg-display-medium) left bottom
			      width height
			      &optional (boole boole-1))
  (declare (ignore medium x y left bottom width height boole))
  (with-output-protection source
    (call-next-method)))

;;; Copy from a pixmap into a medium.
(defmethod copy-area ((medium clg-medium) x y (source pixmap) left bottom width height
		      &optional (boole boole-1))
  (with-slots (drawable port device-transformation) medium
    (multiple-value-bind (min-x min-y max-x max-y)
	(transform-rectangle* device-transformation x y (+ x width) (+ y height))
      (scl:send drawable :bitblt boole (- max-x min-x) (- max-y min-y)
		(realize-pixmap port source) left bottom
		min-x min-y))))

;;; Copy from a medium into another medium.
;;; --- Should there be a COPY-AREA from ON-GENERA::CLG-DISPLAY-MEDIUM to ON-X:???
(defmethod copy-area ((target clg-display-medium) target-x target-y
		      (source clg-medium) source-x source-y
		      width height
		      &optional (boole boole-1))
  ;; width and height are assumed to be unchanged under the various
  ;; transformations.  We use them, however, to realize a rectangle and
  ;; transform the rectangle to get it's anchor, which may be different
  ;; than transforming the anchor point.
  (let* ((target-drawable (slot-value target 'drawable))
	 (source-drawable (slot-value source 'drawable))
	 (target-device-transformation (insured-device-transformation target))
	 ;; --- kludge.  Should probably implement a method for 
	 ;; insured-device-transformation on pixmap mediums.
	 (source-device-transformation
	   (if (typep source 'display-medium)
	       (insured-device-transformation source)
	       (device-transformation source))))
    (multiple-value-setq (target-x target-y)
      (careful-transform-rectangle*
	target-device-transformation
	target-x target-y (+ target-x width) (+ target-y height)))
    (multiple-value-setq (source-x source-y)
      (careful-transform-rectangle*
	source-device-transformation
	source-x source-y (+ source-x width) (+ source-y height)))
    ;; make width and/or height be negative to interact with BITBLT
    ;; correctly in the event these are the same drawables.
    (let ((width  (if (<= target-x source-x) (+ width)  (- width)))
	  (height (if (<= target-y source-y) (+ height) (- height))))
      (tv:bitblt-from-sheet-to-sheet
	boole (round width) (round height)	;--- round???
	source-drawable (integerize-coordinate source-x) (integerize-coordinate source-y)
	target-drawable (integerize-coordinate target-x) (integerize-coordinate target-y)))))

;;; X also has a way to copy from a medium into a pixmap.  Should this be allowed here?
(defmethod copy-area ((pixmap pixmap) dst-x dst-y 
		      (medium clg-display-medium) src-x src-y width height
		      &optional (boole boole-1))
  ;;; This all only works for pixels coordinate systems.
  (with-slots (drawable gcontext port device-transformation) medium
    (multiple-value-bind (device-min-x device-min-y device-lim-x device-lim-y)
	(careful-transform-rectangle* device-transformation
				      src-x src-y (+ src-x width)
				      (+ src-y height))
      (let* ((device-width (- device-lim-x device-min-x))
	     (device-height (- device-lim-y device-min-y))
	     (raster (realize-pixmap port pixmap)))
	(scl:send drawable :bitblt-from-sheet boole
		  (round device-width) (round device-height)	;--- round???
		  (integerize-coordinate device-min-x) (integerize-coordinate device-min-y)
		  raster (integerize-coordinate dst-x) (integerize-coordinate dst-y))))))
