;;; -*- Mode: Lisp; Package: SILICA; Base: 10.; Syntax: Common-Lisp -*-
;;;
;;; Copyright (c) 1989, 1990 by Xerox Corporation.  All rights reserved. 
;;;

(in-package "SILICA")

;;;
;;; CURSORS
;;;

(defclass cursor ()
    ((width	:initarg :width		:accessor width)	
     (height	:initarg :height	:accessor height)
     (x-offset	:initarg :x-offset	:accessor x-offset)
     (y-offset	:initarg :y-offset	:accessor y-offset)
     (mask      :initarg :mask    	:accessor mask)
     (image     :initarg :image         :accessor image)))

(defun make-cursor (&rest args)
  (declare (dynamic-extent args))
  (apply #'make-instance 'cursor args))

(defmethod initialize-instance :after 
	   ((cursor cursor) &key width height mask-data image-data
	    &allow-other-keys)
  (setf (mask cursor) 
	(make-pixmap :width width :height height
		     :data mask-data))
  (setf (image cursor)
	(make-pixmap :width width :height height
		     :data image-data)))

;;;
;;; Portable cursors from Deli
;;;

(defvar *cursors*
  `(
    :get-around-bug-in-pcl		; pcl bug
    ,(make-cursor)
    :dot
    ,(make-cursor 
       :width 16 :height 16 :x-offset 16 :y-offset 16 
       :image-data
       (make-array 32 :element-type '(unsigned-byte 8)
		   :initial-contents
		   '( 0 0 0 0 0 0 3 192 15 240 15 240 31 248 31 248 
		     31 248 31 248 15 240 15 240 3 192 0 0 0 0 0 0))
       :mask-data
       (make-array 32 :element-type '(unsigned-byte 8)
		   :initial-contents 
		   '( 0 0 0 0 7 224 31 248 31 248 63 252 63 252 63 252 63 252
		     63 252 63 252 31 248 31 248 7 224 0 0 0 0)))
    :circle
    ,(make-cursor
       :width 16 :height 16 :x-offset 16 :y-offset 16 
       :image-data
       (make-array 
	 32 :element-type '(unsigned-byte 8)
	 :initial-contents
	 '( 0 0 3 192 15 240 31 248 60 60 56 28 112 14 112 14 112 14
	   112 14 56 28 60 60 31 248 15 240 3 192 0 0))
       :mask-data
       (make-array
	 32 :element-type '(unsigned-byte 8)
	 :initial-contents 
	 '( 7 224 31 248 63 252 127 254 127 254 252 63 248 31 248 31
	   248 31 248 31 252 63 127 254 127 254 63 252 31 248 7 224)))
    :up-down-arrow
    ,(make-cursor
       :width 16 :height 16 :x-offset 16 :y-offset 16 
       :image-data
       (make-array
	 32 :element-type '(unsigned-byte 8)
	 :initial-contents
	 '( 0 0 1 128 3 192 7 224 13 176 25 152 1 128 1 128 1 128 1
	   128 25 152 13 176 7 224 3 192 1 128 0 0)) 
       :mask-data
       (make-array
	 32 :element-type '(unsigned-byte 8)
	 :initial-contents
	 '( 3 192 7 224 15 240 31 248 63 252 63 252 63 252 3 192 3 192
	   63 252 63 252 63 252 31 248 15 240 7 224 3 192)))
    :left-right-arrow
    ,(make-cursor
       :width 16 :height 16 :x-offset 16 :y-offset 16 
       :image-data
       (make-array
	 32 :element-type '(unsigned-byte 8)
	 :initial-contents
	 '( 0 0 0 0 0 0 4 96 12 48 24 24 48 12 127 254 127 254 48 12
	   24 24 12 48 4 32 0 0 0 0 0 0))
       :mask-data
       (make-array
	 32 :element-type '(unsigned-byte 8)
	 :initial-contents
	 '( 0 0 0 0 14 112 30 120 62 124 126 126 255 255 255 255 255
	   255 255 255 126 126 62 124 30 120 14 112 0 0 0 0))) 
    :right-ptr
    ,(make-cursor
       :width 16 :height 16 :x-offset 16 :y-offset 16 
       :image-data
       (make-array
	 32 :element-type '(unsigned-byte 8)
	 :initial-contents
	 '( 0 0 0 8 0 24 0 56 0 120 0 248 1 248 3 248 7 248 0 248 0
	   216 1 136 1 128 3 0 3 0 0 0))
       :mask-data
       (make-array
	 32 :element-type '(unsigned-byte 8)
	 :initial-contents
	 '( 0 12 0 28 0 60 0 124 0 252 1 252 3 252 7 252 15 252 15 252
	   1 252 3 220 3 204 7 128 7 128 3 0)))
    :left-ptr
    ,(make-cursor
       :width 16 :height 16 :x-offset 16 :y-offset 16 
       :image-data
       (make-array
	 32 :element-type '(unsigned-byte 8)
	 :initial-contents
	 '( 0 0 16 0 24 0 28 0 30 0 31 0 31 128 31 192 31 224 31 0 27
	   0 17 128 1 128 0 192 0 192 0 0))
       :mask-data
       (make-array
	 32 :element-type '(unsigned-byte 8)
	 :initial-contents
	 '( 48 0 56 0 60 0 62 0 63 0 63 128 63 192 63 224 63 240 63
	   240 63 128 59 192 51 192 1 224 1 224 0 192))) 
    :cross
    ,(make-cursor
       :width 16 :height 16 :x-offset 16 :y-offset 16 
       :image-data
		   
		     
       (make-array
	 32 :element-type '(unsigned-byte 8)
	 :initial-contents
	 '( 0 0 1 128 1 128 1 128 1 128 1 128 1 128 127 254 127 254 1
	   128 1 128 1 128 1 128 1 128 1 128 0 0)) 
       :mask-data
       (make-array
	 32 :element-type '(unsigned-byte 8)
	 :initial-contents
	 '( 3 192 3 192 3 192 3 192 3 192 3 192 255 255 255 255 255
	   255 255 255 3 192 3 192 3 192 3 192 3 192 3 192)))
    :target
    ,(make-cursor
       :width 16 :height 16 :x-offset 16 :y-offset 16 
       :image-data
       (make-array
	 32 :element-type '(unsigned-byte 8)
	 :initial-contents
	 '( 0 0 1 128 1 128 7 224 9 144 17 136 19 200 126 126 126 126
	   19 200 17 136 9 144 7 224 1 128 1 128 0 0))
       :mask-data
       (make-array
	 32 :element-type '(unsigned-byte 8)
	 :initial-contents
	 '( 3 192 3 192 15 240 31 248 63 252 63 252 255 255 255 255
	   255 255 255 255 63 252 63 252 31 248 15 240 3 192 3 192))) 
    :arrow
    ,(make-cursor
       :width 16 :height 16 :x-offset 16 :y-offset 16 
       :image-data
       (make-array
	 32 :element-type '(unsigned-byte 8)
	 :initial-contents
	 '( 0 0 1 128 3 192 7 224 1 128 17 136 49 140 127 254 127 254
	   49 140 17 136 1 128 7 224 3 192 1 128 0 0))
       :mask-data
       (make-array
	 32 :element-type '(unsigned-byte 8)
	 :initial-contents
	 '( 3 192 3 224 7 224 15 240 23 232 59 220 255 255 255 255 255
	   255 255 255 59 220 23 232 15 240 7 224 3 192 3 192)))))



#||

;;;
;;;  Auxilliary stuff
;;;

(defun read-deli-cursor-file (filename)
  (let ((stream (open filename :direction :input))
	(*read-base* 10)
	(*package* (find-package 'wsii))
	result)
    (declare (special *read-base* *package*))
    (unwind-protect
	 (let ((x-hot (read stream nil nil))
	       (y-hot (read stream nil nil))
	       (cursor-file (read stream nil nil))
	       (mask-file (read stream nil nil)))
	   (unless (and x-hot y-hot)
	     (error "cursor hot-spot not specified in ~s." filename))
	   (unless cursor-file
	     (error "cursor pixrect file not specified in ~s." filename))
	   (unless mask-file
	     (error "cursor mask pixrect file not specified in ~s." filename))
	   (setf cursor-file 
		 (merge-pathnames cursor-file *deli-cursor-library-directory*))
	   (setf mask-file 
		 (merge-pathnames mask-file *deli-cursor-library-directory*))
	   (if (and (probe-file cursor-file)
		    (probe-file mask-file))
	       (let ((mask-pixrect (make-pixrect-from-file mask-file))
		     (cursor-pixrect (make-pixrect-from-file cursor-file)))
		 (unless (and (= (pixmap-width mask-pixrect)
				 (pixmap-width cursor-pixrect))
			      (= (pixmap-height mask-pixrect)
				 (pixmap-height cursor-pixrect)))
		   (error "Cursor ~s mask ~s must have same dimensions."
			  cursor-file mask-file))
		 (setq result 
		       (make-cursor :width (pixmap-width mask-pixrect)
				    :height (pixmap-height mask-pixrect)
				    :x-offset x-hot :y-offset y-hot
				    :image cursor-pixrect
				    :mask  mask-pixrect)))
	       (error 
		 "either cursor file ~s or mask file ~s is not readable."
		 cursor-file mask-file)))
      (close stream))
    result))

 ||#

