;;; -*- Mode: Lisp; Package: CLX-UTILS; Syntax: Common-Lisp -*-
;;;
;;;
;;; Copyright (c) 1989, 1990 by Xerox Corporation.  All rights reserved. 
;;;

(in-package "CLX-UTILS")

;;; Rubberbanding routines.

;; Much of the following code was originally written by Ramana Rao, and later
;; hacked by smL.  In particular, the original polled for the mouse position,
;; this version used motion hints and event-case.  There is a slight loss in
;; performance, but the drop in network traffic when the mouse isn't moving is
;; enough of a win to make up for it.  It still works as fast as it needs to.

;; Sirius independent functions for doing the work.  That way Ramana can use the
;; same code.

;; NOTE:  In the case that these functions cannot grab the pointer, they return
;; NIL. 


(defun clx-prompt-for-screen-location (root display cursor
					    &aux (grab-status nil) (x 0) (y 0))
  (declare (type xlib:display display)
	   (type xlib:window root)
	   (type xlib:cursor cursor)
	   (fixnum x y))
  (unwind-protect
       (progn
	 (setq grab-status
	       (xlib:grab-pointer root
				  #.(xlib:make-event-mask :button-release)
				  ;; ?? Should do this, but debugging
				  ;; is easier without it. 
				  ;; :sync-keyboard-p t
				  :cursor cursor))
	 (case grab-status
	   (:success
	    (xlib:event-case (display :force-output-p t)
	      (:button-release (root-x root-y)
		(setq x root-x y root-y)
		t)))
	   (otherwise
	    (return-from clx-prompt-for-screen-location nil))))
    (when (eq grab-status :success)
      (xlib:ungrab-pointer (xlib:window-display root)))
    (xlib:display-force-output display))
  (values x y))

(defun clx-move-frame (window gc x y w h &optional last-segments) 
  (declare (type xlib:window window)
	   (type xlib:gcontext gc)
	   (fixnum x y w h)
	   (type list last-segments))
  (macrolet ((relist (list &rest elements)
	       ;; Allow us to reuse a list instead of consing up a new one.
	       ;; Take from PCL.
	       (let ((lv (gensym)))
		 `(let ((,lv ,list))
		   ,@(mapcar #'(lambda (x) `(progn (setf (car ,lv) ,x)
					     (pop ,lv)))
		      elements)))))
    (let ((x2 (+ x w))
	  (y2 (+ y h)))
      (declare (type fixnum x2 y2))
      (unless (and last-segments
		   (= x  (the fixnum (first last-segments)))
		   (= y  (the fixnum (second last-segments)))
		   (= x2 (the fixnum (third last-segments)))
		   (= y2 (the fixnum (eighth last-segments))))
	(cond (last-segments
	       (xlib:draw-segments window gc last-segments)
	       (relist last-segments x  y  x2 y
		                     x2 y  x2 y2
				     x2 y2 x  y2
				     x  y2 x  y ))
	      (t (setq last-segments (list x  y  x2 y
					   x2 y  x2 y2
					   x2 y2 x  y2
					   x  y2 x  y ))))
	(xlib:draw-segments window gc last-segments))
      last-segments)))

(defun clx-prompt-for-screen-region-location (root display cursor gc w h)
  (declare (type xlib:window root)
	   (type xlib:display display)
	   (type xlib:cursor cursor)
	   (type xlib:gcontext gc)
	   (fixnum w h))
  (let (;; For tracking region
	(x 0)
	(y 0)
	(move-frame-state nil)
	(grab-status nil))
    (declare (fixnum  x y))
    (unwind-protect
	 
	 (progn
	   (setq grab-status
		 (xlib:grab-pointer root
				    #.(xlib:make-event-mask
				       :button-release
				       :pointer-motion
				       :pointer-motion-hint)
				    ;; ?? Should do this, but debugging
				    ;; is easier without it.
				    ;; :sync-keyboard-p t
				    :cursor cursor))
	   (multiple-value-setq (x y)
	     (xlib:query-pointer root))
	   (setq move-frame-state
		 (clx-move-frame root gc x y w h move-frame-state))
	   (case grab-status
	     (:success
	      (xlib:event-case (display :discard-p t :force-output-p t)
		(:button-release () t)	;Exit input loop
		(:motion-notify (root-x root-y hint-p)
		  (setq x root-x y root-y)
		  (setq move-frame-state
			(clx-move-frame root gc x y w h move-frame-state))
		  ;; Enable motion events
		  (when hint-p (xlib:query-pointer root))
		  (xlib:display-force-output display)
		  nil)))		;To continue input loop
	     (otherwise (return-from
			 clx-prompt-for-screen-region-location nil))))
      (when move-frame-state
	(clx-move-frame root gc 0 0 0 0 move-frame-state)) ; Take down the box
      (when (eq grab-status :success)
	(xlib:ungrab-pointer display))
      (xlib:display-force-output display))
    (values x y)))

(defun clx-prompt-for-screen-region (root display gc
					  upper-left-cursor
					  lower-left-cursor
					  upper-right-cursor
					  lower-right-cursor
					  &optional x y w h)
  (declare (type xlib:window root)
	   (type xlib:display display)
	   (type xlib:gcontext gc)
	   (type xlib:cursor upper-left-cursor lower-left-cursor
		             upper-right-cursor lower-right-cursor)
	   (fixnum x y w h))
  (let (current-cursor
	(move-frame-state nil)
	(origx 0) (origy 0) (origw 0) (origh 0) ;For resizing the frame
	(clamp-v :none) (clamp-h :none)
	(min-w 0) (min-h 0)
	(grab-status nil))
    (declare (fixnum origx origy origw origh))
    (flet ((resize-frame (cx cy)
	     (let (delta)
	       (case clamp-v
		 (:top (if (< (- h (- cy y)) min-h)
			   (setq y (- (+ y h) min-h)
				 h min-h
				 clamp-v :none)
			   (setq h (+ h (- y cy))
				 y cy)))
		 (:bottom (if (< (- cy y) min-h)
			      (setq clamp-v :none
				    h min-h)
			      (setq h (- cy y))))
		 (:none (cond ((<= cy y) 
			       (setq y cy
				     h (+ origy
					  (- origh cy))
				     clamp-v :top))
			      ((>= cy (+ y (- h 1)))
			       (setq y origy
				     h (+ 1 (- cy origy))
				     clamp-v :bottom)))))
	       (case clamp-h
		 (:left (setq delta (- cx x))
			(when (< (- w delta) min-w)
			  (setq delta (- w min-w)
				clamp-h :none))
			(incf x delta)
			(decf w delta))
		 (:right (setq delta (- cx x w))
			 (when (< (+ w delta) min-w)
			   (setq delta (- min-w w)
				 clamp-h :none))
			 (incf w delta))
		 (:none (cond ((<= cx x)
			       (setq x cx
				     w (+ origx
					  (- origw cx))
				     clamp-h :left))
			      ((>= cx (+ x
					 (- w 1)))
			       (setq x origx
				     w (+ 1 (- cx origy))
				     clamp-h :right)))))
	       (setq move-frame-state
		     (clx-move-frame root gc x y w h move-frame-state))))
	   (update-cursor ()
	     (let ((new-cursor (case clamp-v
				 (:top (case clamp-h
					 (:left upper-left-cursor)
					 (:right upper-right-cursor)
					 (:none current-cursor)))
				 (:bottom (case clamp-h
					    (:left lower-left-cursor)
					    (:right lower-right-cursor)
					    (:none current-cursor)))
				 (:none current-cursor))))
	       (unless (eq current-cursor new-cursor)
		 (xlib:change-active-pointer-grab display
						  #.(xlib:make-event-mask
						      :button-press
						      :button-release
						      :pointer-motion
						      :pointer-motion-hint)
						  new-cursor)
		 (setq current-cursor new-cursor)))))
      (unwind-protect
	   (progn
	     (setq grab-status
		   (xlib:grab-pointer root #.(xlib:make-event-mask
					       :button-press
					       :button-release
					       :pointer-motion
					       :pointer-motion-hint)
				      ;; ?? Should do this, but debugging
				      ;; is easier without it.
				      ;; :sync-keyboard-p t
				      :cursor upper-left-cursor))
	     (setq current-cursor upper-left-cursor)
	     (case grab-status
	       (:success
		;; Wait for a button to go down to lock down the upper-left
		;; corner.
		(xlib:event-case (display :discard-p t)
		  (:button-press () t)
		  (:motion-notify (root-x root-y hint-p)
		    (setq x root-x y root-y)
		    (setq move-frame-state
			  (clx-move-frame root gc x y w h move-frame-state))
		    (when hint-p
		      (xlib:query-pointer root))
		    (xlib:display-force-output display)
		    nil))
		;; Rubberband out a box until all the buttons are up.
		(let ((lcx 0) (lcy 0))
		  (declare (fixnum lcx lcy))
		  (xlib:warp-pointer root (+ x (floor w 2))
				          (+ y (floor h 2)))
		  (setq origx x origy y origw w origh h)
		  (xlib:event-case (display :discard-p t)
		    (:button-release () t)
		    (:motion-notify (root-x root-y hint-p)
		      (unless (and (= root-x lcx) (= root-y lcy))
			(resize-frame root-x root-y)
			(update-cursor)
			(setq lcx root-x lcy root-y))
		      ;; Reenable motion events
		      (when hint-p
			(xlib:query-pointer root))
		      (xlib:display-force-output display)
		      nil))))
	       (otherwise
		(return-from clx-prompt-for-screen-region nil))))
	(when move-frame-state
	  (clx-move-frame root gc 0 0 0 0 move-frame-state))
	(when (eq grab-status :success)
	  (xlib:ungrab-pointer (xlib:window-display root)))
	(xlib:display-force-output display))
      (values x y w h))))
