;;; -*- Mode: LISP; Syntax: Common-lisp; Package: SILICA; Base: 10; Lowercase: Yes -*-

;; $fiHeader: regions.lisp,v 1.6 91/03/29 18:02:06 cer Exp $

;;; This had been in the CLIM-UTILS package in 1.0.

(in-package "SILICA")

"Copyright (c) 1990, 1991 Symbolics, Inc.  All rights reserved."

;;; Generic Functions

(defgeneric transform-region (transformation region))
(defgeneric untransform-region (transformation region))

(defgeneric point-position* (point)
  (declare (values x y)))
(defgeneric point-x (point))
(defgeneric point-y (point))

;;--- Many of these methods could stand to be written...
(defgeneric region-equal (region1 region2))
(defgeneric region-contains-point*-p (region x y))
(defgeneric region-contains-region-p (region1 region2))
(defgeneric region-intersects-region-p (region1 region2))

(defgeneric region-set-function (region))
(defgeneric region-set-regions (region &key normalize))
(defgeneric map-over-region-set-regions (function region &key normalize)
  (declare (dynamic-extent function)))

(defgeneric region-union (region1 region2))
(defgeneric region-intersection (region1 region2))
(defgeneric region-difference (region1 region2))

(defgeneric polyline-closed (polyline))
(defgeneric polygon-points (polygon))
(defgeneric map-over-polygon-coordinates (function polygon)
  (declare (dynamic-extent function)))
(defgeneric map-over-polygon-segments (function polygon)
  (declare (dynamic-extent function)))

(defgeneric line-start-point (line))
(defgeneric line-end-point (line))
(defgeneric line-start-point* (line))
(defgeneric line-end-point* (line))

(defgeneric rectangle-min-point (rectangle))
(defgeneric rectangle-max-point (rectangle))
(defgeneric rectangle-edges* (rectangle)
  (declare (values min-x min-y max-x max-y)))
(defgeneric rectangle-min-x (rectangle))
(defgeneric rectangle-min-y (rectangle))
(defgeneric rectangle-max-x (rectangle))
(defgeneric rectangle-max-y (rectangle))
(defgeneric rectangle-width (rectangle))
(defgeneric rectangle-height (rectangle))
(defgeneric rectangle-size (rectangle)
  (declare (values width height)))

(defgeneric ellipse-center-point (ellipse))
(defgeneric ellipse-center-point* (ellipse))
(defgeneric ellipse-radii (ellipse)
  (declare (values radius-1-dx radius-1-dy radius-2-dx radius-2-dy)))
(defgeneric ellipse-start-angle (ellipse))
(defgeneric ellipse-end-angle (ellipse))

(defgeneric opacity-value (opacity))

(defgeneric bounding-rectangle* (region)
  (declare (values left top right bottom)))
(defgeneric bounding-rectangle-set-edges (region left top right bottom)
  (declare (values region)))
(defgeneric bounding-rectangle-set-position* (region x y)
  (declare (values region)))
(defgeneric bounding-rectangle-set-size (region width height)
  (declare (values region)))

(defmacro define-symmetric-region-method (name (region1 region2) &body body)
  `(progn
     (defmethod ,name (,region1 ,region2) ,@body)
     (defmethod ,name (,region2 ,region1) ,@body)))

(defmacro fix-rectangle (left top right bottom)
  `(values (the fixnum (floor ,left))
	   (the fixnum (floor ,top))
	   (the fixnum (ceiling ,right))
	   (the fixnum (ceiling ,bottom))))


(defclass design () ())


(defclass opacity (design) ())

(defmethod print-object ((design opacity) stream)
  (print-unreadable-object (design stream :type t :identity t)
    (format stream "~D" (opacity-value design))))

;; Opacities are unbounded and uniform, so transformations are a no-op
(defmethod transform-region ((transformation transformation) (opacity opacity)) opacity)


#||
;;; Colors are currently defined in silica/paints.lisp
(defclass color (design) ())

;; Colors are unbounded and uniform, so transformations are a no-op
(defmethod transform-region ((transformation transformation) (color color)) color)
||#


;;; Regions

(defclass region (design) ())

(defmethod transform-region ((transformation identity-transformation) region) region)

(defmethod untransform-region ((transformation identity-transformation) region) region)

(defmethod untransform-region ((transformation transformation) region)
  (transform-region (invert-transformation transformation) region))


;;; Nowhere

(defclass nowhere (opacity region) ())

(defmethod make-load-form ((nowhere nowhere))
  '+nowhere+)

(defmethod region-equal ((nowhere1 nowhere) (nowhere2 nowhere)) t)

(defmethod region-contains-point*-p ((nowhere nowhere) x y)
  (declare (ignore x y))
  nil)

(defmethod region-contains-region-p ((nowhere nowhere) (region region)) nil)
(defmethod region-contains-region-p ((region region) (nowhere nowhere)) t)
(defmethod region-contains-region-p ((nowhere1 nowhere) (nowhere2 nowhere)) t)

(define-symmetric-region-method region-intersects-region-p ((nowhere nowhere) (region region))
  nil)
(defmethod region-intersects-region-p ((nowhere1 nowhere) (nowhere2 nowhere)) nil)

(defmethod transform-region (transformation (region nowhere))
  (declare (ignore transformation))
  region)

(defmethod opacity-value ((design nowhere)) 0f0)

(defvar +nowhere+ (make-instance 'nowhere))


;;; Everywhere

(defclass everywhere (opacity region) ())

(defmethod make-load-form ((everywhere everywhere))
  '+everywhere+)

(defmethod region-equal ((everywhere1 everywhere) (everywhere2 everywhere)) t)

(defmethod region-contains-point*-p ((everywhere everywhere) x y) 
  (declare (ignore x y))
  t)

(defmethod region-contains-region-p ((everywhere everywhere) (region region)) t)
(defmethod region-contains-region-p ((region region) (everywhere everywhere)) nil)

(define-symmetric-region-method region-intersects-region-p
				((everywhere everywhere) (region region))
  t)

(defmethod transform-region (transformation (region everywhere))
  (declare (ignore transformation))
  region)

(defmethod opacity-value ((design everywhere)) 1f0)

(defvar +everywhere+ (make-instance 'everywhere))


;;; Points

(defclass point (region) ())

(defmethod print-object ((point point) stream)
  (print-unreadable-object (point stream :type t :identity t)
    (format stream "(~D,~D)" (point-x point) (point-y point))))


(defclass standard-point (point)
    ((x :initarg :x :accessor point-x :type real)
     (y :initarg :y :accessor point-y :type real)))

(define-constructor make-point-1 standard-point (x y)
		    :x x :y y)

(defun make-point (x y)
  (make-point-1 x y))

(defmethod make-load-form ((point standard-point))
  (with-slots (x y) point
    `(make-point ,x ,y)))

(defmethod point-position* ((point standard-point))
  (with-slots (x y) point
    (values x y)))

(defmethod region-equal ((point1 standard-point) (point2 standard-point))
  (with-slots ((x1 x) (y1 y)) point1
    (with-slots ((x2 x) (y2 y)) point2
      (and (= x1 x2) (= y1 y2)))))

(defmethod region-contains-point*-p ((point standard-point) x y)
  (with-slots ((px x) (py y)) point
    (and (= px x) (= py y))))

(defmethod region-contains-region-p ((point1 standard-point) (point2 standard-point))
  (with-slots ((x1 x) (y1 y)) point1
    (with-slots ((x2 x) (y2 y)) point2
      (and (= x1 x2) (= y1 y2)))))

(defmethod region-contains-region-p ((region region) (point standard-point))
  (with-slots (x y) point
    (region-contains-point*-p region x y)))

(defmethod region-intersects-region-p ((point1 standard-point) (point2 standard-point))
  (with-slots ((x1 x) (y1 y)) point1
    (with-slots ((x2 x) (y2 y)) point2
      (and (= x1 x2) (= y1 y2)))))

(define-symmetric-region-method region-intersects-region-p
				((point standard-point) (region region))
  (with-slots ((px x) (py y)) point
    (region-contains-point*-p region px py)))

(defmethod region-intersection ((point1 standard-point) (point2 standard-point))
  (with-slots ((x1 x) (y1 y)) point1
    (with-slots ((x2 x) (y2 y)) point2
      (if (and (= x1 x2) (= y1 y2)) point1 +nowhere+))))

(defmethod transform-region (transformation (point standard-point))
  (with-slots (x y) point
    (multiple-value-bind (x y)
	(transform-point* transformation x y)
      (make-point-1 x y))))

(defmethod bounding-rectangle* ((point standard-point))
  (with-slots (x y) point
    (fix-rectangle x y (1+ x) (1+ y))))


;;; Paths

(defclass path (region) ())


(defclass polyline (path) ())


(defclass line (polyline) ())

(defmethod print-object ((line line) stream)
  (print-unreadable-object (line stream :type t :identity t)
    (multiple-value-bind (start-x start-y) (line-start-point* line)
      (multiple-value-bind (end-x end-y) (line-end-point* line)
	(format stream "(~D,~D)->(~D,~D)" start-x start-y end-x end-y)))))


(defclass standard-line (line)
    ((start-x :initarg :start-x :type real)
     (start-y :initarg :start-y :type real)
     (end-x :initarg :end-x :type real)
     (end-y :initarg :end-y :type real)
     (points :type simple-vector :initarg :points :reader polygon-points)))

(define-constructor make-line-1 standard-line (start-x start-y end-x end-y points)
		    :start-x start-x :start-y start-y :end-x end-x :end-y end-y :points points)

(defun make-line (start-point end-point)
  (make-line-1 (point-x start-point) (point-y start-point)
	       (point-x end-point) (point-y end-point)
	       (vector start-point end-point)))

(define-constructor make-line*-1 standard-line (start-x start-y end-x end-y)
		    :start-x start-x :start-y start-y :end-x end-x :end-y end-y)

(defun make-line* (start-x start-y end-x end-y)
  (make-line*-1 start-x start-y end-x end-y))

(defmethod make-load-form ((line standard-line))
  `(make-line ',(line-start-point line) ',(line-end-point line)))

(defmethod slot-unbound (class (line standard-line) (slot (eql 'points)))
  (declare (ignore class))
  (with-slots (points start-x start-y end-x end-y) line
    (setf points (vector (make-point start-x start-y) (make-point end-x end-y)))))

(defmethod line-start-point ((line standard-line))
  (with-slots (points) line
    (svref points 0)))

(defmethod line-start-point* ((line standard-line))
  (with-slots (start-x start-y) line
    (values start-x start-y)))

(defmethod line-end-point ((line standard-line))
  (with-slots (points) line
    (svref points 1)))

(defmethod line-end-point* ((line standard-line))
  (with-slots (end-x end-y) line
    (values end-x end-y)))

(defmethod polyline-closed ((line standard-line))
  nil)

(defmethod map-over-polygon-coordinates (function (line standard-line))
  (with-slots (start-x start-y end-x end-y) line
    (funcall function start-x start-y)
    (funcall function end-x end-y)
    nil))

(defmethod map-over-polygon-segments (function (line standard-line))
  (with-slots (start-x start-y end-x end-y) line
    (funcall function start-x start-y end-x end-y)
    nil))

(defmethod region-equal ((line1 standard-line) (line2 standard-line))
  (with-slots ((sx1 start-x) (sy1 start-y) (ex1 end-x) (ey1 end-y)) line1
    (with-slots ((sx2 start-x) (sy2 start-y) (ex2 end-x) (ey2 end-y)) line2
      (or (and (= sx1 sx2) (= sy1 sy2) (= ex1 ex2) (= ey1 ey2))
	  (and (= sx1 ex2) (= sy1 ey2) (= ex1 sx2) (= ey1 sy2))))))

;; By using perpendicular-distance from line instead of slope and intercept
;; we don't have to worry about divide by zero in slope and we're also more
;; robust against roundoff error.
(defmethod region-contains-point*-p ((line standard-line) x y)
  (with-slots (start-x start-y end-x end-y) line
    (let ((x1 start-x) (y1 start-y) (x2 end-x) (y2 end-y))
      (when (or (<= x1 x x2)
		(>= x1 x x2))
	(= (+ (* (- y2 y1) x)
	      (* (- x1 x2) y))
	   (- (* x1 y2) (* x2 y1)))))))

(defmethod region-contains-region-p ((line1 standard-line) (line2 standard-line))
  (with-slots (start-x start-y end-x end-y) line2
    (and (region-contains-point*-p line1 start-x start-y)
	 (region-contains-point*-p line1 end-x end-y))))

(defmethod region-intersects-region-p ((line1 standard-line) (line2 standard-line))
  (with-slots ((sx1 start-x) (sy1 start-y) (ex1 end-x) (ey1 end-y)) line1
    (with-slots ((sx2 start-x) (sy2 start-y) (ex2 end-x) (ey2 end-y)) line2
      (let ((sx1 sx1) (sy1 sy1) (ex1 ex1) (ey1 ey1)
	    (sx2 sx2) (sy2 sy2) (ex2 ex2) (ey2 ey2))
	(and (>= (max sx2 ex2) (min sx1 ex1))
	     (>= (max sx1 ex1) (min sx2 ex2))
	     (let ((dx1 (- ex1 sx1)) (dy1 (- ey1 sy1))
		   (dx2 (- ex2 sx2)) (dy2 (- ey2 sy2)))
	       (and (= (* dx1 dy2) (* dx2 dy1)) ;slopes equal
		    (= (* dx1 (- sy1 sy2)) (* dy1 (- sx1 sx2))))))))))

(defmethod region-intersection ((line1 standard-line) (line2 standard-line))
  (if (region-intersects-region-p line1 line2)
      (with-slots ((sx1 start-x) (sy1 start-y) (ex1 end-x) (ey1 end-y)) line1
	(with-slots ((sx2 start-x) (sy2 start-y) (ex2 end-x) (ey2 end-y)) line2
	  (make-line* (max sx1 sx2) (max sy1 sy2) (min ex1 ex2) (min ey1 ey2))))
      +nowhere+))

(defmethod transform-region (transformation (line standard-line))
  (with-slots (start-x start-y end-x end-y) line
    (multiple-value-bind (sx sy)
	(transform-point* transformation start-x start-y)
      (multiple-value-bind (ex ey)
	  (transform-point* transformation end-x end-y)
	(make-line* sx sy ex ey)))))

(defmethod bounding-rectangle* ((line standard-line))
  (with-slots (start-x start-y end-x end-y) line
    (fix-rectangle (min start-x end-x) (min start-y end-y)
		   (max start-x end-x) (max start-y end-y))))


;;; Areas

(defclass area (region) ())


(defclass polygon (area) ())


(defclass rectangle (polygon) ())

(defmethod print-object ((rectangle rectangle) stream)
  (print-unreadable-object (rectangle stream :type t :identity t)
    (multiple-value-bind (left top right bottom)
	(rectangle-edges* rectangle)
      (format stream "/x ~D:~D y ~D:~D/" left right top bottom))))

;;; Glue together the various children of RECTANGLE.
(defmethod region-intersection ((rect1 rectangle) (rect2 rectangle))
  (multiple-value-bind (left1 top1 right1 bottom1) (bounding-rectangle* rect1)
    (multiple-value-bind (left2 top2 right2 bottom2) (bounding-rectangle* rect2)
      (or (ltrb-intersection left1 top1 right1 bottom1
			     left2 top2 right2 bottom2)
	  +nowhere+))))

(defmethod region-union ((rect1 rectangle) (rect2 rectangle))
  (multiple-value-bind (left1 top1 right1 bottom1) (bounding-rectangle* rect1)
    (multiple-value-bind (left2 top2 right2 bottom2) (bounding-rectangle* rect2)
      (let ((new-rectangles (ltrb-union left1 top1 right1 bottom1
					left2 top2 right2 bottom2)))
	(if (= (length new-rectangles) 1)
	    (first new-rectangles)
	    (apply #'make-rectangle-set new-rectangles))))))

(defmethod region-difference ((rect1 rectangle) (rect2 rectangle))
  (multiple-value-bind (left1 top1 right1 bottom1) (bounding-rectangle* rect1)
    (multiple-value-bind (left2 top2 right2 bottom2) (bounding-rectangle* rect2)
      (let ((new-rectangles (ltrb-difference left1 top1 right1 bottom1
					     left2 top2 right2 bottom2)))
	(if new-rectangles
	    (if (= (length new-rectangles) 1)
		(first new-rectangles)
	        (apply #'make-rectangle-set new-rectangles))
	    +nowhere+)))))

(defclass standard-rectangle (rectangle)
    ((min-x :initarg :min-x :reader rectangle-min-x :type real)
     (min-y :initarg :min-y :reader rectangle-min-y :type real)
     (max-x :initarg :max-x :reader rectangle-max-x :type real)
     (max-y :initarg :max-y :reader rectangle-max-y :type real)
     (points :type simple-vector :reader polygon-points)))

(define-constructor make-rectangle-1 standard-rectangle (min-x min-y max-x max-y points)
  :min-x min-x :min-y min-y :max-x max-x :max-y max-y :points points)

(defun make-rectangle (min-point max-point)
  (multiple-value-bind (min-x min-y) (point-position* min-point)
    (multiple-value-bind (max-x max-y) (point-position* max-point)
      (assert (<= min-x max-x))
      (assert (<= min-y max-y))
      (make-rectangle-1 min-x min-y max-x max-y
			(vector min-point (make-point min-x max-y)
				max-point (make-point max-x min-y))))))

(define-constructor make-rectangle*-1 standard-rectangle (min-x min-y max-x max-y)
  :min-x min-x :min-y min-y :max-x max-x :max-y max-y)

(defun make-rectangle* (x1 y1 x2 y2)
  (when (> x1 x2) (rotatef x1 x2))
  (when (> y1 y2) (rotatef y1 y2))
  (make-rectangle*-1 x1 y1 x2 y2))

(defmethod make-load-form ((rectangle standard-rectangle))
  `(make-rectangle* ,(rectangle-min-x rectangle) ,(rectangle-min-y rectangle)
		    ,(rectangle-max-x rectangle) ,(rectangle-max-y rectangle)))

(defmethod slot-unbound (class (rectangle standard-rectangle) (slot (eql 'points)))
  (declare (ignore class))
  (with-slots (points min-x min-y max-x max-y) rectangle
    (setf points (vector (make-point min-x min-y) (make-point min-x max-y)
			 (make-point max-x max-y) (make-point max-x min-y)))))

(defmethod rectangle-min-point ((rectangle standard-rectangle))
  (with-slots (points) rectangle
    (svref points 0)))

(defmethod rectangle-max-point ((rectangle standard-rectangle))
  (with-slots (points) rectangle
    (svref points 2)))

(defmethod rectangle-edges* ((rectangle standard-rectangle))
  (with-slots (min-x min-y max-x max-y) rectangle
    (values min-x min-y max-x max-y)))

(defmethod rectangle-width ((rectangle standard-rectangle))
  (with-slots (min-x max-x) rectangle
    (- max-x min-x)))

(defmethod rectangle-height ((rectangle standard-rectangle))
  (with-slots (min-y max-y) rectangle
    (- max-y min-y)))

(defmethod rectangle-size ((rectangle standard-rectangle))
  (with-slots (min-x min-y max-x max-y) rectangle
    (values (- max-x min-x) (- max-y min-y))))

(defmethod map-over-polygon-coordinates (function (rectangle standard-rectangle))
  (with-slots (min-x min-y max-x max-y) rectangle
    (funcall function min-x min-y)
    (funcall function min-x max-y)
    (funcall function max-x max-y)
    (funcall function max-x min-y)
    nil))

(defmethod map-over-polygon-segments (function (rectangle standard-rectangle))
  (with-slots (min-x min-y max-x max-y) rectangle
    (funcall function min-x min-y min-x max-y)
    (funcall function min-x max-y max-x max-y)
    (funcall function max-x max-y max-x min-y)
    (funcall function max-x min-y min-x min-y)
    nil))

(defmethod region-equal ((rect1 standard-rectangle) (rect2 standard-rectangle))
  (with-slots ((sx1 min-x) (sy1 min-y) (ex1 max-x) (ey1 max-y)) rect1
    (with-slots ((sx2 min-x) (sy2 min-y) (ex2 max-x) (ey2 max-y)) rect2
      (ltrb-equals-ltrb-p sx1 sy1 ex1 ey1
			  sx2 sy2 ex2 ey2))))

(defmethod region-contains-point*-p ((rectangle standard-rectangle) x y)
  (with-slots (min-x min-y max-x max-y) rectangle
    (ltrb-contains-point*-p min-x min-y max-x max-y x y)))

(defmethod region-contains-region-p ((rect1 standard-rectangle) (rect2 standard-rectangle))
  (with-slots ((sx1 min-x) (sy1 min-y) (ex1 max-x) (ey1 max-y)) rect1
    (with-slots ((sx2 min-x) (sy2 min-y) (ex2 max-x) (ey2 max-y)) rect2
      (ltrb-contains-ltrb-p sx1 sy1 ex1 ey1
			    sx2 sy2 ex2 ey2))))

(defmethod region-intersects-region-p ((rect1 standard-rectangle) (rect2 standard-rectangle))
  (with-slots ((sx1 min-x) (sy1 min-y) (ex1 max-x) (ey1 max-y)) rect1
    (with-slots ((sx2 min-x) (sy2 min-y) (ex2 max-x) (ey2 max-y)) rect2
      (ltrb-overlaps-ltrb-p sx1 sy1 ex1 ey1
			    sx2 sy2 ex2 ey2))))

(defmethod transform-region (transformation (rectangle standard-rectangle))
  (with-slots (min-x min-y max-x max-y) rectangle
    (if (rectilinear-transformation-p transformation)
	(multiple-value-bind (x1 y1)
	    (transform-point* transformation min-x min-y)
	  (multiple-value-bind (x2 y2)
	      (transform-point* transformation max-x max-y)
	    (make-rectangle* x1 y1 x2 y2)))
      (let ((coords nil))
	(flet ((transform-coord (x y)
		 (multiple-value-bind (nx ny)
		     (transform-point* transformation x y)
		   (push ny coords)
		   (push nx coords))))
	  (declare (dynamic-extent #'transform-coord))
	  (map-over-polygon-coordinates #'transform-coord rectangle))
	(make-polygon* (nreverse coords))))))

(defmethod bounding-rectangle* ((rectangle standard-rectangle))
  (with-slots (min-x min-y max-x max-y) rectangle
    (fix-rectangle (min min-x max-x) (min min-y max-y)
		   (max min-x max-x) (max min-y max-y))))


;;; General polygons

(defclass polygon-mixin ()
    ((coords :type vector :initarg :coords)
     (points :type vector :initarg :points :reader polygon-points)))

(defmethod map-over-polygon-coordinates (function (polygon polygon-mixin))
  (with-slots (coords points) polygon
    (if (slot-boundp polygon 'coords)
	(let ((ncoords (1- (length coords)))
	      (i -1))
	  (loop
	    (funcall function (aref coords (incf i)) (aref coords (incf i)))
	    (when (= i ncoords) (return)))
	  nil)
	(flet ((map-coordinates (point)
		 (funcall function (point-x point) (point-y point))))
	  (declare (dynamic-extent #'map-coordinates))
	  (map nil #'map-coordinates points))))
  nil)

(defmethod map-over-polygon-segments (function (polygon polygon-mixin))
  (with-slots (coords points) polygon
    (if (slot-boundp polygon 'coords)
	(let* ((ncoords (1- (length coords)))
	       (x1 (aref coords 0))
	       (y1 (aref coords 1))
	       (x x1)
	       (y y1)
	       (i 1))
	  (loop
	    (funcall function x y
		     (setf x (aref coords (incf i))) (setf x (aref coords (incf i))))
	    (when (= i ncoords) (return)))
	  (when (polyline-closed polygon)
	    (funcall function x y x1 y1)))
	(multiple-value-bind (x1 y1)
	    (point-position* (aref points 0))
	  (let ((x x1) (y y1))
	    (dotimes (i (1- (length points)))
	      (multiple-value-bind (nx ny)
		  (point-position* (aref points (1+ i)))
		(funcall function x y nx ny)
		(psetf x nx y ny)))
	    (when (polyline-closed polygon)
	      (funcall function x y x1 y1)))))
    nil))

(defmethod bounding-rectangle* ((polygon polygon-mixin))
  (let ((min-x nil) (min-y nil) (max-x nil) (max-y nil))
    (flet ((add-coord (x y)
	     (minf-or min-x x)
	     (minf-or min-y y)
	     (maxf-or max-x x)
	     (maxf-or max-y y)))
      (declare (dynamic-extent #'add-coord))
      (map-over-polygon-coordinates #'add-coord polygon))
    (fix-rectangle min-x min-y max-x max-y)))


(defclass standard-polyline (polygon-mixin polyline)
    ((closed :initarg :closed :reader polyline-closed)))

(define-constructor make-polyline standard-polyline (point-seq &key closed)
		    :points (coerce point-seq 'vector) :closed closed)

(define-constructor make-polyline* standard-polyline (coord-seq &key closed)
		    :coords (coerce coord-seq 'vector) :closed closed)

(defmethod make-load-form ((polyline standard-polyline))
  (with-slots (closed) polyline
    `(make-polyline ',(polygon-points polyline) :closed ,closed)))

(defmethod transform-region (transformation (polyline standard-polyline))
  (let ((coords nil))
    (flet ((transform-coord (x y)
	     (multiple-value-bind (nx ny)
		 (transform-point* transformation x y)
	       (push ny coords)
	       (push nx coords))))
      (declare (dynamic-extent #'transform-coord))
      (map-over-polygon-coordinates #'transform-coord polyline))
    (make-polyline* (nreverse coords) :closed (slot-value polyline 'closed))))


(defclass standard-polygon (polygon-mixin polygon) ())

(define-constructor make-polygon standard-polygon (point-seq)
		    :points (coerce point-seq 'vector))

(define-constructor make-polygon* standard-polygon (coord-seq)
		    :coords (coerce coord-seq 'vector))

(defmethod make-load-form ((polygon standard-polygon))
  `(make-polygon ',(polygon-points polygon)))

(defmethod polyline-closed ((polygon standard-polygon))
  t)

(defmethod transform-region (transformation (polygon standard-polygon))
  (let ((coords nil))
    (flet ((transform-coord (x y)
	     (multiple-value-bind (nx ny)
		 (transform-point* transformation x y)
	       (push ny coords)
	       (push nx coords))))
      (declare (dynamic-extent #'transform-coord))
      (map-over-polygon-coordinates #'transform-coord polygon))
    (make-polygon* (nreverse coords))))


;;; General ellipses

(defclass elliptical-arc (path) ())


(defclass ellipse (area) ())


(defclass ellipse-mixin ()
    ((center-point :type point :initarg :center-point :reader ellipse-center-point)
     (center-x :initarg :center-x :type real)
     (center-y :initarg :center-y :type real)
     (radius-1-dx :initarg :radius-1-dx :type real)
     (radius-1-dy :initarg :radius-1-dy :type real)
     (radius-2-dx :initarg :radius-2-dx :type real)
     (radius-2-dy :initarg :radius-2-dy :type real)
     (start-angle :initarg :start-angle :reader ellipse-start-angle :type single-float)
     (end-angle :initarg :end-angle :reader ellipse-end-angle :type single-float)))

(defmethod slot-unbound (class (ellipse ellipse-mixin) (slot (eql 'ellipse-center-point)))
  (declare (ignore class))
  (with-slots (center-point center-x center-y) ellipse
    (setf center-point (make-point center-x center-y))))

(defmethod ellipse-center-point* ((ellipse ellipse-mixin))
  (with-slots (center-x center-y) ellipse
    (values center-x center-y)))

(defmethod ellipse-radii ((ellipse ellipse-mixin))
  (with-slots (radius-1-dx radius-1-dy radius-2-dx radius-2-dy) ellipse
    (values radius-1-dx radius-1-dy radius-2-dx radius-2-dy)))


(defclass standard-elliptical-arc (ellipse-mixin elliptical-arc) ())

(define-constructor make-elliptical-arc standard-elliptical-arc
  (center-point radius-1-dx radius-1-dy radius-2-dx radius-2-dy
		&key start-angle end-angle)
  :center-point center-point :center-x (point-x center-point) :center-y (point-y center-point)
  :radius-1-dx radius-1-dx :radius-1-dy radius-1-dy
  :radius-2-dx radius-2-dx :radius-2-dy radius-2-dy
  :start-angle (cond (start-angle (float start-angle 0f0))
		     (end-angle 0f0)
		     (t nil))
  :end-angle (cond (end-angle (float end-angle 0f0))
		   (start-angle (float (* 2 pi) 0f0))
		   (t nil)))

(define-constructor make-elliptical-arc* standard-elliptical-arc
  (center-x center-y radius-1-dx radius-1-dy radius-2-dx radius-2-dy
	    &key start-angle end-angle)
  :center-x center-x :center-y center-y
  :radius-1-dx radius-1-dx :radius-1-dy radius-1-dy
  :radius-2-dx radius-2-dx :radius-2-dy radius-2-dy
  :start-angle (cond (start-angle (float start-angle 0f0))
		     (end-angle 0f0)
		     (t nil))
  :end-angle (cond (end-angle (float end-angle 0f0))
		   (start-angle (float (* 2 pi) 0f0))
		   (t nil)))

(defmethod make-load-form ((ellipse standard-elliptical-arc))
  (with-slots (center-point radius-1-dx radius-1-dy radius-2-dx radius-2-dy
	       start-angle end-angle) ellipse
    `(make-elliptical-arc ',center-point
			  ,radius-1-dx ,radius-1-dy ,radius-2-dx ,radius-2-dy
			  ,@(when start-angle `(:start-angle ,start-angle))
			  ,@(when end-angle `(:end-angle ,end-angle)))))

(defmethod transform-region (transformation (ellipse standard-elliptical-arc))
  (with-slots (center-x center-y radius-1-dx radius-1-dy radius-2-dx radius-2-dy
	       start-angle end-angle) ellipse
    (multiple-value-bind (cx cy)
	(transform-point* transformation center-x center-y)
      (multiple-value-bind (r1-dx r1-dy)
	  (transform-distance transformation radius-1-dx radius-1-dy)
	(multiple-value-bind (r2-dx r2-dy)
	    (transform-distance transformation radius-2-dx radius-2-dy)
	  (make-elliptical-arc* cx cy r1-dx r1-dy r2-dx r2-dy
				;;--- How to transform start and end angles?
				:start-angle start-angle :end-angle end-angle))))))

(defmethod bounding-rectangle* ((ellipse standard-elliptical-arc))
  (with-slots (center-x center-y radius-1-dx radius-1-dy radius-2-dx radius-2-dy
	       start-angle end-angle) ellipse
    (elliptical-arc-box center-x center-y radius-1-dx radius-1-dy radius-2-dx radius-2-dy
			start-angle end-angle 0)))


(defclass standard-ellipse (ellipse-mixin ellipse) ())

(define-constructor make-ellipse standard-ellipse
  (center-point radius-1-dx radius-1-dy radius-2-dx radius-2-dy
		&key start-angle end-angle)
  :center-point center-point :center-x (point-x center-point) :center-y (point-y center-point)
  :radius-1-dx radius-1-dx :radius-1-dy radius-1-dy
  :radius-2-dx radius-2-dx :radius-2-dy radius-2-dy
  :start-angle (cond (start-angle (float start-angle 0f0))
		     (end-angle 0f0)
		     (t nil))
  :end-angle (cond (end-angle (float end-angle 0f0))
		   (start-angle (float (* 2 pi) 0f0))
		   (t nil)))

(define-constructor make-ellipse* standard-ellipse
  (center-x center-y radius-1-dx radius-1-dy radius-2-dx radius-2-dy
	    &key start-angle end-angle)
  :center-x center-x :center-y center-y
  :radius-1-dx radius-1-dx :radius-1-dy radius-1-dy
  :radius-2-dx radius-2-dx :radius-2-dy radius-2-dy
  :start-angle (cond (start-angle (float start-angle 0f0))
		     (end-angle 0f0)
		     (t nil))
  :end-angle (cond (end-angle (float end-angle 0f0))
		   (start-angle (float (* 2 pi) 0f0))
		   (t nil)))

(defmethod make-load-form ((ellipse standard-ellipse))
  (with-slots (center-point radius-1-dx radius-1-dy radius-2-dx radius-2-dy
			    start-angle end-angle) ellipse
    `(make-ellipse ',center-point
		   ,radius-1-dx ,radius-1-dy ,radius-2-dx ,radius-2-dy
		   ,@(when start-angle `(:start-angle ,start-angle))
		   ,@(when end-angle `(:end-angle ,end-angle)))))

(defmethod transform-region (transformation (ellipse standard-ellipse))
  (with-slots (center-x center-y radius-1-dx radius-1-dy radius-2-dx radius-2-dy
	       start-angle end-angle) ellipse
    (multiple-value-bind (cx cy)
	(transform-point* transformation center-x center-y)
      (multiple-value-bind (r1-dx r1-dy)
	  (transform-distance transformation radius-1-dx radius-1-dy)
	(multiple-value-bind (r2-dx r2-dy)
	    (transform-distance transformation radius-2-dx radius-2-dy)
	  (make-ellipse* cx cy r1-dx r1-dy r2-dx r2-dy
			 ;;--- How to transform start and end angles?
			 :start-angle start-angle :end-angle end-angle))))))

(defmethod bounding-rectangle* ((ellipse standard-ellipse))
  (with-slots (center-x center-y radius-1-dx radius-1-dy radius-2-dx radius-2-dy
	       start-angle end-angle) ellipse
    (elliptical-arc-box center-x center-y radius-1-dx radius-1-dy radius-2-dx radius-2-dy
			start-angle end-angle nil)))


;;; Geometry utilities

(defconstant pi-single-float (coerce pi 'single-float))
(defconstant  2pi (coerce (* pi-single-float 2) 'single-float))
(defconstant pi/2 (coerce (/ pi-single-float 2) 'single-float))

(defun radians->degrees (radians)
  (* radians (/ 360 2pi)))

(defun degrees->radians (degrees)
  (* degrees (/ 2pi 360)))

;; This macro wouldn't be necessary if we could count on (expt (expression) 2)
;; being optimized properly
(defmacro square (expression)
  (if (symbolp expression)
      `(* ,expression ,expression)
      (let ((var (gensymbol)))
	`(let ((,var ,expression))
	   (* ,var ,var)))))

;; This runs when we already know that the point is inside the bounding box.
(defun point-close-to-line-p (x y from-x from-y to-x to-y &optional (thickness 1))
  (let ((distance (1+ (ceiling thickness 2)))
	(dx (- to-x from-x))
	(dy (- to-y from-y)))
    (or (and (zerop dx) (zerop dy))
	(<= (square (- (* y dx) (* x dy) (- (* from-y to-x) (* from-x to-y))))
	    (* (square distance) (+ (square dx) (square dy)))))))

;; Computes whether a point is inside an ellipse whose center is (0,0).
;; This calculation is exact.
(defun point-inside-ellipse-p (x y radius-1-dx radius-1-dy radius-2-dx radius-2-dy)
  (<= (+ (square (- (* radius-2-dy x) (* radius-2-dx y)))
	 (square (- (* radius-1-dx y) (* radius-1-dy x))))
      (square (- (* radius-1-dx radius-2-dy) (* radius-1-dy radius-2-dx)))))

;; Computes whether a point is on a stroked ellipse whose center is (0,0).
;; This calculation is not exact - the envelope of an ellipse is not an ellipse
;; and an "average radius" is used - but it should be ok for thickness small
;; compared to radii.  The calculation is exact for circles.
(defun point-on-thick-ellipse-p (x y radius-1-dx radius-1-dy radius-2-dx radius-2-dy
				 half-thickness)
  (let* ((det (- (* radius-1-dx radius-2-dy) (* radius-1-dy radius-2-dx)))
	 (avrad*delta (* (sqrt (abs det)) half-thickness)))
    (<= (square (- det avrad*delta))
	(+ (square (- (* radius-2-dy x) (* radius-2-dx y)))
	   (square (- (* radius-1-dx y) (* radius-1-dy x))))
	(square (+ det avrad*delta)))))

;; Find the singular value decomposition of a 2 by 2 matrix: M = R1.D.R2
;; where R's are rotations and D is diagonal.  The four values returned
;; are the first angle, the two diagonal elements, and the second angle.
;; Used to convert CLIM's representation of ellipses to various window
;; systems' representations.
(defun 2x2-singular-value-decomposition (a b c d)
  (cond ((and (zerop b) (zerop c))
	 (values 0.0 a d 0.0))
	((and (zerop a) (zerop d))
	 (values pi/2 b (- c) 0.0))
	(T
	 (let* ((d+a (+ d a)) (a-d (- a d))
		(c+b (+ c b)) (c-b (- c b))
		(sx+sy (sqrt (+ (square d+a) (square c-b))))
		(sx-sy (sqrt (+ (square a-d) (square c+b))))
		(sx (* 0.5 (+ sx+sy sx-sy)))
		(sy (* 0.5 (- sx+sy sx-sy)))
		(t1+t2 (if (and (zerop c-b) (zerop d+a)) 0.0 (atan c-b d+a)))
		(t1-t2 (if (and (zerop c+b) (zerop a-d)) 0.0 (atan c+b a-d)))
		(t1 (* 0.5 (+ t1+t2 t1-t2)))
		(t2 (* 0.5 (- t1+t2 t1-t2))))
	   (values t2 sx sy t1)))))

;; For a complete ellipse, the box is actually the rectangle that bounds
;; the parallelogram that bounds the ellipse.  That means it's a little
;; bigger than the tightest possible bounding box when the ellipse is
;; not axis-aligned.  It's not worth computing anything tighter because
;; the refined highlighting test will be faster than the computation of
;; a tighter box.
(defun elliptical-arc-box (center-x center-y radius-1-dx radius-1-dy radius-2-dx radius-2-dy
			   theta-1 theta-2 thickness)
  (let* ((filled (null thickness))
	 (thickness (or thickness 0))
	 (lthickness (floor thickness 2))
	 (rthickness (- thickness lthickness)))
    (when (null theta-1)
      (return-from elliptical-arc-box
	(let ((dx (+ (abs radius-1-dx) (abs radius-2-dx)))
	      (dy (+ (abs radius-1-dy) (abs radius-2-dy))))
	  (fix-rectangle (- center-x dx lthickness) (- center-y dy lthickness)
			 (+ center-x dx rthickness) (+ center-y dy rthickness)))))
    (setq theta-1 (mod theta-1 2pi)
	  theta-2 (mod theta-2 2pi))
    ;;--- Fix the NYI stuff some year
    (let* ((x-radius (cond ((zerop radius-1-dx) radius-2-dx)
			   ((zerop radius-2-dx) radius-1-dx)
			   (t (nyi))))
	   (y-radius (cond ((zerop radius-1-dy) radius-2-dy)
			   ((zerop radius-2-dy) radius-1-dy)
			   (t (nyi))))
	   (x1 (+ center-x (* x-radius (cos theta-1))))
	   (y1 (+ center-y (* y-radius (sin theta-1))))
	   (x2 (+ center-x (* x-radius (cos theta-2))))
	   (y2 (+ center-y (* y-radius (sin theta-2))))
	   (left (min x1 x2))
	   (top (min y1 y2))
	   (right (max x1 x2))
	   (bottom (max y1 y2)))
      (when (angle-between-angles-p pi-single-float theta-1 theta-2)
	(minf left (- center-x x-radius)))
      (when (angle-between-angles-p (* pi-single-float 3/2) theta-1 theta-2)
	(minf top (- center-y y-radius)))
      (when (angle-between-angles-p 0 theta-1 theta-2)
	(maxf right (+ center-x x-radius)))
      (when (angle-between-angles-p pi/2 theta-1 theta-2)
	(maxf bottom (+ center-y y-radius)))
      (when filled
	(minf left center-x)
	(minf top center-y)
	(maxf right center-x)
	(maxf bottom center-y))
      (fix-rectangle (- left lthickness) (- top lthickness)
		     (+ right rthickness) (+ bottom rthickness)))))

(defun angle-between-angles-p (theta theta-1 theta-2)
  (unless (< theta-1 theta-2)
    (incf theta-2 2pi))
  (unless (< theta-1 theta)
    (incf theta 2pi))
  (< theta theta-2))


;;; Bounding rectangles

;; Bounding rectangles live in the "ground" coordinate system, such that
;; LEFT = MIN-X, RIGHT = MAX-X, TOP = MIN-Y, AND BOTTOM = MAX-Y.
;; We use these slot names to avoid colliding with the slot names in STANDARD-RECTANGLE.
(defclass bounding-rectangle (rectangle)
    ((left   :initarg :left   :accessor rectangle-min-x)
     (top    :initarg :top    :accessor rectangle-min-y)
     (right  :initarg :right  :accessor rectangle-max-x)
     (bottom :initarg :bottom :accessor rectangle-max-y)))

(define-constructor make-bounding-rectangle-1 bounding-rectangle (left top right bottom)
		    :left left :top top :right right :bottom bottom)

(defun make-bounding-rectangle (left top right bottom)
  (let ((x1 (floor left))
	(y1 (floor top))
	(x2 (floor right))
	(y2 (floor bottom)))
    (when (> x1 x2) (rotatef x1 x2))
    (when (> y1 y2) (rotatef y1 y2))
    (make-bounding-rectangle-1 x1 y1 x2 y2)))

(defmethod make-load-form ((rectangle bounding-rectangle))
  (with-slots (left top right bottom) rectangle
    `(make-bounding-rectangle ,left ,top ,right ,bottom)))

(defmethod transform-region (transformation (rectangle bounding-rectangle))
  (with-slots (left top right bottom) rectangle
    (if (rectilinear-transformation-p transformation)
	(multiple-value-bind (x1 y1)
	    (transform-point* transformation left top)
	  (multiple-value-bind (x2 y2)
	      (transform-point* transformation right bottom)
	    (make-bounding-rectangle x1 y1 x2 y2)))
        (error "You can only transform bounding-rectangles rectilinearly"))))

(defmethod rectangle-max-point ((rectangle bounding-rectangle))
  (make-point (slot-value rectangle 'right) (slot-value rectangle 'bottom)))

(defmethod rectangle-edges* ((rectangle bounding-rectangle))
  (with-slots (left top right bottom) rectangle
    (values left top right bottom)))

(defmethod rectangle-width ((rectangle bounding-rectangle))
  (with-slots (left right) rectangle
    (- right left)))

(defmethod rectangle-height ((rectangle bounding-rectangle))
  (with-slots (top bottom) rectangle
    (- bottom top)))

(defmethod rectangle-size ((rectangle bounding-rectangle))
  (with-slots (left top right bottom) rectangle
    (values (- right left)
	    (- bottom top))))

(defmethod map-over-polygon-coordinates (function (rectangle bounding-rectangle))
  (with-slots (left top right bottom) rectangle
    (funcall function left top)
    (funcall function left bottom)
    (funcall function right bottom)
    (funcall function right top)
    nil))

(defmethod map-over-polygon-segments (function (rectangle bounding-rectangle))
  (with-slots (left top right bottom) rectangle
    (funcall function left top left bottom)
    (funcall function left bottom right bottom)
    (funcall function right bottom right top)
    (funcall function right top left top)
    nil))

;; This and the next three can also serve for output records, which are built
;; on top of BOUNDING-RECTANGLE.
(defmethod region-equal ((rect1 bounding-rectangle) (rect2 bounding-rectangle))
  (with-slots ((sx1 left) (sy1 top) (ex1 right) (ey1 bottom)) rect1
    (with-slots ((sx2 left) (sy2 top) (ex2 right) (ey2 bottom)) rect2
      (ltrb-equals-ltrb-p sx1 sy1 ex1 ey1
			  sx2 sy2 ex2 ey2))))

(defmethod region-contains-point*-p ((rectangle bounding-rectangle) x y)
  (with-slots (left top right bottom) rectangle
    (ltrb-contains-point*-p left top right bottom x y)))

(defmethod region-contains-region-p ((rect1 bounding-rectangle) (rect2 bounding-rectangle))
  (with-slots ((sx1 left) (sy1 top) (ex1 right) (ey1 bottom)) rect1
    (with-slots ((sx2 left) (sy2 top) (ex2 right) (ey2 bottom)) rect2
      (ltrb-contains-ltrb-p sx1 sy1 ex1 ey1
			    sx2 sy2 ex2 ey2))))

(defmethod region-intersects-region-p ((rect1 bounding-rectangle) (rect2 bounding-rectangle))
  (with-slots ((sx1 left) (sy1 top) (ex1 right) (ey1 bottom)) rect1
    (with-slots ((sx2 left) (sy2 top) (ex2 right) (ey2 bottom)) rect2
      (ltrb-overlaps-ltrb-p sx1 sy1 ex1 ey1
			    sx2 sy2 ex2 ey2))))

(defmacro with-bounding-rectangle* ((left top &optional right bottom) region &body body)
  `(multiple-value-bind (,left ,top ,@(when right (list right bottom)))
       (bounding-rectangle* ,region) 
     ,@body))

(defmethod bounding-rectangle* ((rectangle bounding-rectangle))
  (with-slots (left top right bottom) rectangle
    (values left top right bottom)))

;; Guaranteed to cons a new rectangle unless REUSE-RECTANGLE is supplied
(defun bounding-rectangle (region &optional reuse-rectangle)
  (with-bounding-rectangle* (left top right bottom) region
    (cond (reuse-rectangle
	   (setf (slot-value reuse-rectangle 'left)   left)
	   (setf (slot-value reuse-rectangle 'top)    top)
	   (setf (slot-value reuse-rectangle 'right)  right)
	   (setf (slot-value reuse-rectangle 'bottom) bottom)
	   reuse-rectangle)
	  (t
	   (make-bounding-rectangle left top right bottom)))))

;; Set the edges of the rectangle, and return the rectangle as the value
(defmethod bounding-rectangle-set-edges ((rectangle bounding-rectangle) left top right bottom)
  #+ignore (assert (<= left right))
  #+ignore (assert (<= top bottom))
  (unless (and left top right bottom)
    (error "Ahah!"))
  (with-slots ((bl left) (bt top) (br right) (bb bottom)) rectangle
    (setq bl left
	  bt top
	  br right
	  bb bottom))
  rectangle)

(defmacro define-bounding-rectangle-setf (name &optional (accessor name))
  (check-type accessor (member left top right bottom))
  (let* ((fspec (fintern "~A-~A" 'bounding-rectangle name))
	 (new (fintern "~A-~A" 'new name))
	 (edges '(left top right bottom)))
    `(defsetf ,fspec (region) (,new)
       `(with-bounding-rectangle* ,',edges ,region
	  (setq ,',accessor ,,new)
	  (bounding-rectangle-set-edges ,region ,@',edges)
	  ,,new))))

(defun-inline bounding-rectangle-min-x (region)
  (with-bounding-rectangle* (min-x min-y) region
    (declare (ignore min-y))
    min-x))
(define-bounding-rectangle-setf min-x left)

(defun-inline bounding-rectangle-min-y (region)
  (with-bounding-rectangle* (min-x min-y) region
    (declare (ignore min-x))
    min-y))
(define-bounding-rectangle-setf min-y top)

(defun-inline bounding-rectangle-max-x (region) 
  (with-bounding-rectangle* (min-x min-y max-x max-y) region
    (declare (ignore min-x min-y max-y))
    max-x))
(define-bounding-rectangle-setf max-x right)

(defun-inline bounding-rectangle-max-y (region) 
  (with-bounding-rectangle* (min-x min-y max-x max-y) region
    (declare (ignore min-x min-y max-x))
    max-y))
(define-bounding-rectangle-setf max-y bottom)

(defun bounding-rectangle-min-point (region)
  (with-bounding-rectangle* (min-x min-y) region 
    (make-point min-x min-y)))

(defun bounding-rectangle-max-point (region)
  (with-bounding-rectangle* (min-x min-y max-x max-y) region 
    (declare (ignore min-x min-y))
    (make-point max-x max-y)))

(defun-inline bounding-rectangle-position* (region)
  (with-bounding-rectangle* (left top) region 
    (values left top)))

(defun bounding-rectangle-position (region)
  (with-bounding-rectangle* (left top) region 
    (make-point left top)))

;; Set the position of the rectangle, and return the rectangle as the value
(defmethod bounding-rectangle-set-position* ((rectangle bounding-rectangle) x y)
  (with-slots (left top right bottom) rectangle
    (let ((width (- right left))
	  (height (- bottom top)))
      (setq left   x
	    top    y
	    right  (+ x width)
	    bottom (+ y  height))))
  rectangle)

;; Make a new bounding rectangle for the region, and shift its position by DX,DY,
;; and return the new rectangle.
(defun bounding-rectangle-shift-position (region dx dy &optional reuse-rectangle)
  (declare (values region))
  (let ((rectangle (bounding-rectangle region reuse-rectangle)))
    (with-slots (left top right bottom) rectangle
      (incf left   dx)
      (incf top    dy)
      (incf right  dx)
      (incf bottom dy))
    rectangle))

(defun bounding-rectangle-position-equal (region1 region2)
  (multiple-value-bind (x1 y1) (bounding-rectangle-position* region1)
    (multiple-value-bind (x2 y2) (bounding-rectangle-position* region2)
      (and (= x1 x2)
	   (= y1 y2)))))

(defun bounding-rectangle-edges-equal (region1 region2)
  (with-bounding-rectangle* (left1 top1 right1 bottom1) region1
    (with-bounding-rectangle* (left2 top2 right2 bottom2) region2
      (and (= left1 left2)
	   (= top1 top2)
	   (= right1 right2)
	   (= bottom1 bottom2)))))

(defun-inline position-difference* (x1 y1 x2 y2)
  (values (- x1 x2)
	  (- y1 y2)))

(defun bounding-rectangle-position-difference (region1 region2)
  (multiple-value-bind (x1 y1) (bounding-rectangle-position* region1)
    (multiple-value-bind (x2 y2) (bounding-rectangle-position* region2)
      (position-difference* x1 y1 x2 y2))))

(defun-inline bounding-rectangle-width (region)
  (with-bounding-rectangle* (left top right bottom) region
    (declare (ignore top bottom))
    (- right left)))

(defun-inline bounding-rectangle-height (region)
  (with-bounding-rectangle* (left top right bottom) region 
    (declare (ignore left right))
    (- bottom top)))

(defun-inline bounding-rectangle-size (region)
  (declare (values width height))
  (with-bounding-rectangle* (left top right bottom) region 
    (values (- right left)
	    (- bottom top))))

;; Set the size of the rectangle, and return the rectangle as the value
(defmethod bounding-rectangle-set-size ((rectangle bounding-rectangle) width height)
  (with-slots (left top right bottom) rectangle
    (let ((new-right  (+ left width))
	  (new-bottom (+ top height)))
      (setq right  new-right
	    bottom new-bottom)))
  rectangle)

(defun bounding-rectangle-size-equal (region1 region2)
  (with-bounding-rectangle* (left1 top1 right1 bottom1) region1
    (with-bounding-rectangle* (left2 top2 right2 bottom2) region2
      (ltrb-size-equal left1 top1 right1 bottom1
		       left2 top2 right2 bottom2))))

(defun bounding-rectangle-center (region)
  (with-bounding-rectangle* (left top right bottom) region
    (make-point (+ left (floor (- right left) 2))
		(+ top (floor (- bottom top) 2)))))

(defun bounding-rectangle-center* (region)
  (with-bounding-rectangle* (left top right bottom) region
    (values (+ left (floor (- right left) 2))
	    (+ top (floor (- bottom top) 2)))))

(defun bounding-rectangle-ltrb (region)
  (declare (values left top right bottom))
  (with-bounding-rectangle* (left top right bottom) region
    (values left top right bottom)))

(defmacro with-bounding-rectangle-ltrb ((left top &optional right bottom) region &body body)
  `(with-bounding-rectangle* (,left ,top ,@(when right (list right bottom))) ,region
     ,@body))

(defun-inline bounding-rectangle-left (region)
  (with-bounding-rectangle-ltrb (left top) region
    (declare (ignore top))
    left))
(define-bounding-rectangle-setf left)

(defun-inline bounding-rectangle-top (region)
  (with-bounding-rectangle-ltrb (left top) region
    (declare (ignore left))
    top))
(define-bounding-rectangle-setf top)

(defun-inline bounding-rectangle-right (region) 
  (with-bounding-rectangle-ltrb (left top right bottom) region 
    (declare (ignore left top bottom))
    right))
(define-bounding-rectangle-setf right)

(defun-inline bounding-rectangle-bottom (region) 
  (with-bounding-rectangle-ltrb (left top right bottom) region 
    (declare (ignore left top right))
    bottom))
(define-bounding-rectangle-setf bottom)


;;; Region Sets

(defclass region-set (region) ())

;;; Some default methods.

(defmethod region-set-function ((region region)) 'union)

(defmethod region-set-regions ((region region) &key normalize)
  (declare (ignore normalize))
  (list region))

(defmethod map-over-region-set-regions (function (region region) &key normalize)
  (declare (ignore normalize))
  (funcall function region))

(defmethod map-over-region-set-regions (function (region region-set) &rest args &key normalize)
  (declare (dynamic-extent args))
  (declare (ignore normalize))
  (map nil function (apply #'region-set-regions region args)))

#+++ignore
(defmethod region-equal ((set1 region-set) (set2 region-set))
  ;;--- How to do this?
  )

(defmethod region-contains-point*-p ((region-set region-set) x y)
  (flet ((contains-point*-p (region)
	   (when (region-contains-point*-p region x y)
	     (return-from region-contains-point*-p t))))
    (declare (dynamic-extent #'contains-point*-p))
    (map-over-region-set-regions #'contains-point*-p region-set))
  nil)

(defmethod region-contains-region-p ((region-set region-set) (other-region region))
  (flet ((contains-region-p (region)
	   (when (region-contains-region-p region other-region)
	     (return-from region-contains-region-p t))))
    (declare (dynamic-extent #'contains-region-p))
    (map-over-region-set-regions #'contains-region-p region-set))
  nil)

#++ignore
(defmethod region-intersects-region-p ((set1 region-set) (set2 region-set))
  ;;--- How to do this?
  )

(defmethod bounding-rectangle* ((region-set region-set))
  (let ((left nil) (top nil) (right nil) (bottom nil))
    (flet ((add-region (region)
	     (with-bounding-rectangle* (rl rt rr rb) region
	       (minf-or left rl)
	       (minf-or top  rt)
	       (maxf-or right  rr)
	       (maxf-or bottom rb))))
      (declare (dynamic-extent #'add-region))
      (map-over-region-set-regions #'add-region region-set))
    (values left top right bottom)))


;; Exclude the general cases of REGION-EQUAL
(define-symmetric-region-method region-equal ((region region) (nowhere nowhere)) nil)
(define-symmetric-region-method region-equal ((region region) (everywhere everywhere)) nil)
(define-symmetric-region-method region-equal ((point point) (path path)) nil)
(define-symmetric-region-method region-equal ((point point) (area area)) nil)
(define-symmetric-region-method region-equal ((path path) (area area)) nil)
(define-symmetric-region-method region-equal ((line polyline) (arc elliptical-arc)) nil)
(define-symmetric-region-method region-equal ((polygon polygon) (ellipse ellipse)) nil)

;; Exclude the general cases of REGION-CONTAINS-REGION-P
(defmethod region-contains-region-p ((point point) (path path)) nil)
(defmethod region-contains-region-p ((point point) (area area)) nil)
(defmethod region-contains-region-p ((path path) (area area)) nil)
(defmethod region-contains-region-p ((line polyline) (arc elliptical-arc)) nil)

(defmethod region-contains-region-p ((region1 region) (region2 region))
  ;; Dubious special case preserved from old version, which would just return
  ;; NIL when handed any two random region types.  In other words, if someone
  ;; forgot to define a method between two region subclasses, the answer to
  ;; REGION-CONTAINS-REGION-P was simply defined to be NIL.
  (or (eql region1 region2)
      (error "No ~S method defined between objects of type ~S and ~S"
	     'region-contains-region-p (type-of region1) (type-of region2))))

;; Exclude the general cases of REGION-INTERSECTS-REGION-P
(defmethod region-intersects-region-p ((region1 region) (region2 region))
  ;; Dubious special case preserved from old version, which would just return
  ;; NIL when handed any two random region types.  In other words, if someone
  ;; forgot to define a method between two region subclasses, the answer to
  ;; REGION-INTERSECTS-REGION-P was simply defined to be NIL.  The INTERSECTION
  ;; was unknown, not empty.
  (or (eql region1 region2)
      (error "No ~S method defined between objects of type ~S and ~S"
	     'region-intersects-region-p (type-of region1) (type-of region2))))

