;;; -*- Mode: LISP; Syntax: Common-Lisp; Base: 10; Package: ON-LV -*-

(in-package :on-lv)

"Copyright (c) 1991 International Lisp Associates.  All rights reserved."

(defclass lispview-port (x-port)
    ((display :reader port-display)
     (root :reader port-root)
     (type :allocation :class
	   :initform ':lispview
	   :reader port-type)
     )
  )

(defmethod find-port-type ((type (eql :lispview))) 
  (find-class 'lispview-port))

;;;--- figure out how to share this with the X port.
(defun decode-lispview-server-path (server-path)
  (if (eq (car server-path) ':lispview)
      (let* ((keys (cdr server-path))
	     ;; use same defaults as x-port
	     (defaults (cdr (assoc ':x11 *server-path-defaults*)))
	     (host    (or (getf keys :host)    (getf defaults :host)))
	     (display (or (getf keys :display) (getf defaults :display)))
	     (screen  (or (getf keys :screen)  (getf defaults :screen))))
	(values host display screen))
      (error "~S is not a Lispview server path." server-path)))

(defmacro with-decoded-lispview-server-path ((host display screen)
					     server-path
					     &body body)
  `(multiple-value-bind (,host ,display ,screen)
       (decode-lispview-server-path ,server-path)
     ,@body))

(defmethod initialize-instance :after ((port lispview-port) &key 
				server-path)
  (with-decoded-lispview-server-path (host display-id screen-id)
    server-path
    (with-slots (display root) port
      (setf display
	    (make-instance 'lv:display
			   :platform lv:xview
			   :name (format nil "~A:~A.~A" host display-id screen-id)))
      (setf root (lv:root-canvas display)))))

(defmethod port-match ((port lispview-port) server-path)
  (let ((pkeys (cdr (port-server-path port)))
	(akeys (cdr server-path)))
    (and (eq (car server-path) ':lispview)
	 (equalp (getf akeys :host)
		 (getf pkeys :host))
	 (eql  (getf akeys :display)
	       (getf pkeys :display))
	 (eql  (getf akeys :screen)
	       (getf pkeys :screen)))))

(defmethod destroy-port :after ((port lispview-port))
)

(defvar *frame-count* 0)

;;; There is an assumption here that INSTALL-SETTINGS is only
;;; called on a top-level-sheet.  This assumption was inherited
;;; from the X-PORT.
(defmethod install-mirror-settings ((port lispview-port)
				    sheet
				    &key 
				    (plain      nil plain-sp)
				    region
				    (title nil title-supplied)
				    &allow-other-keys)

  (let ((base-window (sheet-mirror sheet)))
    (when base-window
      (let ((old-title (lv:label base-window)))
	;; The install-mirror-settings code gets run multiple
	;; times, so check to see if the base window already has
	;; a title.
	(when (or (null old-title) (equal old-title ""))
	  ;; If the caller didn't supply a title, make one up
	  (let ((title (if title-supplied
			   title
			   (format nil "CLIM Frame ~D"
				   (incf *frame-count*)))))
	    (setf (lv:label base-window)
		  title)
	    (setf (lv:icon base-window)
		  (make-instance 'lv:icon :label title)))))
      (when region
	(setf (lv:bounding-region base-window)
	      (lv:make-region :left (rectangle-min-x region)
			      :top (rectangle-max-x region)
			      :right (rectangle-min-y region)
			      :bottom (rectangle-max-y region)))))))
	    

(defmethod install-port-cursor ((port lispview-port) sheet cursor)
  ;; --- piggyback on X-PORT stuff via CLX?
  )

(defun lispview->clx (port lispview-window)
  ;; copped from CLM
  (let* ((xid (lv::xview-window-xid (lv::device lispview-window)))
	 (clx-display (on-x::x-display port))
	 (clxwin (xlib::make-window :display clx-display
				    :id xid)))
    (xlib::save-id clx-display xid clxwin)
    clxwin))

(defclass clim-base-window (lv:base-window)
  ((port :initarg :port)
   (sheet :initarg :sheet)))

;;; --- we may want to create all components with :MAPPED NIL
(defmethod realize-mirror ((port lispview-port) (sheet ws::top-level-sheet))
  ;; Depending on the frame we might do different things here
  (make-instance 'clim-base-window
		 :display (port-display port)
		 ;; --- Using :PARENT creates the base-window on the
		 ;; local display
		 ;;:parent (port-root port)
		 :port port
		 :sheet sheet)
  )

(defmethod destroy-mirror ((port lispview-port) sheet)
  (when (sheet-mirror sheet)
    ;; --- How do we kill one?
    (setf (lv:mapped (sheet-mirror sheet)) nil)))

;; this is not quite right

(defmethod enable-mirror ((port lispview-port) (sheet ws::top-level-sheet))
  (setf (lv:mapped (sheet-mirror sheet)) t))

;;; Adapted from the method on X-PORT
(defmethod sheet-actual-native-edges* ((port lispview-port)
				       (sheet mirrored-sheet-mixin))
  (let* ((component (sheet-mirror sheet))
	 (region (lv:bounding-region component)))
    (values (lv:region-left region) (lv:region-top region)
	    (lv:region-right region) (lv:region-bottom region))))

(defmethod set-sheet-actual-native-edges* ((port lispview-port) 
					   (sheet mirrored-sheet-mixin)
					   x1 y1 x2 y2)
  (let* ((component (sheet-mirror sheet))
	 (x1 (round x1))
	 (y1 (round y1))
	 (x2 (round x2))
	 (y2 (round y2)))
    (setf (lv:bounding-region component)
	  (lv:make-region :left x1 :top y1 :right x2 :bottom y2))))

(defmethod (setf lv:bounding-region) :after (new-region (cbw clim-base-window))
  (declare (ignore new-region))
  (with-slots (port sheet) cbw
    ;; Don't do this until everything is set up.
    (when (sheet-enabled-p sheet)
      ;; --- If we don't do this, the re-layout code gets all confused.
      ;; What is the right thing to be doing?
      (let ((child (sheet-child sheet))
	    (display (lv:display cbw)))
	(ws::clear-space-req-caches-in-tree child)
	;; Try to suppress "light show" during re-layout.
	;; I also tried binding LV:MAPPED of the CBW to NIL, but
	;; that flashed the whole frame white.
	(lv:with-output-buffering display
	  (mirror-region-updated port sheet)))
      ;; --- until damage events work
      (repaint-sheet sheet +everywhere+))))
