;;; -*- Mode: LISP; Syntax: Common-lisp; Package: CLIM-DEMO; Base: 10 -*-

;;; Document this heavily for addition to CLIM:TUTORIAL;EXAMPLES.LISP.
(in-package "CLIM-DEMO")

"Copyright (c) 1990, International Lisp Associates, Inc.  All rights reserved."

(define-application-frame puzzle
			  ;; No super classes
			  ()
  ;; One slot
  ((puzzle :initform (make-array '(4 4))
	   :accessor puzzle-puzzle))
  ;; Just a single pane, 140 x 90, with a special display function that
  ;; gets called by CLIM each time around the command loop
  (:pane
    (make-clim-pane (nil :hs 140 :vs 90)
		    :display-function '(incremental-redisplay-display-function draw-puzzle)
		    :display-time :command-loop
		    :default-text-style '(:fix :bold :very-large)))
  ;; Provide a macro called DEFINE-PUZZLE-COMMAND to be used to define commands for
  ;; this application
  (:command-definer T)
  ;; Give the frame a command pane and fill it with commands from the menu group named
  ;;"puzzle"
  (:menu-group puzzle)
  ;; Evaluate the form (clim-top-level) to start up the application
  ;; :top-level T means use default-frame-top-level
  (:top-level (clim-top-level)))


;;; run-frame-top-level is the method that is invoked to start the 
;;; application's top-level loop.
;;; Put initialization code in a :before method.
(defmethod run-frame-top-level :before ((puzzle puzzle))
  (initialize-puzzle puzzle))

(define-presentation-type puzzle-cell ()
  ;; Provide own own way of doing highlighting
  :highlight-function ((record stream state)
		       state
		       (multiple-value-bind (xoff yoff)
			   ;; Get the coordinates of this output record in terms of
			   ;; the coordinate system of the stream in whose output history
			   ;; it appears.
			   (ci::convert-from-relative-to-absolute-coordinates 
			     stream
			     (output-record-parent record))
			 (with-bounding-rectangle* (left top right bottom) record
			   (draw-rectangle* stream 
					    (+ left xoff) (+ top yoff)
					    (+ right xoff) (+ bottom yoff)
					    :ink +flipping-ink+)))))

;;; Returns an integer between 0 and 15 corresponding to the specified cell
(defun encode-puzzle-cell (row column)
  (+ (* row 4) column))

(defun decode-puzzle-cell (encoding)
  (floor encoding 4))

(defmethod initialize-puzzle ((puzzle puzzle))
  (let ((puzzle-array (puzzle-puzzle puzzle)))
    (dotimes (row 4)
      (dotimes (column 4)
	(setf (aref puzzle-array row column) (mod (1+ (encode-puzzle-cell row column)) 16))))))

(defmethod draw-puzzle ((puzzle puzzle) stream)
  ;; Don't wrap or scroll, let the output run off the bottom of the viewport.
  (with-end-of-page-action (:allow stream)
    (let ((puzzle-array (puzzle-puzzle puzzle)))
      (formatting-item-list (stream
			      :n-columns 4
			      :n-rows 4)
	(dotimes (row 4)
	  (dotimes (column 4)
	    (let ((value (aref puzzle-array row column)))
	      
	      ;; Explain how CLIM can know whether to refresh the cell that's drawn
	      (updating-output (stream
				 ;; Identify this piece of output by its cell number
				 :unique-id (encode-puzzle-cell row column)

				 ;; If the cell number above ever gets drawn with a value
				 ;; different than the value of :cache-value, then CLIM
				 ;; knows it has to be refreshed
				 :cache-value value)
		(formatting-cell (stream)
		  (with-output-as-presentation (
						;; Making this presentation a puzzle-cell
						;; will render it sensitive to the pointer
						;; when the input context is puzzle-cell,
						;; as it is for some of the commands
						:type 'puzzle-cell

						;; Save in the output history the unique
						;; number corresponding to this cell 
						:object (encode-puzzle-cell row column)
						:stream stream)
		    (format stream "~:[~2,' D~;~*  ~]" (zerop value) value)))))))))))

(defun cell-adjacent-to-open-cell (puzzle r c)
  ;; check row
  (or
    (dotimes (column 4)
      (when (and (/= column c) (zerop (aref puzzle r column)))
	(return (encode-puzzle-cell r column))))
    (dotimes (row 4)
      (when (and (/= row r) (zerop (aref puzzle row c)))
	(return (encode-puzzle-cell row c))))))

(define-puzzle-command com-move-cell
    ;; Command takes one argument, called cell, of type puzzle-cell.
    ((cell 'puzzle-cell))
   (with-frame (frame)				;Get our hands on the frame
     (with-slots (puzzle) frame			;Get our hands on the puzzle slot
       (multiple-value-bind (this-row this-column) (decode-puzzle-cell cell)
	 (let ((open-cell (cell-adjacent-to-open-cell puzzle this-row this-column)))
	   (multiple-value-bind (open-row open-column) (decode-puzzle-cell open-cell)
	     (cond ((= open-row this-row)
		    (cond ((> open-column this-column)
			   (do ((c open-column (1- c)))
			       ((= c this-column))
			     (setf (aref puzzle this-row c)
				   (aref puzzle this-row (1- c)))))
			  (t (do ((c open-column (1+ c)))
				 ((= c this-column))
			       (setf (aref puzzle this-row c)
				     (aref puzzle this-row (1+ c)))))))
		   ((= open-column this-column)
		    (cond ((> open-row this-row)
			   (do ((r open-row (1- r)))
			       ((= r this-row))
			     (setf (aref puzzle r this-column)
				   (aref puzzle (1- r) this-column))))
			  (t (do ((r open-row (1+ r)))
				 ((= r this-row))
			       (setf (aref puzzle r this-column)
				     (aref puzzle (1+ r) this-column)))))))))
	 (setf (aref puzzle this-row this-column) 0)))))

;;; This defines a translator named move-cell that will translate a puzzle-cell
;;; into the command (com-move-cell puzzle-cell) if the puzzle-cell passes the
;;; test function.
;;; When the input context is COMMAND and the pointer is over a puzzle-cell, this
;;; translator knows how to convert that puzzle-cell into a command.
(define-presentation-to-command-translator move-cell
   (puzzle-cell					; Translator applicable only to puzzle-cells
     ;; The tester is called with the puzzle-cell under the pointer as its argument.
     ;; If the tester returns non-NIL then the body of the translator is called
     :tester ((object)
	      (multiple-value-bind (r c)
		  (decode-puzzle-cell object)
		(cell-adjacent-to-open-cell (puzzle-puzzle *frame*) r c))))
   ;; If the body is run object is bound to the puzzle-cell under the pointer.
   (object)
  `(com-move-cell ,object))

(define-puzzle-command com-scramble
    ()						;No arguments
   ;; this isn't really right, because it doesn't preserve "handedness".
   (with-frame (frame)
     (let ((ordering (list 0 1 2 3 4 5 6 7 8 9 10 11 12 13 14 15))
	   (puzzle-array (puzzle-puzzle frame)))
       (setq ordering (sort ordering
			    ;; used to be clim::named-continuation, which within
			    ;; clim is fine but in an example program isn't
			    ;; because it's not a documented functionality.
			    #'(lambda (x y)
				(declare (ignore x y))
				(zerop (random 2)))))
       ;; from group theory.  If you arrange the puzzle in a line that
       ;; goes right, down one, left, down one, right, down one, left,
       ;; you get a sequence.  Remove the blank, since this does not
       ;; affect order of the pieces.  This sequence is 15 long.  In the
       ;; square puzzle, exhanging a piece with the blank has a
       ;; permuation on the sequence.  If the space is moved along this
       ;; unraveling, it doesn't change the sequence of numbers, i.e.,
       ;; it produces a null permutation.  If it does something else, it
       ;; can be shows to be the same as an even number of exhanges,
       ;; which is what this algorithm tests for.  It does this by
       ;; finding the number of exchanges for the first piece, which is
       ;; simply it's distance away from the beginning, and then removes
       ;; it and does the same thing for the next piece.  The ordering
       ;; is the desired target configuration along the same unraveling
       ;; line.
       (let* ((strung-out (remove 0 (mapcar #'(lambda (position)
						(nth (1- position) ordering))
					    '(1 2 3 4 8 7 6 5 9 10 11 12 16 15 14 13))))
	      (can-be-done (do* ((current-string strung-out (remove piece current-string))
				 (pieces '(1 2 3 4 8 7 6 5 9 10 11 12 15 14 13) (cdr pieces))
				 (piece (first pieces) (first pieces))
				 (exchanges (position piece current-string)
					    (+ exchanges (position piece current-string))))
				((null (cdr pieces))
				 (evenp exchanges)))))
	 (cond (nil				;t if there is a notify pane
		(notify-user frame (if can-be-done "Can be done" "Cannot be done")))
	       (can-be-done)
	       (t
		(rotatef (first ordering) (second ordering)))))
       (dotimes (row 4)
	 (dotimes (column 4)
	   (setf (aref puzzle-array row column) (pop ordering)))))))

(define-puzzle-command com-exit-puzzle
    ()
   (with-frame (frame)
     ;; Call to exit the application neatly
     (stop-frame frame)))

;;; Enumerate the command names to appear in the puzzle menu group, and associate with
;;; them commands
(define-menu-group puzzle
  (("Scramble" :command '(com-scramble))
   ("Exit" :command '(com-exit-puzzle))))

;;; Add these commands to the command table named puzzle.  DEFINE-PUZZLE-COMMAND would
;;; do this for us if we had specified
;;; (define-puzzle-command (com-exit-puzzle :command-table t) . . .) for example

(add-command-to-command-table "Scramble" 'com-scramble 'puzzle)
;;; --- This need a name just to be in the command table; this is a limitation that
;;; will be fixed.
(add-command-to-command-table "Move" 'com-move-cell 'puzzle)
(add-command-to-command-table "Exit" 'com-exit-puzzle 'puzzle)

(defvar *puzzles* nil)

(defun do-puzzle (&key server-path create)
  (launch-frame 'puzzle :title "CLIM 15 Puzzle" :where server-path :create create))

