;;; -*- Mode: LISP; Package: CLIM-DEMO; Lowercase: Yes; Syntax: Common-Lisp; -*-

(in-package "CLIM-DEMO")

;;; Database support

(defvar *position-list*      nil "The position database")
(defvar *route-list*         nil "The route database")
(defvar *victor-airway-list* nil "The victor-airway database")
(defvar *aircraft-list*      nil "The aircraft database")

(defvar *max-latitude*  44)
(defvar *min-latitude*  41)
(defvar *max-longitude* 74)
(defvar *min-longitude* 70)

(defvar *magnifier* #-Cloe-Runtime (float 150)
		    #+Cloe-Runtime (float 100))

;; Given longitude,latitude return X,Y
(defun scale-coordinates (longitude latitude)
  (values (* *magnifier* (- *max-longitude* longitude))
	  (* *magnifier* (- *max-latitude* latitude))))

;; Given X,Y return longitude,latitude
(defun unscale-coordinates (x y window)
  (declare (ignore window))
  (values (- *max-longitude* (/ y *magnifier*))
	  (- *max-latitude* (/ x *magnifier*))))

(defvar *present-textually* t "Control presentation style")
(defvar *label-character-style* '(:fix :condensed :smaller))


;;; CLIM doesn't have with-own-coordinates.
(defmacro with-own-coordinates ((&optional stream &key (clear-window t))
				&body body)
  (clim::default-output-stream stream)
  `(progn
     (when ,clear-window
       (window-clear ,stream))
     ,@body))

(defmacro with-temporary-window ((stream-var parent) &body body)
  `(with-temporary-window-1
     (clim::named-continuation with-temporary-window (,stream-var)
       ,@body)
     ,parent))

(clim::defresource temp-window (parent)
  :constructor
    (open-window-stream :parent parent	;"random" size
			:left 100 :top 100 :width 1000 :height 500
			#+Cloe-Runtime
			:margin-components
			#+Cloe-Runtime
			clim::*default-menu-margin-components*)
  :initializer
    (progn (window-clear temp-window)
	   (multiple-value-call #'entity-set-size temp-window (window-inside-size parent))
	   ;; We want to clear it using a deinitializer really.
	   (setf (stream-record-p temp-window) t)))

(defun with-temporary-window-1 (continuation parent)
  (clim::using-resource (menu temp-window parent)
    (entity-set-edges menu 100 100 500 400)
    (unwind-protect
	(progn (window-expose menu)
	       (funcall continuation menu))
      (window-set-visibility menu nil))))

(defun window-root (window)
  (do ((win window (window-parent win)))
      ((null (window-parent win))
       win)))

#||
;;; This should obviously be part of CLIM

;(define-presentation-type integer (&optional low high)
;  :parser ((stream &rest args)
;	   (ignore args low high)
;	   (let* ((token (clim::read-token stream))
;		  (integer (parse-integer token)))
;	     (values integer)))
;  :printer ((object stream &rest args)
;	    (ignore args)
;	    (format stream "~D" object))
;  :object-validator ((object)
;		     (and (integerp object)
;			  (or (null low) (>= object low))
;			  (or (null high) (<= object high))))
;  :describer ((stream)
;	      (format stream "an integer")
;	      (cond ((and low high)
;		     (format stream " between ~D and ~D" low high))
;		    (low
;		     (format stream " greater than or equal to ~D" low))
;		    (high
;		     (format stream " less than or equal to ~D" high)))))

(define-presentation-type float (&optional low high)
  :parser ((stream clim::&downward-rest args)
	   (declare (ignore args low high))
	   (let* ((token (clim::read-token stream))
		  (number (read-from-string token)))
	     (values (float number))))
  :printer ((object stream clim::&downward-rest args)
	    (declare (ignore args))
	    (format stream "~D" object))
  :object-validator ((object)
		     (and (floatp object)
			  (or (null low)
			      (>= object low))
			  (or (null high)
			      (<= object high))))
  :describer ((stream)
	      (format stream "a float")
	      (cond ((and low high)
		     (format stream " between ~D and ~D" low high))
		    (low
		     (format stream " greater than or equal to ~D" low))
		    (high
		     (format stream " less than or equal to ~D" high)))))
||#

(define-presentation-type member (&rest set)
  :parser ((stream clim::&downward-rest args)
	   (declare (ignore args))
	   (values
	     (completing-from-suggestions (stream)
	       (dolist (element set)
		 (clim::suggest (symbol-name element) element)))))
  :printer ((object stream &key &allow-other-keys)
	    (write-string (string-capitalize object) stream))
  :data-args-comparator ((sub super)
			 (setq sub (first sub))
			 (setq super (first super))
			 (every #'(lambda (x) (member x super)) sub))
  :object-validator ((object)
		     (member object set)))


;;; Basic data structures - points and positions

(defclass fp-point
	  ()
  ((latitude :initarg :latitude
	     :accessor point-latitude)
   (longitude :initarg :longitude
	      :accessor point-longitude)))

(defclass position 
	  (fp-point)
  ((altitude :initarg :altitude
	     :accessor position-altitude)
   (deviation :initarg :deviation
	      :accessor position-deviation)))

(defmethod draw-position ((position position) stream &optional label)
  (with-slots (longitude latitude) position
    (multiple-value-bind (x y) (scale-coordinates longitude latitude)
      (let* ((xx (+ x (/ 3 (tan (radian 30))))))
	(clim:draw-circle* xx y 2 :stream stream)
	(draw-label label stream (+ xx 5) y)))))

(defun draw-label (label stream x y)
  (when label
    (clim:draw-string* label (+ x 5) y
		       ;;--- This doesn't work yet
		       ;;--- :character-style *label-character-style*
		       :stream stream)))

(defun distance (from-position to-position)
  (values (geodesic (point-latitude from-position) (point-longitude from-position)
		    (point-latitude to-position) (point-longitude to-position))))

(defun azimuth (from-position to-position)
  (multiple-value-bind (dist azim) 
      (geodesic (point-latitude from-position) (point-longitude from-position) 
                (point-latitude to-position) (point-longitude to-position))
    (declare (ignore dist))
    azim))

(defclass named-position
	  (position)
  ((name :initarg :name
	 :accessor position-name)
   (longname :initarg :longname
	     :accessor position-longname)))

(defmethod describe-position-object ((position named-position) stream)
  (with-slots (name longname) position
    (format stream "~:(~A~) ~A ~A"
      (class-name (class-of position)) name longname)))


(eval-when (compile load eval)

(defun degminsec (degrees &optional (minutes 0) (seconds 0))
  (float (/ (+ (* degrees 3600) (* minutes 60) seconds) 3600)))

(defun getdegminsec (angle-in-degrees)
  (let* ((angle (* 3600 angle-in-degrees))
	 (seconds (rem angle 60))
	 (minutes (rem (/ angle 60) 60))
	 (degrees (/ angle 3600)))
    (values (floor degrees) (floor minutes) (floor seconds))))

(defun coast-segment (latitudes longitudes)
  (let* ((result (make-list (* 2 (length latitudes))))
	 (rpos result))
    (loop
      (when (null rpos) (return result))
      (setf (car rpos) (pop longitudes))
      (setf rpos (cdr rpos))
      (setf (car rpos) (pop latitudes))
      (setf rpos (cdr rpos)))))

)	;eval-when

(defvar *coastline* 
      (list
	(coast-segment				; Boston coastline
	  (list (degminsec 40 48) (degminsec 41 10) (degminsec 41 15) (degminsec 41 20)
		(degminsec 41 25) 
		(degminsec 41 30) (degminsec 41 30) (degminsec 41 35) (degminsec 41 40) 
		(degminsec 41 45) (degminsec 41 45) (degminsec 41 40) (degminsec 41 35) 
		(degminsec 41 30) (degminsec 41 35) (degminsec 41 40) (degminsec 41 45) 
		(degminsec 41 50) (degminsec 41 55) (degminsec 42 00) (degminsec 42 05) 
		(degminsec 42 05) (degminsec 42 00) (degminsec 41 55) (degminsec 41 50)
		(degminsec 41 45) (degminsec 41 45) (degminsec 41 50) (degminsec 41 55)
		(degminsec 42 00) (degminsec 42 05) (degminsec 42 10) (degminsec 42 15) 
		(degminsec 42 20) (degminsec 42 25) (degminsec 42 30) (degminsec 42 35) 
		(degminsec 42 40) (degminsec 42 45) (degminsec 42 50) (degminsec 42 55) 
		(degminsec 43 00) (degminsec 43 05) (degminsec 43 10) (degminsec 43 15) 
		(degminsec 43 20) (degminsec 43 25) (degminsec 43 30) (degminsec 43 35) 
		(degminsec 43 40) (degminsec 43 45) (degminsec 43 50) (degminsec 43 55) 
		(degminsec 43 60))
	  (list (degminsec 73 47) (degminsec 73 07) (degminsec 72 58) (degminsec 71 45)
		(degminsec 71 30)
		(degminsec 71 25) (degminsec 71 02) (degminsec 70 57) (degminsec 70 46)
		(degminsec 70 42) (degminsec 70 39) (degminsec 70 39) (degminsec 70 39)
		(degminsec 70 40) (degminsec 70 28) (degminsec 69 57) (degminsec 69 56)
		(degminsec 69 56) (degminsec 69 58) (degminsec 70 01) (degminsec 70 13)
		(degminsec 70 14) (degminsec 70 05) (degminsec 70 05) (degminsec 70 00)
		(degminsec 70 10) (degminsec 70 24) (degminsec 70 32) (degminsec 70 33)
		(degminsec 70 42) (degminsec 70 39) (degminsec 70 42) (degminsec 70 46)
		(degminsec 71 00) (degminsec 71 00) (degminsec 70 50) (degminsec 70 41)
		(degminsec 70 37) (degminsec 70 48) (degminsec 70 49) (degminsec 70 48)
		(degminsec 70 45) (degminsec 70 40) (degminsec 70 47) (degminsec 70 46)
		(degminsec 70 42) (degminsec 70 23) (degminsec 70 23) (degminsec 70 13)
		(degminsec 70 14) (degminsec 70 12) (degminsec 70 00) (degminsec 69 27)
		(degminsec 69 08)))
	(coast-segment				; Martha's vinyard
	  (list 
	    (degminsec 41 29) (degminsec 41 28) (degminsec 41 25) (degminsec 41 24) 
	    (degminsec 41 23) (degminsec 41 24) (degminsec 41 21) (degminsec 41 21) 
	    (degminsec 41 20) (degminsec 41 18) (degminsec 41 22) (degminsec 41 22) 
	    (degminsec 41 25) (degminsec 41 27) (degminsec 41 29))
	  (list
	    (degminsec 70 36) (degminsec 70 34) (degminsec 70 33) (degminsec 70 31)
	    (degminsec 70 30) (degminsec 70 28) (degminsec 70 27) (degminsec 70 44)
	    (degminsec 70 45) (degminsec 70 46) (degminsec 70 50) (degminsec 70 45)
	    (degminsec 70 44) (degminsec 70 41) (degminsec 70 36)))
	(coast-segment				; Nantucket
	  (list
	    (degminsec 41 24) (degminsec 41 20) (degminsec 41 15) (degminsec 41 14) 
	    (degminsec 41 14) (degminsec 41 16) (degminsec 41 18) (degminsec 41 18) 
	    (degminsec 41 19) (degminsec 41 24))
	  (list
	    (degminsec 70 03) (degminsec 70 00) (degminsec 69 57) (degminsec 70 00)
	    (degminsec 70 07) (degminsec 70 12) (degminsec 70 11) (degminsec 70 03)
	    (degminsec 70 01) (degminsec 70 03)))
	(coast-segment				; Block Island
	  (list
	    (degminsec 41 14) (degminsec 41 13) (degminsec 41 11) (degminsec 41 09) 
	    (degminsec 41 08) (degminsec 41 09) (degminsec 41 12) (degminsec 41 14))
	  (list
	    (degminsec 71 34) (degminsec 71 33) (degminsec 71 34) (degminsec 71 32)
	    (degminsec 71 36) (degminsec 71 37) (degminsec 71 35) (degminsec 71 34)))))

(defun draw-coastline (coastline &optional (stream *standard-output*))
  (with-scaling* (stream (- *magnifier*))
    (with-translation* (stream (- *max-longitude*) (- *max-latitude*))
      (dolist (coast coastline)
	(draw-polygon* coast :filled nil :closed nil :thickness 2 :stream stream)))))


;;; Concrete position objects

(defclass airport
	  (named-position)
  ())

(defmethod draw-position ((airport airport) stream &optional label)
  (with-slots (longitude latitude) airport
    (multiple-value-bind (x y) (scale-coordinates longitude latitude)
      (let ((color-args (and (color-stream-p stream)
			     (list :ink *color-green*))))
	(apply #'draw-circle* x y 5 :stream stream color-args))
      (draw-line* x (- y 2) x (+ y 2) :ink *background* :thickness 2 :stream stream)
      (draw-label label stream (+ x 5) y))))

(defclass waypoint
	  (named-position)
  ())

(defclass VOR
	  (named-position)
  ())

(defmethod describe-position-object ((VOR VOR) stream)
  (with-slots (name longname) VOR
    (format stream "VOR ~A ~A" name longname)))

(defmethod draw-position ((VOR VOR) stream &optional label)
  (with-slots (longitude latitude) VOR
    (multiple-value-bind (x y) (scale-coordinates longitude latitude)
      (let ((xx (+ x (/ 3 (tan (radian 30)))))
	    (color-args (and (color-stream-p stream)
			     (list :ink *color-cyan*))))
	(apply #'draw-hexagon (+ xx 5) (- y 3) (+ xx 5) (+ y 3) stream color-args)
	(apply #'draw-circle* xx y 2 :stream stream color-args)
	(draw-label label stream (+ xx 5) y)))))

(defun draw-hexagon (x1 y1 x2 y2 stream &rest color-args)
  (let* ((n 6)
	 (theta (* pi (1- (/ 2.0 n))))
	 (sin-theta (sin theta))
	 (cos-theta (cos theta)))
    (do ((i 1 (1+ i))
	 (x3) (y3))
	((not (<= i n)))
      (setq x3 (+ (- (- (* x1 cos-theta)
			(* y1 sin-theta))
		     (* x2 (1- cos-theta)))
		  (* y2 sin-theta))
	    y3 (- (- (+ (* x1 sin-theta)
			(* y1 cos-theta))
		     (* x2 sin-theta))
		  (* y2 (1- cos-theta))))
      (apply #'draw-line* x1 y1 x2 y2 :stream stream color-args)
      (setq x1 x2 y1 y2 x2 x3 y2 y3))))

(defclass NDB
	  (named-position)
  ())

(defmethod describe-position-object ((NDB NDB) stream)
  (with-slots (name longname) NDB
    (format stream "NDB ~A ~A" name longname)))

(defclass intersection
	  (named-position)
  ())

(defmethod draw-position ((intersection intersection) stream &optional label)
  (with-slots (longitude latitude) intersection
    (multiple-value-bind (x y) (scale-coordinates longitude latitude)
      (let ((color-args (and (color-stream-p stream)
			     (list :ink *color-magenta*))))
	(apply #'draw-triangle* x (- y 3) (- x 3) (+ y 2) (+ x 3) (+ y 2)
	       :stream stream color-args))
      (draw-label label stream (+ x 5) y))))

(defclass visual-checkpoint
	  (named-position)
  ())


;; User interfaces to concrete position objects

(defun concrete-position-parser (type stream)
  (let ((object (completing-from-suggestions
		  (stream ;;--- :type type
		    ;;--- :partial-completers '(#\space)
		    )
		  (dolist (position *position-list*)
		    (when (typep position type)
		      (clim::suggest (position-name position) position))))))
    object))

(define-presentation-type position ()
  :parser ((stream clim::&downward-rest args)
	   (declare (ignore args))
	   (concrete-position-parser 'position stream))
  :printer ((object stream &key acceptably)
	    (if (or *present-textually* acceptably)
		(format stream "~A" (position-name object))
	      (draw-position object stream
			     (and (typep object 'named-position)
				  (position-name object)))))
  :describer ((stream)
	      (write-string "a position" stream)))

(define-presentation-type airport ()
  :expander 'position
  :parser ((stream clim::&downward-rest args)
	   (declare (ignore args))
	   (concrete-position-parser 'airport stream))
  :describer ((stream) (write-string "an airport" stream)))

(define-presentation-type VOR ()
  :expander 'position
  :parser ((stream clim::&downward-rest args)
	   (declare (ignore args))
	   (concrete-position-parser 'VOR stream))
  :describer ((stream) (write-string "a VOR" stream)))

(define-presentation-type NDB ()
  :expander 'position
  :parser ((stream clim::&downward-rest args)
	   (declare (ignore args))
	   (concrete-position-parser 'NDB stream))
  :describer ((stream) (write-string "an NDB" stream)))

(define-presentation-type intersection ()
  :expander 'position
  :parser ((stream clim::&downward-rest args)
	   (declare (ignore args))
	   (concrete-position-parser 'intersection stream))
  :describer ((stream) (write-string "an intersection" stream)))

(define-presentation-type visual-checkpoint ()
  :expander 'position
  :parser  ((stream clim::&downward-rest args)
	    (declare (ignore args))
	    (concrete-position-parser 'visual-checkpoint stream))
  :describer ((stream) (write-string "a Visual-Checkpoint" stream)))


;;; Route objects

(defclass basic-route-segment
	  ()
  ((at :initarg :at
       :accessor route-segment-at)))

(defclass route-segment
	  (basic-route-segment)
  ((altitude :initarg :altitude :accessor route-segment-altitude)
   (wind-info :initarg :wind-info :accessor route-segment-wind-info)))

(defclass basic-route
	  ()
  ((name :initarg :name :accessor route-name)
   (legs :initarg :legs :accessor route-legs)))

(defmethod describe-position-object ((route basic-route) stream)
  (with-slots (name legs) route
    (format stream "Route ~A ~A" name legs)))

(defmethod draw-route ((route basic-route) stream &rest drawing-args)
  (with-slots (legs) route
    (let* ((start-pos (first legs))
	   (start-lat (route-segment-latitude start-pos))
	   (start-lon (route-segment-longitude start-pos)))
      (do* ((next-legs (cdr legs) (cdr next-legs))
	    next-pos next-lat next-lon)
	   ((null next-legs) nil)
	(setq next-pos (car next-legs)
	      next-lat (route-segment-latitude next-pos)
	      next-lon (route-segment-longitude next-pos))
	(multiple-value-bind (xfrom yfrom) (scale-coordinates start-lon start-lat)
	  (multiple-value-bind (xto yto) (scale-coordinates next-lon next-lat)
	    (apply #'draw-line* xfrom yfrom xto yto
		   :stream stream
		   drawing-args)))
	(setq start-lat next-lat start-lon next-lon)))))

(defclass route
	  (basic-route)
  ())

(defun route-segment-position-name (route-segment)
  (position-name (route-segment-at route-segment)))

(defun route-segment-position-longname (route-segment)
  (position-longname (route-segment-at route-segment)))

(defun route-segment-leg-name (route-segment)
  (concatenate 'string "-" (route-segment-position-name route-segment)))

(defun generate-route-name-from-legs (leg-list)
  (apply #'concatenate 'string 
	 (route-segment-position-name (car leg-list))
	 (mapcar #'route-segment-leg-name (cdr leg-list))))

(defun route-segment-latitude (route-segment)
  (point-latitude (route-segment-at route-segment)))

(defun route-segment-longitude (route-segment)
  (point-longitude (route-segment-at route-segment)))


;;; Victor Airways

(defclass victor-airway-segment
	  (basic-route-segment)
  ((properties :accessor victor-airway-segment-properties)
   (next-leg :accessor victor-airway-segment-next-leg)))

(defclass victor-airway
	  (basic-route)
  ())

(defun route-parser (type list stream)
  (let ((object (completing-from-suggestions
		  (stream ;;--- :type type
		    ;;--- :partial-completers '(#\space)
		    )
		  (dolist (aroute list)
		    (when (typep aroute type)
		      (clim::suggest (route-name aroute) aroute))))))
    object))

(define-presentation-type route ()
  :parser ((stream clim::&downward-rest args)
	   (declare (ignore args))
	   (route-parser 'route *route-list* stream))
  :printer ((object stream &key acceptably)
	    (if (or *present-textually* acceptably)
		(format stream "~A" (route-name object))
	      (let ((drawing-args (if (color-stream-p stream)
				      (list :ink *color-red*)
				    '(:dashed t))))
		(apply #'draw-route object stream drawing-args))))
  :describer ((stream) (write-string "a route" stream))
  :highlight-function ((presentation stream state)
		       (highlight-route (presentation-object presentation) stream state)))

(defun highlight-route (route stream state)
  (declare (ignore state))
  (draw-route route stream :thickness 2 :ink *flipping-ink*))

(define-presentation-type victor-airway ()
  :parser ((stream clim::&downward-rest args)
	   (declare (ignore args))
	   (route-parser 'route *victor-airway-list* stream))
  :printer ((object stream &key acceptably)
	    (if (or *present-textually* acceptably)
		(format stream "~A" (route-name object))
	      (let ((drawing-args (if (color-stream-p stream)
				      (list :ink *color-blue*)
				    '(:dashed nil))))
		(apply #'draw-route object stream drawing-args))))
  :describer ((stream) (write-string "a Victor-airway" stream))
  :highlight-function ((presentation stream state)
		       (highlight-route (presentation-object presentation) stream state)))

;;; Preferred Routes

;;; ADIZ

;;; Airspace

;;; TCA
;;; ARSA
;;; Warning
;;; Restricted
;;; Prohibited
;;; MOA


;;; Aircraft description

(defclass aircraft
	  ()
  ((identification :initarg :identification	; Aircraft tail number
		   :accessor aircraft-identification)
   (type :initarg :type				; eg C-172
	 :accessor aircraft-type)
   (taxi-fuel :initarg :taxi-fuel		; fuel used for taxi&runup (estimate)
	      :accessor aircraft-taxi-fuel)
   (preferred-cruising-altitude :initarg :preferred-cruising-altitude
				:accessor aircraft-preferred-cruising-altitude)
   (normal-cruise-speed :initarg :normal-cruise-speed
			:accessor aircraft-normal-cruise-speed)
   (fuel-consumption-at-normal-cruise :initarg :fuel-consumption-at-normal-cruise
				      :accessor aircraft-fuel-consumption-at-normal-cruise)
   (maximum-usable-fuel :initarg :maximum-usable-fuel
			:accessor aircraft-maximum-usable-fuel)
   (cost-per-hour :initarg :cost-per-hour
		  :accessor aircraft-cost-per-hour)
   (hobs-or-tach :initarg :hobs-or-tach
		 :accessor aircraft-hobs-or-tach)))

(define-presentation-type aircraft ()
  :parser ((stream clim::&downward-rest args)
	   (declare (ignore args))
	   (let ((ac (completing-from-suggestions
		       (stream ;;--- :type 'aircraft
			 ;;--- :partial-completers '(#\space)
			 )
		       (dolist (aircraft *aircraft-list*)
			 (clim::suggest 
			   (aircraft-identification aircraft) aircraft)))))
	     ac))
  :printer ((object stream &key acceptably)
	    (if acceptably
		(format stream "~A" (aircraft-identification object))
	      (format stream "~A" (aircraft-identification object))))
  :describer ((stream) (write-string "an aircraft" stream)))

(defvar *last-plane* nil "The last plane referred to")

(defun edit-aircraft (aircraft)
  (multiple-value-bind 
    (identification type preferred-cruising-altitude normal-cruise-speed
     fuel-consumption-at-normal-cruise maximum-usable-fuel cost-per-hour)
      (with-temporary-window (*query-io* (window-root *query-io*))
	(accepting-values (*query-io*
			    ;;--- :label "Aircraft parameters"
			    )
	  (values
	    (prog1 (accept 'string :prompt "Identification" 
			   :default (aircraft-identification aircraft)
			   :query-identifier 'identification)
		   (terpri *query-io*))
	    (prog1 (accept 'string :prompt "Type" 
			   :default (aircraft-type aircraft)
			   :query-identifier 'type)
		   (terpri *query-io*))
	    (prog1 (accept 'integer :prompt "Preferred cruising altitude" 
			   :default (aircraft-preferred-cruising-altitude aircraft)
			   :query-identifier 'preferred-cruising-altitude)
		   (terpri *query-io*))
	    (prog1 (accept 'integer :prompt "Normal cruise speed" 
			   :default (aircraft-normal-cruise-speed aircraft)
			   :query-identifier 'normal-cruise-speed)
		   (terpri *query-io*))
	    (prog1 (accept 'float :prompt "Fuel consumption at normal cruise" 
			   :default (aircraft-fuel-consumption-at-normal-cruise aircraft)
			   :query-identifier 'fuel-consumption)
		   (terpri *query-io*))
	    (prog1 (accept 'float :prompt "Maximum usable fuel" 
			   :default (aircraft-maximum-usable-fuel aircraft)
			   :query-identifier 'maximum-usable-fuel)
		   (terpri *query-io*))
	    (prog1 (accept 'float :prompt "Cost per hour" 
			   :default (aircraft-cost-per-hour aircraft)
			   :query-identifier 'cost-per-hour)
		   (terpri *query-io*)))))
    (setf (aircraft-identification aircraft) identification)
    (setf (aircraft-type aircraft) type)
    (setf (aircraft-taxi-fuel aircraft) 0)
    (setf (aircraft-preferred-cruising-altitude aircraft) preferred-cruising-altitude)
    (setf (aircraft-normal-cruise-speed aircraft) normal-cruise-speed)
    (setf (aircraft-fuel-consumption-at-normal-cruise aircraft) fuel-consumption-at-normal-cruise)
    (setf (aircraft-maximum-usable-fuel aircraft) maximum-usable-fuel)
    (setf (aircraft-cost-per-hour aircraft) cost-per-hour)))


;;; Flight plans

(defclass flight-plan
	  ()
  ((type :initarg :type
	 :accessor flight-plan-type)
   (aircraft-id :initarg :aircraft-id
		:accessor flight-plan-aircraft-id)
   (aircraft-type :initarg :aircraft-type
		  :accessor flight-plan-aircraft-type)
   (true-speed :initarg :true-speed
	       :accessor flight-plan-true-speed)
   (departure-point :initarg :departure-point
		    :accessor flight-plan-departure-point)
   (departure-time :initarg :departure-time
		   :accessor flight-plan-departure-time)
   (cruising-alt :initarg :cruising-alt
		 :accessor flight-plan-cruising-alt)
   (route :initarg :route
	  :accessor flight-plan-route)
   (destination :initarg :destination
		:accessor flight-plan-destination)
   (ete :initarg :ete
	:accessor flight-plan-ete)
   (remarks :initarg :remarks
	    :accessor flight-plan-remarks)
   (fuel-on-board :initarg :fuel-on-board
		  :accessor flight-plan-fuel-on-board)
   (alternate :initarg :alternate
	      :accessor flight-plan-alternate)
   (pilot :initarg :pilot
	  :accessor flight-plan-pilot)
   (souls :initarg :souls
	  :accessor flight-plan-souls)
   (color :initarg :color
	  :accessor flight-plan-color)))

(defun compute-flight-plan (fp-stream plan)
  (let* ((route (flight-plan-route plan))
	 (plane (flight-plan-aircraft-id plan))
	 (leg-list (route-legs route))
	 (start (first leg-list))
	 (end   (car (last leg-list))))
    (progn ;clim::surrounding-output-with-border (fp-stream)
      (multiple-value-bind (distance true-course)
	  (geodesic (route-segment-latitude start)
		    (route-segment-longitude start)
		    (route-segment-latitude end)
		    (route-segment-longitude end))
	(declare (ignore true-course))
	(format fp-stream 
	    "~&Flight Plan from ~A and ~A:~%The great circle distance is ~3,1F NM.~%"
	  (route-segment-position-name start) (route-segment-position-name end)
	  distance))
      (format fp-stream  "~&Route: [ ")
      (dolist (waypoint leg-list)
	(format fp-stream "~A " (route-segment-position-name waypoint)))
      (format fp-stream "].~%")
      (format fp-stream  "~&Plane: ~A ~A.~%"
	(aircraft-identification plane) (aircraft-type plane))
      (let ((Total-Distance 0)
	    (Total-Time-Enroute 0))
	(do* ((currently-at start)
	      (route-to (cdr leg-list) (cdr route-to)))
	     ((null route-to) nil)
	  (multiple-value-bind (leg-distance leg-true-course)
	      (geodesic (route-segment-latitude currently-at)
			(route-segment-longitude currently-at)
			(route-segment-latitude (car route-to))
			(route-segment-longitude (car route-to)))
	    (declare (ignore leg-true-course))
	    (setq total-distance (+ total-distance leg-distance))
	    (setq currently-at  (car route-to))))
	(format fp-stream "~%")
	(formatting-table (fp-stream)
	  (formatting-row (fp-stream)
	    (with-text-face (:italic fp-stream)
	      (progn ;--- with-underlining (fp-stream)
		(formatting-cell (fp-stream) (write-string "CHECKPOINT" fp-stream))
		(formatting-cell (fp-stream :align-x :right) (write-string "ID" fp-stream))
		(formatting-cell (fp-stream :align-x :right) (write-string "TC" fp-stream))
		(formatting-cell (fp-stream :align-x :right) (write-string "Leg" fp-stream))
		(formatting-cell (fp-stream :align-x :right) (write-string "Rem" fp-stream))
		(formatting-cell (fp-stream :align-x :right) (write-string "MC" fp-stream))
		(formatting-cell (fp-stream :align-x :right) (write-string "MH" fp-stream))
		(formatting-cell (fp-stream :align-x :right) (write-string "GS" fp-stream))
		(formatting-cell (fp-stream :align-x :right) (write-string "ETE" fp-stream))
		(formatting-cell (fp-stream :align-x :right) (write-string "ETA" fp-stream))
		(formatting-cell (fp-stream :align-x :right) (write-string "FUEL" fp-stream)))))
	  (setq total-time-enroute 0)		; yuck, gets calculated twice!
	  (do* ((Currently-At start)
		(rem total-distance)
		(route-to (cdr leg-list) (cdr route-to)))	       
	       ((null route-to) nil)
	    (multiple-value-bind (leg-distance leg-true-course)
		(geodesic (route-segment-latitude currently-at) 
			  (route-segment-longitude currently-at) 
			  (route-segment-latitude (car route-to)) 
			  (route-segment-longitude (car route-to)))
	      (let* ((altitude (flight-plan-cruising-alt plan))
		     (estimated-wind (estimate-wind-at currently-at altitude))
		     (cruising-speed (aircraft-normal-cruise-speed plane))
		     (fuel-rate (aircraft-fuel-consumption-at-normal-cruise plane))
		     (deviation (position-deviation (route-segment-at currently-at)))
		     (MC (+ leg-true-course deviation)))
		(multiple-value-bind (th gs)
		    (true-heading-and-groundspeed leg-true-course cruising-speed 
						  (car estimated-wind) (cadr estimated-wind))
		  (let* ((MH (+ th deviation))
			 (leg-time (/ leg-distance gs))
			 (eta 
			   (+ total-time-enroute leg-time (flight-plan-departure-time plan)))
			 (fuel (* leg-time fuel-rate)))
		    (setq total-time-enroute (+ total-time-enroute leg-time))
		    (formatting-row (fp-stream)
		      (formatting-cell (fp-stream) 
			(format fp-stream "~A"
			  (route-segment-position-longname currently-at)))
		      (formatting-cell (fp-stream :align-x :right)	;ID
			(format fp-stream "~A"
			  (route-segment-position-name currently-at)))
		      (formatting-cell (fp-stream :align-x :right)	;TC
			(format fp-stream "~D" (floor leg-true-course))	)
		      (formatting-cell (fp-stream :align-x :right)	;Leg
			(format fp-stream "~1,1F" leg-distance))
		      (formatting-cell (fp-stream :align-x :right)	;Rem
			(format fp-stream "~1,1F" rem))
		      (formatting-cell (fp-stream :align-x :right)	;MC
			(format fp-stream "~D" (floor MC)))
		      (formatting-cell (fp-stream :align-x :right)	;MH
			(format fp-stream "~D" (floor MH)))
		      (formatting-cell (fp-stream :align-x :right)	;GS
			(format fp-stream "~D" (floor GS)))
		      (formatting-cell (fp-stream :align-x :right)	;ETE
			(format fp-stream "~A" (time-hhmm leg-time)))
		      (formatting-cell (fp-stream :align-x :right)	;ETA
			(format fp-stream "~A" (time-hhmm ETA)))
		      (formatting-cell (fp-stream :align-x :right)	;Fuel
			(format fp-stream "~1,1F" Fuel))))))
	      (setq Currently-at  (car route-to) rem (setq rem (- rem leg-distance))))))
	(formatting-table (fp-stream)
	  (formatting-row (fp-stream)
	    (formatting-cell (fp-stream) 
	      (format fp-stream "~A" (route-segment-position-longname end)))
	    (formatting-cell (fp-stream :align-x :right)	;ID
	      (format fp-stream "~A" (route-segment-position-name end)))))
	(format fp-stream "~%")
	(let* ((departure-time (flight-plan-departure-time plan))
	       (final-eta (+ Total-time-enroute (flight-plan-departure-time plan)))
	       (fuel-on-board (flight-plan-fuel-on-board plan))
	       (fuel-consumption-at-cruise (aircraft-fuel-consumption-at-normal-cruise plane))
	       (total-fuel-used
		 (+ (aircraft-taxi-fuel plane)
		    (* total-time-enroute fuel-consumption-at-cruise)))
	       (average-fuel-usage 
		 (/ (* total-time-enroute fuel-consumption-at-cruise) Total-time-enroute))
	       (cruising-altitude (flight-plan-cruising-alt plan))
	       (reserve-fuel (- fuel-on-board total-fuel-used))
	       (reserve-time (/ reserve-fuel fuel-consumption-at-cruise))
	       (true-airspeed (flight-plan-true-speed plan))
	       (reserve-distance (* reserve-time true-airspeed)))
	  (format fp-stream "~%")
	  (formatting-table (fp-stream)
	    (formatting-row (fp-stream)
	      (with-text-face (:italic fp-stream)
		(formatting-cell (fp-stream) (format fp-stream ""))
		(formatting-cell (fp-stream :align-x :center)
		  (write-string "A N A L Y S I S" fp-stream))))
	    (formatting-row (fp-stream)
	      (formatting-cell (fp-stream) 
		(format fp-stream "Depart at ~A" (time-hhmm departure-time)))
	      (formatting-cell (fp-stream) 
		(format fp-stream "Total Time ~A" (time-hhmm Total-time-enroute)))
	      (setf (flight-plan-ete plan) total-time-enroute)
	      (formatting-cell (fp-stream) 
		(format fp-stream "Final ETA ~A" (time-hhmm final-eta))))
	    (formatting-row (fp-stream)
	      (formatting-cell (fp-stream) 
		(format fp-stream "Fuel on board ~1,1F" fuel-on-board))
	      (formatting-cell (fp-stream)
		(format fp-stream "Total Fuel ~1,1F gallons" total-fuel-used))
	      (formatting-cell (fp-stream) 
		(format fp-stream "Average fuel usage ~1,1F/hr" average-fuel-usage)))
	    (formatting-row (fp-stream)
	      (formatting-cell (fp-stream) 
		(format fp-stream "Total Distance ~1,1F nm" Total-distance))
	      (formatting-cell (fp-stream) 
		(format fp-stream "Total Time ~A" (time-hhmm Total-time-enroute)))	;+++
	      (formatting-cell (fp-stream) 
		(format fp-stream "Total Fuel ~1,1F gallons" total-fuel-used)))	;+++
	    (formatting-row (fp-stream)
	      (formatting-cell (fp-stream) 
		(format fp-stream "Cruise altitude ~A" cruising-altitude))    
	      (formatting-cell (fp-stream) 
		(format fp-stream "TAS ~A" true-airspeed))
	      #+ignore
	      (formatting-cell (fp-stream) 
		(format fp-stream "" #+ignore "CAS ~A" #+ignore 0)))	;unf +++
	    (formatting-row (fp-stream)
	      (formatting-cell (fp-stream) 
		(format fp-stream "Reserve Time ~A" (time-hhmm reserve-time)))
	      (formatting-cell (fp-stream) 
		(format fp-stream "Reserve Dist ~1,1F" reserve-distance))
	      (formatting-cell (fp-stream) 
		(format fp-stream "Reserve Fuel ~1,1F" reserve-fuel)))
	    ;+++ unf
	    (formatting-row (fp-stream)
	      (formatting-cell (fp-stream) 
		(format fp-stream "Cost @ $~1,2F/hr = $~1,2F"
		  (aircraft-cost-per-hour plane)
		  (* total-time-enroute (aircraft-cost-per-hour plane))))))
	  ;; Output any known wind info.
	  (format fp-stream "~%Winds Aloft~%")
	  (dolist (waypoint leg-list)
	    (let ((waypoint-printed nil))
	      (dolist (alt '(sfc 3000 6000 9000 12000))
		(let ((awind (cdr (assoc alt (route-segment-wind-info waypoint)))))
		  (when awind
		    (unless waypoint-printed
		      (format fp-stream "~%~A~%" (route-segment-position-longname waypoint)))
		    (setq waypoint-printed t)
		    (format fp-stream "~7A: ~A@~A~%" alt (car awind) (cadr awind))))))))))
    (format fp-stream "~%")))

;;; This needs to be a lot smarter!
(defun estimate-wind-at (waypoint altitude)
  (or (cdr (assoc altitude (route-segment-wind-info waypoint))) (list 0 0)))


(defun location-parser (stream compass-points error-string)
  (flet ((read-a-space ()
	   (let ((delim (read-gesture :stream stream)))
	     (unless (and (characterp delim)
			  (char-equal delim #\space))
	       ;;--- Should be (sys:parse-ferror error-string)
	       (beep)))))
    (with-blip-characters (#\Space)
      (let ((compass-point (accept `(member ,@compass-points)
				   :prompt nil 
				   :stream stream)))
	(read-a-space)
	(let ((hours (accept '(integer 0 90)
			     :prompt nil 
			     :stream stream)))
	  (read-a-space)
	  (let ((minutes (accept '(float 0.0 60.0)
				 :prompt nil 
				 :stream stream)))
	    (let ((seconds (cond ((not (= minutes (floor minutes)))
				  0)
				 (t
				  (read-a-space)
				  (accept '(integer 0 60) :prompt nil :stream stream)))))
	      (/ (+ seconds (* minutes 60) (* hours 3600))
		 (if (eq compass-point (first compass-points)) 3600.0 -3600.00)))))))))

(defun location-printer (object stream compass-points &key acceptably)
  (if acceptably
      (format stream "~A" object)
    (format stream "~A ~3,'0D ~2,2F" 
      (if (< object 0) (second compass-points) (first compass-points))
      (floor (abs object))
      (- (* (abs object) 60) (* (floor (abs object)) 60)))))

(define-presentation-type longitude ()
  :parser ((stream clim::&downward-rest args)
	   (declare (ignore args))
	   (location-parser stream '(W E) 
			    "Longitude components must be delimited by spaces."))
  :printer ((object stream &key acceptably)
	    (location-printer object stream '(W E) :acceptably acceptably))
  :describer ((stream) (write-string "a Longitude" stream)))

(define-presentation-type latitude ()
  :parser ((stream clim::&downward-rest args)
	   (declare (ignore args))
	   (location-parser stream '(N S)
			    "Latitude components must be delimited by spaces."))
  :printer ((object stream &key acceptably)
	    (location-printer object stream '(N S) :acceptably acceptably))
  :describer ((stream) (write-string "a Latitude" stream)))


(defun time-hhmmss (time-in-hours)
  (let* ((time-in-seconds (floor (* time-in-hours 3600)))
	 (hours (floor time-in-hours))
	 (minutes (- (floor time-in-seconds 60) (* hours 60)))
	 (seconds (- time-in-seconds (* hours 3600) (* minutes 60))))
    (if (zerop hours)
	(format nil "~1D:~2,'0D" minutes seconds)
      (format nil "~1D:~2,'0D:~2,'0D" hours minutes seconds))))

(defun time-hhmm (time-in-hours)
  (let* ((time-in-seconds (floor (* time-in-hours 3600)))
	 (hours (floor time-in-hours))
	 (minutes (- (floor time-in-seconds 60) (* hours 60))))
    (if (zerop hours)
	(format nil ":~2,'0D" minutes)
      (format nil "~1D:~2,'0D" hours minutes))))

(define-presentation-type time ()
  :parser ((stream clim::&downward-rest args)
	   (declare (ignore args))
	   (let ((hours (with-blip-characters ('(#\:))
			  (accept '(integer 0 24)
				  :prompt nil 
				  :stream stream))))
	     (let ((delim (read-gesture :stream stream)))
	       (unless (and (characterp delim)
			    (char-equal delim #\:))
		 ;;--- Should be (sys:parse-ferror "Time components must be delimited by colons.")
		 (beep)))
	     (let ((minutes (accept '(integer 0 60)
				    :prompt nil 
				    :stream stream)))
	       (/ (+ (* hours 60) minutes) 60))))
  :printer ((object stream &key acceptably)
	    (if acceptably
		(format stream "~A" object)
	      (format stream "~A" (time-hhmm object))))
  :describer ((stream) (write-string "a time" stream)))

(define-presentation-type wind ()
  :parser ((stream clim::&downward-rest args)
	   (declare (ignore args))
	   (with-blip-characters (#\@)
	     (let ((direction (accept '(integer 0 360)
				      :prompt nil 
				      :stream stream)))
	       (let ((delim (read-gesture :stream stream)))
		 (unless (and (characterp delim)
			      (char-equal delim #\@))
		   ;;--- Should be (sys:parse-ferror "Wind components must be delimited by atsigns.")
		   (beep)))
	       (let ((speed (accept '(integer 0 300)
				    :prompt nil 
				    :stream stream)))
		 (list direction speed)))))
  :printer ((object stream &key acceptably)
	    (if acceptably
		(format stream "~A" object)
	      (format stream "~A@~A" (first object) (second object))))
  :describer ((stream) (write-string "a wind" stream)))


;;; Viewport scaling



;;; Flight-Planner user interface

(define-application Flight-Planner
		    ((fp-window :initform nil))
  
  :subwindows ((title :title)
	       (display :application)
	       (commands :command-menu)
	       (interactor :interactor))
  :layout ((default
	     (:column 1
	      (title :compute)
	      (display :rest)
	      (commands :compute)
	      (interactor 1/4)))))

(defmethod initialize-fp-window ((fp flight-planner))
  (with-slots (fp-window clim::top-level-window) fp
    (unless fp-window
      (setf fp-window
	    (open-window-stream 
	      :parent (window-root clim::top-level-window)
	      :left 50 :top 50 :width 550 :height 350
	      :save-under T)))))

(defmethod application-standard-output ((p flight-planner))
  (get-pane p 'display))

(defmethod clim::application-query-io ((p flight-planner))
  (get-pane p 'interactor))

(defparameter *color-stream-p* t)
(defun color-stream-p (stream) *color-stream-p*)	;--- kludge

(defun run-flight-planner (&key root)
  (let ((fp (let ((*default-margin-components* nil))
	      (make-application 'flight-planner :parent root))))
    ;;(window-expose (application-top-level-window *fp*)))
    ;; (map nil #'window-clear (slot-value *fp* 'clim::subwindows))
    ;;(map nil #'window-expose (slot-value *fp* 'clim::subwindows))
    (catch 'exit-flight-planner-demo
      (run-application-top-level fp))
    (setf (window-visibility (application-top-level-window fp)) nil)))

(define-Flight-Planner-command
   (com-Exit-Flight-Planner :menu-accelerator "Exit")
  ()
  ;; assume called via run-flight-planner
  (throw 'exit-flight-planner-demo nil))


;;; Database commands and support

(define-presentation-type latitude-and-longitude ()
  :parser ((stream clim::&downward-rest args)
	   (declare (ignore args))
	   (with-blip-characters (#\,)
	     (let ((latitude (accept 'latitude :stream stream)))
	       (let ((char (read-gesture :stream stream)))
		 (unless (and (characterp char)
			      (char-equal char #\,))
		   (beep)))
	       (let ((longitude (accept 'longitude :stream stream)))
		 (values (list latitude longitude))))))
  :printer ((list stream &key &allow-other-keys)
	    (present (first list) 'latitude :stream stream)
	    (write-char #\, stream)
	    (present (second list) 'longitude :stream stream)))

;; Allows you to click anywhere when reading an X-and-Y to indicate that spot
(define-presentation-translator t-to-latitude-and-longitude
   (:blank-area latitude-and-longitude
    :gesture :left)
   (x y window)
  (multiple-value-bind (longitude latitude)
      (unscale-coordinates x y window)		;--- Why does this get bad value for Y ?
    (list latitude longitude)))

(defun route-start-object-p (thing)		;--- kludge!
  (or (typep thing 'airport)
      (typep thing 'intersection)
      (typep thing 'vor)))

(define-presentation-type route-start-object ()
  ;;--- :abbreviation-for '(or airport intersection vor)
  :parser ((stream clim::&downward-rest args)
	   (declare (ignore args))
	   (let* ((start-position (and (interactive-stream-p stream)
				       (clim::input-position stream)))
		  (object 
		    (with-input-context ('airport)
					(object)
			 (with-input-context ('intersection)
					     (object)
			      (with-input-context ('vor)
						  (object)
				   (loop (read-gesture :stream stream)
					 (beep stream))
				 (T (values object)))
			    (t (values object)))
		       (t (values object)))))
	     (when start-position
	       (clim::presentation-replace-input stream object (class-name (class-of object))
						 :buffer-start start-position))
	     object))
  :printer ((thing stream &key &allow-other-keys)
	    (present thing (class-of thing) :stream stream)))

;;; Add <kind>
(define-Flight-Planner-command
  (com-Add-Object :menu-accelerator t)
  ((object '(member position route victor-airway aircraft) ;; :confirm t
	   :prompt "Object")
   ;;--- what about keywords?
   (route-start 'route-start-object
		:default nil
		#+Ignore :when #+Ignore (eq object 'route)))
  (ecase object
    (position
      (let ((new-position (query-new-position))
	    (*present-textually* nil))
	(with-own-coordinates (*standard-output* :clear-window nil)
	  (present new-position)
	  (push new-position *position-list*))))
    (route
      (let* ((new-route (query-new-route :route-start route-start))
	     (*present-textually* nil))
	(with-own-coordinates (*standard-output* :clear-window nil)
	  (present new-route)
	  (push new-route *route-list*))))
    (victor-airway
      (let ((new-victor-airway (query-new-victor-airway))
	    (*present-textually* nil))
	(with-own-coordinates (*standard-output* :clear-window nil)
	  (present new-victor-airway)
	  (push new-victor-airway *victor-airway-list*))))
    (aircraft
      (let ((new-aircraft (query-new-aircraft)))
	(push new-aircraft *aircraft-list*)))))

(define-presentation-to-command-translator add-route
   (T
     :tester ((object)
	      (values (route-start-object-p object) t))
     :gesture :left)
   (object)
  `(com-add-object route ,object))

(defun query-new-position ()
  (multiple-value-bind (name long-name kind lat-and-long alt)
      (with-temporary-window (*query-io* (window-root *query-io*))
	(accepting-values (*query-io*
			    ;;--- :label "Position parameters"
			    )
	  (values
	    (prog1 (accept 'string :prompt "Name" :default nil)
		   (terpri *query-io*))
	    (prog1 (accept 'string :prompt "Long name" :default "")
		   (terpri *query-io*))
	    (prog1 (accept '(member airport vor intersection visual-checkpoint)
			   :prompt "Kind of position" :default nil)
		   (terpri *query-io*))
	    (prog1 (accept 'latitude-and-longitude :prompt "Latitude, Longitude")
		   (terpri *query-io*))
	    (prog1 (accept '(integer 0 60000) :prompt "Altitude" :default 0)
		   (terpri *query-io*)))))
    (make-instance kind
		   :name name
		   :longname long-name
		   :latitude (first lat-and-long)
		   :longitude (second lat-and-long)
		   :altitude alt)))

(defun waypoint-object-p (thing)		;--- kludge!
  (or (null thing)
      (typep thing 'airport)
      (typep thing 'intersection)
      (typep thing 'vor)))

(define-presentation-type waypoint-object ()
  ;;--- :abbreviation-for '(or null airport intersection vor)
  :parser ((stream clim::&downward-rest args)
	   (declare (ignore args))
	   (with-input-context ('airport)
			       (object)
		(with-input-context ('intersection)
				    (object)
		     (with-input-context ('vor)
					 (object)
			  (loop (let ((gesture (read-gesture :stream stream)))
				  (when (and (characterp gesture)
					     (char-equal gesture #\Return))
				    (return nil)))
				(beep stream))
			(T (values object)))
		   (t (values object)))
	      (t (values object))))
  :printer ((thing stream &key &allow-other-keys)
	    (present thing (class-of thing) :stream stream)))

(defun query-new-route (&key (name nil) (route-start nil))
  (do* ((overfly ())
	(point (or route-start
		   (accept 'route-start-object :prompt "Start"))
	       (accept 'waypoint-object :prompt "Waypoint"
		       :default nil)))
       ((null point)
	(setq overfly (nreverse overfly))
	(make-instance 'route 
		       :name (or name (generate-route-name-from-legs overfly))
		       :legs overfly))
    (push (make-instance 'route-segment :at point :wind-info nil) overfly)))

(defun query-new-victor-airway (&optional (name (accept 'string
							:prompt "Name of this Victor Airway")))
  (do* ((overfly ())
	(point (accept 'route-start-object  :prompt "Start")
	       (accept 'waypoint-object :prompt "Waypoint"
		       :default nil)))
       ((null point)
	(make-instance 'victor-airway :name name :legs (nreverse overfly)))
    (push (make-instance 'victor-airway-segment :at point) overfly)))

(defun query-new-aircraft ()
  (multiple-value-bind 
    (identification type preferred-cruising-altitude normal-cruise-speed
     fuel-consumption-at-normal-cruise maximum-usable-fuel cost-per-hour)
      (with-temporary-window (*query-io* (window-root *query-io*))
	(accepting-values (*query-io*
			    ;;--- :label "Aircraft parameters"
			    )
	  (values
	    (prog1 (accept 'string :prompt "Identification")
		   (terpri *query-io*))
	    (prog1 (accept 'string :prompt "Type")
		   (terpri *query-io*))
	    (prog1 (accept 'integer :prompt "Preferred cruising altitude")
		   (terpri *query-io*))
	    (prog1 (accept 'integer :prompt "Normal cruise speed" :default 110)
		   (terpri *query-io*))
	    (prog1 (accept 'float :prompt "Fuel consumption at normal cruise" :default 6)
		   (terpri *query-io*))
	    (prog1 (accept 'float :prompt "Maximum usable fuel")
		   (terpri *query-io*))
	    (prog1 (accept 'float :prompt "Cost per hour" :default 50)
		   (terpri *query-io*)))))
    (make-instance 'aircraft
		   :identification identification
		   :type type
		   :taxi-fuel 0
		   :preferred-cruising-altitude preferred-cruising-altitude
		   :normal-cruise-speed normal-cruise-speed
		   :fuel-consumption-at-normal-cruise fuel-consumption-at-normal-cruise
		   :maximum-usable-fuel maximum-usable-fuel
		   :cost-per-hour cost-per-hour)))

(defun concrete-object-p (object)		;--- kludge!
  (or (typep object 'position)
      (typep object 'route)
      (typep object 'victor-airway)
      (typep object 'aircraft)))

(define-presentation-type concrete-object ()
  ;;--- :abbreviation-for '(or position route victor-airway aircraft))
  :parser ((stream clim::&downward-rest args)
	   (declare (ignore args))
	   (with-input-context ('position)
			       (object)
		(with-input-context ('route)
				    (object)
		     (with-input-context ('victor-airway)
					 (object)
			  (with-input-context ('aircraft)
					      (object)
			       (loop (read-gesture :stream stream)
				     (beep stream))
			     (T (values object)))
			(T (values object)))
		   (t (values object)))
	      (t (values object))))
  :printer ((thing stream &key &allow-other-keys)
	    (present thing (class-of thing) :stream stream)))

;;; Delete <object>
(define-Flight-Planner-command
  com-Delete-Object
  ((object 'concrete-object :prompt "Object")
   ;;--- what about key?
   (presentation 'presentation :default nil)	;--- presentation??
   #+++ignore (window 'tv:sheet :default nil))	;---
  (etypecase object
    (position
      (format *query-io* "~&Deleting position ~a.~%" object)
      (setq *position-list* (delete object *position-list*)))
    (route
      (format *query-io* "~&Deleting route ~a.~%" object)
      (setq *route-list* (delete object *route-list*)))
    (victor-airway
      (format *query-io* "~&Deleting victor-airway ~a.~%" object)
      (setq *victor-airway-list* (delete object *victor-airway-list*)))
    (aircraft
      (format *query-io* "~&Deleting aircraft ~a.~%" object)
      (setq *aircraft-list* (delete object *aircraft-list*))))
  #+++ignore	;---
  (when presentation
    (dw:erase-displayed-presentation presentation window)))

(define-presentation-to-command-translator delete-object
   (T
     :tester ((object)
	      (concrete-object-p object))
     :gesture :shift-middle)
   (object presentation window)
  `(com-delete-object ,object ,presentation #+++ignore ,window))

;;; Describe <object>
(define-Flight-Planner-command
  com-Describe-Object
    ((object 'concrete-object :prompt "Object"))
   (let ((stream *query-io*))
     (fresh-line stream)
     (describe-position-object object stream)))

(define-presentation-to-command-translator describe-object
   (T
     :tester ((object)
	      (concrete-object-p object))
     :gesture :middle)
   (object)
  `(com-describe-object ,object))

;;; Edit <object>
(define-Flight-Planner-command
   (com-Edit-Object :menu-accelerator t)
  ((argument 'concrete-object ;; :confirm t
	     :prompt "Object"))
  (etypecase argument
    (position)
    (route
      (edit-route argument))
    (victor-airway
      (edit-victor-airway argument))
    (aircraft
      (edit-aircraft argument))))

(define-presentation-to-command-translator edit-object
   (T
     :tester ((object)
	      (concrete-object-p object))
     :gesture :shift-left)
   (object)
  `(com-edit-object ,object))

(define-Flight-Planner-command
   (com-Flight-Plan :menu-accelerator t)
  ((route 'route :prompt "Route"))
  (let* ((plane *last-plane*)
	 plan 
	 (type)
	 equip airsp
	 (orig (route-segment-at (first (route-legs route))))
	 (dest (route-segment-at (car (last (route-legs route)))))
	 remks
	 (pilot "") (souls 1)
	 color alt fuel alts
	 (deptm 0))
    (with-temporary-window (*query-io* (window-root *query-io*))
      (accepting-values (*query-io*
			  ;;--- :label "Flight Plan"
			  ;;--- :resynchronize-every-pass t
			  )
	(setq type  (accept '(member VFR IFR DVFR) :Prompt "Type"
			    :default 'VFR
			    :query-identifier 'type))
	(terpri *query-io*)
	(setq plane (accept 'aircraft :prompt "Aircraft Identification"
			    :query-identifier 'aircraft))
	(terpri *query-io*)
	(setq equip (accept 'string   :prompt "Aircraft Type/Special Equipment" 
			    :default (or (and plane (aircraft-type plane)) "C172/U")
			    :query-identifier 'special-equipment))
	(terpri *query-io*)
	(setq airsp (accept 'integer   :prompt "True Airspeed (kts)" 
			    :default (or (and plane (aircraft-normal-cruise-speed plane)) 110)
			    :query-identifier 'true-airspeed))
	(terpri *query-io*)
	(setq deptm (accept 'time     :prompt "Proposed Departure Time"
			    :default (/ (+ (* 12 60) 00) 60)
			    :query-identifier 'departure-time))
	(terpri *query-io*)
	(setq alt   (accept 'integer  :prompt "Cruising Altitude" 
			    :default (or (and plane (aircraft-preferred-cruising-altitude plane)) "3000")
			    :query-identifier 'crusing-altitude))
	(terpri *query-io*)
	(setq remks (accept 'string   :prompt "Remarks"
			    :default ""
			    :query-identifier 'remarks))
	(terpri *query-io*)
	(setq fuel  (accept 'integer  :prompt "Fuel on Board" 
			    :default (or (and plane (aircraft-maximum-usable-fuel plane)) 0)
			    :query-identifier 'fuel-on-board))
	(terpri *query-io*)
	(setq alts  (accept '(null-or-type airport) :prompt "Alternate Airport"
			    :default nil
			    :query-identifier 'alternate-airport))
	(terpri *query-io*)
	(setq pilot (accept 'string   :prompt "Pilot's Name, Address & Telephone Number & Aircraft Home Base"
			    :query-identifier 'pilot-name))
	(terpri *query-io*)
	(setq souls (accept '(integer 1 500) :prompt "Number Aboard"
			    :query-identifier 'number-aboard))
	(terpri *query-io*)
	(setq color (accept 'string   :prompt "Color of Aircraft"
			    :query-identifier 'color-of-aircraft))
	(terpri *query-io*)))
    (setq plan (make-instance 'flight-plan
			      :type type
			      :aircraft-id plane
			      :aircraft-type equip
			      :true-speed airsp
			      :departure-point orig
			      :departure-time deptm
			      :cruising-alt  alt
			      :route route
			      :destination dest
			      ;:ete xxx
			      :remarks remks
			      :fuel-on-board fuel
			      :alternate alts
			      :pilot pilot
			      :souls souls
			      :color color))
    (window-clear fp-window)
    (window-expose fp-window)
    (compute-flight-plan fp-window plan)
    (present "Click here to remove this display" 'string :stream fp-window)
    (with-input-context ('string)
			()
	 (loop
	   (read-gesture :stream fp-window))
       (T nil))
    (window-set-visibility fp-window nil)))

(define-presentation-to-command-translator flight-plan
   (route
     :gesture :middle)
   (object)
  `(com-flight-plan ,object))

(define-Flight-Planner-command
  (com-Show-Distance :menu-accelerator t)
    ((start 'route-start-object :translator-gesture :left)
     (end 'route-start-object :translator-gesture :left))
  (multiple-value-bind (distance tc)
      (geodesic (point-latitude start) 
		(point-longitude start) 
		(point-latitude end) 
		(point-longitude end))
    (format *query-io* 
       "~&The distance between ~a and ~a is ~1,2F NM, and the true course is ~1,2F.~%"
      (position-name start) (position-name end) distance tc)))

(define-Flight-planner-command
   (com-Show-Map :menu-accelerator t)
  ()
   (initialize-fp-window *application*)
  (let ((*present-textually* nil))
    (with-own-coordinates ()
      (draw-coastline *coastline*)
      (mapc #'present *route-list*)
      (mapc #'present *position-list*)
      (mapc #'present *victor-airway-list*))))


;;; Misc. functions and constants

(defun square (n) (* n n))

(defun radian (degrees) (* degrees (/ pi 180)))
(defun degree (radians) (* radians (/ 180 pi)))

(defun Geodesic (K M L N)			;arguments are in degrees
  (let* ((CC 0.0033901)
	 (O 3443.95594)				;semi major axis of Earth
	 (A (atan (* (- 1 CC) (tan (radian K)))))	;radians
	 (COSA (cos A))
	 (SINA (sin A))
	 (B (atan (* (- 1 CC) (tan (radian L)))))	;radians
	 (COSB (cos B))
	 (SINB (sin B))
	 (D (* SINA SINB))
	 (E (radian (- M N)))			;radians
	 (ABSE (abs E))
	 (COSE (cos E))
	 (SINABSE (sin ABSE))
	 (FF (+ (* SINA SINB) (* COSA COSB COSE)))
	 (S (* (square SINABSE) (square COSB)))
	 (TT (square (- (* SINB COSA) (* SINA COSB COSE))))
	 (H (sqrt (+ S TT)))
	 (I (/ (- (square H)
		  (* (square SINABSE) (square COSA) (square COSB)))
	       (square H)))
	 (J (* (atan (/ H FF))))		;radian
	 (G (+ J (* (/ (+ (square CC) CC) 2) (+ (* J (- 2 I)) (* H (- (* 2 D) (* I CC)))))))
	 (V (+ D (* H (/ 1 (tan (/ (* 180 J) pi)))) 
	       (* (square H) FF (+ (* 8 D (- (* I FF) D)) (- 1 (* 2 (square (abs FF))))))))
	 (P (+ G (* (/ (square CC) (* 16 H)) (+ (* 8 (square J) (- I 1) V) (* H J)))))
	 (R (* (- 1 CC) O P))			;R is distance

	 (A1 (* J (+ CC (square CC))))
	 (A3 (+ (* (/ (* D CC) (* 2 H)) (square (abs H))) (* 2 (square (abs J)))))
	 (A2 (* (/ (* I (square CC)) 4) (+ (* H FF) (* -5 J) 
				       (* 4 (square (abs J)) (/ 1 (tan J))))))
	 (Q (+ A1 (- A2) A3))
	 (U (+ (* (/ (* (sin E) COSA COSB) H) Q) E))
	 (W (- (* SINB COSA) (* (cos U) SINA COSB)))
	 (X (* (sin U) COSB))
	 (A4 (atan (/ X W)))
	 (Y (if (< A4 0) (+ A4 pi) A4))
	 (Z (if (< E 0) (+ Y pi) Y)))
    (if (and (zerop Z) (< L K)) (setq Z pi))
    (values R (degree Z))))
;    (list A B D E FF S TT H I J G V P 'dist R a1 a3 a2 q u w x a4 y 'dir z)))

;;; A position on the Geodesic globe.
;;; Miscellaneous functions

;;; Wind correction

;; tc=true course
;; th=true heading
;; v =true airspeed
;; gs=ground speed
(defun wind-speed-and-direction (tc th v gs)
  (let* ((w (- th tc))
	 (ht (- (* v (cos (radian w))) gs))
	 (cx (* v (sin (radian w))))
	 (ws (square (+ (abs (square ht)) (abs (square cx)))))
	 (w2 (* (degree (asin (/ cx ws))) (if (minusp (- v gs)) -1 1)))
	 (w1 (+ tc w2 (if (> gs v) 180 0)))
	 (wd (- w1 (* (floor (/ w1 360)) 360))))
    (setq wd (floor (+ (* 100 wd) 0.5) 100))
    (setq ws (floor (+ (* 100 ws) 0.5) 100))
    (values wd ws)))

;; tc=true course
;; v =true airspeed
;; wd=wind direction
;; b =wind speed
(defun true-heading-and-groundspeed (tc v wd b)
  (let* ((a (+ 180 wd (- 360 tc)))
	 (ht (* b (cos (radian a))))
	 (w (degree (asin (* b (/ (sin (radian a)) v)))))
	 (gs (+ (* v (cos (radian w))) ht))
	 (tt (- tc w))
	 (th (- tt (* (floor tt 360) 360))))
;    (setq th (floor (+ (* 100 wd) 0.5) 100))
;    (setq gs (floor (+ (* 100 wd) 0.5) 100))
    (values th gs)))

;(true-heading-and-groundspeed 229 125 270 14)
;(true-heading-and-groundspeed 229 125 0 0)

;;; Crosswind components

#+ignore
(defun wind-components (wind-angle wind-speed)
  (let ((angle (radian wind-angle)))
    (values (* (cos angle) wind-speed)		;The headwind component
	    (* (sin angle) wind-speed))))	;The crosswind component

;(wind-components 45 50)

#+ignore
(defun density-altitude (pressure-altitude temperature)
  (* 145426 (- 1 (expt (/ (expt (/ (- 288.16 (* pressure-altitude 0.001981)) 288.16) 5.2563)
			  (/ (+ 273.16 temperature) 288.16))
		       0.235))))

;(density-altitude 11000 10)

#+ignore
(defun feet-per-minute (feet-per-mile ground-speed)
  (* feet-per-mile (/ ground-speed 60.0)))

#+ignore
(defun true-airspeed (indicated-airspeed altitude temperature)
  (let ((D (/ altitude (- 63691.776 (* 0.2190731712 altitude)))))
    (* indicated-airspeed (sqrt (/ (+ 273.16 temperature) (/ 288 (expt 10 D)))))))

#+ignore
;; mach = mach number, temperature is true OAT Celcius
(defun mach-to-true-airspeed (mach temperature)
  (* 38.967 mach (sqrt (+ 273.16 temperature))))

#+ignore
(defun leg-time (leg-distance leg-speed)
  (/ leg-distance leg-speed))

#+ignore
(defun leg-speed (leg-distance leg-time)
  (/ leg-distance leg-time))

#+ignore
(defun leg-distance (leg-speed leg-time)
  (* leg-speed leg-time))

#+ignore
(defun bank-angle-for-standard-rate-turn (speed)
  (degree (atan (/ (* speed 9.2177478 1.15) 3860))))

#+ignore
(defun G-force-in-bank (bank-angle)
  (/ 1 (cos (radian bank-angle))))

#+ignore
(defun diameter-of-turn (TAS bank-angle)
  (/ (square TAS) (* 34208 (tan (radian bank-angle)))))

#+ignore
(defun wind-correction-angle (wa ws tas)
  (degree (asin (/ (* ws (sin (radian wa))) tas))))

#+ignore
(defun speed-loss-due-to-crabbing (tas wca tas)
  (- tas (* tas (cos (radian wca)))))


;;; A simple cheat setup database

(defun add-position (name kind latitude longitude altitude deviation long-name)
  (when (or (> latitude *max-latitude*)
	    (< latitude *min-latitude*)
	    (> longitude *max-longitude*)
	    (< longitude *min-longitude*))
    (return-from add-position nil))
  (push (make-instance kind 
		       :name name 
		       :longname long-name
		       :latitude latitude 
		       :longitude longitude 
		       :altitude altitude
		       :deviation deviation) 
	*position-list*))

(defun add-aircraft (identification type altitude speed fuel-consumption max-fuel cost)
  (let ((aircraft (make-instance 'aircraft
				 :identification identification
				 :type type
				 :taxi-fuel 0
				 :preferred-cruising-altitude altitude
				 :normal-cruise-speed speed
				 :fuel-consumption-at-normal-cruise fuel-consumption
				 :maximum-usable-fuel max-fuel
				 :cost-per-hour cost)))
    (push aircraft *aircraft-list*)))

(defun customize-database ()
  ;; Airports
  (add-position "HFD" 'airport (degminsec 41 44) (degminsec 72 39) 19  15 "Hartford-Brainard")
 
  ;; Intersections
  (add-position "DREEM" 'intersection (degminsec 42 21.6) (degminsec 71 44.3) 0 15 "DREEM")
  (add-position "GRAYM" 'intersection (degminsec 42 06.1) (degminsec 72 01.9) 0 15 "GRAYM")
  (add-position "WITNY" 'intersection (degminsec 42 03) (degminsec 72 14.2) 0 15 "WITNY")
  (add-position "EAGRE" 'intersection (degminsec 41 45) (degminsec 72 20.6) 0 15 "EAGRE")

  ;; VOR's

  ;; Aircraft
  ;;            ident      type       alt   sp     fuel cost
  (add-aircraft "NCC-1701" "Starship" 35000 550 25 1000 10000)
  (add-aircraft "xyzzy"    "C172"     3500  110  6   50    50)
  
  )

(defun set-up ()
  (setq *position-list* nil
	*route-list* nil
	*victor-airway-list* nil
	*aircraft-list* nil)
  (dolist (bits *default-nav-data*)
    (apply #'(lambda (num name type freq longname lat1 lat2 lon1 lon2 dev ew elev)
	       (declare (ignore num))
	       (add-position 
		 name 
		 (case type
		   (A 'airport)
		   (V 'vor)
		   ((VA AV) 'airport)
		   ((AN NA) 'airport)
		   (C 'visual-checkpoint)	;actually visual cp
		   (N 'ndb))			;actually ndb
		 (degminsec lat1 lat2)
		 (degminsec lon1 lon2)
		 elev
		 dev
		 longname))
	   bits))
  (customize-database)
  t)


(define-demo "Flight Planner"
  (progn
    (unless *position-list*
      (set-up))
    (run-flight-planner :root clim-demo::*demo-root*)))

