;;; -*- Mode: Lisp; Package: CLIM-DEMO; Base: 10.; Syntax: Common-Lisp -*-
;;;
;;; Copyright (c) 1986, 1987, 1988, 1989, 1990 by Mark A. Son-Bell.  All rights reserved. 
;;;
;;; Algorithms straight out of PDP books
;;;

(in-package "CLIM-DEMO")

#+ansi-90
(eval-when (:compile-toplevel :load-toplevel :execute)
(cl:define-declaration values (spec env)
  (declare (ignore env))
  (values :declare spec)))

#-ansi-90
(defmacro with-standard-io-syntax (&body body)
  #+Genera
  `(scl:with-standard-io-environment ,@body)
  #-Genera
  `(let ((*print-base* 10.)
	 (*read-base* 10.)
	 (*print-array* t)
	 (*print-radix* nil)
	 (*readtable* (copy-readtable nil))
	 (*package* (find-package "USER"))
	 (*print-escape* t)
	 (*print-circle* nil)
	 (*print-pretty* nil)
	 (*print-level* nil)
	 (*print-length* nil)
	 (*print-case* :upcase))
     ,@body))

;;; Some helping crap
(define-presentation-type y-or-n ()
  :parser ((stream &key default)
	   ;; --- need to upgrade parser-defining syntax, too.
	    default
	    (let ((token (read-char stream)))
	      (case token
		(#\y t)
		(#\n nil)
		(otherwise (input-not-of-required-type #+clim-0.9 stream token 'y-or-n)))))
  :printer ((value stream &key &allow-other-keys)
	    (if value
		(write-string "Yes" stream)
		(write-string "No" stream))))

(define-presentation-type character ()
  :parser ((stream &key default)
	   ;; --- need to upgrade parser-defining syntax, too.
	    default
	    (read-char stream))
  :printer ((value stream &key &allow-other-keys)
	    (write-char value stream)))

(define-presentation-type list (&rest options)
   :parser ((stream &key default)
	    (values (accept `expression
			    :stream stream
			    :prompt nil
			    :default default)))
   ;;--- I don't seem to be using this right at the moment.
   :object-validator ((object)
		      (listp object))
   :printer ((object stream &key acceptably)
	     (present object `expression
		      :acceptably acceptably
		      :stream stream)))

;;; RASSOC for hash tables
(defun get-hash-key (desired-value table)
  (maphash #'(lambda (key value)
	       (when (eql value desired-value)
		 (return-from get-hash-key key)))
	   table))

;;;
(defparameter *tablet-window-side* 256.)
(defparameter *n-connection-inputs* 100.)
(defparameter *connection-input-window-side* 128.)

(defvar *tablet-window*)
;	(open-window-stream :parent clim::root
;			    :label "Drawing Tablet"
;			    :height *tablet-window-side*
;			    :width *tablet-window-side*
;			    :top 40
;			    :left 40
;			    :scroll-bars nil
;			    :record-p nil))
;
;(defvar *connection-input-window*
;	(open-window-stream :parent clim::root
;			    :label "Input Signals"
;			    :height *connection-input-window-side*
;			    :width *connection-input-window-side*
;			    :top (entity-top *tablet-window*)
;			    :left (+ 5 (entity-right *tablet-window*))
;			    :scroll-bars nil
;			    :record-p nil))

;(close *tablet-window*)
;(close *connection-input-window*)

(defparameter *tablet-magnification* 4)

(defun make-nn-color (r g b)
  (flet ((bucketize (n)
	   (let ((range .2))
	     (* range (ffloor n range)))))
    (setq r (min 1.0 (bucketize r)))
    (setq g (min 1.0 (bucketize g)))
    (setq b (min 1.0 (bucketize b))))
  (make-color-rgb r g b))


;;; At least in Symbolics 8.0.2, the order of defaulting can't
;;; be relied on.  I haven't checked CLtL.  --RWK

(defstruct (glyph (:named) 
		  (:conc-name)
		  (:constructor make-glyph-internal))
  (dim0)
  (dim1)
  (pixels))

(defun make-glyph (&key dim0 (dim1 dim0) 
		   (pixels  (make-array `(,dim0 ,dim1) :element-type 'bit :initial-element 0)))
  (make-glyph-internal :dim0 dim0 :dim1 dim1 :pixels pixels))

(defmacro glyph-aref (glyph &rest indices)
  `(aref (glyph-pixels ,glyph) ,@indices))

(defun canonicalize-glyph-zoom (glyph stream)
  (declare (values left top right bottom))
  (let* ((width (glyph-dim0 glyph))
	 (height (glyph-dim1 glyph))
	 final-left final-top final-right final-bottom)
    (with-scaling (stream *tablet-magnification*)
      (do ((left 0 (1+ left))
	   (top 0 (1+ top))
	   (right width (1- right))
	   (bottom height (1- bottom)))
	  ((and final-left final-top final-right final-bottom))
	(macrolet ((do-one-side (name)
		    (let ((final-name (intern (format nil "FINAL-~A" name)))
			  (erase-line-points (case name
					       ((left right) `(ii (1- top) ii (1+ bottom)))
					       ((top bottom) `((1- left) ii (1+ right) ii))))
			  (draw-line-points (case name
					      ((left right) `(ii top ii bottom))
					      ((top bottom) `(left ii right ii)))))
		      `(unless ,final-name
			 (let ((ii ,(case name	;Clear previous line
				      ((left top) `(1- ,name))
				      ((right bottom) `(1+ ,name)))))
			   (block ,name
			     ;; Erase previous boundary
			     (unless (or ,final-name
					 ,(case name
					    ((left top) `(minusp ii))
					    (right `(= (1+ width) ii))
					    (bottom `(= (1+ height) ii))))
			       (draw-line* stream ,@erase-line-points :line-thickness 2
					   :ink +background+))
			     ;; Check for points in this row or column
			     (dotimes (i ,(case name
					    ((left right) 'height)
					    ((top bottom) 'width)))
			       (unless (zerop ,(case name
						 (left `(glyph-aref glyph ,name i))
						 (right `(glyph-aref glyph (1- ,name) i))
						 (top `(glyph-aref glyph i ,name))
						 (bottom `(glyph-aref glyph i (1- ,name)))))
				 (setq ,final-name ,name)
				 (return-from ,name))))
			   ;; Draw this boundary, unless we've come to the end
			   (unless ,final-name
			     (setq ii ,name)
			     (draw-line* stream ,@draw-line-points :line-thickness 2
					 :ink +foreground+)))))))
	  (with-output-recording-options (stream :record-p nil)
	    (do-one-side left)
	    (do-one-side top)
	    (do-one-side right)
	    (do-one-side bottom))))
      ;; Draw final box
      (draw-rectangle* stream final-left final-top final-right final-bottom
		       :filled nil :line-thickness 2 :ink +foreground+))
    (values final-left final-top final-right final-bottom)))

;;;
(defun input-glyph-from-tablet-points (points-as-glyph stream left top right bottom)
  (declare (values scaled-glyph-for-connections))
  (window-expose stream)
  (let* ((side (isqrt *n-connection-inputs*))
	 (actual-width (- right left))
	 (actual-height (- bottom top))
	 (mxx (/ side actual-width))
	 (myy (/ side actual-height))
	 (tx (- left))
	 (ty (- top))
	 (scaled-glyph-for-connections (make-glyph :dim0 side)))
    (flet ((fast-transform-point* (x y mxx mxy myx myy tx ty)
	     (values (+ (* x mxx) (* y mxy) tx)
		     (+ (* x myx) (* y myy) ty))))
      (do ((x left (1+ x)))
	  ((= x right))
	(do ((y top (1+ y)))
	    ((= y bottom))
	  (unless (zerop (glyph-aref points-as-glyph x y))
	    (multiple-value-bind (new-x new-y)
		(fast-transform-point* x y 1 0 0 1 tx ty)
	      (multiple-value-bind (new-x new-y)
		  (fast-transform-point* new-x new-y mxx 0 0 myy 0 0)
		(setq new-x (round (min (max new-x 0) (1- side)))
		      new-y (round (min (max new-y 0) (1- side))))
		(setf (glyph-aref scaled-glyph-for-connections new-x new-y) 1))))))
      (window-clear stream)
      (draw-input-glyph scaled-glyph-for-connections stream)
      scaled-glyph-for-connections)))
	  
(defun canonicalize-glyph (points-as-glyph &optional (tablet-stream *tablet-window*)
			   (glyph-stream (slot-value *frame* 'current-input)))
  (window-expose tablet-stream)
  (multiple-value-bind (left top right bottom)
      (canonicalize-glyph-zoom points-as-glyph tablet-stream)
    (input-glyph-from-tablet-points points-as-glyph glyph-stream left top right bottom)))

(defun get-tablet-points (&optional (tablet-stream *tablet-window*))
  (declare (values points-as-glyph))
  (window-clear tablet-stream)
  (window-expose tablet-stream)
  (let* ((set-bits? nil)
	 (vp (pane-viewport tablet-stream))
	 (vph (rectangle-height vp))
	 (vpw (rectangle-width vp))
	 (glyph-dim-0 (round vpw *tablet-magnification*))
	 (glyph-dim-1 (round vph *tablet-magnification*))
	 (points-as-glyph (make-glyph :dim0 glyph-dim-0 :dim1 glyph-dim-1)))
    (block track
      (loop
;	(catch 'clim::abort-gesture-seen	;--- Something's busted in CLIM
	  (with-scaling (tablet-stream *tablet-magnification*)
	    (tracking-pointer (tablet-stream)
	      (:pointer-motion (x y)
	       ;;--- Baaaarf!  Why doesn't scaling work for input, too???
	       (setq x (round x *tablet-magnification*)
		     y (round y *tablet-magnification*))
	       (when set-bits?
		 (when (and (< x glyph-dim-0)	;--- Shouldn't have to do this
			    (< y glyph-dim-1)
                            (> x 0) (> y 0)
			    (zerop (glyph-aref points-as-glyph x y)))
		   (draw-rectangle* tablet-stream x y (1+ x) (1+ y))
		   (setf (glyph-aref points-as-glyph x y) 1))))
	      (:button-press (x y button-name)
                             (declare (ignore x y))
	       (case button-name
		 (:left (setq set-bits? t))
		 (:middle (setq set-bits? nil))
		 (:right (setq set-bits? nil)
		    (return-from track nil))))))
;	  )
      ))
    points-as-glyph))

;(get-tablet-points)

(defun draw-input-tablet-grid (n-units &optional (stream *tablet-window*))
  (window-expose stream)
  (window-clear stream)
  (do ((i 0 (1+ i)))
      ((> i n-units))
    (draw-line* stream i 0 i n-units :line-thickness (if (zerop i) 2 1))
    (draw-line* stream 0 i n-units i :line-thickness (if (zerop i) 2 1)))
  (force-output stream))

;(draw-input-tablet-grid 16)

(defun draw-input-glyph (glyph &optional (stream *tablet-window*) (draw-as-grid t) scale)
  (let* ((frame *frame*)
	 (gcs (and frame (gethash glyph (neural-network-gcs-cache frame))))
	 (side (glyph-dim0 glyph)))
    (unless scale
      (setq scale (1- (round (rectangle-height (pane-viewport stream)) side))))
    (unless (and gcs (= scale 1))
      (with-scaling (stream scale)
	(dotimes (x side)
	  (dotimes (y (glyph-dim1 glyph))
	    (let ((off? (zerop (glyph-aref glyph x y))))
	      (if (or (> scale 1)
		      (null frame))		;Always do it slow if not in application mode
		  (if draw-as-grid
		      (draw-rectangle* stream x y (1+ x) (1+ y) :filled (not off?)
                                       :line-thickness 0)
		      (unless off?
			(draw-rectangle* stream x y (1+ x) (1+ y))))
		  (unless off?
		    (setq gcs (list* x y gcs)))))))
	(when gcs
	  (setf (gethash glyph (neural-network-gcs-cache frame)) gcs))))
    (when (and gcs (= scale 1))
      (draw-points* stream gcs))))		;--- Too bad no BITBLT



;;;
(defvar *desired-output*)

(defparameter *eta* 0.80)
(defparameter *alpha* 0.90)
(defparameter *min-passes* 5)

(defvar *symbols* #(zero one two three four five six seven eight nine))

;;; EXP accelerator
(defun get-output-function-array ()
  (let ((a (make-array 8000.)))
    (let ((j -40.0))
      (dotimes (i 8000.)
	(setf (aref a i) (/ 1 (1+ (exp (- j)))))
	(incf j .01)))
    a))

(defvar *output-function-array* (get-output-function-array))

(defmacro slow-output-function (sum)
  `(/ 1 (1+ (exp (- ,sum)))))

;;; Use this one if coercing a single-float into a fixum is significantly faster than EXP
(defmacro output-function (sum)
  `(let ((index (+ 4000.
		   #+genera (sys:%convert-single-to-fixnum (* ,sum 100.))
		   #-genera (values (round (* ,sum 100.))))))
     (if (<= 0 index 8001.)
	 (aref *output-function-array* index)
	 (slow-output-function ,sum))))

;;; Unit (neuron) classes/structs
(defstruct (bpc (:conc-name "CONNECTION-")  ;"back-prop-connection"
		(:named))
  (fu nil)					;"from-unit"
  (tu nil)					;"to-unit"
  (wt (- (random 0.6) 0.3)))			;"weight"

(defstruct (basic-back-prop-unit (:conc-name "UNIT-")
				 (:named))
  (idx nil)					;"index"
  (val nil)					;"val"
  ;; move the slots shared by more than one "subclass" structure
  ;; to here, so we don't get conflicts in the accessor names
  (ci nil)
  (s1 nil)
  (er nil))

(defstruct (iu (:include basic-back-prop-unit) (:conc-name "UNIT-")	;"input-unit"
	       (:named))
  (co nil))						;"connections-out"

(defstruct (hu (:include basic-back-prop-unit) (:conc-name "UNIT-")	;"hidden-unit"
	       (:named))
  (co nil)
  ;;(er nil)
  (lyr nil))

(defstruct (ou (:include basic-back-prop-unit) (:conc-name "UNIT-")	;"output-unit"
	       (:named))
  ;;(er nil)
  )

(defmacro structure-typeof (structure)
  #+(and Genera (not Genera-release-8)) `(aref ,structure 0)
  #-(and Genera (not Genera-release-8)) `(type-of ,structure))

(defmacro structure-typep (structure type)
  #+(and Genera (not Genera-release-8)) `(eq (aref ,structure 0) ,type)
  #-(and Genera (not Genera-release-8)) `(typep ,structure ,type))

;;;
(defun connect-unit-to-previous-layer (unit layer unit-index &optional weights)
  (let* ((n-connections (length layer))
	 (connections (make-array n-connections)))
    (dotimes (i n-connections)
      (let* ((layer-unit (aref layer i))
	     (connection (make-bpc :fu layer-unit :tu unit
				   :wt (if weights
					   (elt weights i)
					   (- (random 0.6) 0.3)))))
	(setf (aref connections i) connection)
	(setf (aref (unit-co layer-unit) unit-index) connection)))
    (setf (unit-ci unit) connections)))

(defun make-back-prop-layer (n-units layer-number n-connections-out
			     &optional (type 'hu))
  ;; Makes a vector of units of type TYPE, and fills in their CONNECTIONS-OUT
  ;; slot with a vector of length N-CONNECTIONS-OUT.
  (let ((layer (make-array n-units)))
    (dotimes (i n-units)
      (setf (aref layer i)
	    (case type
	      (ou
		(make-ou :idx i))
	      (iu
		(make-iu :idx i
			 :co (make-array n-connections-out)))
	      (hu
		(make-hu :idx i
			 :lyr layer-number
			 :co (make-array n-connections-out))))))
    layer))

(defun make-back-prop-network (n-input-units n-output-units hidden-layers-n-units
			       &optional layers-connection-weights)
  (let* ((n-layers (+ (length hidden-layers-n-units) 2))
	 (network (make-array n-layers 
			      ;;:leader-length 1
			      ;;:named-structure-symbol 'back-prop-network
			      )))
    (setf (aref network 0)
	  (make-back-prop-layer n-input-units 0 (first hidden-layers-n-units) 'iu))
    (flet ((do-a-layer (this-n type i next-n &optional layer-connection-weights)
	     (let* ((prev-layer (aref network (1- i)))
		    (this-layer (make-back-prop-layer this-n i next-n  type)))
	       (dotimes (i this-n)
		 (let ((unit (aref this-layer i))
		       (unit-connection-weights (and layer-connection-weights
						     (elt layer-connection-weights i))))
		   (connect-unit-to-previous-layer unit prev-layer i
						   unit-connection-weights)))
	       (setf (aref network i) this-layer))))
      (do ((sublist hidden-layers-n-units (cdr sublist))
	   (i 1 (1+ i)))
	  ((null sublist))
	(do-a-layer (first sublist) 'hu i
		    (if (cdr sublist)
			(second sublist)
			n-output-units)
		    (and layers-connection-weights
			 (elt layers-connection-weights (1- i)))))
      (let ((output-layer-index (1- n-layers)))
	(do-a-layer n-output-units 'ou output-layer-index 0
		    (and layers-connection-weights
			 (elt layers-connection-weights (1- output-layer-index))))))
    network))

;;; This is equation 1, labelled "(15)" on p. 329 - the Activation Function
(defun unit-output-value (network unit &optional cache-p update-display)
  (if (structure-typep unit 'iu)
      (unit-val unit)
      (progn
	(setf (unit-er unit) nil)		;Clear the error cache every time we calculate
	(let ((new-value			; a new value
		(output-function
		  (let ((sum 0)
			(connections-in (unit-ci unit)))
		    (dotimes (i (length connections-in))
		      (let ((connection (aref connections-in i)))
			(incf sum (* (connection-wt connection)
				     (unit-output-value network
							(connection-fu connection)
							cache-p
							update-display)))))
		    sum))))
	  (when cache-p
	    (setf (unit-val unit) new-value)
	    (when update-display
	      (show-unit network unit (neural-network-outputs *frame*) (slot-value *frame* 'connections) t)))
	  new-value))))

;;; This is equations 3 & 4, p.329
(defun unit-error-signal (unit)
  (if (structure-typep unit 'iu)
      0
      (or (unit-er unit)
	  (setf (unit-er unit)			;Fill the cache
		(let ((value (unit-val unit)))
		  (if (structure-typep unit 'hu)
		      (* value
			 (- 1 value)
			 ;; Now sum the errors for the units connected to this one in the
			 ;; output direction
			 (let ((sum 0)
			       (connections-out (unit-co unit)))
			   (dotimes (i (length connections-out))
			     (let ((connection (aref connections-out i)))
			       (incf sum (* (connection-wt connection)
					    (unit-error-signal
					      (connection-tu connection))))))
			   sum))
		      (* (- (elt *desired-output* (unit-idx unit)) value)
			 value
			 (- 1 value))))))))

(defmacro connection-weight-delta (connection)
  `(let ((to-unit (connection-tu ,connection)))
     ;; *eta* is the learning rate
     (* (neural-network-eta *frame*)
	(unit-val to-unit)			;Assumes we've run the inputs through once and
	(unit-error-signal to-unit))		; cached the result
     ;; In order to agree with (16) on p. 330, we should add here
     ;;   (* *alpha* (previous-weight-delta output-index input-index level))
     ;;  where *alpha* is the constant of learning momentum, but this would mean
     ;;  we'd have to keep the previous weight delta around, and we don't.
     ))

(defun layer-adjust-connection-weights (network layer &optional (update-display t))
  "Adjust the strengths of the connections -to- the units in LAYER."
  (map nil
       #'(lambda (unit)
	   (let ((unit-type (structure-typeof unit)))	;--- TYPE-OF
	     (map nil
		  #'(lambda (connection)
		      (incf (connection-wt connection)
			    (connection-weight-delta connection))
		      ;; For the moment, only worry about connections to outputs
		      (when (eq unit-type 'ou)
			(when update-display
			  (show-connection network connection))))
		  (unit-ci unit))))
       layer))

(defun network-result (network input-glyph &optional (result-type 'list) (update-display t))
  "Sets the value caches in the units."
  (let ((network-input-layer (elt network 0))
	(i -1))
    (dotimes (x (glyph-dim0 input-glyph))
      (dotimes (y (glyph-dim0 input-glyph))
	(setf (unit-val (elt network-input-layer (incf i)))
	      (glyph-aref input-glyph x y)))))
  (map result-type
       #'(lambda (unit)
	   (let ((value (unit-output-value network unit t update-display)))
	     (setf (unit-val unit) value)
	     value))
       (elt network (1- (length network)))))

(defun adjust-network (network inputs target-outputs &optional (update-display t))
  (network-result network inputs nil)		;Set the value caches
  (let ((*desired-output* target-outputs)
	(n (length network)))
    ;; Work from the outputs back
    (do ((layer-index (1- n) (1- layer-index)))
	((= 0 layer-index))
      (layer-adjust-connection-weights network (elt network layer-index) update-display))))

;;;
;(defvar *network* (make-back-prop-network *n-connection-inputs*
;					  (length *symbols*)
;					  '(16)))	;This one we can vary at will


;;;
;(defvar *backprop-input-stream*
;	(open-window-stream :parent clim::root
;			    :height #+Imach 80 #-Imach 100
;			    :width 300
;			    :top #+imach 338 #-imach 50
;			    :left #+Imach 50 #-Imach 600))

;(window-expose *backprop-input-stream*)
;(close *backprop-input-stream*)

;(defparameter *connection-window-width* (- (window-inside-width clim::root) 30))
;(defparameter *connection-window-height* (- (window-inside-height clim::root) 100))
;
(defvar *connections-window*)
;	(open-window-stream :parent clim::root
;			    :label "Connection Weights"
;			    :scroll-bars nil
;			    :height *connection-window-height*
;			    :width *connection-window-width*
;			    :blinker-p nil
;			    :top #-imach (+ (rectangle-bottom *tablet-window*) 10)
;			         #+imach 20
;			    :left #-imach (rectangle-left *tablet-window*)
;			          #+imach 10))

;(close *connections-window*)
;(window-clear *connections-window*)

(defvar *border* 20)

(defun unit-bounding-box (network unit)
  "Assumes coordinate system has been transformed past the border: 0,0 is the top
of the first input unit."
  (declare (values left top right bottom))
  (let* ((max-layer-index (1- (length network)))
	 (n-hidden-units (1- max-layer-index))
	 (type (structure-typeof unit))
	 (layer-index (case type	;Too bad TYPECASE is too slow.
			(iu 0)
			(hu (unit-lyr unit))
			(ou max-layer-index)))
	 (layer (elt network layer-index))
	 (unit-index (unit-idx unit))
	 (unit-width (/ 250 (length layer)))
	 (left (* unit-index unit-width))
	 (width (* unit-width .7))
	 (hidden-unit-height (/ 60 n-hidden-units))
	 (top (case type
		(iu 0)
		(hu (+ 20 (* hidden-unit-height (1- layer-index))))
		(ou 80)))
	 (height (case type
		   (iu 5)
		   (hu 10)		;(* hidden-unit-height .3)
		   (ou 95)))
	 (side (min width height)))
    (values (round left) (round top) (round (+ left side)) (round (+ top side)))))

(defun show-connection (network connection &optional (stream (slot-value *frame* 'connections))
			value-and-weight)
  (let ((from-unit (connection-fu connection))
	(to-unit (connection-tu connection)))
    (multiple-value-bind (from-l ignore from-r from-b)
	(unit-bounding-box network from-unit)
      (declare (ignore ignore))
      (multiple-value-bind (to-l to-t to-r ignore)
	  (unit-bounding-box network to-unit)
        (declare (ignore ignore))
	(let ((w (* 3 (connection-wt connection)))
	      (in-value (or (when value-and-weight
			      (unit-val from-unit))
			    0)))
 	  (multiple-value-bind (red green blue)
	      (if (minusp w)
		  (values (- w) in-value 0)
		  (values 0 in-value w))
	    (let ((from-x (round (+ from-l from-r) 2))
		  (to-x (round (+ to-l to-r) 2))
		  (ink (make-nn-color red green blue)))
	      (draw-line* stream from-x from-b to-x to-t :line-thickness 1 :ink +white+)
	      (force-output stream)
	      (draw-line* stream from-x from-b to-x to-t :line-thickness 1 :ink ink)
	      (force-output stream))))))))

(defparameter *color-pale-green* (make-color-rgb 0 .5 0))

(defun show-unit (network unit outputs
		  &optional (stream (slot-value *frame* 'connections)) show-connections-p)
  (multiple-value-bind (left top right bottom)
      (unit-bounding-box network unit)
    (let ((type (structure-typeof unit)))
      (case type
	(iu )
	(hu
	  (let* ((value (unit-val unit))
		 (ink (if value
			  (make-nn-color 0 value 0)
			  #-3600 *color-pale-green* #+3600 +green+)))
	    (draw-rectangle* stream left top right bottom
			     :filled (not (null value)) :line-thickness 2 :ink ink)
	    (when (= (unit-lyr unit) 1)
	      (let* ((frame-pane (frame-pane *frame*))
		     (current-input-pane (slot-value *frame* 'current-input))
		     (current-input-left
		       (transform-point*
			 (fetch-delta-transformation current-input-pane frame-pane)
			 0 0)))
		(multiple-value-bind (ci-left ignore ci-right)
		    (bounding-rectangle* (pane-viewport current-input-pane))
                  (declare (ignore ignore))
		  ;; Get the center of the current input pane, in the frame's coordinates
		  (let* ((frame-x (+ current-input-left
				     (round (- ci-right ci-left) 3)))
			 ;; Remember that the connections pane is itself under a
			 ;; (scaling) transform - it may also have some x translation.
			 (x-for-drawing
			   (untransform-point*
			     (medium-transformation (sheet-medium stream))
			     (- frame-x
				(transform-point*
				  (fetch-delta-transformation stream frame-pane)
				  0 0))
			     0)))
		    (draw-line* stream (/ (+ left right) 2) top x-for-drawing 0
				:line-thickness 3 :line-dashed t)))))))
	(ou
	  (let ((value (unit-val unit))
                (middle-x (/ (+ left right) 2)))
	    (draw-rectangle* stream left top right bottom	;Erase previous
			     :filled t :ink +white+)
	    (draw-rectangle* stream left top right bottom
			     :filled nil :line-thickness 3
			     :ink (if value
				      (make-nn-color value 0 0)
				      +red+))
	    (draw-text* stream (string (elt outputs (unit-idx unit))) middle-x (+ top 2)
                        :align-x :center :align-y :top)
	    (let ((string (if (null value) 
                            "<**>"
                            (format nil s-b-back-propagation.lisp.6, Line #688, After "(defun learn-glyph (network glyph desired-output-name outputs &optional (n-times 10) update-display)"
    ;;(draw-input-glyph glyph *connection-input-window*)
    (adjust-network network
		    glyph
		    (map 'list #'(lambda (s) (if (string-equal desired-output-name s) 1 0)) outputs)
		    update-display)))
**** File B:>clim>code>demo>mas-b-back-propagation.lisp.18, Line #738, After "(defun learn-glyph (network glyph desired-output-name outputs &optional (n-times 10) update-display)"
    #-(or ccl) (declare (ignore n))
    ;;(draw-input-glyph glyph *connection-input-window*)
    (adjust-network network
		    glyph
		    (map 'list #'(lambda (s) (if (string-equal desired-output-name s) 1 0))
			 outputs)
		    update-display)))
	(unless (or (eq type 'iu)
		    (and (eq type 'hu)
			 (= (unit-lyr unit) 1)))
	  (map nil #'(lambda (connection)
		       (show-connection network connection stream t))
	       (unit-ci unit)))))))
      
(defun show-network (network outputs
		     &optional (stream (slot-value *frame* 'connections)) show-connections-p)
  ;; Do units
  (map nil
       #'(lambda (layer)
	   (unless (eq layer (elt network 0))
	     (map nil
		  #'(lambda (unit)
		      (show-unit network unit outputs stream show-connections-p))
		  layer)))
       network))

(defmacro in-connection-display-transform ((stream &key expose clear) &body body)
  `(progn
     (when ,clear (window-clear ,stream))
     (when ,expose (window-expose ,stream))
     (let* ((vp (pane-viewport ,stream))	;--- Is this cheating???
	    (actual-width (rectangle-width vp))
	    (actual-height (rectangle-height vp))
	    (border *border*))
       (with-translation (,stream border border)
	 ;; All units below on a 250x100 scale
	 (with-scaling (,stream (/ (- actual-width (* border 2)) 250)
			 (/ (- actual-height (* border 2)) 100))
	   ,@body)))))

;(in-connection-display-transform (*connections-window* :clear t :expose t)
;  (show-network *network*))


;;;
(defun learn-glyph (network glyph desired-output-name outputs &optional (n-times 10) update-display)
  (dotimes (n n-times)
    ;;(draw-input-glyph glyph *connection-input-window*)
    (adjust-network network
		    glyph
		    (map 'list #'(lambda (s) (if (string-equal desired-output-name s) 1 0)) outputs)
		    update-display)))

(defun get-output-from-glyph (network glyph outputs &optional (update-display t))
  (declare (values output-name result glyph))
  (let ((max-output 0)
	(max-i 0)
	(result (network-result network glyph 'list update-display))
	(i 0))
    (dolist (output result)
      (when (> output max-output)
	(setq max-output output
	      max-i i))
      (incf i))
    (values (elt outputs max-i) result glyph)))

(defun get-input-glyph (tablet-stream glyph-stream)
  (window-clear tablet-stream)
  (window-expose tablet-stream)
  (window-clear glyph-stream)
  (window-expose glyph-stream)
  (canonicalize-glyph (get-tablet-points tablet-stream) tablet-stream glyph-stream))

(defun run-input (network glyph outputs istream cstream)
  (declare (values output-name))
  ;;---  Baarf!
  (in-connection-display-transform (cstream :clear nil :expose t)
    ;; Get the new input glyph
    (let ((output-name (get-output-from-glyph network glyph outputs)))
      (write-string (string output-name) istream)
      output-name)))

;;;--- Obsolete
;(defun learn-from-input (network glyph outputs istream connections-stream)
;  (let ((*connections-window* connections-stream))	  ; For the SHOW- functions
;    (in-connection-display-transform (connections-stream :clear nil :expose t)
;      ;; Get the new input glyph
;      (multiple-value-bind (output-name ignore glyph)
;	  (get-output-from-glyph network glyph outputs)
;	;; Find out if we got the right symbol
;	(let ((correct? (accept 'y-or-n :stream istream
;				:prompt (format nil "~%Is ~A correct? " output-name))))
;	  (unless correct?
;	    (let ((desired (accept 'string :stream istream
;				   :prompt (format nil "~%What should it have been? "))))
;	      (force-output istream)
;	      (when (find desired outputs :test #'string-equal)
;		(block learn
;		  (loop
;		    (learn-glyph network glyph desired outputs 1)
;		    ;; Keep doing it until the net has learned
;		    (when (string-equal (get-output-from-glyph network glyph outputs) desired)
;		      (return-from learn))))))))))))

;(learn-from-input *network* *backprop-input-stream* *tablet-window*
;		  *connection-input-window* *connections-window*)

;;; In "Application" form

(define-application-frame neural-network ()
  ((network :initform nil :accessor neural-network-network)
   (n-inputs :initform *n-connection-inputs* :accessor neural-network-n-inputs)
   (outputs :initform (copy-seq *symbols*) :accessor neural-network-outputs)
   ;; Random parameters
   (eta :initform *eta* :accessor neural-network-eta)
   (alpha :initform *alpha* :accessor neural-network-alpha)
   (npasses :initform *min-passes* :accessor neural-network-npasses)
   ;;
   (current-input-glyph :initform nil :accessor neural-network-current-input-glyph)
   (glyph-table :initform (make-hash-table :test #'equal :size 20)
		:accessor neural-network-glyph-table)
   (glyph-coordinate-sequence-cache :initform (make-hash-table :size 20)
				    :accessor neural-network-gcs-cache)
   (glyph-output-association-table :initform (make-hash-table :size 20)	;glyph => output-name
				   :accessor neural-network-goa-table)
   (network-table :initform (make-hash-table :test #'equal :size 20)
		  :accessor neural-network-network-table)
   (current-menu-group :initform 'main-menu :accessor neural-network-current-menu-group)
   ;; the panes/subwindows
   (title )
   (top-menu)
   (parameters )
   (networks )
   (interactor )
   (input-glyphs )
   (current-input )
   (input-tablet )
   (outputs-pane )
   (connections )
   )
  ;; eventually this wants to be (:panes ...) with whatever the syntax is for specifying
  ;; multiple configurations...
  (:pane
    (with-frame-slots (title top-menu parameters networks interactor input-glyphs
			     current-input input-tablet outputs-pane connections)
      (vertically ()
	(make-clim-pane (title :scroll-bars nil :hs 800 :vs 70)
			:display-function '(display-title))
	(make-clim-pane (top-menu :scroll-bars nil :hs 800 :vs 30)
                        :default-text-style '(:sans-serif :roman :normal)
			:display-function '(display-current-command-menu))
	(horizontally ()
	  (make-clim-pane (parameters :scroll-bars nil :hs 160 :vs 130)
			  :display-function 
                          '(incremental-redisplay-display-function display-parameters)
                          :record-p nil)
	  (make-clim-pane (networks :scroll-bars :vertical :hs 200 :hs- 50 :vs 100
				    :label "Networks")
			  :display-function 
                          '(incremental-redisplay-display-function display-networks))
	  (make-clim-interactor (interactor :hs 500 :vs 100 :hs+ 200)))
	(horizontally ()
	  (make-clim-pane (input-glyphs :scroll-bars :vertical :hs 200 :vs 150
					:label "Input Glyphs")
			  :display-function
                          '(incremental-redisplay-display-function display-inputs))
	  (make-clim-pane (current-input :scroll-bars nil :hs 200 :vs 150
					 :label "Current Inputs")
			  :display-function '(display-current-input) :record-p nil)
	  (make-clim-pane (input-tablet :scroll-bars nil :hs 200 :vs 150
					:label "Tablet")
			  :display-function nil :record-p nil)
	  (make-clim-pane (outputs-pane :scroll-bars nil :hs 200 :vs 150
					:label "Outputs")
			  :display-function 
                          '(incremental-redisplay-display-function display-outputs)
                          :record-p nil))
	(make-clim-pane (connections :scroll-bars nil :hs 800 :vs 350
				     :label "Network Connections")
			:display-function '(display-connections) :record-p nil))))
  (:command-definer t)
  ;; you won't have to say this in 1.0.
  (:top-level (clim-top-level))
  
)

;;; Dire kludge.  I don't know why the display function isn't beeing called on these two.
(defmethod frame-repaint-pane ((frame neural-network) (pane #+clim-0.9 clim-stream-pane
							    #-clim-0.9 extended-stream-pane)
			       &optional opt-0 opt-1 opt-2)
  (declare (ignore opt-0 opt-1 opt-2))
  (with-slots (current-input connections) frame
    (if (or (eq pane current-input)
            (eq pane connections))
      (let ((disp-fun (slot-value pane 'ws::display-function)))
        (apply (first disp-fun) frame pane (rest disp-fun)))
      (call-next-method))))


;;; Presentation types

;;; --- this wants to take FRAME as a data argument, but since I can't
;;; define a command with a translator on a type of the form
;;; `(input-glyph *frame*) I guess that I will just use the special
;;; in the body...
(define-presentation-type input-glyph ()
  :parser ((stream &key default)
	   default
	   (completing-from-suggestions (stream)
	     (maphash #'(lambda (key val)
			  (suggest key val))
		      (neural-network-glyph-table *frame*))))
  ;; I can't really write a good printer since you can't easily
  ;; lookup the name given the glyph object.
  ;; I suggest using with-output-as-presentation instead.
  ;; --- Of course, the CP is going to invoke the printer with just
  ;; the object...
  :printer ((glyph stream &key &allow-other-keys)
	    (block find-glyph
	      (maphash #'(lambda (key val)
			   (when (eql val glyph)
			     (write-string key stream)
			     (return-from find-glyph nil)))
		       (neural-network-glyph-table *frame*)))))

;;; See comments on the input-glyph PT re: FRAME
(define-presentation-type network ()
  :parser ((stream &key default)
	   default
	   (completing-from-suggestions (stream)
	     (maphash #'(lambda (key val)
			  (suggest key val))
		      (neural-network-network-table *frame*))))
  ;; see comments on :printer of input-glyph
  :printer ((network stream &key &allow-other-keys)
	    (block find-network
	      (maphash #'(lambda (key val)
			   (when (eql val network)
			     (write-string key stream)
			     (return-from find-network nil)))
		       (neural-network-network-table *frame*)))))

;;; At last, a type that doesn't depend on per-frame data
(define-presentation-type output ()
  :parser ((stream &key default)
	   default
	   (completing-from-suggestions (stream)
	     (clim-utils::dovector (sym *symbols*)
	       (suggest (string sym) sym))))
  :printer ((sym stream &key &allow-other-keys)
	    (write-string (string sym) stream)))

(define-menu-group main-menu
  ((">Input" :command '(com-input-mode))
   (">Network" :command '(com-network-mode))
   (">Learn" :command '(com-learn-mode))
;   ("Show" :command '(com-show-connections ,*unsupplied-argument* ,*unsupplied-argument*))
;   ("Learn" :command `(com-learn-input ,*unsupplied-argument* ,*unsupplied-argument* ,*unsupplied-argument* ))
   ("Set" :command `(com-set-parameter ,*unsupplied-argument* #+Ignore ,*unsupplied-argument*))
   ("Exit" :command '(com-exit-neural-net))))

(define-neural-network-command (com-exit-neural-net :command-name "Exit")
    ()
   (with-frame (frame)
     (clim:stop-frame frame)))

(defmacro define-neural-network-menu-navigation-command (menu)
  (let* ((menu-name (intern (concatenate 'string (symbol-name menu) "-MENU")))
	 (command-symbol (intern (concatenate 'string "COM-" (symbol-name menu) "-MODE"))))
    `(define-neural-network-command (,command-symbol :command-name t)
	 ()
	(with-frame (frame)
	  (setf (neural-network-current-menu-group frame) ',menu-name)
	  (pane-needs-redisplay (slot-value frame 'top-menu))))))

(define-neural-network-menu-navigation-command main)
(define-neural-network-menu-navigation-command network)
(define-neural-network-menu-navigation-command input)
(define-neural-network-menu-navigation-command learn)

;(add-menu-group-entry ...)
;(remove-menu-group-entry
;  (ci::frame-find-menu-group)
;  (ws:find-menu-group-prototype

(define-neural-network-command (com-show-connections :command-name t)
    ((layer 'integer)
     (value? 'clim::boolean))
   (with-frame (frame)
     (let* ((network (neural-network-network frame))
	    (cstream (slot-value frame 'connections))
	    (n (length network)))
       (when (< -1 layer n)
	 ;;--- Baaarrf!
	 (in-connection-display-transform (cstream :clear nil :expose t)
	   (map nil
		#'(lambda (unit)
		    (map nil
			 #'(lambda (connection)
			     (show-connection network connection cstream value?))
			 (unit-ci unit)))
		(elt network layer))))
       (values))))

(define-neural-network-command (com-set-parameter :command-name t)
    ((parameter-name '(member n-inputs outputs eta alpha npasses))
     #+Ignore
     (value 'expression))
   (with-frame (frame)
     (let ((value (accept 'expression :stream (slot-value frame 'interactor)
			  :prompt "Value")))

       (setf (slot-value frame parameter-name) value))
     (pane-needs-redisplay
       (case parameter-name
	 (outputs (slot-value frame 'outputs-pane))
	 (otherwise (slot-value frame 'parameters))))
     (values)))

;;;;  The NETWORK configuration commands
(define-menu-group network-menu
  ((">Main" :command '(com-main-mode))
   (">Input" :command '(com-input-mode))
   (">Learn" :command '(com-learn-mode))
   ;;--- Does this mean one of these command lists gets consed for every command execution???
   ("Create" :command `(com-create-network ,*unsupplied-argument*))
   ("Name" :command `(com-name-network ,*unsupplied-argument*))
   ("Load" :command `(com-load-network ,*unsupplied-argument*))
   ("Save" :command `(com-save-network ,*unsupplied-argument* ,*unsupplied-argument*))
   ("Remove" :command `(com-remove-network ,*unsupplied-argument*))
   ("Select" :command `(com-select-network ,*unsupplied-argument*))))

(define-neural-network-command (com-create-network :command-name t)
    ((hidden-unit-layers 'integer))				;-- I'd really like to
								;-- use LIST here, but
								;-- it's not working
   (with-frame (frame)
     (let* ((hidden-units (let ((list (make-list hidden-unit-layers))
				(stream (slot-value frame 'interactor)))
			    (do ((sublist list (cdr sublist))
				 (i 1 (1+ i)))
				((null sublist))
			      (setf (first sublist)
				    (accept 'integer :prompt
					    (format nil "~%N Units in Hidden Layer ~D" i)
					    :stream stream))
			      (read-char stream))	;--- Baaaarf (have to read out the terminator char)
			    list))
	    (network (make-back-prop-network (neural-network-n-inputs frame)
					     (length (neural-network-outputs frame))
					     hidden-units))
	    (cstream (slot-value frame 'connections)))
       (setf (neural-network-network frame) network)
       (pane-needs-redisplay cstream))
     (values)))

(define-neural-network-command (com-name-network :command-name t)
    ((name 'string))
   (with-frame (frame)
     (setf (gethash name (neural-network-network-table frame))
	   (neural-network-network frame))
     (pane-needs-redisplay (slot-value frame 'networks))
     (values)))

(defun load-network (file)
  (declare (values name network))
  (with-open-file (input-stream file)
    (with-standard-io-syntax
      (multiple-value-bind (name connections)
                           (eval (read input-stream))
        (let* ((n-h&o-layers (length connections))
               (n-h-layers (1- n-h&o-layers))
               (n-inputs (length (elt (elt connections 0) 0)))
               (n-outputs (length (elt connections n-h-layers)))
               (network (make-back-prop-network n-inputs n-outputs
                                                (loop for i below n-h-layers
                                                      collect (length (elt connections i)))
                                                connections)))
          (values name network))))))

(define-neural-network-command (com-load-network :command-name t)
    ((file 'pathname))
   (with-frame (frame)
     (multiple-value-bind (name network)
	 (load-network file)
       (setf (gethash name (neural-network-network-table frame)) network))
     (pane-needs-redisplay (slot-value frame 'networks))
     (values)))

(defun dump-network (name network file)
  (with-open-file (output-stream file :direction :output
                                 :if-exists :new-version)
;;;--- This scheme is just too slow
;    (let ((*print-array* t)
;	  (*print-circle* t)
;	  (*print-pretty* nil)
;	  (*print-level* nil)
;	  (scl:*print-structure-contents* t)	;-- Why do we need this one???
;	  (scl:*print-readably* t))
;      ;;--- What a horrible kludge - I guess this means I should make
;      ;;--- NETWORKs defstructs
;      (setf (scl::array-leader network 1) nil)
;      (print network output-stream)
;      (setf (scl::array-leader network 1) 'back-prop-network))
    (format output-stream "(VALUES ~%  ~S~%  #(" name)
    (with-standard-io-syntax
      (loop for i from 1 below (length network)
            do
            (let ((layer (aref network i)))
              (format output-stream "~%    #(")
              (loop for unit being the array-elements of layer
                    do
                    (let ((connections-in (unit-ci unit)))
                      (format output-stream "~%      #(")
                      (loop for connection being the array-elements of connections-in
                            do (format output-stream "~S " (connection-wt connection)))
                      (format output-stream ")")))
              (format output-stream ")")))
      (format output-stream "))"))))
    
(define-neural-network-command (com-save-network :command-name t)
    ((network 'network)
     (file 'pathname))
   (with-frame (frame)
     (let ((name (get-hash-key network (neural-network-network-table frame))))
       (when name
	 (dump-network name network file)))
     (values)))

(define-neural-network-command (com-remove-network :command-name t)
    ((network 'network))
   (with-frame (frame)
     (let* ((table (neural-network-network-table frame))
	    (name (get-hash-key network table)))
       (when network
	 (when (eq network (neural-network-network frame))
	   (setf (neural-network-network frame) nil)
	   (pane-needs-redisplay (slot-value frame 'connections))
	   (pane-needs-redisplay (slot-value frame 'networks)))
	 (when name
	   (remhash name (neural-network-network-table frame)))))
     (values)))

(define-neural-network-command (com-select-network :command-name t)
    ((network 'network :translator-gesture :left))
   (with-frame (frame)
     (setf (neural-network-network frame) network)
     (pane-needs-redisplay (slot-value frame 'connections))
     (values)))

;;;; The INPUT configuration commands
(define-menu-group input-menu
  ((">Main" :command '(com-main-mode))
   (">Network" :command '(com-network-mode))
   (">Learn" :command '(com-learn-mode))
   ("Get" :command '(com-get-input-from-tablet))
   ("Name" :command `(com-name-input ,*unsupplied-argument*))
   ("Load" :command `(com-load-inputs ,*unsupplied-argument*))
   ("Save" :command `(com-save-input ,*unsupplied-argument* ,*unsupplied-argument*))
   ("Select" :command `(com-select-input ,*unsupplied-argument*))))

(define-neural-network-command (com-name-input :command-name t)
    ((name 'string))
   (with-frame (frame)
     (setf (gethash name (neural-network-glyph-table frame))
	   (neural-network-current-input-glyph frame))
     (pane-needs-redisplay (slot-value frame 'input-glyphs))
     (values)))

(define-neural-network-command (com-load-inputs :command-name t)
    ((file 'pathname))
   (with-frame (frame)
     (block read-loop
       (with-open-file (input-stream file)
	 (let ((*package* (find-package "CLIM-DEMO")))
	   (loop 
	     (multiple-value-bind (name glyph)
		 (eval (with-standard-io-syntax (read input-stream nil nil)))
	       (if name
		   (setf (gethash name (neural-network-glyph-table frame)) glyph)
		   (return-from read-loop)))))))
     (pane-needs-redisplay (slot-value frame 'input-glyphs))
     (values)))

(defun print-glyph-array (array stream)
  #-ccl-2 (print array stream)
  #+ccl-2 (write-string "#2a" stream)
  (loop for x below (array-dimension array 0)
        for char first #\( then #\Space
        do
        (write-char char stream)
        (loop for y below (array-dimension array 1)
              for char first #\( then #\Space
              do
              (write-char char stream)
              (prin1 (aref array x y) stream)
              finally (write-char #\) stream))
        finally (write-char #\) stream)))

(defun dump-glyph (name glyph file)
  (with-open-file (output-stream file :direction :output
                                 :if-exists :new-version)
    (format output-stream "(VALUES ~S" name)
    (format output-stream "~%        (~S :DIM0 ~D :DIM1 ~D~
                           ~%                    :PIXELS "
            'make-glyph (glyph-dim0 glyph) (glyph-dim1 glyph))
    (with-standard-io-syntax
      (print-glyph-array (glyph-pixels glyph) output-stream))
    (format output-stream "~%))")))

(define-neural-network-command (com-save-input :command-name t)
    ((glyph 'input-glyph)
     (file 'pathname))
   (with-frame (frame)
     (let ((name (get-hash-key glyph (neural-network-glyph-table frame))))
       (when name
	 (dump-glyph name glyph file)))
     (values)))

(define-neural-network-command (com-get-input-from-tablet :command-name t)
    ()
   (with-frame (frame)
     (let* ((tablet-stream (slot-value frame 'input-tablet))
	    (glyph-stream (slot-value frame 'current-input))
	    ;; Make sure to bind this special
	    (*n-connection-inputs* (neural-network-n-inputs frame))
	    (input-glyph (get-input-glyph tablet-stream glyph-stream)))
       (when input-glyph
	 (setf (neural-network-current-input-glyph frame) input-glyph)))
     (values)))

#+ignore
;;; old version 
(define-neural-network-command (com-select-input :command-name t)
    ((name 'string))
   (with-frame (frame)
     (let* ((glyph-stream (slot-value frame 'current-input))
	    (input-glyph (gethash name (neural-network-glyph-table frame))))
       (cond (input-glyph
	      (setf (neural-network-current-input-glyph frame) input-glyph)
	      (window-clear glyph-stream)
	      (window-expose glyph-stream)
	      (draw-input-glyph input-glyph glyph-stream))
	     (t (format (slot-value frame 'interactor) "~%No glyph named ~S" name))))
     (values)))

(define-neural-network-command (com-select-input :command-name t)
    ;; ---I used to have:
    ;;    ((input-glyph `(input-glyph ,*frame*) :translator-gesture :left))
    ;; but you can't have a translator to a non-constant type.
    ;; So, I re-wrote the presentation type to reference *frame* instead
    ((input-glyph `input-glyph :translator-gesture :left))
   (with-frame (frame)
     (let* ((glyph-stream (slot-value frame 'current-input)))
       (setf (neural-network-current-input-glyph frame) input-glyph)
       (window-clear glyph-stream)
       (window-expose glyph-stream)
       (draw-input-glyph input-glyph glyph-stream))
     (values)))

;;;; The LEARN configuration commands
(define-menu-group learn-menu
  ((">Main" :command '(com-main-mode))
   (">Input" :command '(com-input-mode))
   (">Network" :command '(com-network-mode))
   ("Run" :command `(com-run-current-input))
   ("Learn" :command `(com-learn-input ,*unsupplied-argument* ,*unsupplied-argument* ,*unsupplied-argument*))
   ("Train" :command `(com-train-network))))

(defun learn-glyph-from-input (network glyph desired-output outputs cstream
			       &optional (min-passes *min-passes*))
  (let ((frame *frame*)
	(istream (slot-value *frame* 'interactor)))
    (unless (eq glyph (neural-network-current-input-glyph frame))
      (setf (neural-network-current-input-glyph frame) glyph)
      (display-current-input frame (slot-value frame 'current-input)))
    (in-connection-display-transform (cstream :clear nil :expose t)
      (let ((npasses min-passes))
	(loop for pass from 1
	      do (progn (format istream " ~D" pass)
			(learn-glyph network glyph desired-output outputs 1)
			(decf npasses))
	      until (and (not (plusp npasses))
			 (string-equal (get-output-from-glyph network glyph outputs)
				       desired-output)))))))

(define-neural-network-command (com-learn-input :command-name t)
    ((glyph 'input-glyph :default (neural-network-current-input-glyph *frame*)
	    :translator-gesture :middle)
     ;; Can't use :translator-gesture :middle here 'cause the arguments
     ;; will be read via AVV, which will bind *frame* to the avv
     ;; application.  Sigh.
     (network 'network :default (neural-network-network *frame*))
     (for-output 'output))
   (with-frame (frame)
     (learn-input-internal frame glyph network for-output)))

(define-neural-network-command (com-learn-current-input :command-name t)
    ()
   (with-frame (frame)
     (let* ((glyph (neural-network-current-input-glyph frame))
	    (network (neural-network-network frame))
	    (output (gethash glyph (neural-network-goa-table frame))))
       (cond (output
	      ;; count on *standard-input* supporting output...
	      (format *standard-input* "Learning glyph ~A in network ~A for output ~A"
		      (get-hash-key glyph (neural-network-glyph-table frame))
		      (get-hash-key network (neural-network-network-table frame))
		      output)
	      (learn-input-internal frame glyph network output))
	     (t (format *standard-input*
			"No output associated with the current input glyph."))))))

(defun learn-input-internal (frame glyph network for-output)   
  (unless (eq glyph (neural-network-current-input-glyph frame))	;This is SET-CURRENT-INPUT-GLYPH
    (setf (neural-network-current-input-glyph frame) glyph)
    (display-current-input frame (slot-value frame 'current-input)))
  (unless (eq network (neural-network-network frame))	;This is SET-CURRENT-NETWORK
    (setf (neural-network-network frame) network)
    (display-connections frame (slot-value frame 'connections)))
  (let* ((outputs (neural-network-outputs frame))
	 (cstream (slot-value frame 'connections))
	 ;; Make sure to bind this special
	 (*n-connection-inputs* (neural-network-n-inputs frame)))
    ;; Fix up the association cache
    (let ((current-association (gethash glyph (neural-network-goa-table frame))))
      (unless (string-equal current-association for-output)
	(pane-needs-redisplay (slot-value frame 'input-glyphs)))
      (setf (gethash glyph (neural-network-goa-table frame)) for-output))
    ;;
    (learn-glyph-from-input network glyph for-output outputs cstream))
  (values))

(define-neural-network-command (com-run-current-input :command-name t)
    ()
   (with-frame (frame)
     (let ((glyph (neural-network-current-input-glyph frame)))
       (when glyph
	 (let ((network (neural-network-network frame)))
	   (when network
	     (let* ((outputs (neural-network-outputs frame))
		    (istream (slot-value frame 'interactor))
		    (cstream (slot-value frame 'connections))
		    ;; Make sure to bind this special
		    (*n-connection-inputs* (neural-network-n-inputs frame)))
	       (when glyph
		 (run-input network glyph outputs istream cstream)))
	     (values)))))))

(define-neural-network-command (com-train-network :command-name t)
    #+ignore ((passes 'integer)				;:default `(neural-network-min-learn-passes ,*frame*)
     (verify-as-you-go 'clim::boolean))		;:default nil
    ()
    (let ((verify-as-you-go t)
          (passes 5))
   (with-frame (frame)
     (let ((network (neural-network-network frame)))
       (when network
	 (let ((goa-cache (neural-network-goa-table frame)))
	   (when goa-cache
	     (let* ((outputs (neural-network-outputs frame))
		    (istream (slot-value frame 'interactor))
		    (cstream (slot-value frame 'connections))
		    ;; Make sure to bind this special
		    (*n-connection-inputs* (neural-network-n-inputs frame))
		    ;;---
		    (glyph-table (neural-network-glyph-table frame))
		    tested)
	       (flet ((do-one (glyph-name &optional (glyph (gethash glyph-name glyph-table)))
			(let ((desired-output (gethash glyph goa-cache)))
			  (when desired-output
			    (format istream "~%Learning ~A (~A)  " glyph-name desired-output)
			    (learn-glyph-from-input network glyph desired-output
						    outputs cstream passes)
			    ;; Make sure all the ones we've already trained still work
			    (when verify-as-you-go
			      (loop for onemotime = nil
				    do
				(loop for (g-n glyph d-o) on tested by 'cdddr
				      do
				  (progn
				    (format istream "~%  Verifying ~A (~A) ==> " g-n d-o)
				    (let ((r-o (run-input network glyph outputs
							  istream cstream)))
				      (format istream "~A " r-o)
				      (unless (string-equal d-o r-o)
					(format istream
						"~%    Ooops.  Forgot ~A (~A) ... relearning  "
						g-n d-o)
					(setq onemotime t)
					(learn-glyph-from-input network glyph d-o outputs
								cstream passes)))))
				    while onemotime)
			    (setq tested (list* glyph-name glyph desired-output tested)))))))
	       (maphash #'do-one (neural-network-glyph-table frame))
	       (values))))))))))

;;;--- Borrowed from CLIM:DEMO;GRAPHICS-DEMOS
(defun new-ila-logo (stream)
  "Translated from the following PostScript code:
/ila-l
 { newpath

   1 1 5 {
   /iteration exch def
   /offset iteration 2 mul 17 exch sub def

   /origin iteration 1 sub 20 mul def
   /ii origin offset add def
   /jj 100 offset sub def

   origin origin m
   origin jj l
   ii 100 l
   ii ii l
   100 ii l
   jj origin
   origin origin l
   closepath fill

   } for
   
 } def
"
  ;;--- Isn't there a canned reflection transform? 
  (with-drawing-options (stream :transformation (make-transformation 1 0 0 -1 20 120))	;-- There's some weird extra 20 pixels

    (do ((i 1 (1+ i)))
	((> i 5))
      (let* ((offset (- (* i 2) 17))
	     (origin (* (1- i) 20))
	     (ii (+ origin offset))
	     (jj (- 100 offset)))
	(draw-polygon* stream (list origin origin origin jj ii 100 ii ii 100 ii jj origin)
		       :ink +red+)))))

(defmethod display-title ((application neural-network) stream)
  (window-clear stream)
  (scroll-home stream)
  ;;--- Why is the text style not being obeyed?
  (multiple-value-bind (width height)
      (entity-size (sheet-region stream))
    (let* ((title "CLIM Neural Net")		;"Hide the Thalami"
	   (title-style '(:sans-serif :bold :very-large))
	   (text-width (stream-string-width stream title :text-style title-style)))
      (draw-text* stream title (round (- width text-width) 2) (round height 2)
                  :text-style title-style
                  :align-y :center)
      ;; Draw the ILA logo
      (with-scaling (stream (/ (- height 15) 100))
	(new-ila-logo stream))
      (values))))

(defmethod display-current-command-menu ((frame neural-network) pane)
  ;;-- Baaarf - just make the (window-clear pane) after Bill fixes the
  ;;   repaint recursion bug
  (display-command-menu frame pane (neural-network-current-menu-group frame)))

(defmethod display-parameters ((application neural-network) stream)
  (with-slots (n-inputs #+ignore n-outputs eta alpha npasses) application
    (with-text-style ('(:sans-serif :bold :small) stream)
      (updating-output (stream)
	(formatting-table (stream)
	  (macrolet ((do-one-param (name &optional (format-string "~4D"))
		       `(formatting-row (stream)
			  (formatting-cell (stream)
			    (write-string (string-capitalize ',name) stream))
			  (formatting-cell (stream)
			    (format stream ,format-string ,name)))))
	    (do-one-param n-inputs)
;          (do-one-param n-outputs)
	    (do-one-param eta "~4,1F")
	    (do-one-param alpha "~4,1F")
	    (do-one-param npasses))))))
    (values))

(defmethod display-inputs ((application neural-network) stream)
  (with-slots (glyph-table glyph-output-association-table) application
;;;--- Old way
;    (formatting-table (stream)
;      (maphash #'(lambda (name glyph &rest ignore)
;		   (formatting-row (stream)
;		     (formatting-cell (stream)
;		       (present name 'string :stream stream))
;		     (formatting-cell (stream)
;		       (draw-input-glyph glyph stream nil 1))
;		     (let ((matching-output (gethash glyph glyph-output-association-table)))
;		       (when matching-output
;			 (formatting-cell (stream)
;			   (present (string matching-output) 'string :stream stream))))))
;;; York way
    (updating-output (stream)
      (formatting-table (stream :inter-column-spacing 5)
        (formatting-row (stream)
	  (with-text-style ('(:serif :bold :small) stream)
	    (formatting-cell (stream)
	      (write-string "Name  " stream))
	    (formatting-cell (stream)
	      (write-string "Shape  " stream))
	    (formatting-cell (stream)
	      (write-string "Output" stream))))
;        (formatting-row (stream)
;          (formatting-cell (stream)
;            (write-string " " stream))
;          (formatting-cell (stream)
;            (write-string " " stream))
;          (formatting-cell (stream)
;            (write-string " " stream)))
        (maphash #'(lambda (name glyph &rest ignore)
                     (declare (ignore ignore))
                     (with-output-as-presentation (:stream stream
                                                           :object glyph
                                                           :type `input-glyph
                                                           :single-box t)
                       (formatting-row (stream)
                         (formatting-cell (stream)
                           (write-string name stream))
                         (formatting-cell (stream)
                           (draw-input-glyph glyph stream nil 1))
                         (let ((matching-output (gethash glyph glyph-output-association-table)))
                           (formatting-cell (stream)
                             (write-string (if matching-output
                                             (string matching-output)
                                             "???")
                                           stream))))))
                 glyph-table))))
    (values))

(defmethod display-current-input ((application neural-network) stream)
  (window-clear stream)
  (with-slots (current-input-glyph current-input) application
    (when current-input-glyph
      (draw-input-glyph current-input-glyph current-input)))
  (values))

(defmethod display-networks ((application neural-network) stream)
  (with-slots (network-table) application
    (updating-output (stream)
      (formatting-table (stream)
        (maphash #'(lambda (name network)
                     (formatting-row (stream)
                       (with-output-as-presentation (:stream stream
                                                             :object network
                                                             :type `network)
                         (formatting-cell (stream)
                           (write-string name stream)))))
                 network-table))))
  (values))

(defmethod display-outputs ((application neural-network) stream)
  (with-slots (outputs) application
    (updating-output (stream)
      (formatting-table (stream :multiple-columns t)
        (map nil #'(lambda (name)
                     (formatting-row (stream)
                       (formatting-cell (stream)
                         (present name 'output :stream stream))))
             outputs))))
  (values))

(defmethod display-connections ((application neural-network) stream)
  (window-clear stream)
  (with-slots (connections network outputs) application
    (when network
      (in-connection-display-transform (connections :clear nil :expose t)
        (show-network network outputs connections t))))
  (values))

;;; A list of neural net applications.
(defun run-neural-network (&key (server-path *default-server-path*))
  (ci::launch-frame 'neural-network
                    :title "Neural Network"
                    :where server-path
                    :width 700 :height 490
                    :if-does-not-exist :create
                    ))

;(run-neural-network)

