;;; -*- Mode: LISP; Syntax: Common-lisp; Package: CLIM-DEMO; Base: 10 -*-

(in-package "CLIM-DEMO")

"Copyright (c) 1989, International Lisp Associates.  All rights reserved."

(defmacro with-demo-medium ((var-spec frame) &body body)
  #+genera (declare (scl:arglist ((medium-var stream-var) frame) &body body))
  (multiple-value-bind (medium-var stream-var)
      (etypecase var-spec
        (list 
         (values (first var-spec)
                 (or (second var-spec) (clim-utils:gensymbol "STREAM"))))
        (symbol
         (values var-spec (clim-utils:gensymbol "STREAM"))))
    `(let ((,stream-var (graphics-demo-demo-pane ,frame)))
       (using-clim-medium (,medium-var ,stream-var)
                          ,@body))))

(defun spin (frame)
  (explain frame "A simple example of the use of affine transforms.
Take a simple function that draws a picture and invoke
it repeatedly under various rotations.")
  (with-demo-medium (stream frame)
    (flet ((draw (stream)
	     (draw-rectangle* stream 0 0 50 50 :ink +blue+)
	     (draw-polygon* stream '(50 50 50 75 75 50) :ink +cyan+)
	     (draw-circle* stream 70 30 20 :ink +red+)))
      (do ((angle 0 (+ angle (/ pi 4))))
	  ((> angle clim::2pi) nil)
	(with-rotation (stream angle)
	  (with-translation (stream 100 0)
	    (draw stream)))))))

(defun ila-logo (stream)
  (flet ((draw-l (stream)
           (draw-polygon* stream '(0 0 0 70 10 80 10 10 80 10 70 0) :ink +red+)
           #|
	   (draw-rectangle* stream 0 10 10 80 :ink +red+)
	   (draw-rectangle* stream 0 70 80 80 :ink +red+)
	   (draw-polygon* stream '(10 0 10 10 0 10) :ink +red+)
   	   (draw-polygon* stream '(80 70 90 70 80 80) :ink +red+)
	   (draw-triangle* 0 0 0 10 10 10 :stream stream :ink +red+)
           (draw-triangle* 80 70 80 80 91 80 :stream stream :ink +red+)
           |#
	   ))
    (dotimes (i 4)
      (with-translation (stream (* i 18) (* i 18))
	(with-scaling (stream (/ (- 5 i) 5))
	  (draw-l stream))))))

(defun ila-spin (frame stream)
  (explain frame "A slightly more complex example using both
rotation and scaling.")
  (with-translation (stream 0 -25)
    (do ((angle 0 (+ angle (/ pi 4)))
         (scale 1 (* scale 7/8)))
        ((< scale .07) nil)
      ;; ((> angle clim::2pi) nil)
      (with-rotation (stream angle)
        (with-scaling (stream scale)
          (with-translation (stream 100 0)
            (ila-logo stream)))))))

(defun big-spin (frame)
  (let ((stream (graphics-demo-demo-pane frame)))
    (with-output-recording-options (stream :record-p nil)
      (with-scaling (stream 1.7)
        (with-demo-medium (stream frame)
          (ila-spin frame stream))))))

#+ignore
(defparameter *lot-of-text*
  "This is a lot of text to test the text drawing speed.")

#+ignore
(defun lot-of-text (frame)
  (explain frame "Text drawing should be fast, but it's not.")
  (let ((stream (graphics-demo-demo-pane frame)))
    (with-output-recording-options (stream :record-p nil)
      (dotimes (i 5)
        (window-clear stream)
        (dotimes (j 20)
          (draw-text* stream *lot-of-text* 5 (+ (* j 15) 5))))))
  (explain frame "Stream text drawing is even slower.")
  (let ((stream (graphics-demo-demo-pane frame)))
    (with-output-recording-options (stream :record-p nil)
      (dotimes (i 5)
        (window-clear stream)
        (dotimes (j 20)
          (write-string *lot-of-text* stream)
          (terpri stream))))))

(defun draw-crosshairs-on-window (ws stream &optional (scale-p nil) (x nil) (y nil) (size nil))
  (let ((ink +flipping-ink+))
    (multiple-value-bind (width height)
	(entity-size stream)
      (unless size
	(setq size (max width height)))
      (unless (and x y)
	(setf x (truncate width 4))
	(setf y (truncate height 4)))
      (draw-line* ws x (- y size) x (+ y size) :ink ink)
      (draw-line* ws (- x size) y (+ x size) y :ink ink)
      (when scale-p
	(do ((x1 x (- x1 scale-p))
	     (x2 x (+ x2 scale-p))
	     (y1 y (- y1 scale-p))
	     (y2 y (+ y2 scale-p)))
	    ((and (>= x2 size) (>= y2 size)) nil)
	  (let ((x3 (- x (truncate scale-p 2)))
		(x4 (+ x (truncate scale-p 2)))
		(y3 (- y (truncate scale-p 2)))
		(y4 (+ y (truncate scale-p 2))))
	    (draw-line* ws x3 y1 x4 y1 :ink ink)
	    (draw-line* ws x3 y2 x4 y2 :ink ink)
	    (draw-line* ws x1 y3 x1 y4 :ink ink)
	    (draw-line* ws x2 y3 x2 y4 :ink ink)))))))

(defun demo-cbs-logo (frame)
  (explain frame "Flipping ink can be used to create interesting pictures.")
  (with-demo-medium ((ws stream) frame)
    (draw-circle* ws 0 0 200 :ink +foreground+)
    (draw-ellipse* ws 0 0 200 0 0 100 :ink +background+)
    (draw-circle* ws 0 0 100 :ink w:+light-blue+)
    (draw-crosshairs-on-window ws stream 25 0 0 200)))

(defun draw-regular-polygon (stream x1 y1 x2 y2 n &optional (ink +foreground+))
  (multiple-value-bind (left top)
                       (values 0 0)
    (let ((theta (* pi (1- (/ 2.0 n))))
          (n (abs n))
          (x0 x1)
          (y0 y1)
          (x-1 x2)
          (y-1 y2)
          (poly nil))
    
      (do ((i 2 (1+ i))
           (sin-theta (sin theta))
           (cos-theta (cos theta))
           (x3) (y3))
          ((not (< i n)))
        (setq x3 (+ (- (- (* x1 cos-theta)
                          (* y1 sin-theta))
                       (* x2 (1- cos-theta)))
                    (* y2 sin-theta))
              y3 (- (- (+ (* x1 sin-theta)
                          (* y1 cos-theta))
                       (* x2 sin-theta))
                    (* y2 (1- cos-theta))))
        (push (+ left (round x3)) poly)
        (push (+ top (round y3)) poly)
        (setq x1 x2 y1 y2 x2 x3 y2 y3))
      (push (+ top (round x0)) poly)
      (push (+ top (round y0)) poly)
      (push (+ top (round x-1)) poly)
      (push (+ top (round y-1)) poly)
      (setq poly (nreverse poly))
      (draw-polygon* stream poly :ink ink))))

(defun demo-sleep (ws secs)
  ;(finish-output ws)
  ;(sleep secs)
  ;#+ignore
  (loop
    (multiple-value-bind (gesture type)
       (stream-read-gesture ws :timeout secs)
      (cond ((eql gesture #\space) (return t))
            ((eq type :timeout) (return nil)))
      (beep)
      (decf secs))))

(defun polygon-demo (frame)
  (with-demo-medium ((ws stream) frame)
    (dolist (number-of-sides '(3 #+Ignore 4 5 #+Ignore 6 #+Ignore 7 8))
      (window-clear stream)
      (do ((i 10 (+ i 5)))
	  ((> i 100) nil)
	;;--- assumption about size of viewport and current transform
	(draw-regular-polygon ws -200 (+ i) -200 (- i) number-of-sides +flipping-ink+))
      (force-output stream)
      (demo-sleep stream 1))))

;;; Not really a constant, since Symbolics will warn, but don't
;;; try to change it either (note the #, below).  --RWK
(defvar +random-ink-list+
	(list +red+ +green+ +blue+
	      +cyan+ +magenta+ +yellow+ +black+))

(defun random-ink ()
  (nth (random #-Ansi-90 '#,(length +random-ink-list+)
               #+Ansi-90 (length +random-ink-list+))
       +random-ink-list+))

(defun draw-lots-of-circles (frame &optional (radius 5))
  (explain frame "")
  (with-demo-medium ((ws stream) frame)
    (let ((separation (+ 2 (* 2 radius))))
      (multiple-value-bind (wid hei)
           (bounding-rectangle-size (sheet-region stream))
        (do ((y separation (+ y separation)))
            ((> y (- hei separation)) nil)
          (do ((x separation (+ x separation)))
              ((> x (- wid separation)) nil)
            (draw-circle* ws x y radius :filled nil
                          :ink (random-ink))))))))

;;; --- this should scale to the size of the window
(defun a-mazing-demo (frame)
  (explain frame "This simple maze drawer uses the graphics
scaling feature to adjust the maze size
to the window in which it is displayed.")
  (with-demo-medium ((ws stream) frame)
    (multiple-value-bind (w h)
	(bounding-rectangle-size (sheet-region stream))
      ;; --- seems to be designed for 700x600 window, so scale appropriately
      (let ((xs (/ w 700)) (ys (/ h 600)))
	(with-scaling (ws xs ys)
	  (draw-lines* ws '( 30  40 670  40 670  40 670 560 670 560  30 560
			     30 560  30  80  30  80  70  80  70  80  70 520
			     70 520 630 520 630 520 630  80 630  80 590  80
			    590  80 590 480 590 480 110 480 110 480 110 120
			    110 120 510 120 510 120 510 400 510 400 190 400
			    190 400 190 160
			    110  80 550  80 550  80 550 440 550 440 150 440
			    150 440 150 160 150 160 470 160 470 160 470 360
			    470 360 230 360 230 360 230 200 230 200 430 200
			    430 200 430 320 430 320 270 320 270 320 270 240
			    270 240 390 240 390 240 390 280)
		       :ink +foreground+ :line-thickness 3)
	  ;;draw start
	  (draw-circle* ws 25 60 5 :ink +flipping-ink+)
	  ;; draw finish
	  (draw-circle* ws 330 280 5 :ink +flipping-ink+)
  
	  (force-output stream)
	  (demo-sleep stream 5)

	  ;; draw a solution path
	  (draw-lines* ws '( 30  60 570  60 570  60 570 460 570 460 130 460
			    130 460 130 140 130 140 490 140 490 140 490 380
			    490 380 210 380 210 380 210 180 210 180 450 180
			    450 180 450 340 450 340 250 340 250 340 250 220
			    250 220 410 220 410 220 410 300 410 300 330 280)
		       :ink w:+orange+)
	  )))))

;;; The EXPLAINs should probably be in some def-graphics-demo form rather
;;; than scattered in the code...
(define-application-frame graphics-demo ()  
  ((title-pane)
   (demo-pane :reader graphics-demo-demo-pane)
   (explanation-pane))
  (:pane (with-frame-slots (title-pane demo-pane explanation-pane)
	   (vertically ()
	     (setq title-pane (make-pane 'ws:label-pane
					 :text "Graphics Demo"
					 :text-style (parse-text-style
						       '(:sans-serif :bold :large))))
	     (make-clim-pane (demo-pane :hs 400
					:vs #+imach 320 #-imach 380
					:scroll-bars nil))
	     (make-clim-pane (explanation-pane :vs 44 :hs 400
					       :scroll-bars nil)))))
  (:top-level (do-demo)))

(defmethod explain ((frame graphics-demo) text)
  (with-slots (explanation-pane) frame
    (when explanation-pane
      (window-clear explanation-pane)
      (with-text-style ('(:serif :roman :normal) explanation-pane)
        (write-string text explanation-pane)))))

(defun run-graphics-demos (&key (sleep-time 5) loop
                                (server-path *default-server-path*))
  (macrolet ((run-one (&body body)
	       `(progn (window-clear stream)
		       (pane-needs-redisplay stream)
		       ,@body
		       (demo-sleep stream sleep-time))))
    (let* ((frame (make-frame 'graphics-demo))
           (port (find-port :server-path server-path))
	   (framem (find-frame-manager :server-path server-path)))
      (unwind-protect
	  (progn (adopt-frame framem frame)
                 (ci::initialize-stream-queues frame)
		 (enable-frame frame)
                 (ws:repaint-pane (clim:frame-pane frame) clim:+everywhere+)
		 (let ((stream (graphics-demo-demo-pane frame)))
		   (multiple-value-bind (w h)
		       (bounding-rectangle-size (sheet-region stream))
                     (ci::with-cursor-visibility (nil stream)
                       (with-output-recording-options (stream :record-p nil)
                         (process-next-event port :timeout 1)
                         ;(redisplay-frame-panes frame)
                         (let ((xoff (round w 2))
                               (yoff (round h 2)))
                           (loop
                             #+ignore (run-one (lot-of-text frame))
                             (with-translation (stream (round w 2) (round h 2))
                               (run-one (spin frame))
                               (run-one (big-spin frame))
                               (with-drawing-options (stream :ink (make-color-rgb 0 .5 1))
                                 (run-one (demo-cbs-logo frame)))
                               (run-one (polygon-demo frame))
                               ;;--- expects (0,0) to be in upper left
                               (with-translation (stream (- xoff) (- yoff))
                                 (run-one (draw-lots-of-circles frame 20))
                                 (run-one (a-mazing-demo frame)))
                               (unless loop (return))))))))))
	(disable-frame frame)))))
  
