;;; -*- Mode: LISP; Syntax: Common-lisp; Package: CLIM-DEMO; Base: 10 -*-

(in-package 'CLIM-DEMO)

"Copyright (c) 1990, International Lisp Associates.  All rights reserved."

(define-application FSEDIT
		    ((root :initform
			   (setup-fsedit-file 
			     (directory-pathname-as-file (user-homedir-pathname))))
		     (sort-type :initform :name)
		     (sort-forward :initform t))
  :subwindows ((title :title
		      :display-string "File System")
	       (display :application
			:display-function display-files)
	       (interactor :interactor))
  :layout ((main (:column :rest
		  (title :compute)
		  (interactor 1/4)
		  (display :rest)))))

(defstruct (fsedit-file
	    (:conc-name fsf-))
  pathname
  (creation-date nil)
  (author nil)
  (is-directory-p)
  wild-pathname
  (file-list nil)
  (show-directory-contents nil))

(defun directory-pathname-as-file (pathname)
  (let* ((pathname (pathname pathname))
	 (dir (pathname-directory pathname))
	 (new-dir (butlast dir))
	 (new-name (first (last dir))))
    (when (null new-dir) (setf new-dir :ROOT))
    (make-pathname :host (pathname-host pathname)
		   :device (pathname-device pathname)
		   :directory new-dir
		   :defaults (parse-namestring new-name)
		   #+Genera :type #+Genera "DIRECTORY")))

(defun file-pathname-as-wild-directory (pathname &optional (version :newest))
  (let* ((pathname (pathname pathname))
	 (dir (pathname-directory pathname))
	 (new-dir-part #-Genera (file-namestring pathname) #+Genera (pathname-name pathname)))
    (when (eql dir :ROOT) (setf dir nil))
    ;; Handle "/foo/bar/" (as opposed to "/foo/bar") case:
    (when (and (stringp new-dir-part) (zerop (length new-dir-part)))
      (setf new-dir-part nil))
    (make-pathname :host (pathname-host pathname)
		   :device (pathname-device pathname)
		   :directory (append dir (and new-dir-part (list new-dir-part)))
		   :name :wild :type :wild :version version)))

#+:excl
;;; Internal hackery because there isn't a way to tell this otherwise.
;;; Written 21 Feb 1990 by Charley Cox...
(defun file-is-directory-p (pathname)
  (eql 1 
       (excl::filestat-type
	(excl::filesys-filestat (namestring (pathname pathname))))))

#+Genera
(defun file-is-directory-p (pathname)
  (getf (cdr (fs:file-properties pathname)) :directory))

(defun setup-fsedit-file (pathname)
  (let* ((pathname (pathname pathname))
	 (wild-pathname (file-pathname-as-wild-directory pathname))
	 (fsf (make-fsedit-file :pathname pathname :wild-pathname wild-pathname)))
    (get-file-properties fsf)
    fsf))
  
(defun get-file-properties (fsf)
  (let ((pathname (fsf-pathname fsf)))
    (setf (fsf-creation-date fsf) (file-write-date pathname)
	  (fsf-author fsf) (file-author pathname)
	  (fsf-is-directory-p fsf) (file-is-directory-p pathname))))

(defun expand-directory (fsf)
  (when (fsf-is-directory-p fsf)
    (setf (fsf-file-list fsf)
      (mapcar #'setup-fsedit-file (directory (fsf-wild-pathname fsf))))))


(defmethod display-files ((application fsedit) stream)
  (with-slots (root sort-type sort-forward) application
     (updating-output (stream :unique-id 'root :cache-value root :cache-test #'equal)
       (format stream "~A" (namestring (fsf-pathname root))))
     (formatting-table (stream)
       (display-file-internal stream root sort-type sort-forward))))

(defun display-file-internal (stream fsf sort-type sort-forward
			       &optional (indent 0) &aux (new-indent (1+ indent)))
  ;; Cache: pathname is the key, FSF is the cache value (could probably
  ;; use creation-date, but...)
  (updating-output (stream :unique-id (fsf-pathname fsf) :id-test #'equal
			   :cache-value fsf :cache-test #'equal
			   :copy-cache-value t)
    (formatting-row (stream)
      (multiple-value-bind (sec min hr day mo yr)
	  (decode-universal-time (fsf-creation-date fsf))
	(formatting-cell (stream)
	  (write-string (if (fsf-is-directory-p fsf) "D" " ") stream))
	(with-output-as-presentation (:type 'pathname :object (fsf-pathname fsf) :stream stream)
	  (formatting-cell (stream)
	    (dotimes (i indent) (write-char #\Space) stream)
	    (write-string (file-namestring (fsf-pathname fsf)) stream)))
	(formatting-cell (stream)
	  (write-string (fsf-author fsf) stream))
	(formatting-cell (stream)
	  (format stream "~2,'0D/~2,'0D/~2,'0D ~2,'0D:~2,'0D:~2,'0D"
		  day mo (mod yr 100) hr min sec)))))
  (when (and (fsf-is-directory-p fsf)
	     (fsf-show-directory-contents fsf))
    (sort-fsf-file-list fsf sort-type sort-forward)
    (dolist (fsf (fsf-file-list fsf))
      (display-file-internal stream fsf sort-type sort-forward new-indent))))

(defun fsf-namestring (fsf)
  (file-namestring (fsf-pathname fsf)))

(defun sort-fsf-file-list (fsf sort-type sort-forward)
  (multiple-value-bind (sort-predicate sort-key)
      (ecase sort-type
	(:name (values (if sort-forward #'string-lessp #'string-greaterp) #'fsf-namestring))
	(:creation-date (values (if sort-forward #'< #'>) #'fsf-creation-date)))
    (setf (fsf-file-list fsf)
	    (sort (fsf-file-list fsf) sort-predicate :key sort-key))))

(defmethod find-fsf-for-pathname ((fsedit fsedit) pathname)
  (labels ((find (fsf)
	     (when (equal pathname (fsf-pathname fsf))
	       (return-from find-fsf-for-pathname fsf))
	     (mapcar #'find (fsf-file-list fsf))))
    (find (slot-value fsedit 'root))))

(define-fsedit-operation com-open-directory
			 ((pathname 'pathname))
  (let ((fsf (find-fsf-for-pathname *application* pathname)))
    (when fsf
      (expand-directory fsf)
      (setf (fsf-show-directory-contents fsf) t))))

(define-presentation-to-operation-translator com-open-directory
   (pathname :gesture :left
	     :tester ((object)
		      (let ((fsf (find-fsf-for-pathname *application* object)))
			(and fsf (not (fsf-show-directory-contents fsf))))))
   (object)
   `(com-open-directory ,object))

(define-fsedit-operation com-close-directory ((pathname 'pathname))
  (let ((fsf (find-fsf-for-pathname *application* pathname)))
    (when fsf
      (setf (fsf-show-directory-contents fsf) nil))))

(define-presentation-to-operation-translator com-close-directory
   (pathname :gesture :left
	     :tester ((object)
		      (let ((fsf (find-fsf-for-pathname *application* object)))
			(and fsf (fsf-show-directory-contents fsf)))))
   (object)
   `(com-close-directory ,object))