;;; -*- Base: 10; Package: CLIM-DEMO; Mode: LISP; Syntax: Common-Lisp -*-
;;;
;;; Copyright (c) 1990 by International Lisp Associates.  All rights reserved. 
;;;

(in-package "CLIM-DEMO")

"Copyright (c) 1991 by International Lisp Associates.  All rights reserved."

(defclass color-feedback-pane (pane-background-mixin layout-mixin leaf-pane) () )

;;; Why do we have to do this?  Isn't there some mixin we can provide for
;;; this behavior?
(defmethod compose-space ((pane color-feedback-pane))
  (ws::make-space-req :hs 100 :vs 100))

;;; --- The missing link here is figuring out how to initialize the state of various
;;; --- gadgets from init keywords to the frame.  The problem is that the gadgets
;;; --- don't actually get made until the frame is grafted, and that's long after
;;; --- initialize-instance time.  So we do it here, on the sheet-grafted method for
;;; --- color-feedback-pane.
(defmethod ws::sheet-grafted ((pane color-feedback-pane))
  (let ((frame (pane-frame pane)))
    ;; make sure that this pane's background matches the current slot in the frame
    (update-feedback-background frame)
    ;; make sure that the state of the radio box matches the slot in the frame
    (update-toggles frame)
    ;; make sure that the states of the sliders match the slots in the frame
    (update-sliders frame)))

;;; --- This should already be done by the substrate, but until then...
(pushnew 'ws::hline-pane (getf ws::*abstract-pane-implementations* 
			       'ws::horizontal-divider))

(define-application-frame foo-color-editor ()
  ((feedback-pane )
   (current-value-list :initform '(0 0 0))
   (current-mode :initform :rgb)
   (gadgets :initform nil))
  (:pane
    (with-frame (frame)
      (with-frame-slots (feedback-pane gadgets)
	;; I use this rather than having a million slots, one for each gadget.
	;; I don't know what the real "right" solution is.
	(macrolet ((gadget (thing)
		     `(first (push ,thing gadgets))))
	  (vertically ()
              (make-pane 'ws::scroller-pane
		    :vs 40
		    :hs+ +fill+ :vs+ +fill+
		    :hs- +fill+ :vs- +fill+
		    :contents (make-pane 
				 'ci::new-extended-stream-pane
				 :display-function '(display-palette))
		    :scroll-bars nil)
	    (realize-pane 'ws::horizontal-divider)
	    (spacing (:space 30 :halign :center :valign :center)
	      ;; need some pane class that has background-repainter-mixin mixed in.
	      (bordering (:thickness 2)
		(setf feedback-pane
		  (make-pane 'color-feedback-pane :background +black+))))
	    (realize-pane 'ws::horizontal-divider)
	    (ws::with-radio-box
	     :fill
	     (gadget
	      (ws::radio-box-current-selection
	       (realize-pane 'ws::toggle-button
			     :label "RGB"
			     :id :rgb
			     :client frame)))
	     :fill
	     (gadget
	      (realize-pane 'ws::toggle-button
			    :label "IHS"
			    :id :ihs
			    :client frame))
	     :fill)
	    (realize-pane 'ws::horizontal-divider)
	      ;; plus some space between the slider bars
	    (let ((separation 10))
	      (vertically ()
		separation
		(horizontally ()
		  separation
		  (make-pane 'ws:label-pane :text "R/I")
		  5
		  (gadget
		   (realize-pane 'ws::slider
				 :orientation :horizontal 
				 :id 0 :client frame))
		  separation)
		separation
		(horizontally ()
		  separation
		  (make-pane 'ws:label-pane :text "G/H")
		  5
		  (gadget
		   (realize-pane 'ws::slider
				 :orientation :horizontal 
				 :id 1 :client frame))
		  separation)
		separation
		(horizontally ()
		  separation
		  (make-pane 'ws:label-pane :text "B/S")
		  5
		  (gadget
		   (realize-pane 'ws::slider
				 :orientation :horizontal 
				 :id 2 :client frame))
		  separation)
		separation)))))))
  (:command-definer T)

  ;; give it a top-level just so we can wait-until-done
  (:top-level (clim-top-level) )
  )

(defmethod initialize-instance :after ((frame foo-color-editor) &key color mode)
  ;; perhaps I should have specified :mode as the init keyword for current-mode.
  (when mode
    (setf (slot-value frame 'current-mode) mode))
  (when color
    (setf (foo-color-editor-color frame) color)))

;;; Here we change the currently edited color.  Note the horrible kludge
;;; of seeing if we have a current feedback pane.  (That's how we tell if we're
;;; currently adopted.  I dunno how this should work in the world in which the frame
;;; *was* adopted but has since been disowned on its way to being adopted by another
;;; frame manager.  The slots aren't going to get unbound.   What's the "right" solution?)
(defmethod (setf foo-color-editor-color) (color (frame foo-color-editor))
  (let ((mode (slot-value frame 'current-mode))
	(feedback-pane (and (frame-manager frame)
			    (slot-value frame 'feedback-pane))))
    (when feedback-pane
      (setf (ws::pane-background feedback-pane) color))
    (setf (slot-value frame 'current-value-list)
	  (multiple-value-list
	    (funcall
	      (ecase mode
		(:rgb #'color-rgb)
		(:ihs #'color-ihs))
	      color)))
    (when feedback-pane
      (update-sliders frame)
      (repaint-sheet feedback-pane +everywhere+))))

(defmethod foo-color-editor-color ((frame foo-color-editor))
  (with-slots (current-value-list current-mode) frame
    (apply 
      (ecase current-mode
	(:ihs #'make-color-ihs)
	(:rgb #'make-color-rgb))
      current-value-list)))

(defmethod update-feedback-background ((frame foo-color-editor))
  ;; cons out the wazoo...
  (setf (foo-color-editor-color frame) (foo-color-editor-color frame)))

(define-presentation-type color-sample () )

;;; Pick a better size, etc., for the samples.
(defmethod display-palette ((frame foo-color-editor) stream)
  (let ((samples (list (list +red+ +green+ +blue+)
		       (list +magenta+ +yellow+ +cyan+))))
    (formatting-table (stream :inter-column-spacing 20 :inter-row-spacing 10)
      (dolist (row samples)
	(formatting-row (stream)
	  (dolist (sample row)
	    (formatting-cell (stream :align-x ':center)
	      (with-output-as-presentation (:stream stream
					    :type 'color-sample
					    :object sample)
		(draw-rectangle* stream 0 0 20 10 :ink sample))))))))
  (ws::viewport-scroll (ws::new-pane-viewport stream) :y 0))

(define-foo-color-editor-command (com-set-color :command-name t)
    ((color 'color-sample :translator-gesture :select))
   (with-frame (frame)
     (setf (foo-color-editor-color frame) color)))

;;; Here's how we find the gadgets we're interested in.  I almost think that
;;; such a registry would be a useful addition to CLIM.
(defmethod find-gadget-id ((frame foo-color-editor) id)
  (find id (slot-value frame 'gadgets) :key #'ws::gadget-id))

(defmethod update-sliders ((frame foo-color-editor))
  (let ((slider-0 (find-gadget-id frame 0))
	(slider-1 (find-gadget-id frame 1))
	(slider-2 (find-gadget-id frame 2)))
    (multiple-value-bind (zero one two)
	(values-list (slot-value frame 'current-value-list))
      ;; --- the IHS to RGB conversion code can produce numbers
      ;; outside the range 0.0 to 1.0
      (clim-utils::minf zero 1.0)
      (clim-utils::maxf zero 0.0)
      (clim-utils::minf one 1.0)
      (clim-utils::maxf one 0.0)
      (clim-utils::minf two 1.0)
      (clim-utils::maxf two 0.0)
      (unless (= (ws::gadget-value slider-0) zero)
	(ws::set-gadget-value slider-0 zero :call-callbacks nil))
      (unless (= (ws::gadget-value slider-1) one)
	(ws::set-gadget-value slider-1 one :call-callbacks nil))
      (unless (= (ws::gadget-value slider-2) two)
	(ws::set-gadget-value slider-2 two :call-callbacks nil)))))

;;; Maybe we can figure out some way of packaging up this kind of behavior, too.
;;; It seems overly complicated, and SWM's approach of saying something like
;;; (accept '(member :ihs :rgb) :default :ihs) seems much more clean.
(defmethod update-toggles ((frame foo-color-editor))
  (let* ((ihs (find-gadget-id frame :ihs))
	 (ihs-p (ws::gadget-value ihs))
	 (rgb (find-gadget-id frame :rgb))
	 (rgb-p (ws::gadget-value rgb))
	 (current-mode (slot-value frame 'current-mode)))
    (ecase current-mode
      (:ihs
	(when rgb-p
	  (ws::set-gadget-value rgb nil))
	(unless ihs-p
	  (ws::set-gadget-value ihs t)))
      (:rgb
	(when ihs-p
	  (ws::set-gadget-value ihs nil))
	(unless rgb-p
	  (ws::set-gadget-value rgb t))))))

;;; Now, the chase.  When we change from :IHS to :RGB or vice versa, recompute
;;; the displayed color.
(defmethod ws::value-change-callback ((gadget ws::toggle-button)
				      (client foo-color-editor)
				      id
				      new-value)
  (when new-value
    (let ((color (foo-color-editor-color client)))
      (setf (slot-value client 'current-mode) id)
      (setf (foo-color-editor-color client) color))))

;;; When we change (or drag) a slider, update the displayed color.
(defmethod ws::value-change-callback ((gadget ws::slider)
				      (client foo-color-editor)
				      id
				      new-value)
  (setf (elt (slot-value client 'current-value-list) id) new-value)
  (update-feedback-background client))

;;; When we change (or drag) a slider, update the displayed color.
(defmethod ws::drag-callback ((gadget ws::slider)
			      (client foo-color-editor)
			      id
			      new-value)
  (setf (elt (slot-value client 'current-value-list) id) new-value)
  (update-feedback-background client))

;;; The interface routine.  Try (edit-color :color +green+ :mode :ihs)
(defun edit-color (&key color mode (frame-manager (find-frame-manager)))
  ;; Jeez, I'd like to use launch-frame, but its syntax is wrong
  (let ((frame (make-frame 'foo-color-editor
			   :frame-manager frame-manager
			   :color color
			   :mode mode
			   :title "Color Editor")))
    (launch-frame frame :wait-until-done nil)
    ;; I suppose there's the tiniest of timing windows here...
    (values (foo-color-editor-color frame) frame)))
