;;; -*- Mode: LISP; Syntax: Common-lisp; Package: CLIM-DEMO; Base: 10 -*-

(in-package "CLIM-DEMO")

(defvar *demos* (make-hash-table :test #'equal))

(defclass clim-demo ()
    ((name :reader clim-demo-name
	   :initarg :name)
     (frame-class :reader clim-demo-frame-class
		  :initarg :frame-class)
     (parameter-specs :reader clim-demo-parameter-specs
		      :initarg :parameter-specs)))

(defmethod run-demo ((demo symbol) &rest parameters &key &allow-other-keys)
  (declare (dynamic-extent parameters))
  (apply #'run-demo (string demo) parameters))

(defmethod run-demo ((demo string) &rest parameters &key &allow-other-keys)
  (declare (dynamic-extent parameters))
  (let ((demo-obj (gethash demo *demos*)))
    (unless demo-obj
      (error "The demo named ~S was not found." demo))
    (apply #'run-demo demo-obj parameters)))

(defmethod run-demo ((demo clim-demo) &rest parameters
		     &key &allow-other-keys)
  (with-slots (frame-class name) demo
    (apply #'launch-frame frame-class :title name parameters)))

(defun define-demo-internal (name frame-class parameter-specs)
  (check-type name string)
  (check-type frame-class (or symbol class))
  (check-type parameter-specs list)
  (validate-parameter-specs name parameter-specs)
  (setf (gethash name *demos*)
	(make-instance 'clim-demo
		       :name name
		       :frame-class (find-class frame-class)
		       :parameter-specs (mapcar #'copy-list parameter-specs))))

(defun validate-parameter-specs (demo-name parameter-specs)
  (dolist (spec parameter-specs)
    (destructuring-bind (name type &key (prompt (string-capitalize (string name)))) spec
      (check-type name symbol)
      (check-type prompt string)
      (unless (presentation-type-specifier-p type)
	(warn "Parameter ~S in demo ~S uses unknown type ~S."
	      name demo-name type))
      (loop (unless spec
	      (return))
	    (let ((option-name (pop spec)))
	      (unless (member option-name '(:default :prompt))
		(warn "Unknown parameter spec option ~S in demo ~S"
		      option-name demo-name))
	      (unless spec
		(warn "Odd number of option keyword argumens in parameter specs for demo ~S."
		      demo-name))
	      (pop spec)))))
  parameter-specs)

(defmacro define-demo (name frame-class superclasses slots &body options)
  (setq name (string name))
  (let ((parameter-specs (assoc :parameters options)))
    (when parameter-specs
      (setq options (remove parameter-specs options)))
    `(progn (define-application-frame ,frame-class (,@superclasses demo-frame-mixin)
	      ,slots ,@options)
	    (define-demo-internal ,name ',frame-class ',parameter-specs))))

(defun exit-demo () (throw 'command-executed nil))

(defun make-demo-menu ()
  (let ((menu-items nil))
    (labels ((collect-menu-items (key value)
	       (declare (ignore key))
	       (push `(,(clim-demo-name value) :value ,value) menu-items)))
      (maphash #'collect-menu-items *demos*))
    (setq menu-items (sort menu-items #'string-lessp :key #'car))
    menu-items))

(define-application-frame clim-demos () ((message-pane))
  (:pane (with-frame-slots (message-pane)
	   (make-clim-pane (message-pane :hs 400 :vs 200 :scroll-bars nil))))
  (:menu-group demos-menu-group)
  (:top-level (clim-top-level)))

(define-clim-demos-command com-exit-demos ()
   (throw 'exit-all-demos nil))

(define-clim-demos-command com-run-demo ((demo 'clim-demo))
   ;; Later, worry about setting parameters for the demo.
   (run-demo demo))

(defmethod run-frame-top-level :around ((frame clim-demos))
  (update-menu-group frame)
  (catch 'exit-all-demos
    (call-next-method)))

(defmethod update-menu-group ((frame clim-demos))
  (with-slots (frame-manager) frame
    (let ((proto (ws:find-menu-group-prototype 'demos-menu-group)))
      (dolist (menu-item (make-demo-menu))
	(let ((name (first menu-item))
	      (demo (getf (rest menu-item) :value)))
	  (add-menu-group-entry proto name :command (build-command 'com-run-demo demo)
				:after nil :if-exists :supersede)))
      (setf (ws::frame-menu-group frame)
	    (ws:instantiate-menu-group-from-prototype 'demos-menu-group))
      (let ((fm frame-manager)) ;; DISOWN-FRAME side effects on FRAME-MANAGER...
	(when fm
	  (disown-frame fm frame)
	  (adopt-frame fm frame))))))

(define-menu-group demos-menu-group 
  (("Exit" :command (build-command 'com-exit-demos))))

;;; --- Not yet updated

(defun start-demo ()
  (labels ((demo-menu-drawer (stream type &rest args)
	     (with-text-style ('(:serif :roman :very-large) stream)
	       (apply #'clim::draw-standard-menu stream type args)))
	   (demo-menu-choose (list associated-window)
	     (clim::with-menu (menu associated-window)
	       (setf (clim::window-label menu)
		     '("Clim Demonstrations" :text-style (:fix :bold :normal)))
	       (menu-choose-from-drawer
		 menu 'foo
		 ;;#'clim::draw-standard-menu
		 #'(lambda (stream type)
		     (demo-menu-drawer stream type list nil))
		 :unique-id 'demo-menu
		 :unique-id-test #'eql
		 :cache-value *demos*
		 :cache-value-test #'equal
		 :cache T))))
    (catch 'exit-demo
      (loop
	(let* ((demo-name (demo-menu-choose (nreverse (map 'list #'car *demos*)) *demo-root*))
	       (demo-form (cdr (assoc demo-name *demos* :test #'string-equal))))
	  (when demo-form
	    ;;--- kludge to keep from blowout when demo not yet loaede
	    (if (fboundp (first demo-form))
		(eval demo-form)
		(beep *demo-root*))))))))
