;;; -*- Mode: LISP; Syntax: Common-lisp; Package: CLIM-DEMO; Base: 10; Lowercase: Yes -*-

(in-package "CLIM-DEMO")

"Copyright (c) 1989, International Lisp Associates.  All rights reserved."

;;; A simple gate-level CAD program.

;;; First we define the application-specific data structures, the components and their
;;; connection terminals.  The only part of the user interface specified at this
;;; level is the displayed representation (appearance) of the entities when they
;;; are drawn on the screen.

(defclass basic-thing ()
     ((x :initarg :x :accessor thing-x)
      (y :initarg :y :accessor thing-y)
      (size :initarg :size :accessor thing-size)))

(defmethod thing-position ((thing basic-thing))
  (with-slots (x y) thing
    (values x y)))

(defmethod move ((thing basic-thing) new-x new-y)
  (with-slots (x y) thing
    (setf x new-x y new-y)))

(defmethod ci::presentation-p ((thing basic-thing)) t)

(defmethod ci::presentation-single-box ((thing basic-thing)) nil)

(defmethod ci::displayed-output-record-element-p ((thing basic-thing)) t)

(defmethod presentation-object ((thing basic-thing))
  thing)

;;;****************************************************************

;;; This is *color-red* for color systems, otherwise *flipping-ink*.
(defvar *highlight-ink*)

(defvar *component-size* 
	#+Imach 18
	#-Imach 25
 "Default display size of a component.")

;;; A connection belongs to a component.  The component may have any number of
;;; input and output connections, although currently only one output is supported.
(defclass connection
	  (basic-thing)
     ((component :initform nil :initarg :component :accessor connection-component)
      (other-connections :initform nil :accessor connection-other-connections)
      ;; Give wire router some hints
      (early :initarg :early :reader connection-early-p)
      (wire-offset :initarg :wire-offset :reader connection-wire-offset))
  (:default-initargs :size 5 :early nil :wire-offset 20))

(defmethod draw-self ((connection connection) stream &key (ink w:+foreground+))
  (with-slots (x y size) connection
    (draw-circle* stream x y size
		  ;; compute filled from the value,
		  ;; white for on, black for off
		  :filled (not (connection-value connection))	;required method
		  :ink ink)))

(defmethod ci::highlight-output-record-1 ((connection connection) stream state)
  (with-slots (x y size) connection

    (if (eq *highlight-ink* w:+flipping-ink+)
	(draw-circle* stream x y (1+ size)
		      :filled t
		      :ink *highlight-ink*)
	(ecase state
	  (:highlight
	    (draw-self connection stream :ink *highlight-ink*))
	  (:unhighlight
	    (draw-self connection stream))))))

(defmethod thing-edges ((conn connection))
  (let ((fudge 2))
    (with-slots (x y size) conn
      ;; size is a radius, but make the box larger so that connections
      ;; are easier to point to
      (values (- x size fudge) (- y size fudge)
	      (+ x size fudge) (+ y size fudge)))))

(defun connect (output input)
  ;; Inputs can have only one incoming connection, so remove this input
  ;; from its former incoming connection's outputs list.
  (with-slots (other-connections) input
    (when other-connections
      (setf (connection-other-connections (first other-connections))
	    (remove input (connection-other-connections (first other-connections)))))
    (setf other-connections (list output)))
  ;; Add this input to the list of other-connections of the output.
  (push input (connection-other-connections output)))

;;; Sort of hairy, but it always computes a connection's position relative to
;;; the current position of its owning component.  So, when the component is moved
;;; the new connection position is reflected immediately.
(defun compute-connection-position (connection)
  (with-slots (component) connection
    ;;--- CLOS bug, can't use COMPONENT in subsequent WITH-SLOTS
    (let ((foo component))
      (with-slots (x y inputs outputs) foo
	;; We don't deal with multiple outputs
	(assert (<= (length outputs) 1) nil
		"Don't know how to handle multiple outputs")
	(cond ((member connection outputs)
	       ;; The output has a constant location
	       (return-from compute-connection-position
		 (values (+ x *component-size*) y)))
	      ((member connection inputs)
	       ;; Divide up the available space (the height of the
	       ;; component) among the inputs, then figure out which input we are
	       ;; interested in and therefore how far down it is.
	       (let ((spacing (/ (* *component-size* 2) (1+ (length inputs))))
		     ;; Start at the top of the component
		     (y-pos (- y *component-size*)))
		 (let ((index (position connection inputs)))
		   (return-from compute-connection-position
		     (values x (+ y-pos (* spacing (1+ index))))))))
	      (t (error "Connection ~S is not among the connections of its component ~S"
			connection component)))))))

(defclass input
	  (connection)
     ())

(defmethod presentation-type ((input input))
  'input)

;;; An input connection's logic value is determined by the value of the output
;;; connection that is feeding it.
(defmethod connection-value ((conn input))
  (with-slots (other-connections) conn
    (assert (<= (length other-connections) 1)
	    nil
	    "Don't know how to handle multiple inputs to one connection.")
    ;; Floating inputs default to "off"
    (when other-connections
      (connection-value (first other-connections)))))

(defclass output
	  (connection)
     ())

(defmethod presentation-type ((output output))
  'output)

;;; An output connection's logic value is computed from the inputs by the
;;; component.
(defmethod connection-value ((conn output))
  (connection-value (slot-value conn 'component)))

;;;****************************************************************

(defclass component
	  (basic-thing)
     ((inputs :initform nil)
      (outputs :initform nil))
  (:default-initargs :size *component-size*))

;;; Fill in the inputs and outputs from the init args.
(defmethod initialize-instance :after ((component component) &key (n-inputs 1) (n-outputs 1) &allow-other-keys)
  ;; This early-p stuff is all a big kludge to get slightly better wire routing
  (let ((early-p nil)
	(offset-1 20)
	(offset-2 30))
    (flet ((make-one (type component)
	     (prog1 (make-instance type :component component
				   :early early-p
				   :wire-offset offset-1)
		    (rotatef offset-1 offset-2)
		    (setq early-p (not early-p)))))
      (dotimes (n n-inputs)
	#+Genera-Release-8 (declare (ignore n))
	(push (make-one 'input component) (slot-value component 'inputs)))
      (setq early-p nil)
      (setq offset-1 20 offset-2 30)
      (dotimes (n n-outputs)
	#+Genera-Release-8 (declare (ignore n))
	(push (make-one 'output component) (slot-value component 'outputs)))))
  ;; Place the newly-created connections relative to the component
  (move component (thing-x component) (thing-y component)))

(defun all-connections (component)
  (with-slots (inputs outputs) component
    (append inputs outputs)))

;;; When a component is added to the database, add its connections
(defmethod add-new-object :after (cd (new-object component))
  (dolist (conn (all-connections new-object))
    (add-new-object cd conn)))

(defmethod move :after ((component component) new-x new-y)
	   (declare (ignore new-x new-y))
	   (dolist (conn (all-connections component))
	     (multiple-value-bind (x y)
		 ;; place the connections relative to their component
		 (compute-connection-position conn)
	       (move conn x y))))

;;; Call this on a component to display the whole thing
(defmethod draw-self ((component component) stream &key (ink w:+foreground+))
  (draw-body component stream :ink ink)
  (draw-connections component stream :ink ink)
  (draw-wires component stream :ink ink))

;;; Default body is a half-circle
(defmethod draw-body ((comp component) stream &key (ink w:+foreground+))
  (with-slots (x y)
	      comp
    (draw-circle* stream x y *component-size*
		  ;; Elegant, ain't we?
		  ;; Why the hell can't genera draw half circles in :alu :flip?
		  ;; Note the superb attention to detail in the selection
		  ;; of the *ONLY* magic numbers that appear to work.
		  ;;
		  :start-angle (+ #+genera .00001 (* pi 3/2))
		  :end-angle (+ #+genera .000001 (* pi 1/2))
		  :ink ink)))

;;; make a component behave as an output record
(defmethod thing-edges ((comp component))
  (with-slots (x y) comp
    (values x (- y *component-size*)
	    (+ x *component-size*) (+ y *component-size*))))

(defmethod presentation-type ((comp component))
  'component)

(defmethod ci::highlight-output-record-1 ((comp component) stream state)
  (if (eq *highlight-ink* w:+flipping-ink+)
      (multiple-value-bind (cl ct cr cb) (thing-edges comp)
	(draw-rectangle* stream cl ct cr cb :ink *highlight-ink*))
      (ecase state
	(:highlight (draw-body comp stream :ink *highlight-ink*))
	(:unhighlight (draw-body comp stream :ink w:+foreground+)))))

(defmethod draw-connections ((comp component) stream &key (ink w:+foreground+))
  (dolist (conn (all-connections comp))
    (draw-self conn stream :ink ink)))

(defvar *draw-junctions* t)

;;; This guy is responsible for wire layout.  It doesn't have any global
;;; knowledge, so it can't avoid running over things or draw little humps
;;; where unconnected wires cross.
(defmethod draw-wires ((comp component) stream &key (ink w:+foreground+))
  (with-slots (inputs outputs) comp
    (labels ((round-val (val &optional (direction :down))
	       (let ((chunk-size 20))
		 (ecase direction
		   (:up (incf val chunk-size))
		   (:down (decf val chunk-size) ))
		 (* chunk-size (round val chunk-size))))
	     (draw-junction (x y)
	       (when *draw-junctions*
		 (draw-rectangle* stream (- x 2) (- y 2) (+ x 3) (+ y 3) :ink ink)))
	     ;; Various routing helper functions.
	     ;; This one forks near X2, rounding down to the next CHUNK-SIZE
	     ;; X coord.
	     (draw-path-fork-late (x1 y1 x2 y2)
	       (draw-line* stream x1 y1 (round-val x2) y1 :ink ink)
	       (draw-junction (round-val x2) y1)
	       (draw-line* stream (round-val x2) y1 (round-val x2) y2 :ink ink)
	       (draw-junction (round-val x2) y2)
	       (draw-line* stream (round-val x2) y2 x2 y2 :ink ink))
	     ;; This one forks near X1, rouding up to the next CHUNK-SIZE
	     ;; X coord.
	     (draw-path-fork-early (x1 y1 x2 y2)
	       (draw-line* stream x1 y1 (round-val x1 :up) y1 :ink ink)
	       (draw-junction (round-val x1 :up) y1)
	       (draw-line* stream (round-val x1 :up) y1 (round-val x1 :up) y2 :ink ink)
	       (draw-junction (round-val x1 :up) y2)
	       (draw-line* stream (round-val x1 :up) y2 x2 y2 :ink ink))

	     #+Ignore ; --- This isn't used on 7/26/90 --- Doughty
	     ;; This one forks near X2, splitting off OFFSET units away.
	     (draw-path-fork-late-offset (x1 y1 x2 y2 offset)
	       (let ((x-mid (- x2 offset)))
		 (draw-line* stream x1 y1 x-mid y1 :ink ink)
		 (draw-line* stream x-mid y1 x-mid y2 :ink ink)
		 (draw-line* stream x-mid y2 x2 y2 :ink ink)))

	     #+Ignore ; --- This isn't used on 7/26/90 --- Doughty
	     ;; Path policy functions.  The one currently named DRAW-WIRE wins.
	     ;; This one forks late, extracting the offset from the connection.
	     ;; (see the code that creates connections)
	     (draw-wire-conn-offset (connection direction)
	       (dolist (other-conn (connection-other-connections connection))
		 (let ((conn connection))
		   ;; Always draw line from :FROM to :TO
		   (ecase direction
		     (:to (rotatef conn other-conn))
		     (:from ))
		   (multiple-value-bind (x y) (thing-position conn)
		     (multiple-value-bind (ox oy) (thing-position other-conn)
		       (draw-path-fork-late-offset x y ox oy (connection-wire-offset other-conn)))))))
	     ;; This one forks early or late depending on a value stored in the connection
	     ;; at make-instance time.
	     (draw-wire #+ignore -early-late (connection direction)
	       (dolist (other-conn (connection-other-connections connection))
		 (let ((conn connection))
		   ;; Always draw line from :FROM to :TO
		   (ecase direction
		     (:to (rotatef conn other-conn))
		     (:from ))
		   (multiple-value-bind (x y) (thing-position conn)
		     (multiple-value-bind (ox oy) (thing-position other-conn)
		       (if (connection-early-p other-conn)
			   (draw-path-fork-early x y ox oy)
			   (draw-path-fork-late x y ox oy)))))))

	     #+Ignore ; --- This isn't used on 7/26/90 --- Doughty
	     ;; This one simply forks early for all connections.
	     (draw-wire-early (connection direction)
	       (multiple-value-bind (x y) (thing-position connection)
		 (dolist (oc (connection-other-connections connection))
		   (multiple-value-bind (ox oy)
		       (thing-position oc)
		     ;;(draw-line x y ox oy :stream stream :ink ink)
		     ;; The draw-path guys need to know left-to-right ordering
		     ;; to do their jobs.
		     (ecase direction
		       (:to (draw-path-fork-early ox oy x y))
		       (:from (draw-path-fork-early x y ox oy)))))))

	     #+Ignore ; --- This isn't used on 7/26/90 --- Doughty
	     ;; This one simply forks late for all connections.
	     (draw-wire-late (connection direction)
	       (multiple-value-bind (x y) (thing-position connection)
		 (dolist (oc (connection-other-connections connection))
		   (multiple-value-bind (ox oy)
		       (thing-position oc)
		     ;;(draw-line x y ox oy :stream stream :ink ink)
		     ;; The draw-path guys need to know left-to-right ordering
		     ;; to do their jobs.
		     (ecase direction
		       (:to (draw-path-fork-late ox oy x y))
		       (:from (draw-path-fork-late x y ox oy))))))))
      (dolist (i inputs)
	(draw-wire i :to))
      (dolist (o outputs)
	(draw-wire o :from)))))

;;; Various components

(defclass and-gate
	  (component)
     ()
  (:default-initargs :n-inputs 2))

(defmethod connection-value ((ag and-gate))
  (every #'connection-value (slot-value ag 'inputs)))

(defmethod equation-part ((ag and-gate))
  (let ((equation nil))
    (dolist (in (slot-value ag 'inputs))
      (let ((out (first (connection-other-connections in))))
	(when out
	  (unless (null equation)
	    (push "&" equation))
	  (push (equation-part (connection-component out))
		equation))))
    equation))

(defclass or-gate
	  (component)
     ()
  (:default-initargs :n-inputs 2))

(defmethod connection-value ((og or-gate))
  (some #'connection-value (slot-value og 'inputs)))

(defmethod equation-part ((og or-gate))
  (let ((equation nil))
    (dolist (in (slot-value og 'inputs))
      (let ((out (first (connection-other-connections in))))
	(when out
	  (unless (null equation)
	    (push "|" equation))
	  (push (equation-part (connection-component out))
		equation))))
    equation))

;;; Default body is an almost-half-circle, so we get a different look
;;; for OR gates.  Looks marginal and XOR's funny under Genera.
(defmethod draw-body ((comp or-gate) stream &key (ink w:+flipping-ink+))
  (with-slots (x y)
	      comp
    (draw-circle* stream x y *component-size*
		  ;; Elegant, ain't we?
		  ;; Why the hell can't genera draw half circles in :alu :flip?
		  ;; Note the superb attention to detail in the selection
		  ;; of the *ONLY* magic numbers that appear to work.
		  ;;
		  :start-angle (+ #+genera .00001 (* pi 3/2) -0.3)
		  :end-angle (+ #+genera .000001 (* pi 1/2) 0.3)
		  :ink ink)))

(defclass logic-constant
	  (component)
     ((name :initform nil)
      (value :initarg :value))
  (:default-initargs :n-inputs 0))

(defvar *name-code* (1- (char-code #\A)))

(defmethod initialize-instance :after ((lc logic-constant) &key &allow-other-keys)
	   (when *name-code*
	     (setf (slot-value lc 'name)
		   (string (code-char (incf *name-code*))))))

(defmethod connection-value ((component logic-constant))
  (slot-value component 'value))

(defmethod equation-part ((lc logic-constant))
  (slot-value lc 'name))

;;; Draw the logic "variable" name next to the component, or erase it.
(defmethod draw-self :after ((lc logic-constant) stream &key (ink w:+flipping-ink+))
  (with-slots (x y name) lc
    (when name
      (cond ((eq ink +background+)
	     (draw-rectangle* stream (- x 10) (- y 10) x (+ y 20)
			      :ink ink))
	    (t (draw-text* stream name (- x 10) y :ink ink))
	    ;; We now have draw-text
	    #+ignore
	    (t (stream-set-cursor-position* stream (- x 10) (- y 10))
	       (write-string name stream))))))

(defclass logic-one
	  (logic-constant)
    ()
  (:default-initargs :value t))

(defclass logic-zero
	  (logic-constant)
     ()
  (:default-initargs :value nil))

(defvar *component-types* '(("And Gate" :value and-gate)
			    ("Or Gate" :value or-gate)
			    ("Logic One" :value logic-one)
			    ("Logic Zero" :value logic-zero)))

;
;;; ****************************************************************

;;; The User Interface

;;; First define a "application" that manages the application's state variables
;;; and defines a high-level division of screen real estate.
(define-application-frame cad-demo ()
  ((object-list :initform nil)
   (display-pane))
  (:settings :message-pane t)
  (:pane
    (ws:with-frame-slots (display-pane)
     (ws:vertically ()
      (ws:make-pane 'ws::spacer-pane
	    :contents
	    (make-clim-pane (display-pane
			      :hs 400 :vs #+Imach 300 #-Imach 380
			      :scroll-bars nil)
			    :default-text-style '(:fix :bold :very-large)
			    :display-function '(display-stuff)
			    :record-p nil)))))
  (:menu-group cad-demo-menu-group)
  (:top-level (clim-top-level))
  )

(defun display-pt-prompt (presentation-type frame)
  (let ((name (presentation-type-name presentation-type)))
    (case name
      (input (notify-user frame "Click on an input (on the left side of a gate)."))
      (output (notify-user frame "Click on an output (on the right side of a gate)."))
      (component (notify-user frame "Click on a gate to be moved."))
      (command-name (notify-user frame "Click on a gate, a connection, or a menu item."))
      (cad-position (notify-user frame "Click on a position to place the gate."))
      (otherwise (notify-user frame "")))))

(defun cad-demo-arg-parser (command-table stream)
  (with-input-context (`(command :command-table ,command-table))
		      (command type)
       ;; where is this supposed to get FRAME from?
       (let ((frame *frame*))
	 (flet ((cad-demo-parser (stream presentation-type &key &allow-other-keys)
		  (display-pt-prompt presentation-type frame)
		  (with-input-context (presentation-type)
				      (object type)
		       (read-token stream :click-only T)
		     (t (values object type))))
		(cad-demo-delimiter (stream args-to-go)
		  (declare (ignore stream args-to-go))))
	   (multiple-value-prog1
	     (invoke-command-parser-and-collect
	       command-table #'cad-demo-parser #'cad-demo-delimiter stream)
	     (notify-user frame ""))))
     (t (values command type))))

(defmethod read-frame-command :around ((frame cad-demo) stream)
  (declare (ignore stream))
  (flet ((kludge (partial-command command-table stream start-location)
	   (declare (ignore command-table start-location))
	   (let ((*standard-input* stream))
	     (frame-read-remaining-arguments-for-partial-command frame partial-command))))
    (let ((*command-parser* #'cad-demo-arg-parser)
	  (*partial-command-parser*  #'kludge))
      (call-next-method))))

(defmethod frame-read-remaining-arguments-for-partial-command ((frame cad-demo) command)
  (let ((command-table *command-table*))
    (flet ((menu-parser (stream presentation-type &key &allow-other-keys)
	     (let ((arg-p command)
		   (arg (pop command)))
	       (cond ((and arg-p (not (ci::unsupplied-argument-p arg)))
		      (return-from menu-parser (values arg presentation-type)))
		     (t (display-pt-prompt presentation-type frame)
			(with-input-context (presentation-type :override T)
					    (object type)
			     (read-token stream :click-only T)
			   (t (values object type)))))))
	   (menu-delimiter (stream args-to-go)
	     (declare (ignore stream args-to-go))))
      (multiple-value-prog1
	(invoke-command-parser-and-collect
	  command-table #'menu-parser #'menu-delimiter *standard-input*)))))

(defmethod add-new-object ((cd cad-demo) new-object)
  (push new-object (slot-value cd 'object-list)))

(defmethod frame-repaint-pane ((cd cad-demo) pane
				     &optional bounding-rectangle x-offset y-offset)
  (declare (ignore bounding-rectangle x-offset y-offset))
  (with-slots (display-pane) cd
    (if (eq pane display-pane)
	(display-stuff cd pane)
	(call-next-method))))

(defmethod frame-find-presentation ((cd cad-demo) pane type x y continuation)
  (declare (ignore type))					;it's encoded in continuation
  (with-slots (object-list display-pane) cd
    (cond ((eq pane display-pane)
	   (dolist (object object-list)
	     (multiple-value-bind (left top right bottom) (thing-edges object)
	       (when (and (<= left x right)
			  (<= top y bottom)
			  (funcall continuation object))
		 (return object)))))
	  (t (call-next-method)))))

;;; The display function for the application-controlled output pane.  The
;;; application substrate automatically runs this.
(defmethod display-stuff ((application cad-demo) stream)
  ;(window-clear stream)
  ;(scroll-home stream)
  (using-clim-medium (medium stream)
     (dolist (object (slot-value application 'object-list))
       (draw-self object medium))))

;;; Utility routines

;;; Should already exist on the POINT datatype
(define-presentation-type cad-position () )

;;; Only over blank areas.
(define-presentation-translator select-position (:blank-area cad-position)
				   (x y)
  (cons x y))

;;; Now define the commands or commands of the application.

(defvar *component-prototypes* nil)

(defun make-component-prototypes ()
  (setq *component-prototypes* nil)
  ;; inhibit giving names to logic vars
  (let ((*name-code* nil))
    (dolist (ct (map 'list 'third *component-types*))
      (push (make-instance ct :x 0 :y 0) *component-prototypes*))))

;(make-component-prototypes)

#+Ignore
;;; Return the class name of the selected component
(defun select-component (parent)
  (labels ((draw-icon-menu (menu presentation-type)
	     (formatting-table (menu :inter-row-spacing 5)
	       (dolist (icon *component-prototypes*)
		 (with-output-as-presentation (:stream menu
					       :object icon
					       :type presentation-type)
		   (formatting-row (menu)
		     (formatting-cell (menu)
		       (with-user-coordinates (menu)
			 (draw-self icon menu)
			 (multiple-value-bind (x y)
			     (stream-cursor-position* menu)
			   (stream-set-cursor-position*
			     menu
			     ;; fudge for the fact that the presentation encloses the
			     ;; half of the circle that's invisible
			     (- x 20) (+ y *component-size*)))
			 (write-string (string (class-name (class-of icon))) menu)
			 ))))))
	     nil))
    (ci::with-menu (menu parent)
      (let ((component (menu-choose-from-drawer
			 menu 'ci::menu-item #'draw-icon-menu)))
	(class-name (class-of component))))))

;;; Try to start with a reasonable drawing for the demo.
(defun make-drawing (cd)
  (setq *name-code* (1- (char-code #\A)))
  (setf (slot-value cd 'object-list) nil)
  (flet ((mi (type x y)
	   (let ((obj (make-instance type
				     :x (* x (/ *component-size* 25)) 
				     :y (* y (/ *component-size* 25)))))
	     (add-new-object cd obj)
	     obj))
	 (splice (out-comp in-comp conn-number)
	   (let ((out-conn (first (slot-value out-comp 'outputs)))
		 (in-conn (elt (slot-value in-comp 'inputs) conn-number)))
	     (connect out-conn in-conn))))
  (let (;; column 1
	(one1 (mi 'logic-one 100 100))
	(zero1 (mi 'logic-zero 100 200))
	(one2 (mi 'logic-one 100 300))
	(zero2 (mi 'logic-zero 100 450))
	;; column two
	(and1 (mi 'and-gate 200 150))
	(or1 (mi 'or-gate 200 350))
	;; colum three
	(or2 (mi 'or-gate 300 108))
	(and2 (mi 'and-gate 300 300))
	(or3 (mi 'or-gate 300 420))
	;; column four
	(or4 (mi 'or-gate 400 150))
	(and3 (mi 'and-gate 400 350))
	;; column five
	(or5 (mi 'or-gate 500 250))
	)
    (splice one1 and1 0)
    (splice zero1 and1 1)
    (splice one2 or1 0)
    (splice zero2 or1 1)
    (splice and1 and2 0)
    (splice or1 and2 1)
    (splice one1 or2 0)
    (splice and1 or2 1)
    (splice and1 and2 0)
    (splice or1 and2 1)
    (splice or1 or3 0)
    (splice zero2 or3 1)
    (splice or2 or4 0)
    (splice zero1 or4 1)
    (splice and2 and3 0)
    (splice or3 and3 1)
    (splice or4 or5 0)
    (splice and3 or5 1)
    )))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; Commands
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(define-cad-demo-command (com-create-component :command-name "Create")
			 ()
   (with-frame (frame)
     (let* ((display-pane (slot-value frame 'display-pane))
	    (type (menu-choose *component-types* :associated-window display-pane
			       :cache T :unique-id 'component-types))
	    (position 
	      (with-input-context ('cad-position)
				  (object)
                                  (progn (display-pt-prompt 'cad-position frame)
                                         (read-gesture :stream display-pane))
		 (t object)))
	    (object (make-instance type :x (car position) :y (cdr position))))
       (add-new-object frame object)
       (draw-self object display-pane))))

;;; Takes two operands, an input terminal and an output terminal
;;; --- This needs to propagate value changes down the line, or
;;; rather redraw those components whose values have changed.
(define-cad-demo-command (com-connect-gates :command-name "Connect")
    ((output 'output :translator-gesture :left)
     (input 'input :translator-gesture :left))
  (with-frame (frame)
    (let ((win (slot-value frame 'display-pane)))
      (draw-self (connection-component input) win :ink w:+background+)
      (draw-self (connection-component output) win :ink w:+background+)
      (connect output input)
      (draw-self (connection-component input) win)
      (draw-self (connection-component output) win))))

;;; Moves a component.  The ":translator-gesture" option specifes that a 
;;; component-to-command translator be automatically defined.  This means that
;;; the user can just point at the component to be moved and click to invoke
;;; this command on that component.
(define-cad-demo-command (com-move-component :command-name "Move")
    ((component 'component :translator-gesture :left))
   (with-frame (frame)
     (let ((stream (slot-value frame 'display-pane)))
       (draw-self component stream :ink w:+background+)
       (notify-user frame "Position this gate with the mouse.  Click when done.")
       (multiple-value-bind (x y)
	   (dragging-output (stream)		;dragging-raster??
	     (draw-body component stream :ink w:+flipping-ink+))	;for XORing
	 (move component x y))
       (draw-self component stream))))

(define-cad-demo-command (com-clear :command-name T)
    ()
   (with-frame (frame)
     (with-slots (display-pane object-list) frame
       (setf object-list nil)
       (window-clear display-pane))))

(define-cad-demo-command (com-refresh :command-name T)
    ()
    (pane-needs-redisplay (with-frame (frame)
			    (slot-value frame 'display-pane))))

(define-cad-demo-command (com-show :command-name T)
    ((output 'output :translator-gesture :middle))
   ;; how to keep the application from redisplaying around the command loop?
   (let ((component (connection-component output)))
     (with-frame (frame)
       (notify-user frame "~A" (equation-part component))
       (sleep 5))))

(define-cad-demo-command (com-setup :command-name T)
    ()
  (with-frame (frame)
    (make-drawing frame)
    (com-refresh frame)))

(define-cad-demo-command (com-exit-demo :menu-accelerator "Exit")
    ()
    (with-frame (frame)
      (stop-frame frame)))

#+Ignore
(define-cad-demo-command (com-swap-layouts :menu-accelerator t)
    ()
  (let ((current-layout (application-current-layout *application*)))
    (set-layout *application*
		(case current-layout
		  (main 'other)
		  (other 'main)))))

#||

Things to do


add commands to scale up and down
but first get better menu formatting!

||#

;;; A per-root alist of cad demo objects.
(defvar *cad-demos* nil)

(defun color-stream-p (port)
  #+ccl (declare (ignore port))
  (cond #+Genera
	((typep port 'on-genera::genera-port)
	 (dolist (inf (tv:sheet-inferiors (on-genera::genera-screen port)))
	   (scl:ignore-errors
	     (return (color:color-stream-p inf)))))
	#+XLIB
	((typep port 'on-x::x-port)
	 (> (slot-value port 'on-x::depth) 1))
	#+ccl
	((and ccl:*color-available*
              (or (not (numberp ccl:*color-available*))
                  (> ccl:*color-available* 1)))
         t)
	(t nil)))

(defmethod run-frame-top-level :around ((frame cad-demo))
  (let ((*highlight-ink* (if (color-stream-p (port frame)) +red+ +flipping-ink+)))
    (call-next-method)))

(defun run-cad-demo (&key (server-path *default-server-path*))
  (launch-frame 'cad-demo 
		:title "CLIM Cad Demo"
		:message-pane T
		:where server-path))

(define-menu-group cad-demo-menu-group 
  (("Connect" :command (build-command 'com-connect-gates))
   ("Move" :command (build-command 'com-move-component))
   ("Setup" :command '(com-setup))
   ("Show" :command (build-command 'com-show))
   ("Refresh" :command (build-command 'com-refresh))
   ("Clear" :command (build-command 'com-clear))
   ("Create" :command '(com-create-component))
   ("Exit" :command (build-command 'com-exit-demo))))
