;;; -*- Syntax: Common-Lisp; Package: CLIM-DEMO; Base: 10; Mode: LISP -*-

(in-package "CLIM-DEMO")

"Copyright (c) 1990 International Lisp Associates.  All rights reserved."

;;; Define a simple CLIM program.  This program maintains a simple address book.
;;;
;;; First, we need a minimal address database.

;;; A structure to hold each address
(defclass address ()
    ((name :initarg :name :accessor address-name)
     (address :initarg :address :accessor address-address)
     (number :initarg :number :accessor address-number))
  (:default-initargs :name "Unsupplied" :address "Unsupplied" :number "Unsupplied")
  )

;;; Database maintenance.
(defun make-address (&key name address number)
  (make-instance 'address :name name :address address :number number))

;;; A support utility.
(defun last-name (name)
  (subseq name
	  (1+ (or (position #\Space name :test #'char-equal :from-end T)
		  -1))))

;;; And a function which operates on the address class.
(defun address-last-name (address)
  (last-name (address-name address)))

;;; A place to keep addresses.
(defvar *addresses* nil)

(defun add-address (address)
  ;; obviously could deal with multiple address entries
  ;; with same name here, but that's outside the scope of this demo.
  (push address *addresses*)
  (setq *addresses* (sort *addresses* #'string-lessp :key #'address-last-name))
  (values)
  )

(add-address (make-address :name "Bill York"
			   :address "ILA, Mountain View"
			   :number "415-968-3656"))
(add-address (make-address :name "Dennis Doughty"
			   :address "ILA, Cambridge"
			   :number "617-576-1151"))
(add-address (make-address :name "Mark Son-Bell"
			   :address "ILA, Cambridge"
			   :number "617-576-1151"))
(add-address (make-address :name "Richard Lamson"
			   :address "ILA, San Francisco"
			   :number "415-661-5477"))

;;; --------------------------------
;;; Define the user interface here.
;;;
;;; First, we define a presentation type for address, which enables us to make them
;;; mouse-sensitive.  We define the :printer for the presentation-type to print out just
;;; the personal name of the address entry.

(define-presentation-type address ())

(clim::define-presentation-method present (object (type address) stream
						 (view clim::textual-view)
						 &key acceptably)
  (write-string (address-name object) stream))

;;; Define a method for displaying the "Rolodex" form of entry.  
;;; This will be redisplayed efficiently by CLIM's updating output facility.
;;; [Note that the addition of calls to UPDATING-OUTPUT with specific cache values
;;; could be inserted around each of the fields here to improve the performance if the
;;; amount of information on display became large.  The trade-off would be the relative
;;; speed difference between whatever mechanism would be used to compare unique-ids and 
;;; cache-values (typically EQL) versus the default mechanism for comparing strings
;;; (STRING-EQUAL).]
(defmethod display-address ((address-to-display address) stream)
  (with-slots (name address number) address-to-display
    (write-string "Name:  " stream) 
    (write-string name stream)
    (terpri stream)
    (write-string "Address:  " stream)
    (write-string address stream)
    (terpri stream)
    (write-string "Number:  " stream)
    (write-string number stream)))

;;; Define the application-frame for our application
(define-application-frame address-book
			  ()
  ;; This application has two state variables, the currently displayed
  ;; address and the window from which user queries should be read.
  ((current-address :initform nil)
   (interaction-pane )
   (name-pane)
   )
  (:pane
    ;; WITH-FRAME-SLOTS is a macro that gives us WITH-SLOTS-style access to the
    ;; named slots of the frame being created.
    (with-frame-slots (interaction-pane name-pane)
      ;; VERTICALLY and HORIZONTALLY are layout macros.
      ;; This description is creating a vertical stack of two thin horizontal panes above
      ;; one long horizontal pane.
      (vertically ()
	(horizontally ()
	  ;; MAKE-CLIM-PANE is the canonical way to construct a CLIM pane.
	  ;; [Other pane types, such as pushbuttons, etc. are outside the scope
	  ;; of this example.]
	  ;; This call to MAKE-CLIM-PANE is constructing the pane that will
	  ;; show the current address.  It should incrementally redisplay the function
	  ;; display-current-address.
	  (make-clim-pane
	    () 
	    :display-function
	    '(incremental-redisplay-display-function display-current-address)
	    :display-time :command-loop)
	  (make-clim-pane (name-pane :hs 150 :hs+ 0)
			  :display-function
			  '(incremental-redisplay-display-function display-names))
			  )
	(make-clim-pane (interaction-pane)
			))))
  ;; (:command-definer T) indicates that we will be defining commands for this frame.
  (:command-definer T)

  ;; (:menu-group address-book) indicates that we would like the commands in the
  ;; address-book menu group to be displayed in an window-system specific fashion.
  (:menu-group address-book)

  ;; As noted in the documentation, in CLIM 0.9 this clause is always necessary.
  (:top-level (clim-top-level)))

;;; This is the display-function for the upper-left pane, which specified 
;;; :display-function '(incremental-redisplay-display-function display-current-address).
(defmethod display-current-address ((frame address-book) stream)
  (let ((current-address (slot-value frame 'current-address)))
    (when current-address (display-address current-address stream))))

;;; This is the display-function for the upper-right pane, which specified
;;; :display-function '(display-names).
(defmethod display-names ((frame address-book) stream)
  (dolist (address *addresses*)
    ;; PRESENT invokes the :PRINTER for the ADDRESS presentation-type, defined above.
    ;; It also makes each address printed out mouse-sensitive.
    (present address 'address :stream stream)
    (terpri stream)))

;;; Defining the address-book menu group.  This will ensure that the "New" and "Quit"
;;; commands are made visible to the user as some form of command button.
(define-menu-group address-book
  (("New" :command '(com-new-address))
   ("Delete" :command `(com-delete-address ,*unsupplied-argument*))
   ("Quit" :command '(com-quit-address-book))))

;;; Define the commands for this application.  Note that in CLIM 0.9, some non-NIL value
;;; of :command-name must be specified in order to make the commands accessible.
(define-address-book-command (com-quit-address-book :name "Quit")
    ()
   ;; WITH-FRAME allows us to get access to the frame that is executing this command.
   ;; STOP-FRAME causes termination of the CLIM top-level command loop.
   (with-frame (frame) (stop-frame frame)))

;;; This command changes the currently selected address.
(define-address-book-command (com-select-address :name "Select Address")
    ;; this indicates that the command takes one argument, an address.
    ;; the :translator-gesture causes clicking left on objects of type address
    ;; to invoke this command.  [The default binding for the :select gesture is a left click]
    ((address 'address :gesture :select))

   (with-frame (frame)
     (setf (slot-value frame 'current-address) address)))

;;; This command allows us to enter a new address.
(define-address-book-command (com-new-address :name "New Address")
    ()
   (let ((name nil)
	 (address nil)
	 (number nil))
     (with-frame (frame)
       (let ((stream (slot-value frame 'interaction-pane)))
	 (window-clear stream)
	 (terpri stream)
	 ;; ACCEPTING-VALUES collects all calls to ACCEPT within its body
	 ;; into dialog entries and allows parallel, random editing of the fields.
	 ;; In this case, a dialog that looks like:
	 ;; Name: a string
	 ;; Address: a string
	 ;; Number: a string
	 ;; is produced, where each "a string" is sensitive and can be edited.
	 (let ((abort-p
		 (accepting-values (stream)
		   (setq name (accept 'string :default name :stream stream :prompt "Name"))
		   (terpri stream)
		   (setq address (accept 'string :default address :stream stream
					 :prompt "Address"))
		   (terpri stream)
		   (setq number (accept 'string :default number :stream stream
					:prompt "Number")))))
	   (window-clear stream)
	   (unless (eq abort-p ':abort)
	     (add-address (make-address :name name :address address :number number))
	     (pane-needs-redisplay (slot-value frame 'name-pane))))))))

(define-address-book-command (com-delete-address :name "Delete Address")
    ((address 'address :gesture :middle))
   (setf *addresses* (delete address *addresses*))
   (with-frame (frame)
     (pane-needs-redisplay (slot-value frame 'name-pane))))

;;; The function to run the address-book
(defun address-book (&optional address-book-or-server-path)
  (let ((address-book nil)
	(frame-manager nil))
    (etypecase address-book-or-server-path
      (null (setq frame-manager (find-frame-manager)))
      ((or list port)
       (setq frame-manager (find-frame-manager :where address-book-or-server-path)))
      (address-book (setq address-book address-book-or-server-path
			  frame-manager (frame-manager address-book))))
    (launch-frame 'address-book :frame-manager frame-manager :where address-book
		  :width 500 :height 400
		  :title "CLIM Address Book")
    ))
