;;; -*- Syntax: Common-Lisp; Package: CLIM-INTERNALS; Base: 10; Mode: LISP -*-

(in-package "CLIM-INTERNALS")

"Copyright (c) 1988, 1989, 1990 International Lisp Associates.  All rights reserved."

(defclass tracking-pointer-distributor (self-distributor-base-mixin
					standard-input-contract)
    ((held-button :initform nil)
     (generator :initarg :generator)
     )
  )

(defmethod handle-clicks? ((distributor tracking-pointer-distributor)) 
  nil)

(define-constructor make-tracking-pointer-distributor tracking-pointer-distributor 
  (generator)
  :generator generator)

(defmacro invoke-tracking-pointer-clause (clause-name &rest args)
  `(let ((clause (funcall generator ,clause-name)))
     (when clause
       (funcall clause ,@args))))

(defun tracking-pointer-internal (sheet generator
				  &optional distributor-class
					    initial-button-down
					    initial-x initial-y)
  (let* ((port (port sheet))
	 (distributor
	   (if distributor-class
	       (make-instance distributor-class :generator generator)
	       (make-tracking-pointer-distributor generator))))
					      
    (progn ;w::with-pointer-grab (port)
      (with-distributor (port distributor)

	(when (or initial-button-down initial-x initial-y)
	  (multiple-value-bind (x y state) (poll-pointer sheet)
	    (when (and initial-x initial-y
		       (not (and (eql x initial-x) (eql y initial-y))))
	      (invoke-tracking-pointer-clause :pointer-motion sheet x y))
	    (when (and initial-button-down
		       ;; Who thought state-match-p should only work at compile time?
		       (not
			 (ecase initial-button-down
			   (:left (state-match-p state :left))
			   (:middle (state-match-p state :middle))
			   (:right (state-match-p state :right))
			   (:none (state-match-p state (not (or :left :middle :right)))))))
	      (if (eql initial-button-down :none) 
		  (invoke-tracking-pointer-clause 
		    :button-press sheet x y
		    (cond ((state-match-p state :left) :left)
			  ((state-match-p state :middle) :middle)
			  ((state-match-p state :right) :right)))
		  (invoke-tracking-pointer-clause :button-release initial-button-down)))))

	(let ((event-queue (sheet-event-queue distributor)))
	  (flet ((queue-not-empty-p ()
		   (not (queue-empty-p event-queue))))
	    (declare (dynamic-extent #'queue-not-empty-p))
	    (loop
	      (port-event-wait port #'queue-not-empty-p :wait-reason "Tracking Pointer")
              #+ignore (process-wait #'queue-not-empty-p "Tracking Pointer")
	      (handle-queued-input-event 
		distributor (queue-get event-queue)))))))))

#+Ignore ;; Debugging.
(defmethod handle-queued-input-event :around ((distributor tracking-pointer-distributor) event)
  (print event *trace-output*)
  (call-next-method))

(defmethod handle-queued-input-event ((distributor tracking-pointer-distributor)
				      (event button-press-event))
  (let* ((generator (slot-value distributor 'generator))
	 (button-press (funcall generator :button-press))
	 (presentation-button-press (funcall generator :presentation-button-press))
	 (button (event-button event))
	 (sheet (event-sheet event))
	 (x (event-x event))
	 (y (event-y event)))
    (when (typep sheet 'extended-stream-pane)
      (setf (slot-value distributor 'held-button) button)
      (when presentation-button-press
	;; must figure out what's under the mouse
	(let ((presentation (find-innermost-applicable-presentation '(t) sheet x y)
			    #+ignore ; was
			    (find-presentation-at-position sheet x y nil)))
	  (when presentation
	    (return-from handle-queued-input-event
	      (funcall presentation-button-press presentation sheet x y button)))))
      (when button-press
	(funcall button-press sheet x y button)))))

(defmethod handle-queued-input-event ((distributor tracking-pointer-distributor)
				      (event button-release-event))
  (let ((button (event-button event)))
    (with-slots (generator held-button) distributor
      ;; --- is there some timing window here if we press button 1 then 2 then release
      ;; --- 1 (keeping 2 pressed) or do we not care about that?
      (setf held-button nil)
      (invoke-tracking-pointer-clause :button-release button))))

(defmethod handle-queued-input-event ((distributor tracking-pointer-distributor)
				      (event key-press-event))
  (with-slots (generator) distributor
    (let ((char (event-char event))
	  (keysym (event-keysym event)))
      (invoke-tracking-pointer-clause :key-press keysym char))))

(defmethod handle-queued-input-event ((distributor tracking-pointer-distributor)
				      (event key-release-event))
  (with-slots (generator) distributor
    (let ((char (event-char event))
	  (keysym (event-keysym event)))
      (invoke-tracking-pointer-clause :key-release keysym char))))

(defmethod handle-queued-input-event ((distributor tracking-pointer-distributor) (event pointer-motion-event))
  (let* ((held-button (slot-value distributor 'held-button))
	 (generator (slot-value distributor 'generator))
	 (presentation-motion-hold (funcall generator :presentation-hold))
	 (presentation-motion (funcall generator :presentation))
	 (pointer-hold (funcall generator :pointer-motion-hold))
	 (pointer-motion (funcall generator :pointer-motion))
	 (sheet (event-sheet event))
	 (x (event-x event))
	 (y (event-y event)))
    (when (typep sheet 'extended-stream-pane)
      (macrolet ((try-choice (choice &rest args)
		   `(when ,choice
		      (return-from handle-queued-input-event
			(funcall ,choice ,@args)))))
	(when (or presentation-motion-hold presentation-motion)
	  ;; must figure out what's under the mouse
	  (let ((presentation
		  (find-innermost-applicable-presentation '(t) sheet x y)
		  #+ignore ; was:
		  (find-presentation-at-position sheet x y nil)))
	    (when presentation
	      (when held-button
		(try-choice presentation-motion-hold presentation sheet x y held-button))
	      (try-choice presentation-motion presentation sheet x y))))
	(when held-button
	  (try-choice pointer-hold sheet x y held-button))
	(try-choice pointer-motion sheet x y)))))

(defmethod handle-queued-input-event ((distributor tracking-pointer-distributor) event)
  (declare (ignore event))
  ;;; --- Ignore all events we don't understand, for now.
  )

(eval-when (eval compile load)

;;; For compile-time checking.
(defvar *tracking-pointer-clauses* nil)		;each clause looks like (name arglist)

(defmacro register-tracking-pointer-clause (name arglist)
  `(push-unique '(,name ,arglist) *tracking-pointer-clauses* :key #'car))

(defun validate-tracking-pointer-clause (name arglist)
  #+Genera (declare (values new-arglist ignores))
  (let ((item (find name *tracking-pointer-clauses* :key #'car)))
    (unless item
      (error "~S is not a known TRACKING-POINTER clause." name))
    ;; if we had an error system we'd use it here to generate a better
    ;; error message
    (canonicalize-and-match-lambda-lists (second item) arglist)))

;;; The clauses.
(register-tracking-pointer-clause :pointer-motion (sheet x y))
(register-tracking-pointer-clause :button-press (sheet x y button-name))
(register-tracking-pointer-clause :button-release (button-name))
(register-tracking-pointer-clause :key-press (keysym char))
(register-tracking-pointer-clause :key-release (keysym char))
(register-tracking-pointer-clause :pointer-motion-hold (sheet x y button-name))
(register-tracking-pointer-clause :presentation (presentation sheet x y))
(register-tracking-pointer-clause :presentation-button-press (presentation sheet x y button-name))
(register-tracking-pointer-clause :presentation-hold (presentation sheet x y button-name))

)

(defmacro tracking-pointer ((sheet &key distributor-class
					initial-button-down initial-x initial-y)
			    &body clauses)
  (let (case-clauses function-defs function-references 
	(event-handler-function-name (gensymbol 'event-handler)))
    (dolist (clause clauses)
      (let* ((type (pop clause))
	     (arglist (pop clause))
	     (body clause)
	     (function-name (gensymbol type))
	     (ignores nil))
	(multiple-value-setq (arglist ignores)
	  (validate-tracking-pointer-clause type arglist))
	(push `(,type #',function-name) case-clauses)
	(push `(,function-name ,arglist
		(declare ,@(when ignores `((ignore ,@ignores))))
		,@body)
	      function-defs)
	(push `#',function-name function-references)))
    `(block nil
       (labels
	 ((,event-handler-function-name (.type.)
	   (case .type. ,@case-clauses))
	  ,@function-defs)
	 (declare (dynamic-extent #',event-handler-function-name ,@function-references))
	 ;; cause CLOE compiler to produce closures for the LABELs above.
	 #+Cloe-Runtime ,@function-references
	 (tracking-pointer-internal
	   ,sheet
	   #',event-handler-function-name
	   ,distributor-class
	   ,initial-button-down ,initial-x ,initial-y)))))

#||
(defun test-tracking-pointer (sheet)
  (tracking-pointer (sheet)
    (:pointer-motion (&key x y sheet)
     (when (< 90 x 100)
       (tv:notify nil "win: ~S (~D,~D)" sheet x y))
     )
    (:presentation (&key presentation)
     (tv:notify nil "Over presentation ~S" presentation))
    (:button-press (&key button-name)
     (tv:notify nil "~S pressed" button-name))
    (:key-press (&key keysym)
     (when (typep keysym 'meta-keysym)
       (return-from test-tracking-pointer))
     (tv:notify nil "~A pressed" keysym))
    (:key-release (&key keysym)
     (tv:notify nil "~A released" keysym))
    ))

(defun test-tracking-pointer-1 (stream &optional clear-p (count 20))
  (when clear-p (window-clear stream))
  (let ((i 0))
    (with-output-recording-options
     (stream :record-p nil :draw-p t)
     (tracking-pointer (stream)
       (:pointer-motion (&key sheet x y)
	(incf i)
	(when (> i count)
	  (return-from test-tracking-pointer-1))
	(draw-rectangle* sheet x y (+ x 10) (+ y 10)))))))
||#
