;;; -*- Mode: LISP; Syntax: Common-lisp; Package: CLIM-INTERNALS; Base: 10; Lowercase: Yes -*-

(in-package "CLIM-INTERNALS")

"Copyright (c) 1988, 1989, 1990 International Lisp Associates.  All rights reserved."

(defun decode-alist-member-object (element)
  #+Genera (declare (values object))
  (cond ((atom element) element)
	(t (getf (rest element) :value))))

(defun decode-alist-member-element (element)
  #+Genera (declare (values name object))
  (cond ((atom element)
	 (values (symbol-name element)
		 element))
	(t
	 (values (first element)
		 (getf (rest element) :value)))))

(define-presentation-type alist-member (&key alist)
  :data-args-comparator ((sub super)
			 (let ((sub-alist (getf sub :alist))
			       (super-alist (getf super :alist)))

			   ;; --- What is a guaranteed portable way of doing this?
			   (equal (intersection sub-alist super-alist
						:test #'equal)
				  sub-alist)))
  
  :parser ((stream &key &allow-other-keys)
	   (values
	     (completing-from-suggestions (stream)
	       (dolist (element alist)
		 (multiple-value-call #'suggest (decode-alist-member-element element))))))
  :printer ((object stream &key &allow-other-keys)
	    (let ((element (find object alist
				 :key #'decode-alist-member-object)))
	      (write-string (decode-alist-member-element element) stream)))
  :object-validator ((object)
		     (find object alist
			   :key #'decode-alist-member-object))
  :describer ((stream)
	      (write-string "one of " stream)
	      ;; -- CLIM doesn't have a general list formatter yet
	      (let (length)
		(dorest (rest-of-elements alist)
		  (setq length (length rest-of-elements))
		  (format stream "~A" (decode-alist-member-element (car rest-of-elements)))
		  (cond ((> length 2)
			 (write-string ", " stream))
			((= length 2)
			 (write-string " or " stream))))))
  :accept-values-displayer ((stream object query-identifier)
			    (genera-accept-values-choose-from-sequence 
			      stream alist
			      #'decode-alist-member-object
			      object
			      #'eql 
			      type ;; shouldn't TYPE be in the arglist somewhere?
			      query-identifier #'(lambda (x ignore)
                                                   (declare (ignore ignore))
                                                   x)
			      #'(lambda (continuation object stream)
				  (with-text-face (:bold stream)
				    (funcall continuation object stream)))))
  )

(define-presentation-type boolean ()
  :abbreviation-for '(alist-member :alist (("Yes" :value t) ("No" :value nil)))) 

;;; No longer necessary??
#+Ignore
(defun complete-member (string completion-type set)
  (let ((possibilities (remove-if #'(lambda (thing)
				      (let ((m (mismatch string (string thing)
							 :test #'char-equal)))
					(and m
					     (< m (length string)))))
				  set)))
    (ecase completion-type
      (:complete (process-completions possibilities))
      (:complete-limited (process-completions possibilities #'string " " (length string))))))

#+Ignore
(defun complete-string-list (string completion-type string-list)
  (let ((possibilities (remove-if #'(lambda (thing)
				      (let ((m (mismatch string (string thing)
							 :test #'char-equal)))
					(and m (< m (length string)))))
				  string-list)))
    (ecase completion-type
      (:complete (process-completions possibilities))
      (:complete-limited (process-completions possibilities #'string " " (length string))))))

;;; The FUNCTION to complete-input is called with two arguments,
;;; string-so-far and a completion type, which is one of :complete or :complete-limited
;;; for now.  It returns string, success, object, nmatches (first-interesting-index?)
#+Genera
(defun pathname-complete (string completion-type
			  &optional (default *default-pathname-defaults*))
  (declare (ignore completion-type))
  ;; aw, shit.  The only way to pass the default off to this function
  ;; is by lexically capturing it within the :parser clause of the d-p-t.
  ;; I'm getting pretty worried about performance in non-lispm implementations.
  ;; We can always 
  (multiple-value-bind (string success)
      (fs:complete-pathname default string nil :newest :read)
    (values string 
	    success
	    (and success (pathname string))
	    1)))

#+:Coral
(defun pathname-complete (string completion-type
			  &optional (default *default-pathname-defaults*))
  (declare (ignore default))			;--- for now
  ;; slow but accurate
  (let ((pathname (pathname string))
	all-completions)
    (cond ((pathname-version pathname)
	   ;; get around file-system braino I don't know how to resolve
	   (setq all-completions (directory pathname)))
	  (t (setq all-completions (directory (concatenate 'string string "*")))))
    (ecase completion-type
      (:complete
	(process-completions all-completions #'namestring))
      (:complete-limited
	(process-completions all-completions #'namestring " " (length string))))))

#-(or Genera :Coral)
(defun pathname-complete (string completion-type
			  &optional (default *default-pathname-defaults*))
  (declare (ignore default))			;--- for now
  ;; --- This still needs to be thoroughly tested in Franz.
  ;; slow but accurate
  (let ((pathname (pathname string))
	all-completions)
    (cond ((pathname-version pathname)
	   ;; get around file-system braino I don't know how to resolve
	   (setq all-completions (directory pathname)))
	  (t (setq all-completions
		   (directory 
		     (make-pathname :host (pathname-host pathname)
				    :device (pathname-device pathname)
				    :directory (pathname-directory pathname))))))
    ;; now prune out all completions that don't start with
    (let ((name (pathname-name pathname))
	  (type (pathname-type pathname)))
      (setq all-completions
	    (delete-if-not #'(lambda (pn)
			       (let* ((pn-name (pathname-name pn))
				      (pn-type (pathname-type pn)))
				 (cond (type
					(and
					  (string-equal pn-name name)
					  (let ((s (search type pn-type :test #'char-equal)))
					    (and s (zerop s)))))
				       (t
					(let ((s (search name (pathname-name pn) 
							 :test #'char-equal)))
					  (and s (zerop s)))))))
			   all-completions)))
    (ecase completion-type
      (:complete
	(process-completions all-completions #'namestring))
      (:complete-limited
	(process-completions all-completions #'namestring " " (length string))))))

(define-presentation-type pathname ()
  :parser ((stream &key default)
	   (let ((buffer-start (input-position stream)))
	     (multiple-value-bind (pathname success string)
		 (complete-input stream #'pathname-complete :allow-any-input t)
	       (declare (ignore success))
	       (unless pathname
		 (setq pathname (merge-pathnames string default)))
	       (setq pathname (merge-pathnames pathname default))
	       ;; we need to do our own presentation-replace-input because we
	       ;; said :ALLOW-ANY-INPUT T
	       (presentation-replace-input stream pathname type :buffer-start buffer-start)
	       pathname)))
  :printer ((thing stream &key &allow-other-keys)
	    (write-string (namestring thing) stream)))

(define-presentation-type string ()
  :parser ((stream &key default)
	   ;; --- need to upgrade parser-defining syntax, too.
	   default
	   (values (read-token stream)))
  :printer ((string stream &key &allow-other-keys)
	    (write-string string stream)))

(defun simple-lisp-object-parser (type stream &optional coerce-test coerce-function)
  (flet ((presentation-typep (object type)
	   (or (null (presentation-type-validate-objects-p type))
	       (validate-object object type))))
    (declare (dynamic-extent #'presentation-typep))
    (loop
      (let ((token (read-token stream)))
	(multiple-value-bind (object index)
	    #+Ignore
	    (let ((future-common-lisp:*read-eval* nil))		;disable "#."
	      (read-from-string token nil token))
	    (read-from-string token nil token)
	  (when (eq object token)
	    (parse-error "Unexpected EOF"))
	  ;; Too bad read-from-string doesn't take a :junk-allowed argument
	  ;; Simulate what it would do
	  (unless (>= index (length token))
	    (when (find-if-not #'whitespace-character-p token :start index)
	      (parse-error "Extra junk ~S found after the ~A."
			   (subseq token index)
			   (describe-presentation-type type nil))))
	  (when (presentation-typep object type)
	    (return-from simple-lisp-object-parser (values object type)))
	  (when coerce-function
	    (when (funcall coerce-test object)
	      (setq object (funcall coerce-function object))
	      (when (presentation-typep object type)
		(return-from simple-lisp-object-parser (values object type)))))
	  (input-not-of-required-type token type))))))

(define-presentation-type number (&optional (low '*) (high '*))
  :parser ((stream &key default)
	   default
	   (simple-lisp-object-parser type stream))
  :printer ((number stream &key &allow-other-keys)
	    (format stream "~A" number))
  :object-validator ((object)
		     (and (numberp object)
			  (or (eq low '*) (>= object low))
			  (or (eq high '*) (< object high))))
  :describer ((stream)
	      (format stream "a number")
	      (cond ((and low high)
		     (format stream " between ~D and ~D" low high))
		    (low
		     (format stream " greater than or equal to ~D" low))
		    (high
		     (format stream " less than or equal to ~D" high)))))

#|
(defmacro defaulting-token ((token &optional default) &body body)
  `(or (and (string-equal ,token "") default)
       ,@body))
|#

(define-presentation-type integer (&optional (low '*) (high '*))
  :parser ((stream &key default)
	   default
	   (simple-lisp-object-parser type stream))
  :printer ((number stream &key &allow-other-keys)
	    (format stream "~D" number))
  :object-validator ((object)
		     (and (integerp object)
			  (or (eq low '*) (>= object low))
			  (or (eq high '*) (< object high))))
  :describer ((stream)
	      (format stream "an integer")
	      (if (eq low '*)
		  (if (eq high '*)
		      nil
		      (format stream " less than or equal to ~D" high))
		  (if (eq high '*)
		      (format stream " greater than or equal to ~D" low)
		      (format stream " between ~D and ~D" low high)))))

(define-presentation-type float (&optional (low '*) (high '*))
  :parser ((stream &key default)
	   default
	   (simple-lisp-object-parser type stream #'rationalp #'float))
  :printer ((number stream &key &allow-other-keys)
	    (format stream "~F" number))
  :object-validator ((object)
		     (and (floatp object)
			  (or (eq low '*) (>= object low))
			  (or (eq high '*) (< object high))))
  :describer ((stream)
	      (format stream "a float")
	      (if (eq low '*)
		  (if (eq high '*)
		      nil
		      (format stream " less than or equal to ~D" high))
		  (if (eq high '*)
		      (format stream " greater than or equal to ~D" low)
		      (format stream " between ~D and ~D" low high)))))

(define-presentation-type symbol ()
  :parser ((stream &key default)
	   default
	   (simple-lisp-object-parser type stream))
  :object-validator ((object)
		     (symbolp object)))

#||
(defun dtest (stream type object &optional sleep)
  (stream-clear-input stream)		;why??
  (window-clear stream)
  (window-expose stream)
  (present object type :stream stream)
  (dotimes (i 3) (terpri stream))
  (write-string (format nil "Enter a ~(~A~): " (with-type-decoded (type) type type)) stream)
  (unwind-protect
      (with-output-recording-options (stream :draw-p t :record-p nil)	;don't cons in IE
	(with-input-editing (stream)
	  (accept type :stream stream)))
    (when sleep (sleep 2))
    (window-set-visibility stream nil)))

(defun etest (stream &aux (type2 '(alist-member :alist (("Yes" :value t) ("No" :value nil))))
			  (type1 'boolean))
  (stream-clear-input stream)		;why??
  (window-clear stream)
  (window-expose stream)
  (present t type2 :stream stream)		;--- this is how <Help> would present it.
  (dotimes (i 3) (terpri stream))
  (write-string (format nil "Enter a ~(~A~): " (with-type-decoded (type1) type1 type1)) stream)
  (unwind-protect
      (with-output-recording-options (stream :draw-p t :record-p nil)	;don't cons in IE
	(with-input-editing (stream)
	  (accept type1 :stream stream)))
    (window-set-visibility stream nil)))

(defun ctest (stream)
  (stream-clear-input stream)		;why??
  (window-clear stream)
  (window-expose stream)
  (present *default-pathname-defaults* 'pathname :stream stream)
  (dotimes (i 3) (terpri stream))
  (write-string "Enter a pathname: " stream)
  (unwind-protect
      (with-output-recording-options (stream :draw-p t :record-p nil)	;don't cons in IE
	(with-input-editing (stream)
	  (accept 'pathname :stream stream)))
    (window-set-visibility stream nil)))
||#

(defvar *char-associations* '((#\( . #\))
			      (#\" . #\")
			      (#\| . #\|)))

;;; Ignore issues like comments for now.
(defun read-recursive (stream input-buffer desired-delimiter)
  (loop
    (with-input-context ('expression)
			(object)
	 (let ((char (read-gesture :stream stream))
	       (other-delimiter nil))
	   (cond ((activation-character-p char)
		  (if desired-delimiter
		      (beep stream)
		      (return)))
		 ((blip-character-p char)
		  (beep stream))
		 (t
		  (vector-push-extend char input-buffer)
		  (cond ((and desired-delimiter 
			      (char-equal char desired-delimiter))
			 (return))
			((setq other-delimiter (cdr (assoc char *char-associations*)))
			 (read-recursive stream input-buffer other-delimiter)
			 (unless desired-delimiter (return)))
			((and (whitespace-character-p char) (not desired-delimiter))
			 (return))
			(t nil)))))
       (t (let ((start-position (and (interactive-stream-p stream)
				     (input-position stream))))
	    (with-temporary-string (string)
	      (with-output-to-string (stream string)
		(let* ((index (fill-pointer input-buffer))
		       (previous-character (and (not (zerop index))
						(aref input-buffer (1- index))))
		       (non-whitespace-p 
			 (and previous-character
			      (not (whitespace-character-p previous-character)))))
		  (when non-whitespace-p
		    (write-string " " stream)))
		(present object 'expression :stream stream))
	      ;; not only put this thing in the input editor's buffer
	      (when start-position
		(replace-input stream string :buffer-start start-position))
	      ;; but into the buffer we ourselves are maintaining.
	      (doseq (char string)
		(vector-push-extend char input-buffer))))
	  (unless desired-delimiter (return))))))

(defun print-recursive (thing stream)
  (flet ((body (thing stream)
	   (cond ((atom thing)
		  (write-string (with-output-to-string (s)
				  (write thing :escape t :stream s))
				stream))
		 ((eq (first thing) 'quote)
		  (write-string "'" stream)
		  (print-recursive (cadr thing) stream))
		 (t
		  (write-string "(" stream)
		  (dorest (elements thing)
		    (let ((element (first elements)))
		      (print-recursive element stream))
		    (when (rest elements) (write-string " " stream)))
		  (write-string ")" stream)))))
    ;;; --- WITH-OUTPUT-AS-PRESENTATION should actually do this for us!
    (if (output-recording-stream-p stream)
	(with-output-as-presentation (:stream stream
				      :object thing
				      :type 'expression)
	  (body thing stream))
	(body thing stream))))

(define-presentation-type expression ()		;"form"?
  :parser ((stream &rest args)
	   (declare (dynamic-extent args)
		    (ignore args))
	   (with-temporary-string (string)
	     (read-recursive stream string nil)
	     (read-from-string string)))
  :printer ((thing stream &key &allow-other-keys)
	    (print-recursive thing stream)))

;;; Quote all but self-evaluating things when translating them.
(define-presentation-translator quoted-expression
				(expression expression
					    :tester ((object)
						     (values (not (constantp object))
							     T)))
				(object)
  (list 'quote object))

(defun completion-default-name-key (item)
  (typecase item
    (string item)
    (null "NIL")
    (cons (string (car item)))
    (symbol (string-capitalize (symbol-name item)))
    (otherwise (princ-to-string item))))

(define-presentation-type completion (&optional sequence
				      &key (test 'eql) (value-key 'identity)
				           (name-key 'completion-default-name-key)
					   (partial-completers '(#\Space)))
  :parser ((stream &key default &allow-other-keys)
	   (declare (ignore default))				;--- for now
	   (values
	     (completing-from-suggestions (stream :partial-completers partial-completers)
	       (flet ((completion (item)
			(suggest (funcall name-key item) (funcall value-key item))))
		 (declare (dynamic-extent #'completion))
		 (map nil #'completion sequence)))))
  :printer ((object stream &key acceptably &allow-other-keys)
	    (declare (ignore acceptably))			;--- for now
	    (write-string (funcall name-key (find object sequence :key value-key :test test))
			  stream)))

(define-presentation-type sequence (sequence-type)
  :parser ((stream &key default)
	   (block parser
	     (let ((result nil)
		   char)
	       (flet ((add-space ()
			(replace-input 
			  stream " " :buffer-start (input-position stream)))
		      (add-comma-space ()
			(replace-input 
			  stream ", " :buffer-start (input-position stream))))
		 (declare (dynamic-extent #'add-space #'add-comma-space))
		 (loop
		   (with-blip-characters (#\,)
		     (push (accept sequence-type :stream stream) result))
		   (loop
		     ;; This clause is only to handle the case where
		     ;; you do <click> to satisfy the accept above, and now
		     ;; you want to click on the 2nd thing w/o typing the comma
		     ;; first.
		     (catch 'click
		       (with-input-context (sequence-type)
					   (object ptype)
			    (setq char (read-char stream))
			  (T (push object result)
			     (add-comma-space)
			     (presentation-replace-input 
			       stream object sequence-type
			       :buffer-start (input-position stream))
			     (throw 'click nil)))
		       ;; don't do this part if click
		       (when (activation-character-p char)
			 (stream-unread-char stream char)
			 (return-from parser (or (reverse result) default)))
		       (return
			 (cond ((char= char #\,)
				(if (not (rescanning-p stream))
				    (add-space)
				    ;; otherwise, read the space out and throw it away
				    (let ((char (read-char stream)))
				      (assert (char= char #\Space)))))
			       (t (stream-unread-char stream char)))))))))))
  :printer ((object stream &key acceptably)
	    acceptably
	    (do ((sequence object (cdr sequence)))
		((null sequence))
	      (present (first sequence) sequence-type :stream stream)
	      (unless (= (length sequence) 1)
		(write-string ", " stream))))
  )

(defun token-element-string (element)
  (typecase element
    (null (symbol-name element))		;"NIL"
    (cons (string (first element)))
    (symbol (symbol-name element))
    (string element)
    (otherwise (present-to-string element))))

(define-presentation-type member (&rest choices)
  :data-args-comparator ((sub super)
			 (setq sub (first sub))
			 (setq super (first super))
			 (null (intersection (set-difference sub super) sub))
			 )
  :object-validator ((object)
		     (member object choices))
  :parser ((stream &key default)
	   (multiple-value-bind (object success string)
	       (completing-from-suggestions (stream)
		 (dolist (choice choices)
		   (suggest (token-element-string choice) choice)))
	     (cond ((and object success)
		    object)
		   ((string-equal string "") default)
		   (t (input-not-of-required-type object 'member)))))
  :printer ((object stream &key acceptably) acceptably
	    (write-string (token-element-string object) stream))
  :accept-values-displayer ((stream object query-identifier)
			    (flet ((highlighter (continuation object stream)
					 (with-text-face (:bold stream)
					   (funcall continuation object stream)))
				       (selector (a ignore)
                                         (declare (ignore ignore))
					 a))
			      (declare (dynamic-extent #'highlighter #'selector))
			      (genera-accept-values-choose-from-sequence 
				stream choices
				#'decode-alist-member-object
				object
				#'eql 
				type ;; shouldn't TYPE be in the arglist somewhere?
				query-identifier
				#'selector
				#'highlighter)))
  )

;;; ----------------
(define-presentation-type OR (&rest types)
  ;; we can't do object-validator yet because we don't have
  ;; PRESENTATION-TYPEP
;  :object-validator ((object)
;		     (flet ((validator (type)
;			      (presentation-typep type object)))
;		       (declare (dynamic-extent #'validator))
;		       (some #'validator types)))
  :data-args-comparator ((subargs superargs)
			 ;; --- rewrite this to not cons closures.
			 (every #'(lambda (subtype)
				    (some #'(lambda (supertype)
					      (presentation-subtypep subtype supertype))
					  (first superargs)))
				(first subargs)))
  :describer ((stream)
	      (let ((length (length types)))
		(dotimes (i length)
		  (describe-presentation-type (elt types i) stream)
		  (cond ((= i (1- length)))
			((= length 2) (write-string " or " stream))
			((= i (- length 2)) (write-string ", or " stream))
			(t (write-string ", " stream))))))
  :printer ((object stream &rest args)
	    (declare (dynamic-extent args))
	    (block did-it
	      (dolist (type types)
		(block this-try
		  (handler-bind ((error
				   #'(lambda (error)
                                       (declare (ignore error))
				       (return-from this-try (values)))))
		    (return-from did-it
		      (apply #'present object type :stream stream args)))))
	      (error "The object ~S is not of type ~S" object type)))
  :parser ((stream &key default)
	   (block accept
	     (let ((location (input-position stream))
		   last-error)
	       (dolist (type types)
		 (block fail
		   (handler-bind ((parse-error
				    #'(lambda (error)
					;; That parser didn't work, try another
					;; on the same input
					(setq last-error error)
					(setf (input-position stream) location)
					(return-from fail))))
		     (multiple-value-bind (object object-type)
			 (accept type :stream stream :prompt nil :default default)
		       (return-from accept
			 (values object (or object-type type)))))))
	       ;; No parser worked.  Resignal the most recent parse-error.
	       (if (typep last-error 'input-not-of-required-type)
		   (input-not-of-required-type
		     (input-not-of-required-type-string last-error)
		     type)
		   (parse-error "~A" last-error)))))
  )


(defun and-presentation-typep (object type)
  (with-type-decoded (name parameters) type
    (case name
      (satisfies (funcall (first parameters) object))
      (not (not (and-presentation-typep object (first parameters))))
      (otherwise 
	;; best we can do.
	(or (null (presentation-type-validate-objects-p type))
	    (validate-object object type))))))

(defun and-data-args-comparator (subtypes supertypes)
  (flet ((and-data-args-comparator-helper (supertype)
	   (with-type-decoded (supertype-name) supertype
	     (if (member supertype-name '(not satisfies))
		 (member supertype subtypes :test #'equal)
		 (flet ((internal-helper (subtype) 
			  (presentation-subtypep subtype supertype)))
		   (declare (dynamic-extent #'internal-helper))
		 (some #'internal-helper subtypes))))))
    (declare (dynamic-extent #'and-data-args-comparator-helper))
    (every #'and-data-args-comparator-helper supertypes)))

(define-presentation-type and (&rest types)
  :object-validator ((object)
		     (flet ((helper (type)
			      (and-presentation-typep object type)))
		       (declare (dynamic-extent #'helper))
		     (every #'helper types)))
  :describer ((stream)
	      (describe-presentation-type (first types) stream)
	      (let ((first t))
		(dolist (type (rest types))
		  (if first (setq first nil) (write-string " and" stream))
		  (let ((not nil))
		    (block nil
		      (loop
			(with-type-decoded (name parameters) type
			  (cond ((eq name 'satisfies)
				 (format stream " that ~:[satisfies~;doesn't satisfy~] ~S"
					 not (first parameters))
				 (return))
				((eq name 'not) 
				 (setq not (not not))
				 (setq type (first parameters)))
				(t
				 (format stream " that ~:[is~;is not~] " not)
				 (describe-presentation-type type stream)
				 (return))))))))))
  :data-args-comparator ((subargs superargs)
			 (and-data-args-comparator (first subargs) (first superargs))
			 )
  :printer ((object stream &rest args)
	    (declare (dynamic-extent args))
	    (apply #'present object (first types) :stream stream args))
  :parser ((stream &rest stuff)
	   (declare (dynamic-extent stuff))
	   (block accept
	     (loop
	       (multiple-value-bind (object input-type)
		   (apply #'accept (first types) :stream stream :prompt nil stuff)
		 (when (every #'(lambda (type)
				  (and-presentation-typep object type))
			      (rest types))
		   (return-from accept (values object input-type)))
		 (input-not-of-required-type object type))))))

(define-presentation-type token-or-type (tokens type)
  :abbreviation-for `(or (alist-member :alist ,tokens)
			 ,type))

(define-presentation-type type-or-string (type)
  :abbreviation-for `(or ,type string))

