;;; -*- Mode: LISP; Syntax: Common-lisp; Package: CLIM-INTERNALS; Base: 10; Lowercase: Yes -*-

;; $fiHeader: output-recording-protocol.lisp,v 1.5 91/03/26 12:48:25 cer Exp $

(in-package "CLIM-INTERNALS")

"Copyright (c) 1990, 1991 Symbolics, Inc.  All rights reserved.
 Portions copyright (c) 1989, 1990 International Lisp Associates."

;; The protocol class for an object that obeys the output record protocol,
;; that is, can hold output record elements
(defclass output-record () ())
(defun-inline output-record-p (object)
  (typep object 'output-record))

;; The protocol class for an element in an output record.
(defclass output-record-element () ())
(defun-inline output-record-element-p (object)
  (typep object 'output-record-element))

;; The protocol class for output record elements that are leaves.
(defclass displayed-output-record-element () ())
(defun-inline displayed-output-record-element-p (object)
  (typep object 'displayed-output-record-element))


;;; output-record-element-mixin has slots for
;;;     (bounding rectangle, parent, start-position, end-position, contents-ok)
;;;     plus space to store old bounding rectangle and position for incremental redisplay
;;;  graphics-output-record-element
;;;  text-output-record-element
;;;  output-record-mixin has slots for (generation-tick, old-elements)
;;;      both used for incremental redisplay.
;;;   linear-output-record
;;;    table, row, column, cell, presentation
;;;   coordinate-sorted-set-output-record
;;;   kd-tree-output-record

;;; The output-record-element "protocol"
;;;   :x-position, :y-position, :parent init args.
;;;   bounding rectangle protocol

;;; The output record protocol:
;;;  output-record-elements: (record)
;;;  add-output-record-element: (record element)
;;;  delete-output-record-element: (record element)
;;;  clear-output-record: (record)
;;;  replay-1: (record stream &optional region (x-offset 0) (y-offset 0)
;;;  recompute-extent: (record)
;;;  recompute-extent-for-new-element (there's a default): (record element)
;;;  recompute-extent-for-changed-element: (record element)
;;;  map-over-output-record-elements-overlapping-region:
;;;   (record region continuation
;;;    &optional (x-offset 0) (y-offset 0) &rest continuation-args)
;;;  map-over-output-record-elements-containing-point*:
;;;   (record x y continuation
;;;    &optional (x-offset 0) (y-offset 0) &rest continuation-args)

;;; The incremental redisplay protocol:
;;;  see file incremental-redisplay-protocol.text, and incremental-redisplay.lisp.
;;;

;; Bounding rectangle position and set-position are in relative coordinates, 
;;   relative to (OUTPUT-RECORD-POSITION* (OUTPUT-RECORD-PARENT RECORD)).
;; The bounding rectangle measures just the ink.
;; (OUTPUT-RECORD-START-POSITION* RECORD) refers to the position of the
;; cursor at the start of RECORD.  It is also the origin of the
;; coordinate system for all inferiors of RECORD.
;;--- Remember that there are typecases throughout CLIM which
;;--- expect OUTPUT-RECORD-ELEMENT-MIXIN to be the name of the
;;--- abstract class.
(defclass output-record-element-mixin
	  ;; The bounding rectangle bounds the ink, not the surrounding whitespace.
	  ;; It is relative to parent's start-position 
	  (bounding-rectangle output-record-element)
    ;; start position is relative to the parent's start-position
    ;; start position is where the cursor was when the new
    ;; output-record was started.
    ((start-x :initform 0 :initarg :start-x)
     (start-y :initform 0 :initarg :start-y)
     ;; end position is relative to start position.
     ;; end position is where the cursor was when we finished the
     ;; output-record.
     (end-x :initform 0 :initarg :end-x)
     (end-y :initform 0 :initarg :end-y)
     ;; old-start-position is relative to old-start-position of parent
     (old-start-x :initform 0)
     (old-start-y :initform 0)
     ;; old extent is relative to parents' old-start-position.
     ;;--- This should be renamed to OLD-BOUNDING-RECTANGLE
     (old-extent :initform nil			;a bounding rectangle, for redisplay
		 :accessor output-record-old-extent)
     (contents-ok :initform nil :accessor output-record-contents-ok)
     (parent :accessor output-record-parent :initarg :parent))
  (:default-initargs :parent nil :left 0 :top 0 :right 0 :bottom 0))

;;; Give initial rectangle of 0 size, it will get expanded as inferiors are added.
(defmethod initialize-instance :after ((record output-record-element-mixin)
				       &key (x-position 0) (y-position 0))
  ;; --- removed a #-Silica 12/27/91 doughty
  (with-slots (left top right bottom) record
    (setf left x-position
	  top  y-position
	  right  x-position
	  bottom y-position)))

;;; Shadow the method on RECTANGLE with this one that keeps the start-position and 
;;; bounding rectangle in synch.
(defmethod bounding-rectangle-set-position* ((record output-record-element-mixin) nx ny)
  (with-slots (left top right bottom start-x start-y parent) record
    ;; Move the start position by as much as we do the record.
    (setq start-x (+ start-x (- nx left)))
    (setq start-y (+ start-y (- ny top)))
    (let ((width (- right left))
	  (height (- bottom top)))
      (setf left nx top ny)
      (setf right  (+ nx width)
	    bottom (+ ny height)))))

(defun-inline output-record-position* (record)
  (output-record-start-position* record))

(defmethod output-record-set-position* ((record output-record-element-mixin) x y)
  (bounding-rectangle-set-position* record x y))

(defmethod output-record-start-position ((record output-record-element-mixin))
  (with-slots (start-x start-y) record
    (make-point start-x start-y)))

(defmethod output-record-start-position* ((record output-record-element-mixin))
  (with-slots (start-x start-y) record
    (values start-x start-y)))

;;; Keep the start-position and bounding rectangle in synch
(defmethod output-record-set-start-position* ((record output-record-element-mixin) nx ny)
  (with-slots (start-x start-y) record
    (with-bounding-rectangle* (left top right bottom) record
      (let ((dx (- nx start-x))
	    (dy (- ny start-y)))
	(bounding-rectangle-set-edges record
				      (+ left dx)  (+ top dy)
				      (+ right dx) (+ bottom dy))
	(setf start-x nx start-y ny)))))

(defmethod output-record-end-position* ((record output-record-element-mixin))
  (with-slots (end-x end-y) record
    (values end-x end-y)))
  
(defmethod output-record-set-end-position* ((record output-record-element-mixin) nx ny)
  (with-slots (end-x end-y) record
    (setf end-x nx)
    (setf end-y ny)))

(defmethod output-record-old-start-position ((record output-record-element-mixin))
  (with-slots (old-start-x old-start-y) record
    (make-point old-start-x old-start-y)))

(defmethod output-record-old-start-position* ((record output-record-element-mixin))
  (with-slots (old-start-x old-start-y) record
    (values old-start-x old-start-y)))

(defmethod output-record-set-old-start-position* ((record output-record-element-mixin) nx ny)
  (with-slots (old-start-x old-start-y) record
    (setf old-start-x nx)
    (setf old-start-y ny)))

(defmethod output-record-elements ((record output-record-element-mixin))
  nil)

;;; For specialization by PRESENTATIONs, for example
(defmethod output-record-refined-sensitivity-test ((record output-record-element-mixin) x y)
  (declare (ignore x y))
  T)

(defun compute-output-record-offsets (record)
  (let ((parent (output-record-parent record)))
    (if (null parent)
	(values 0 0)
      (multiple-value-bind (x y)
	  (compute-output-record-offsets parent)
	(multiple-value-bind (our-x our-y) (output-record-position* record)
	  (values (+ our-x x) (+ our-y y)))))))

(defmethod region-equal
	   ((record1 output-record-element-mixin) (record2 output-record-element-mixin))
  (with-bounding-rectangle* (left1 top1 right1 bottom1) record1
    (with-bounding-rectangle* (left2 top2 right2 bottom2) record2
      (if (eq (output-record-parent record1) (output-record-parent record2))
	  (ltrb-equals-ltrb-p left1 top1 right1 bottom1
			      left2 top2 right2 bottom2)
	(multiple-value-bind (xoff1 yoff1) (compute-output-record-offsets record1)
	  (multiple-value-bind (xoff2 yoff2) (compute-output-record-offsets record2)
	    (translate-positions xoff1 yoff1 left1 top1 right1 bottom1)
	    (translate-positions xoff2 yoff2 left2 top2 right2 bottom2)
	    (ltrb-equals-ltrb-p left1 top1 right1 bottom1
				left2 top2 right2 bottom2)))))))

(defmethod region-contains-point*-p
	   ((record output-record-element-mixin) x y)
  (with-bounding-rectangle* (left top right bottom) record
    (multiple-value-bind (xoff yoff) (compute-output-record-offsets record)
      (ltrb-contains-point*-p left top right bottom
			      (+ x xoff) (+ y yoff)))))

(defmethod region-contains-region-p
	   ((record1 output-record-element-mixin) (record2 output-record-element-mixin))
  (with-bounding-rectangle* (left1 top1 right1 bottom1) record1
    (with-bounding-rectangle* (left2 top2 right2 bottom2) record2
      (if (eq (output-record-parent record1) (output-record-parent record2))
	  (ltrb-contains-ltrb-p left1 top1 right1 bottom1
				left2 top2 right2 bottom2)
	(multiple-value-bind (xoff1 yoff1) (compute-output-record-offsets record1)
	  (multiple-value-bind (xoff2 yoff2) (compute-output-record-offsets record2)
	    (translate-positions xoff1 yoff1 left1 top1 right1 bottom1)
	    (translate-positions xoff2 yoff2 left2 top2 right2 bottom2)
	    (ltrb-contains-ltrb-p left1 top1 right1 bottom1
				  left2 top2 right2 bottom2)))))))

(defmethod region-intersects-region-p
	   ((record1 output-record-element-mixin) (record2 output-record-element-mixin))
  (with-bounding-rectangle* (left1 top1 right1 bottom1) record1
    (with-bounding-rectangle* (left2 top2 right2 bottom2) record2
      (if (eq (output-record-parent record1) (output-record-parent record2))
	  (ltrb-overlaps-ltrb-p left1 top1 right1 bottom1
				left2 top2 right2 bottom2)
	(multiple-value-bind (xoff1 yoff1) (compute-output-record-offsets record1)
	  (multiple-value-bind (xoff2 yoff2) (compute-output-record-offsets record2)
	    (translate-positions xoff1 yoff1 left1 top1 right1 bottom1)
	    (translate-positions xoff2 yoff2 left2 top2 right2 bottom2)
	    (ltrb-overlaps-ltrb-p left1 top1 right1 bottom1
				  left2 top2 right2 bottom2)))))))

(defun region-contains-offset-region-p (region1 region2 xoff yoff)
  (with-bounding-rectangle* (left1 top1 right1 bottom1) region1
    (with-bounding-rectangle* (left2 top2 right2 bottom2) region2
      (ltrb-contains-ltrb-p left1 top1 right1 bottom1
			    (+ left2 xoff)  (+ top2 yoff)
			    (+ right2 xoff) (+ bottom2 yoff)))))

(defun region-intersects-offset-region-p (region1 region2 xoff yoff)
  (with-bounding-rectangle* (left1 top1 right1 bottom1) region1
    (with-bounding-rectangle* (left2 top2 right2 bottom2) region2
      (ltrb-overlaps-ltrb-p left1 top1 right1 bottom1
			    (+ left2 xoff)  (+ top2 yoff)
			    (+ right2 xoff) (+ bottom2 yoff)))))

(defun offset-region-contains-point*-p (region xoff yoff x y)
  (with-bounding-rectangle* (left top right bottom) region
    (ltrb-contains-point*-p (+ left xoff)  (+ top yoff)
			    (+ right xoff) (+ bottom yoff)
			    x y)))

;;; This maps over all of the elements of the record
#+Genera (zwei:defindentation (map-over-output-record-elements 1 1))
(defun map-over-output-record-elements (record continuation
					&optional (x-offset 0) (y-offset 0)
					&rest continuation-args)
  (declare (dynamic-extent continuation continuation-args))
  (apply #'map-over-output-record-elements-overlapping-region
	 record nil continuation x-offset y-offset continuation-args))

;;; This must map over the elements in such a way that, when it maps over
;;; overlapping elements, the topmost (most recently inserted) element is
;;; hit last.  This is because this function is used for things such as
;;; replaying, where the most recently drawn thing must come out on top
;;; (i.e., must be drawn last).  If the region is NIL, then this maps over
;;; all of the elements in the output record.
#+Genera (zwei:defindentation (map-over-output-record-elements-overlapping-region 2 1))
(defgeneric map-over-output-record-elements-overlapping-region
	    (record region continuation
	     &optional x-offset y-offset &rest continuation-args)
  (declare (dynamic-extent continuation continuation-args)))

;;; This must map over the elements in such a way that, when it maps over
;;; overlapping elements, the topmost (most recently inserted) element is 
;;; hit first, that is, the opposite order of MAP-...-ELEMENTS-OVERLAPPING-REGION.
;;; This is because this function is used for things like locating the
;;; presentation under the pointer, where the topmost thing wants to be
;;; located first.
#+Genera (zwei:defindentation (map-over-output-record-elements-containing-point* 3 1))
(defgeneric map-over-output-record-elements-containing-point*
	    (record x y continuation
	     &optional x-offset y-offset &rest continuation-args)
  (declare (dynamic-extent continuation continuation-args)))

;;; X-offset and Y-offset represent the accumulated offset between the
;;; regions's native coordinates and "our" coordinates and must be added
;;; to our local coordinates (or subtracted from the region, if
;;; possible) in order to validly compare them.
;;; 
;;; In the absence of x- and y- offsets, region should be in the
;;; coordinate system of the record - i.e. relative to 
;;;  (OUTPUT-RECORD-POSITION* RECORD).
;;; This is the same coordinate system as the output-record-elements
;;; we are mapping over.
(defmethod map-over-output-record-elements-overlapping-region
	   ((record output-record-element-mixin) region continuation
	    &optional (x-offset 0) (y-offset 0) &rest continuation-args)
  (declare (ignore region continuation x-offset y-offset continuation-args)
	   (dynamic-extent continuation continuation-args))
  nil)

(defmethod map-over-output-record-elements-containing-point*
	   ((record output-record-element-mixin) x y continuation
	    &optional (x-offset 0) (y-offset 0) &rest continuation-args)
  (declare (ignore x y continuation x-offset y-offset continuation-args)
	   (dynamic-extent continuation continuation-args))
  nil)

(defmethod map-over-output-record-elements-containing-point*
	   ((record t) x y continuation
	    &optional (x-offset 0) (y-offset 0) &rest continuation-args)
  (declare (dynamic-extent continuation continuation-args))
  (apply #'map-over-output-record-elements-overlapping-region
	 record (make-point x y) continuation
	 x-offset y-offset continuation-args))

;; If some coordinate is relative to a given output-record, then
;; CONVERT-FROM-RELATIVE-TO-ABSOLUTE-COORDINATES returns an x,y offset
;; to be ADDED to any coordinates relative to OUTPUT-RECORD to give you
;; absolute coordinates.
(defun convert-from-relative-to-absolute-coordinates (stream output-record)
  (declare (values x-offset y-offset))
  (cond ((eql output-record (output-recording-stream-current-output-record-stack stream))
	 (let ((position (output-recording-stream-output-record-absolute-position stream)))
	   (values (point-x position) (point-y position))))
	((null (output-record-parent output-record))
	 (values 0 0))
	(t
	 (multiple-value-bind (x y)
	     (convert-from-relative-to-absolute-coordinates
	       stream (output-record-parent output-record))
	   (multiple-value-bind (our-x our-y) (output-record-position* output-record)
	     (values (+ our-x x) (+ our-y y)))))))

;; If some coordinate is in absolute coordinates, then
;; CONVERT-FROM-ABSOLUTE-TO-RELATIVE-COORDINATES returns an x,y offset
;; to be ADDED to any absolute coordinates to give you coordinates
;; relative to OUTPUT-RECORD.
(defun convert-from-absolute-to-relative-coordinates (stream output-record)
  (declare (values x-offset y-offset))
  (cond ((eql output-record (output-recording-stream-current-output-record-stack stream))
	 (let ((position (output-recording-stream-output-record-absolute-position stream)))
	   (values (- (point-x position)) (- (point-y position)))))
	((null (output-record-parent output-record))
	 (values 0 0))
	(t
	 (multiple-value-bind (x y)
	     (convert-from-absolute-to-relative-coordinates
	       stream (output-record-parent output-record))
	   (multiple-value-bind (our-x our-y) (output-record-position* output-record)
	     (values (- x our-x) (- y our-y)))))))

;; if ANCESTOR is an output-record with DESCENDANT as a descendant
;; output-record (member of the transitive closure of INFERIORS of
;; ancestor) then:  CONVERT-FROM-ANCESTOR-TO-DESCENDANT-COORDINATES
;; returns an x,y offset pair that can be ADDED to any coordinates
;; relative to ANCESTOR in order to get coordinates relative to
;; DESCENDANT.
(defun convert-from-ancestor-to-descendant-coordinates (ancestor descendant)
  (declare (values x-offset y-offset))
  (cond ((eql descendant ancestor)
	 (values 0 0))
	((null descendant)
	 (error "~S was not an ancestor of ~S" ancestor descendant))
	(t
	 (multiple-value-bind (x y)
	     (convert-from-ancestor-to-descendant-coordinates
	       ancestor (output-record-parent descendant))
	   (multiple-value-bind (our-x our-y) (output-record-position* descendant)
	     (values (- x our-x) (- y our-y)))))))

;; if ANCESTOR is an output-record with DESCENDANT as a descendant
;; output-record (member of the transitive closure of INFERIORS of
;; ancestor) then:  CONVERT-FROM-DESCENDANT-TO-ANCESTOR-COORDINATES
;; returns an x,y offset pair that can be ADDED to any coordinates
;; relative to DESCENDANT in order to get coordinates relative to
;; ANCESTOR.
(defun convert-from-descendant-to-ancestor-coordinates (descendant ancestor)
  (declare (values x-offset y-offset))
  (cond ((eql descendant ancestor)
	 (values 0 0))
	((null descendant)
	 (error "~s was not an ancestor of ~s" ancestor descendant))
	(t
	 (multiple-value-bind (x y)
	     (convert-from-descendant-to-ancestor-coordinates
	       (output-record-parent descendant) ancestor)
	   (multiple-value-bind (our-x our-y) (output-record-position* descendant)
	     (values (+ our-x x) (+ our-y y)))))))

(defclass displayed-output-record-element-mixin
	  (output-record-element-mixin displayed-output-record-element)
    ())

(defclass output-record-mixin
	  (output-record)			;:required-protocol output-record-element-mixin
     ;; old-elements is the list of >unmatched< output-records from last redisplay-pass.
     ;; if you implement your own find-inferior-output-record, and iff you match from
     ;; old-elements, you are required to remove the match from old-elements.
     ;; (probably by using decache-inferior-output-record).
     ((old-elements :initform nil :accessor output-record-old-elements)
      (generation-tick :initform 0 :initarg :generation-tick
		       :accessor output-record-generation-tick)))

(defmethod elements-never-overlap-p ((record output-record-mixin)) nil)

(defun with-output-record-internal (continuation stream record &optional abs-x abs-y)
  ;; Close the text record before and after, 
  (close-current-text-output-record stream)
  (let ((current-output-position
	  (output-recording-stream-output-record-absolute-position stream)))
    (unless abs-y
      (multiple-value-setq (abs-x abs-y)
	(stream-cursor-position* stream)))
    (letf-globally (((point-x current-output-position) abs-x)
		    ((point-y current-output-position) abs-y)
		    ((output-recording-stream-current-output-record-stack stream) record))
      (funcall continuation record)
      (multiple-value-bind (end-x end-y)
	  (stream-cursor-position* stream)
	(output-record-set-end-position*
	  record (- end-x abs-x)
		 (- end-y abs-y)))
      (close-current-text-output-record stream))))

;;; Rest of stuff started in clim-defs...
(defun construct-output-record-1 (type &rest init-args)
  (declare (dynamic-extent init-args))
  (let ((constructor (gethash type *output-record-constructor-cache*)))
    (if constructor
	(apply constructor init-args)
	(apply #'make-instance type init-args))))

#||
;;; A hash table associating vectors holding free output records.
;;; The idea is to not cons on allocate/free, but be fast.
(defvar *record-resource-table* (make-hash-table))
(defvar *use-record-resources* nil)

(defun allocate-record (type)
  (when *use-record-resources*
    (multiple-value-bind (record-vector found-p)
	(gethash type *record-resource-table*)
      (unless found-p
	(setq record-vector
	      (setf (gethash type *record-resource-table*)
		    (make-array 20 :fill-pointer 0))))
      (vector-pop record-vector))))

(defun free-record (record)
  (when *use-record-resources*
    (let ((type (class-name (class-of record))))
      (multiple-value-bind (record-vector found-p)
	  (gethash type *record-resource-table*)
	(unless found-p
	  (setq record-vector
		(setf (gethash type *record-resource-table*)
		      (make-array 20 :fill-pointer 0))))
	(setf (output-record-parent record) nil)
	(vector-push-extend record record-vector)))))
||#

(defun with-new-output-record-internal (continuation stream record-type constructor
					&rest init-args &key parent &allow-other-keys)
  (declare (dynamic-extent init-args))
  (with-rem-keywords (init-args init-args '(:parent))
    (let* ((current-output-record (output-recording-stream-current-output-record-stack stream))
	   (new-output-record (and (stream-redisplaying-p stream)
				   current-output-record
				   (apply #'find-inferior-output-record-internal
					  current-output-record record-type init-args))))
      (multiple-value-bind (cursor-x cursor-y)
	  (stream-cursor-position* stream)
	(multiple-value-bind (x y)
	    (multiple-value-bind (px py)
		(point-position*
		  (output-recording-stream-output-record-absolute-position stream))
	      (position-difference* cursor-x cursor-y px py))
	  (if new-output-record
	      (copy-display-state new-output-record nil)
	      (setq new-output-record
		    ;;--- Used to call ALLOCATE-RECORD, then initialize by
		    ;;--- setting the edges (or INITIALIZE-INSTANCE)
		    (if constructor
			(apply constructor
			       :x-position x :y-position y init-args)
		        (apply #'construct-output-record-1 record-type
			       :x-position x :y-position y init-args))))
	  (output-record-set-start-position* new-output-record x y)
	  (with-output-record-internal continuation stream new-output-record
				       cursor-x cursor-y)
	  (when (stream-redisplaying-p stream)
	    (recompute-contents-ok new-output-record))
	  ;; We set the parent after doing everything else so that calls
	  ;; to RECOMPUTE-CONTENTS-OK inside the dynamic extent of the
	  ;; continuation won't take forever.
	  (let ((parent (or parent
			    current-output-record
			    (output-recording-stream-output-record stream))))
	    (when parent (add-output-record-element parent new-output-record)))
	  ;; Whenever an output record gets added, make sure
	  ;; that the sheet's region is updated.
	  #+Silica
	  (let ((out-rec (output-recording-stream-output-record stream))
		(width (bounding-rectangle-width stream))
		(height (bounding-rectangle-height stream)))
	    (when out-rec
	      (multiple-value-bind (rl rt rr rb) 
		  (bounding-rectangle* out-rec)
		(when (or (< rl 0) (< width rr)
			  (< rt 0) (< height rb))
		  (stream-update-region stream (- rr rl) (- rb rt))))))
	  new-output-record)))))

(defun with-room-for-graphics-1 (stream record-type move-cursor continuation &key height)
  (let ((record
	  (with-output-recording-options (stream :draw-p nil :record-p t)
	    (with-first-quadrant-coordinates (stream)
	      (with-new-output-record (stream record-type)
		(funcall continuation stream))))))
    (multiple-value-bind (x y) (output-record-position* record)
      ;;--- Hey, there is something wierd going on here.  The problem is that
      ;;--- OUTPUT-RECORD-POSITION* and OUTPUT-RECORD-SET-POSITION* seem to obey
      ;;--- different coordinate system conventions.  Geez.
      (when height
	(incf y (- height (bounding-rectangle-height record))))
      (output-record-set-position* record x y))
    (tree-recompute-extent record)
    (replay record stream)
    (when move-cursor
      (move-cursor-beyond-output-record stream record))
    record))

(defun replay (record stream &optional region)
  (when (stream-draw-p stream)
    (multiple-value-bind (x-offset y-offset)
	(convert-from-relative-to-absolute-coordinates stream (output-record-parent record))
      ;; Output recording should be off, but let's be forgiving...
      (letf-globally (((stream-record-p stream) nil))
	(replay-1 record stream region x-offset y-offset)))))

;; Replay all the the elements of RECORD that overlap REGION.
(defmethod replay-1 ((record output-record-mixin) stream
		     &optional region (x-offset 0) (y-offset 0))
  ;;--- Doing things this way bypasses any REPLAY-1 methods supplied on 
  ;;--- non-standard classes that satisfy the output record protocol.
  ;;--- Too bad, this relative coordinates stuff is a disaster anyway.
  (labels ((replay-internal (record x-offset y-offset)
	     (if (output-record-p record)
		 (multiple-value-bind (xoff yoff) (output-record-position* record)
		   (map-over-output-record-elements-overlapping-region record region
		     #'replay-internal (- x-offset) (- y-offset)
		     (+ x-offset xoff) (+ y-offset yoff)))
	         (replay-1 record stream region x-offset y-offset))))
    (declare (dynamic-extent #'replay-internal))
    (replay-internal record x-offset y-offset)))

(defun move-cursor-beyond-output-record (stream record)
  (multiple-value-bind (x-offset y-offset)
      (convert-from-relative-to-absolute-coordinates
	stream (output-record-parent record))
    (with-bounding-rectangle* (left top right bottom) record
      (declare (ignore left top))
      (with-end-of-page-action (:allow stream)
	(stream-set-cursor-position*
	  stream
	  (+ right x-offset)
	  (- (+ bottom y-offset) (stream-line-height stream)))))))

(defmethod recompute-extent-for-changed-element ((record output-record-mixin) 
						 element
						 old-left old-top old-right old-bottom)
  ;; old edges are passed in parent's coordinate system because
  ;; their reference point may have changed.
  ;; (assert (element-completely-contained-within-extent-of record element))
  (with-slots (parent) record
    (with-bounding-rectangle* (left top right bottom) record
      ;; We must recompute the extent if the element is not completely contained
      ;; or if it used to "define" one of the old edges.
      ;; A picture would help, but we're not going to draw it here. :-)
      (multiple-value-bind (xoff yoff)
	  (convert-from-descendant-to-ancestor-coordinates record parent)
	(when (or (not (region-contains-offset-region-p record element xoff yoff))
		  (= old-left left)
		  (= old-top top)
		  (= old-right right)
		  (= old-bottom bottom))
	  (recompute-extent record))))))

(defmethod recompute-extent ((record output-record-mixin))
  (with-slots (parent) record
    (with-bounding-rectangle* (old-left old-top old-right old-bottom) record
      (let ((once nil)
	    (min-x 0) (min-y 0) (max-x 0) (max-y 0))
	(flet ((recompute-extent-of-element (element)
		 (with-bounding-rectangle* (left top right bottom) element
		   (cond (once
			  (minf min-x left)
			  (minf min-y top)
			  (maxf max-x right)
			  (maxf max-y bottom))
			 (t
			  (setq min-x left
				min-y top
				max-x right
				max-y bottom
				once  t))))))
	  (declare (dynamic-extent #'recompute-extent-of-element))
	  (map-over-output-record-elements record #'recompute-extent-of-element))
	(multiple-value-bind (xoff yoff)
	    (convert-from-descendant-to-ancestor-coordinates record parent)
	  (if once
	      (progn (assert (ltrb-well-formed-p min-x min-y max-x max-y))
		     (bounding-rectangle-set-edges
		       record
		       (+ min-x xoff) (+ min-y yoff)
		       (+ max-x xoff) (+ max-y yoff)))
	      ;; no inferiors.
	      (bounding-rectangle-set-edges record 0 0 0 0))
	  ;; Pass these coordinates in parent's coordinate system (I think)
	  (translate-positions xoff yoff
	    old-left old-top old-right old-bottom))
	(when parent
	  (recompute-extent-for-changed-element
	    parent record old-left old-top old-right old-bottom))))))

(defmethod recompute-extent-for-new-element ((record output-record-mixin) element)
  (with-slots (parent) record
    (with-bounding-rectangle* (left top right bottom) record
      (let ((old-left left)
	    (old-top top)
	    (old-right right)
	    (old-bottom bottom))
	(with-bounding-rectangle* (eleft etop eright ebottom) element
	  (multiple-value-bind (xoff yoff)
	      (convert-from-descendant-to-ancestor-coordinates record parent)
	    (translate-positions xoff yoff
	      eleft etop eright ebottom
	      ;; pass these coordinates in parent's coordinate system.
	      old-left old-top old-right old-bottom))
	  (cond ((= (output-record-element-count record) 1)
		 (bounding-rectangle-set-edges record eleft etop eright ebottom))
		(t (bounding-rectangle-set-edges record
						 (min left eleft) (min top etop)
						 (max right eright) (max bottom ebottom)))))
	(when parent
	  (recompute-extent-for-changed-element
	    parent record old-left old-top old-right old-bottom))))))

;;; This is for adjusting extents after a bunch of leaves have been moved.
(defmethod tree-recompute-extent ((record output-record-element-mixin))
  (with-bounding-rectangle* (old-left old-top old-right old-bottom) record
    (let ((parent (output-record-parent record)))
      (multiple-value-bind (xoff yoff)
	  (convert-from-descendant-to-ancestor-coordinates record parent)
	;; we must pass the old coordinates in the parent's coordinate system
	;; because tree-recompute-extent-1 may adjust the reference point.
	(translate-positions xoff yoff old-left old-top old-right old-bottom))
      (tree-recompute-extent-1 record)
      (when parent
	(recompute-extent-for-changed-element
	  parent record old-left old-top old-right old-bottom)))))

;;; Each implementation is required to provide output-record-elements.
;;; Maybe use "write-trampolines" or equivalent to define the protocol thusly.
;(defmethod output-record-elements ((record output-record-mixin))
;  (error "No specific method supplied."))

;;; Common to all implementations.
;;; ADD-OUTPUT-RECORD-ELEMENT assumes that ELEMENT's start-position and bounding 
;;; rectangle have already been normalized to RECORD's coordinate system.
(defmethod add-output-record-element :after ((record output-record-mixin) element) ;was :around
  (setf (output-record-parent element) record)
  (recompute-extent-for-new-element record element))

;;; Ditto.
(defmethod delete-output-record-element :after
	   ((record output-record-mixin) element &optional (errorp t))
  (declare (ignore errorp))
  (with-bounding-rectangle* (left top right bottom) element
    (multiple-value-bind (xoff yoff)
	(convert-from-descendant-to-ancestor-coordinates element record)
      (translate-positions xoff yoff left top right bottom)
      (recompute-extent-for-changed-element record element left top right bottom)))
  (setf (output-record-parent element) nil))	;in case other things are still pointing to it.

;;; Recurse down all inferiors returning them to the "resource" table.
#+++Ignore
(defmethod clear-output-record :before ((record output-record-mixin))
	   (free-output-record record))

;;; Invoked by CLEAR-OUTPUT-RECORD.
#+++Ignore
(defmethod free-output-record ((record output-record-element-mixin))
  (free-record record)
  (map-over-output-record-elements record #'free-output-record))

(defmethod clear-output-record :after ((record output-record-mixin))
  (bounding-rectangle-set-edges record 0 0 0 0))

;;; Mix this into an output history that's associated with a stream.
(defclass stream-output-history-mixin ()
    ((stream :initarg :stream :accessor output-history-stream))
  )

(defmethod bounding-rectangle-set-edges :after
	   ((history stream-output-history-mixin)
	    new-min-x new-min-y new-max-x new-max-y)
  ;; update the sheet's region rectangle.
  ;; for now, just call some high-level tool
  (let* ((sheet (slot-value history 'stream))
	 (viewport (ws::new-pane-viewport sheet)))
    (when viewport
      (multiple-value-bind (width height) (bounding-rectangle-size (sheet-region viewport))
	(setq new-min-x (min new-min-x 0)
	      new-min-y (min new-min-y 0)
	      new-max-x (max new-max-x width)
	      new-max-y (max new-max-y height))))
    (setf (sheet-region sheet)
	  (make-rectangle* new-min-x new-min-y new-max-x new-max-y))))


;;; Linear output records store their elements in a vector
(defclass linear-output-record (output-record-mixin output-record-element-mixin)
    ((elements :initform nil)
     (fill-pointer :initform 0 :type fixnum)))

(define-output-record-constructor linear-output-record (&key x-position y-position (size 5))
  :x-position x-position :y-position y-position :size size)

(defmethod initialize-instance :after ((record linear-output-record) 
				       &key (size 5))
  ;; probably want to save size away somewhere so that the
  ;; guy who actually makes the array can reference it...
  (declare (ignore size))
  ;; size defaults to very small to save space
  ;; most dependent classes will have supplied default-initargs with better
  ;; chosen default.
  (with-slots (elements fill-pointer) record
    ;; We run initialize-instance to re-initialize the record, so don't re-alloc the array
    (etypecase elements
      ((or null output-record-element) (setf elements nil))
      (array (setf fill-pointer 0)))))

;;; For debugging.
(defmethod output-record-elements ((record linear-output-record))
  (with-slots (elements fill-pointer) record
    (typecase elements
      (null nil)
      (array
	(let ((result (make-list fill-pointer)))
	  (replace result elements :end1 fill-pointer :end2 fill-pointer)
	  result))
      ;; It must be an OUTPUT-RECORD-ELEMENT
      (otherwise (list elements)))))

(defmethod output-record-element ((record linear-output-record) index)
  (with-slots (elements) record
    (svref elements index)))

(defmethod output-record-element-count ((record linear-output-record))
  (with-slots (elements fill-pointer) record
    (typecase elements
      (null 0)
      (array fill-pointer)
      ;; It must be an OUTPUT-RECORD-ELEMENT
      (otherwise 1))))

(defmethod clear-output-record ((record linear-output-record))
  (with-slots (elements fill-pointer) record
    (typecase elements
      (null nil)
      (array (setf fill-pointer 0))
      ;; It must be an OUTPUT-RECORD-ELEMENT
      (otherwise (setf elements nil)))))

(defmethod add-output-record-element ((record linear-output-record) element)
  (with-slots (elements fill-pointer) record
    (typecase elements
      (null
	(setf elements element))
      (array
	(multiple-value-setq (elements fill-pointer)
	  (simple-vector-push-extend element elements fill-pointer)))
      ;; It must be an OUTPUT-RECORD-ELEMENT
      (otherwise
	(let ((first elements))
	  (setf elements (make-array 5))
	  (setf fill-pointer 2)
	  (setf (svref elements 0) first)
	  (setf (svref elements 1) element))))))

(defmethod delete-output-record-element
	   ((record linear-output-record) element &optional (errorp t))
  (with-slots (elements fill-pointer) record
    (typecase elements
      (null (error "The element ~S was not found in ~S" element record))
      (array
	(let ((index (position element elements :end fill-pointer)))
	  (cond (index
		 (let ((new-fp (the fixnum (1- fill-pointer)))
		       (vector elements))
		   (declare (type simple-vector vector) (fixnum new-fp)
			    #+Genera (sys:array-register vector))
		   (unless (= (the fixnum index) new-fp)
		     ;; Shift the whole vector downward
		     (do ((i (the fixnum index) (1+ i)))
			 ((= i new-fp))
		       (declare (fixnum i) (optimize (speed 3) (safety 0)))
		       (setf (svref vector i) (svref vector (1+ i)))))
		   (setf fill-pointer new-fp)))
		(errorp
		 (error "The element ~S was not found in ~S" element record)))))
      ;; It must be an OUTPUT-RECORD-ELEMENT
      (otherwise
	(unless (eql elements element)
	  (error "The element ~S was not found in ~S" element record))
	(setf elements nil))))
  t)

(defmethod map-over-output-record-elements-overlapping-region
	   ((record linear-output-record) region continuation 
	    &optional (x-offset 0) (y-offset 0) &rest continuation-args)
  (declare (dynamic-extent continuation continuation-args))
  (declare (optimize (safety 0)))
  (with-slots (elements fill-pointer) record
    (typecase elements
      (null nil)
      (array
	(if (or (null region) (eql region +everywhere+))
	    (dovector (element elements :start 0 :end fill-pointer :simple-p t)
	      (apply continuation element continuation-args))
	  (with-bounding-rectangle* (left1 top1 right1 bottom1) region
	    (translate-positions x-offset y-offset left1 top1 right1 bottom1)
	    ;; Subtract out the record offset from the region, to make comparison fair
	    (multiple-value-bind (xoff yoff)
		(output-record-position* record)
	      (translate-positions (- xoff) (- yoff) left1 top1 right1 bottom1))
	    (dovector (element elements :start 0 :end fill-pointer :simple-p t)
	      (with-bounding-rectangle* (left2 top2 right2 bottom2) element
		(when (ltrb-overlaps-ltrb-p left1 top1 right1 bottom1
					    left2 top2 right2 bottom2)
		  (apply continuation element continuation-args)))))))
      (otherwise
	(if (or (null region) (eql region +everywhere+))
	    (apply continuation elements continuation-args)
	  (multiple-value-bind (xoff yoff)
	      (output-record-position* record)
	    (when (region-intersects-offset-region-p
		    elements region (- x-offset xoff) (- y-offset yoff))
	      (apply continuation elements continuation-args)))))))
  nil)

(defmethod map-over-output-record-elements-containing-point*
	   ((record linear-output-record) x y continuation 
	    &optional (x-offset 0) (y-offset 0) &rest continuation-args)
  (declare (dynamic-extent continuation continuation-args))
  (declare (optimize (safety 0)))
  (translate-positions x-offset y-offset x y)
  (with-slots (elements fill-pointer) record
    (typecase elements
      (null nil)
      (array
	(multiple-value-bind (xoff yoff)
	    (output-record-position* record)
	  (translate-positions (- xoff) (- yoff) x y))
	(dovector (element elements :start 0 :end fill-pointer :from-end t :simple-p t)
	  (with-bounding-rectangle* (left top right bottom) element
	    (when (ltrb-contains-point*-p left top right bottom x y)
	      (apply continuation element continuation-args)))))
      (otherwise
	(multiple-value-bind (xoff yoff) (output-record-position* record)
	  (when (offset-region-contains-point*-p elements xoff yoff x y)
	    (apply continuation elements continuation-args))))))
  nil)


;;; Text line output recording.  A stream which does text output creates one of
;;; these per line of text.  Lines are delimited by either #\RETURNs or by
;;; wrapping.  Line boundaries are not recalculated when window size is changed; a
;;; higher-level kind of output record is required for that (protocol not yet
;;; defined, but should be obvious).

;;; The string is a vector of those characters which were output on the current
;;; line.  The initial-text-style is the very first style which
;;; appeared on the line.  The current-text-style is the style of
;;; the last character which appeared on the line.  The baseline is used to
;;; determine where to draw the glyphs.

;;; The text-style-changes is an NCONCed list of conses of the form
;;; (style . position).  The characters between the beginning of the record
;;; and the first change are in the initial-text-style.  Unfortunately,
;;; you can't just use STREAM-WRITE-STRING-INTERNAL on the substring involved
;;; because that function returns in the middle when it encounters a character it
;;; can't deal with, such as tabs and non-graphic characters.  There should never
;;; be a #\Return character in a text-output-record-element.

(defclass text-output-record-element
	  (displayed-output-record-element-mixin)
    ((string :initarg :string)
     (wrapped-p :initform nil :initarg :wrapped-p)
     (ink :initarg :ink)))

(defclass styled-text-output-record-element
	  (text-output-record-element)
    ((initial-text-style :initform nil :initarg :initial-style)
     (text-style-changes :initform nil)
     (current-text-style :initform nil :initarg :current-style)
     (baseline :initform 0 :initarg :baseline :type fixnum)))

(define-constructor make-text-output-record-element
		    text-output-record-element (ink string)
		    :ink ink :string string)

(define-constructor make-styled-text-output-record-element
		    styled-text-output-record-element (ink string)
		    :ink ink :string string)

(define-constructor make-styled-text-output-record-element-1
		    styled-text-output-record-element
  (ink string wrapped-p style baseline)
  :ink ink :string string :wrapped-p wrapped-p
  :initial-style style :current-style style :baseline baseline)

(defun safe-slot-value (instance slot-name)
  (if (slot-boundp instance slot-name)
      (slot-value instance slot-name)
      "Unbound"))

(defmethod print-object ((object text-output-record-element) stream)
  (print-unreadable-object (object stream :type t :identity t)
    (format stream "~S /x ~A:~A y ~A:~A/"
	    (safe-slot-value object 'string)
	    (safe-slot-value object 'left)
	    (safe-slot-value object 'right)
	    (safe-slot-value object 'top)
	    (safe-slot-value object 'bottom))))

(defmethod output-record-unique-id ((text text-output-record-element))
  (slot-value text 'string))

(defmethod replay-1 ((record text-output-record-element) stream
		     &optional region (x-offset 0) (y-offset 0))
  (declare (ignore region))
  (let* ((string (slot-value record 'string))
	 (start 0)
	 (end (length string))
	 (text-style (medium-default-text-style stream))
	 (baseline (- (text-style-height text-style stream)
		      (text-style-descent text-style stream)))
	 (glyph-buffer (stream-output-glyph-buffer stream))
	 (color (slot-value record 'ink)))
    (declare (fixnum start end baseline))
    (#-Silica progn
     #+Silica silica:using-display-medium #+Silica (medium :clim stream)
     (macrolet
      ((do-it (end-position)
	 `(loop
	    (when (>= start ,end-position) (return))
	    (multiple-value-bind (write-char next-char-index
				  new-cursor-x new-baseline new-height font)
		(stream-scan-string-for-writing stream #+Silica medium 
						string start ,end-position
						text-style cursor-x
						most-positive-fixnum 
						glyph-buffer)
	      ;; --- need draw-glyphs, which will take a port-specific font object,
	      ;; as well as the :INK option.
	      #+Silica
	      (draw-text* medium string
			  cursor-x (the fixnum (+ cursor-y (- baseline new-baseline)))
			  :start start :end next-char-index :text-style text-style
			  :align-y :top
			  :ink color)

	      ;; GLYPH-BUFFER NIL => pass the string to the port-specific code.
	      #-Silica
	      (if glyph-buffer
		  (stream-write-string-internal
		    stream glyph-buffer 0 (the fixnum (- next-char-index start))
		    font color
		    cursor-x (the fixnum (+ cursor-y (- baseline new-baseline))))
		  (stream-write-string-internal
		    stream string start next-char-index
		    font color
		    cursor-x (the fixnum (+ cursor-y (- baseline new-baseline)))))
	      (setf cursor-x new-cursor-x start next-char-index)
	      (when write-char
		(cond ((eql write-char #\Tab)	;Only non-lozenged exception char?
		       (setf cursor-x (stream-next-tab-column stream cursor-x text-style)))
		      (t 
		       (multiple-value-bind (new-cursor-x new-cursor-y)
			   (stream-draw-lozenged-character
			     stream write-char cursor-x cursor-y new-baseline new-height
			     text-style most-positive-fixnum nil t)
			 (setf cursor-x new-cursor-x
			       cursor-y new-cursor-y))))
		(incf start))))))
      (multiple-value-bind (cursor-x cursor-y) (output-record-start-position* record)
	(declare (fixnum cursor-x cursor-y))
	(translate-fixnum-positions x-offset y-offset cursor-x cursor-y)
	(do-it end)
	#-Silica
	(when (slot-value record 'wrapped-p)
	  (draw-character-wrap-indicator
	    stream cursor-y (bounding-rectangle-height record) (stream-text-margin stream) nil)))))))

(defmethod replay-1 ((record styled-text-output-record-element) stream
		     &optional region (x-offset 0) (y-offset 0))
  (declare (ignore region))
  (let* ((string (slot-value record 'string))
	 (start 0)
	 (end (length string))
	 (text-style (slot-value record 'initial-text-style))
	 (baseline (slot-value record 'baseline))
	 (glyph-buffer (stream-output-glyph-buffer stream))
	 (color (slot-value record 'ink)))
    (declare (fixnum start end baseline))
    (#-Silica progn
     #+Silica silica:using-display-medium #+Silica (medium :clim stream)
     (macrolet
      ((do-it (end-position)
	 `(loop
	    (when (>= start ,end-position) (return))
	    (multiple-value-bind (write-char next-char-index
				  new-cursor-x new-baseline new-height font)
		(stream-scan-string-for-writing stream #+Silica medium string
						start ,end-position
						text-style cursor-x
						most-positive-fixnum 
						glyph-buffer)
	      #-Silica
              (if glyph-buffer
		  (stream-write-string-internal
		    stream glyph-buffer 0 (the fixnum (- next-char-index start))
		    font color
		    cursor-x (the fixnum (+ cursor-y (- baseline new-baseline))))
                  (stream-write-string-internal
		    stream string start next-char-index
		    font color
		    cursor-x (the fixnum (+ cursor-y (- baseline new-baseline)))))
	      ;; --- need draw-glyphs, which will take a port-specific font object,
	      ;; as well as the :INK option.
	      #+Silica
	      (draw-text* medium string
			  cursor-x (the fixnum (+ cursor-y (- baseline new-baseline)))
			  :start start :end next-char-index :text-style text-style
			  :align-y :top
			  :ink color)
	      (setf cursor-x new-cursor-x start next-char-index)
	      (when write-char
		(cond ((eql write-char #\Tab)	;Only non-lozenged exception char?
		       (setf cursor-x (stream-next-tab-column stream cursor-x text-style)))
		      (t 
		       (multiple-value-bind (new-cursor-x new-cursor-y)
			   (stream-draw-lozenged-character
			     stream write-char cursor-x cursor-y new-baseline new-height
			     text-style most-positive-fixnum nil t)
			 (setf cursor-x new-cursor-x
			       cursor-y new-cursor-y))))
		(incf start))))))
      (multiple-value-bind (cursor-x cursor-y) (output-record-start-position* record)
	(translate-positions x-offset y-offset cursor-x cursor-y)
	(dolist (text-style-change (slot-value record 'text-style-changes))
	  (let ((new-text-style (car text-style-change))
		(change-position (cdr text-style-change)))
	    (do-it change-position)
	    (setf text-style new-text-style
		  start change-position)))
	(do-it end)
	#-Silica
	(when (slot-value record 'wrapped-p)
	  (draw-character-wrap-indicator
	    stream cursor-y (bounding-rectangle-height record) (stream-text-margin stream) nil)))))))

;;; #-Silica --- doughty removed this #-Silica on 1/4/92 because he couldn't figure out
;;;          --- why it should only be done in non-Silica
(defmethod bounding-rectangle-set-edges :around
	   ((record text-output-record-element) new-left new-top new-right new-bottom)
  (declare (ignore new-left new-top new-right new-bottom))
  (let ((parent (output-record-parent record)))
    (if (not (null parent))
	(with-bounding-rectangle* (old-left old-top old-right old-bottom) record
	  (multiple-value-bind (xoff yoff)
	      (convert-from-descendant-to-ancestor-coordinates record parent)
	    (translate-positions xoff yoff old-left old-top old-right old-bottom)
	    (call-next-method)
	    (recompute-extent-for-changed-element parent record
						  old-left old-top old-right old-bottom)))
	(call-next-method))))

(defmethod add-string-output-to-text-record ((record text-output-record-element)
					     text-string start end text-style
					     new-width new-height new-baseline)
  (declare (ignore text-style new-baseline))
  (declare (fixnum start end))
  (when (>= start end)
    (return-from add-string-output-to-text-record))
  (let* ((count (the fixnum (- end start)))
	 (string (prepare-text-record-for-appending record count nil))
	 (fill-pointer (fill-pointer string)))
    (multiple-value-bind (width height) (bounding-rectangle-size record)
      (setf (fill-pointer string) (the fixnum (+ fill-pointer count)))
      (replace string text-string :start1 fill-pointer :start2 start :end2 end)
      (incf width new-width)
      (maxf height new-height)
      (bounding-rectangle-set-size record width height))))

(defmethod add-string-output-to-text-record ((record styled-text-output-record-element)
					     text-string start end text-style
					     new-width new-height new-baseline)
  (declare (fixnum start end))
  (when (>= start end)
    (return-from add-string-output-to-text-record))
  (let* ((count (the fixnum (- end start)))
	 (string (prepare-text-record-for-appending record count text-style))
	 (fill-pointer (fill-pointer string)))
    (multiple-value-bind (width height) (bounding-rectangle-size record)
      (setf (fill-pointer string) (the fixnum (+ fill-pointer count)))
      (replace string text-string :start1 fill-pointer :start2 start :end2 end)
      (incf width new-width)
      (maxf height new-height)
      (maxf (slot-value record 'baseline) new-baseline)
      (bounding-rectangle-set-size record width height))))

(defmethod add-character-output-to-text-record ((record text-output-record-element)
						character text-style
						new-width new-height new-baseline)
  (declare (ignore text-style new-baseline))
  (let* ((string (prepare-text-record-for-appending record 1 nil))
	 (fill-pointer (fill-pointer string)))
    (multiple-value-bind (width height) (bounding-rectangle-size record)
      (setf (fill-pointer string) (1+ fill-pointer)
	    (aref string fill-pointer) character)
      (incf width new-width)
      (maxf height new-height)
      (bounding-rectangle-set-size record width height))))
  
(defmethod add-character-output-to-text-record ((record styled-text-output-record-element)
						character text-style
						new-width new-height new-baseline)
  (let* ((string (prepare-text-record-for-appending record 1 text-style))
	 (fill-pointer (fill-pointer string)))
    (multiple-value-bind (width height) (bounding-rectangle-size record)
      (setf (fill-pointer string) (1+ fill-pointer)
	    (aref string fill-pointer) character)
      (incf width new-width)
      (maxf height new-height)
      (maxf (slot-value record 'baseline) new-baseline)
      (bounding-rectangle-set-size record width height))))
  
(defmethod prepare-text-record-for-appending
    ((record text-output-record-element) space-needed style)
  (declare (fixnum space-needed))
  (declare (ignore style))
  (let* ((string (slot-value record 'string))
	 (fill-pointer (fill-pointer string)))
    (declare (fixnum fill-pointer))
    (when (> (the fixnum (+ fill-pointer space-needed)) (array-dimension string 0))
      (setf string (adjust-array string (the fixnum (+ fill-pointer space-needed 16))))
      (setf (slot-value record 'string) string))
    string))

(defmethod prepare-text-record-for-appending
    ((record styled-text-output-record-element) space-needed style)
  (declare (fixnum space-needed))
  (with-slots (initial-text-style current-text-style
	       text-style-changes baseline) record
    (let* ((string (slot-value record 'string))
	   (fill-pointer (fill-pointer string)))
      (unless (eql style current-text-style)
	(if (null initial-text-style)
	    (setf initial-text-style style)
	    (let ((change-record (cons style fill-pointer)))
	      (setf text-style-changes 
		    (nconc text-style-changes (list change-record)))))
	(setf current-text-style style))
      (when (> (the fixnum (+ fill-pointer space-needed)) (array-dimension string 0))
	(setf string (adjust-array string (the fixnum (+ fill-pointer space-needed 16))))
	(setf (slot-value record 'string) string))
      string)))

(defun text-recompute-contents-id-test (id1 id2)
  (or (eql id1 id2)
      (and (stringp id2)
	   (string= id1 id2))))

;; We don't do a WITH-NEW-OUTPUT-RECORD-INTERNAL for TEXT-OUTPUT-RECORD-ELEMENTs.
;; However, CLOSE-CURRENT-TEXT-OUTPUT-RECORD does a RECOMPUTE-CONTENTS-OK, too.
(defmethod recompute-contents-ok ((text text-output-record-element))
  (with-slots (string wrapped-p) text
    (let* ((output-record (output-record-parent text))
	   (match (and output-record
		       (find-inferior-output-record
			 output-record t 'text-output-record-element
			 :unique-id string :id-test #'text-recompute-contents-id-test))))
      (when match
	;; The old extent is a copy of MATCH's bounding rectangle
	(setf (output-record-old-extent text) (bounding-rectangle match))
	(when (and (bounding-rectangle-size-equal match text)
		   (eql wrapped-p (slot-value match 'wrapped-p))
		   (eql (class-of text) (class-of match)))
	  (setf (output-record-contents-ok text) t)
	  ;; make sure that old bounding-rect is the same relative position from
	  ;; old-start-position as the bounding-rect is from start-position
	  (multiple-value-bind (delta-x delta-y)
	      (multiple-value-bind (ex ey) (bounding-rectangle-position* text)
		(multiple-value-bind (sx sy) (output-record-start-position* text)
		  (position-difference* ex ey sx sy)))
	    (multiple-value-bind (old-start-x old-start-y)
		(multiple-value-bind (px py) (bounding-rectangle-position* match)
		  (position-difference* px py delta-x delta-y))
	      (output-record-set-old-start-position* text old-start-x old-start-y))))))))

(defmethod recompute-contents-ok ((text styled-text-output-record-element))
  (with-slots (string wrapped-p initial-text-style current-text-style text-style-changes)
	      text
    (let* ((output-record (output-record-parent text))
	   (match (and output-record
		       (find-inferior-output-record
			 output-record t 'styled-text-output-record-element
			 :unique-id string :id-test #'text-recompute-contents-id-test))))
      (when match
	;; The old extent is a copy of MATCH's bounding rectangle
	(setf (output-record-old-extent text) (bounding-rectangle match))
	;; --- maybe make a method out of this to get efficient slot access?
	(when (and (bounding-rectangle-size-equal match text)
		   (eql wrapped-p (slot-value match 'wrapped-p))
		   (eql (class-of text) (class-of match))
		   (eql initial-text-style
			(slot-value match 'initial-text-style))
		   (eql current-text-style
			(slot-value match 'current-text-style))
		   (equal text-style-changes
			  (slot-value match 'text-style-changes)))
	  (setf (output-record-contents-ok text) t)
	  ;; make sure that old bounding-rect is the same relative position from
	  ;; old-start-position as the bounding-rect is from start-position
	  (multiple-value-bind (delta-x delta-y)
	      (multiple-value-bind (ex ey) (bounding-rectangle-position* text)
		(multiple-value-bind (sx sy) (output-record-start-position* text)
		  (position-difference* ex ey sx sy)))
	    (multiple-value-bind (old-start-x old-start-y)
		(multiple-value-bind (px py) (bounding-rectangle-position* match)
		  (position-difference* px py delta-x delta-y))
	      (output-record-set-old-start-position* text old-start-x old-start-y))))))))

(defun find-text-baseline (record stream)
  ;; This finds the lowest baseline of the text in RECORD, which will be slower than, say,
  ;; the first baseline but more likely to look good with misaligned things.
  (let ((baseline 0)
	(style (medium-default-text-style stream)))
    (labels ((find-or-recurse (element y-offset)
	       (typecase element
		 (styled-text-output-record-element
		   (maxf baseline (+ y-offset (slot-value element 'baseline))))
		 (text-output-record-element
		   (maxf baseline
			 (+ y-offset (- (text-style-height style stream)
						    (text-style-descent style stream)))))
		 (t
		   (multiple-value-bind (xoff yoff) (output-record-position* element)
		     (declare (ignore xoff))
		     (map-over-output-record-elements element
		       #'find-or-recurse 0 0 (+ yoff y-offset)))))))
      (declare (dynamic-extent #'find-or-recurse))
      (find-or-recurse record 0))
    baseline))


;;; Defclass of BASIC-OUTPUT-RECORDING, etc. is in STREAM-CLASS-DEFS
(defmethod initialize-instance :after ((stream basic-output-recording) &rest args)
  (declare (ignore args))
  (with-slots (output-record) stream
    ;; --- our basic-output-recording expects extended output...
    (multiple-value-bind (x y) (stream-cursor-position* stream)
      ;; I don't understand why the output record's initial position was set to
      ;; some untransformed "viewport" coordinate.  The cursor position is the
      ;; right place, no?
      (output-record-set-position* output-record x y)
      #+ignore
      (multiple-value-bind (wx wy) (stream-untransform-point x y stream)
	(bounding-rectangle-set-position* output-record wx wy)))))

(defmethod clear-output-history ((stream basic-output-recording))
  (when (output-recording-stream-output-record stream)
    (clear-output-record (output-recording-stream-output-record stream)))
  (setf (output-recording-stream-text-output-record stream) nil)
  (setf (output-recording-stream-highlighted-presentation stream) nil)
  #+Silica
  (progn
    (stream-update-region stream 0 0)
    (ws::scroll-home stream)))

(defmethod add-output-record ((stream basic-output-recording) element)
  (with-slots (output-record current-output-record-stack) stream
    (let ((the-output-record (or current-output-record-stack output-record)))
      (add-output-record-element the-output-record element)
      ;;(add-output-record-element output-record element)
      ;; --- We should optimize when the new element doesn't change the
      ;; size of the history
      #+Silica
      (let ((width (bounding-rectangle-width stream))
	    (height (bounding-rectangle-height stream)))
	(with-bounding-rectangle* (rl rt rr rb) the-output-record
	  (when (or (< rl 0) (< width rr)
		    (< rt 0) (< height rb))
	    (stream-update-region stream (- rr rl) (- rb rt))))))))

(defmethod output-recording-stream-replay ((stream basic-output-recording) &optional region)
  (when (stream-draw-p stream)
    (with-slots (output-record text-output-record-element record-p) stream
      (when (or output-record text-output-record-element)
	(letf-globally ((record-p nil))
	  (when output-record
	    (replay-1 output-record stream region 0 0))
	  (when text-output-record-element
	    (replay-1 text-output-record-element stream region 0 0)))))))

(defun erase-output-record (output-record stream)	;--- specialize on stream?
  (multiple-value-bind (xoff yoff)
      (convert-from-relative-to-absolute-coordinates 
	;; --- I'm certainly going to forget to use the PARENT at some point!
	stream (output-record-parent output-record))
    (with-bounding-rectangle* (left top right bottom) output-record
      (with-output-recording-options (stream :record-p nil)
	#-Silica
	(if (or (= left right) (= top bottom))
	    ;; Handle specially, for a line is wider than a rectangle of zero width or height
	    (draw-line-internal stream xoff yoff
				left top right bottom
				+background+ nil)
	    (draw-rectangle-internal stream xoff yoff
				     left top right bottom
				     +background+ nil))
	#+Silica
	;; --- might need offsets?
	(draw-rectangle* 
	  stream
	  (+ left xoff) (+ top yoff)
	  (+ right yoff) (+ bottom yoff)
	  :ink +background+ :filled T))))
  (when (output-record-parent output-record)
    (delete-output-record-element (output-record-parent output-record) output-record))
  ;; Use the output record itself as the replay region, and replay
  ;; the stuff that might have been obscured by the erased output
  (frame-replay *application-frame* stream output-record))

(defmethod with-output-recording-options-internal ((stream basic-output-recording)
						   draw-p record-p continuation)
  (letf-globally (((stream-record-p stream) record-p)
		  ((stream-draw-p stream) draw-p))
    (funcall continuation)))

(defmethod get-text-output-record ((stream basic-output-recording) style)
  (let ((default-style (medium-default-text-style stream)))
    (let ((record (output-recording-stream-text-output-record stream)))
      (when record
	;; If we're changing styles mid-stream, need to convert this
	;; text record to the more expensive form
	(when (and (not (eq style default-style))
		   (not (typep record 'styled-text-output-record-element)))
	  (setq record (stylize-text-output-record record default-style stream)))
	(return-from get-text-output-record record)))
    (let* ((string (make-array 16 :element-type 'extended-char	;--- 16?
				  :fill-pointer 0 :adjustable t))
	   (record (if (not (eq style default-style))
		       (make-styled-text-output-record-element (medium-ink stream) string)
		       (make-text-output-record-element (medium-ink stream) string))))
      (setf (output-recording-stream-text-output-record stream) record)
      (multiple-value-bind (abs-x abs-y)
	  (point-position*
	    (output-recording-stream-output-record-absolute-position stream))
	(multiple-value-bind (cx cy) (stream-cursor-position* stream)
	  ;; output-record-set-start-position
	  (output-record-set-start-position*
	    record (- cx abs-x) (- cy abs-y))))
      ;; Moved to close-current-text-output-record, since we don't need this thing
      ;; in the history until then.  This should save an extra recompute-extent call
      ;; (one in here, one when the string is added).
      ;; (add-output-record stream record)
      record)))

;; The cost of stylizing an existing record is actually fairly low, and we
;; don't do it very often, because of the optimization in GET-TEXT-OUTPUT-RECORD
;; that creates a stylized record as early as possible.
(defmethod stylize-text-output-record ((record text-output-record-element) style stream)
  (with-slots (ink string wrapped-p
	       left top right bottom start-x start-y end-x end-y) record
    (let ((new-record (make-styled-text-output-record-element-1
			ink string wrapped-p
			style (- (text-style-height style stream)
				 (text-style-descent style stream)))))
      (with-slots ((new-left left) (new-top top) (new-right right) (new-bottom bottom)
		   (new-sx start-x) (new-sy start-y) (new-ex end-x) (new-ey end-y)
		   (new-wrapped-p wrapped-p)) new-record
	(setq new-left left
	      new-top top
	      new-right right
	      new-bottom bottom
	      new-sx start-x
	      new-sy start-y
	      new-ex end-x
	      new-ey end-y))
      (setf (output-recording-stream-text-output-record stream) new-record)
      new-record)))

;;; The following two are only called when STREAM-RECORD-P is true and the
;;; characters are printable (see CHARACTER-DRAWING.LISP).
(defmethod add-string-output-to-output-record ((stream basic-output-recording)
					       string start end text-style
					       width height baseline)
  (declare (fixnum start end))
  (when (< start end)
    (let ((record (get-text-output-record stream text-style)))
      (add-string-output-to-text-record record string start end text-style
					width height baseline))))

(defmethod add-character-output-to-output-record ((stream basic-output-recording)
						  character text-style
						  width height baseline)
  (let ((record (get-text-output-record stream text-style)))
    (add-character-output-to-text-record record character text-style
					 width height baseline)))

(defmethod close-current-text-output-record ((stream basic-output-recording)
					     &optional wrapped)
  ;; It's faster to access the slot directly instead of going through 
  ;; OUTPUT-RECORDING-STREAM-TEXT-OUTPUT-RECORD
  (let ((text-record (slot-value stream 'text-output-record-element)))
    (when text-record
      (when wrapped
	(setf (slot-value text-record 'wrapped-p) t))
      (add-output-record stream text-record)
      (when (stream-redisplaying-p stream)
	(recompute-contents-ok text-record))
      (setf (slot-value stream 'text-output-record-element) nil))))

(defmethod stream-force-output :after ((stream basic-output-recording))
  (close-current-text-output-record stream))

(defmethod stream-finish-output :after ((stream basic-output-recording))
  (close-current-text-output-record stream))

;; When setting cursor position, have to dump old text record.
;; This is necessary in order to capture the correct cursor position in
;; text output records.  If we did not close the current text record,
;; a sequence such as WRITE-STRING/SET-CURSORPOS/WRITE-STRING would
;; create only a single output record, and intervening whitespace would
;; be lost if the two WRITE-STRINGs took place on the same line.
(defmethod stream-set-cursor-position* :before ((stream basic-output-recording) x y)
  (declare (ignore x y))
  (close-current-text-output-record stream))

;; This gets used to reposition the cursor when drawing text.  We need to
;; close the text output record when there was a line wrap, but not when
;; we are simply incrementing the cursor beyond the just-written glyph.
(defmethod stream-set-cursor-position*-internal :before ((stream basic-output-recording) x y)
  (declare (ignore x))
  (multiple-value-bind (old-x old-y) (stream-cursor-position* stream)
    (declare (ignore old-x))
    (unless (eql y old-y)
      (close-current-text-output-record stream))))

;; Copy just the text from the window to the stream.  If REGION is supplied,
;; only the text overlapping that region is copied.
;; This loses information about text styles, presentations, and graphics, and
;; doesn't deal perfectly with tab characters and changing baselines.
(defun copy-textual-output-history (window stream &optional region)
  (let* ((char-width (stream-character-width window #\space))
	 (line-height (stream-line-height window))
	 (history (output-recording-stream-output-record window))
	 (array (make-array (ceiling (bounding-rectangle-height history) line-height)
			    :fill-pointer 0 :adjustable t :initial-element nil)))
    (labels ((collect (record x-offset y-offset)
	       (multiple-value-bind (start-x start-y)
		   (output-record-start-position* record)
		 (translate-positions x-offset y-offset start-x start-y)
		 (when (typep record 'text-output-record-element)
		   (vector-push-extend (list* start-y start-x (slot-value record 'string))
				       array))
		 (map-over-output-record-elements-overlapping-region
		   record region #'collect
		   (- x-offset) (- y-offset) start-x start-y))))
      (declare (dynamic-extent #'collect))
      (collect history 0 0))
    (sort array #'(lambda (r1 r2)
		    (or (< (first r1) (first r2))
			(and (= (first r1) (first r2))
			     (< (second r1) (second r2))))))
    (let ((current-x 0)
	  (current-y (first (aref array 0))))
      (dotimes (i (fill-pointer array))
	(let* ((item (aref array i))
	       (y (pop item))
	       (x (pop item)))
	  (unless (= y current-y)
	    (dotimes (j (round (- y current-y) line-height))
	      #-excl (declare (ignore j))
	      (terpri stream)
	      (setq current-x 0))
	    (setq current-y y))
	  (unless (= x current-x)
	    (dotimes (j (round (- x current-x) char-width))
	      #-excl (declare (ignore j))
	      (write-char #\space stream))
	    (setq current-x x))
	  (write-string item stream)
	  (incf current-x (stream-string-width window item)))))))


;;; New class sheet-output-recording is for intermediary methods that can exist
;;; only when both the basic Silica sheet (window) protocol and the output recording
;;; protocol are mixed together.

;;;--- Move to stream-class-defs
(defclass sheet-output-recording () ())

;;; This method should cover a multitude of sins.
#+Silica
(defmethod silica:handle-repaint :after 
	   ((stream sheet-output-recording) region &key)
	   ;;--- Who should establish the clipping region?
	   ;; Who should clear the region?
	   (output-recording-stream-replay stream region))

;;; For Silica
;;;--- Consider these old methods on a case-by-case basis to see if the
;;; general handle-repaint method subsumes them.

;;; --- should merge our process-update-region with handle-repaint
;;; Do we use it anywhere where Silica isn't generating handle-repaint?

;;; Mix in window-output-recording when you have mixed together
;;; something supporting the window protocol and something supporting
;;; the output recording protocol.
#-Silica
(progn

(defmethod window-process-update-region :around ((stream window-output-recording))
  (let ((update-region (slot-value stream 'update-region)))
    (when update-region
      (with-output-recording-options (stream :draw-p t :record-p nil)
	(let ((highlighted-presentation (slot-value stream 'highlighted-presentation)))
	  (when highlighted-presentation
	    (highlight-output-record stream highlighted-presentation :unhighlight))
	  (call-next-method)
	  (dolist (region update-region)
	    (with-clipping-region (stream region)
	      (frame-replay *application-frame* stream region)))
	  (when highlighted-presentation
	    (highlight-output-record stream highlighted-presentation :highlight))))
      (window-flush-update-region stream))))

;;;--- We need some version of this code to do the area copying.
(defmethod window-set-viewport-position* :around ((stream window-output-recording)
						  new-x new-y)
  (declare (ignore new-x new-y))
  (with-bounding-rectangle* (left top right bottom) (window-viewport stream)
    (call-next-method)
    ;; now replay
    (with-bounding-rectangle* (nl nt nr nb) (window-viewport stream)
      (cond
	;; if some of the stuff that was previously on display is still on display
	;; bitblt it into the proper place and redraw the rest.
	((ltrb-overlaps-ltrb-p left top right bottom
			       nl nt nr nb)
	 ;; move the old stuff to the new position
	 (window-shift-visible-region stream left top right bottom
				      nl nt nr nb)
	 (window-process-update-region stream))
	;; otherwise, just redraw the whole visible viewport
	;; Adjust for the left and top margins by hand so clear-area doesn't erase
	;; the margin components.
	(t (multiple-value-bind (ml mt) (window-margins stream)
	     (declare (fixnum ml mt))
	     (multiple-value-bind (vw vh) (window-inside-size stream)
	       (declare (fixnum vw vh))
	       (window-clear-area stream
				  ml mt (the fixnum (+ ml vw)) (the fixnum (+ mt vh)))))
	   (frame-replay *application-frame* stream (window-viewport stream)))))))

(defmethod window-refresh :after ((stream window-output-recording))
  ;; don't bother me, it takes too long and is useless since
  ;; we'll refresh this again when it eventually becomes visible
  (when (window-drawing-possible stream)
    (frame-replay *application-frame* stream (window-viewport stream))
    (let ((text-record (output-recording-stream-text-output-record stream)))
      (when text-record (replay text-record stream)))
    (redisplay-decorations stream)))

;;; I don't think that this is needed.
(defmethod window-note-size-or-position-change :after ((stream window-output-recording)
						       left top right bottom)
  (declare (ignore left top right bottom))
  #+Ignore
  (when (window-visibility stream)
    (window-refresh stream)))

;;; --- Define Silica version of this.
(defmethod window-clear :before ((stream window-output-recording))
  (clear-output-history stream))

) ; end of #-Silica PROGN


;;; Genera compatibility

#+Genera
(defmethod stream-compatible-output-as-presentation
	   ((stream basic-output-recording)
	    continuation xstream
	    &key (object nil) (type t) single-box &allow-other-keys)
  (dw:with-type-decoded (type-name nil pr-args) type
    (if (or (null type)
	    (and (eq type-name 'sys:expression)
		 (not (getf pr-args :escape *print-escape*))
		 (stringp object)))
	(funcall continuation xstream)
        (multiple-value-bind (object clim-type changed-p)
	    (dw-type-to-clim-type object type)
	  (if changed-p
	      (with-output-as-presentation (:stream xstream
					    :object object
					    :type clim-type
					    :single-box single-box)
		(funcall continuation xstream))
	      (funcall continuation xstream))))))

#+Genera
(defmethod stream-compatible-output-as-presentation-1
	   ((stream basic-output-recording)
	    continuation continuation-args
	    &key (object nil) (type t) single-box &allow-other-keys)
  (dw:with-type-decoded (type-name nil pr-args) type
    (if (or (null type)
	    (and (eq type-name 'sys:expression)
		 (not (getf pr-args :escape *print-escape*))
		 (stringp object)))
	(apply continuation continuation-args)
        (multiple-value-bind (object clim-type changed-p)
	    (dw-type-to-clim-type object type)
	  (if changed-p
	      (with-output-as-presentation (:stream stream
					    :object object
					    :type clim-type
					    :single-box single-box)
		(apply continuation continuation-args))
	      (apply continuation continuation-args))))))

