;;; -*- Mode: LISP; Syntax: Common-lisp; Package: CLIM-INTERNALS; Base: 10; Lowercase: Yes -*-

;; $fiHeader: interactive-defs.lisp,v 1.4 91/03/26 12:48:08 cer Exp $

(in-package "CLIM-INTERNALS")

"Copyright (c) 1990, 1991 Symbolics, Inc.  All rights reserved.
 Portions copyright (c) 1988, 1989, 1990 International Lisp Associates."

;; For communication through parsers to lower levels.
;; Later, clever use of macrolet can replace this.
#||
;;; --- These are in STREAM/STREAM-DEFS for some reason.  I dunno why.  12/23/91 doughty
(defvar *input-wait-test* nil)
(defvar *input-wait-handler* nil)
(defvar *pointer-button-press-handler* nil)

;;; ---This is gesture-based now
(defparameter *abort-characters* '(#+Genera #\Abort
				   #+Cloe-Runtime #\Escape
				   #+Lucid #\control-\z
				   #+excl #\c-\z))

;;; These are in stream/stream-input-protocol, and gesture-based.
(defvar *accelerator-characters* nil)
(defvar *accelerator-numeric-argument* nil)
||#

;; Activation characters terminate the entire input line.  They are usually
;; non-printing characters such as Return or End.
(defvar *activation-characters* nil)

;; Until we have a real per-implementation key table, we don't
;; know whether the implementation puts #\Newline or #\Return on the
;; key marked "Return".
(defvar *standard-activation-characters* '(#+Genera #\End #\Newline #\Return))

(defmacro with-activation-characters ((additional-characters &key override) &body body)
  (when (characterp additional-characters)
    (setq additional-characters `'(,additional-characters)))
  `(with-stack-list* (*activation-characters*
		       ,additional-characters
		       ,(cond ((constantp override)
			       (if (null override) '*activation-characters* nil))
			      (t
			       `(unless ,override *activation-characters*))))
     ,@body))

(defun activation-character-p (character)
  (dolist (set *activation-characters*)
    (when (if (listp set)
	      (member character set :test #'equal)
	      (funcall set character))
      (return-from activation-character-p t))))


;; Blip characters terminate a field in an input line.  They are usually
;; printing characters such as Space or Tab
(defvar *blip-characters* nil)

(defmacro with-blip-characters ((additional-characters &key override) &body body)
  (when (characterp additional-characters)
    (setq additional-characters `'(,additional-characters)))
  `(with-stack-list* (*blip-characters*
		       ,additional-characters
		       ,(cond ((constantp override)
			       (if (null override) '*blip-characters* nil))
			      (t
			       `(unless ,override *blip-characters*))))
     ,@body))

(defun blip-character-p (character)
  (dolist (set *blip-characters*)
    (when (if (listp set)
	      (member character set :test #'equal)
	      (funcall set character))
      (return-from blip-character-p t))))


;;; Reading and writing of tokens

(defparameter *quotation-character* #\")

;; READ-TOKEN reads characters until it encounters an activation character,
;; a blip character, or something else (like a mouse click).
(defun read-token (stream &key input-wait-handler pointer-button-press-handler click-only)
  (with-temporary-string (string :length 50 :adjustable t)
    (let* ((gesture nil)
	   (quote-seen nil)
	   (old-blip-chars *blip-characters*)
	   (*blip-characters* *blip-characters*))
      (flet ((return-token (&optional unread)
	       (when unread
		 (unread-gesture unread :stream stream))
	       (when (and (activation-character-p unread)
			  (interactive-stream-p stream))
		 (rescan-for-activation stream))
	       (return-from read-token
		 (values (evacuate-temporary-string string)))))
	(loop
	  (setq gesture
		(read-gesture :stream stream
			      :input-wait-handler
			        (or input-wait-handler
				    *input-wait-handler*)
			      :pointer-button-press-handler
			        (or pointer-button-press-handler
				    *pointer-button-press-handler*)))
	  (cond ((and click-only
		      (not (typep gesture #+Silica 'button-press-event
					  #-Silica 'pointer-button-event)))
		 (beep stream))
		((typep gesture #+Silica 'button-press-event
				#-Silica 'pointer-button-event)
		 ;; No need to do anything, since this should have been handled
		 ;; in the presentation type system already
		 )
		((characterp gesture)
		 (cond ((and (zerop (fill-pointer string))
			     (eql gesture *quotation-character*))
			(setq quote-seen t)
			(setq *blip-characters* nil))
		       ((and quote-seen
			     (eql gesture *quotation-character*))
			(setq quote-seen nil)
			(setq *blip-characters* old-blip-chars))
		       ((activation-character-p gesture)
			(return-token gesture))
		       ((blip-character-p gesture)
			;; ditto?
			(return-token gesture))
		       ((ordinary-char-p gesture)
			(vector-push-extend gesture string)
			;;--- haven't updated WRITE-CHAR yet
			#+ignore (write-char gesture stream))
		       (t (beep stream))))
		(t (return-token gesture))))))))

(defun write-token (token stream &key acceptably)
  (cond ((and acceptably (some #'blip-character-p token))
	 (write-char *quotation-character* stream)
	 (write-string token stream)
	 (write-char *quotation-character* stream))
	(t
	 (write-string token stream))))


;;; Input editor macros

;; The collected numeric argument, not fully implemented
(defvar *numeric-argument* nil)

;; The kill ring
(defvar *kill-ring*)
(defvar *kill-ring-application* nil)

;; Used for passing information from ACCEPT through the input editor to here
(defvar *presentation-type-for-yanking* nil)

;; WITH-INPUT-EDITING simply encapsulates the stream and sets up an editing
;; context that allows rescanning, etc.
(defmacro with-input-editing ((&optional stream
			       &key input-sensitizer initial-contents
				    (class `'input-editing-stream))
			      &body body)
  (default-query-stream stream with-input-editing)
  `(flet ((with-input-editing-body (,stream) ,@body))
     (declare (dynamic-extent #'with-input-editing-body))
     (with-input-editing-internal ,stream ,class
				  #'with-input-editing-body
				  ,input-sensitizer ,initial-contents)))

(defmacro with-input-editor-typeout ((&optional stream) &body body)
  (default-query-stream stream with-input-editor-typeout)
  `(flet ((with-ie-typeout-body (,stream) ,@body))
     (declare (dynamic-extent #'with-ie-typeout-body))
     (with-ie-typeout-internal ,stream #'with-ie-typeout-body)))


;;; Support for the Help key while inside ACCEPT

(defvar *accept-help* nil)
(defvar *accept-help-displayer* 'ie-display-accept-help)

(defun ie-display-accept-help (function stream &rest args)
  (declare (dynamic-extent function args))
  (with-input-editor-typeout (stream)
    (apply function stream args)))

(defmacro with-input-editor-help (stream &body body)
  `(flet ((with-input-editor-help-body (,stream) ,@body))
     (declare (dynamic-extent #'with-input-editor-help-body))
     (funcall *accept-help-displayer* #'with-input-editor-help-body stream)))

;; ACTION is either :HELP or :POSSIBILITIES
(defun display-accept-help (stream action string-so-far)
  (with-input-editor-help stream
    (flet ((find-help-clauses-named (help-name)
	     (let ((clauses nil))
	       (dolist (clause *accept-help* clauses)
		 (when (eql (caar clause) help-name)
		   (push clause clauses)))))
	   (display-help-clauses (help-clauses)
	     (dolist (clause help-clauses)
	       (let ((type (first clause))
		     (args (rest clause)))
		 (declare (ignore type))
		 (fresh-line stream)
		 (typecase (first args)
		   (string (format stream (first args)))
		   (function
		     (apply (first args) stream action string-so-far (rest args))))))))
      (declare (dynamic-extent #'find-help-clauses-named #'display-help-clauses))
      (let ((top-level-help-clauses
	      (find-help-clauses-named :top-level-help))
	    (subhelp-clauses
	      (find-help-clauses-named :subhelp)))
	(cond ((null top-level-help-clauses)
	       (fresh-line stream)
	       (write-string "No top-level help specified.  Check the code." stream))
	      (t (display-help-clauses top-level-help-clauses)))
	(when subhelp-clauses
	  (display-help-clauses subhelp-clauses))))))

;; OPTIONS is a list of a help type followed by a help string (or a function
;; of two arguments, a stream and the help string so far) A "help type" is
;; either a single keyword (either :TOP-LEVEL-HELP or :SUBHELP), or a list
;; consisting of the type and a suboption (:OVERRIDE, :APPEND, or
;; :ESTABLISH-UNLESS-OVERRIDDEN).
;; Specifying :SUBHELP means "Append to previous subhelp, unless an outer
;; context has established an :OVERRIDE".
;; Specifying (:SUBHELP :APPEND) means append no matter what.
;; Specifying (:SUBHELP :OVERRIDE) means "This is the subhelp, subject to
;; lower-level explicit :APPENDs, unless someone above has already :OVERRIDden us.
;; Specifying (<type> :ESTABLISH-UNLESS-OVERRIDDEN) means "Establish <type>
;; at this level, unless someone above has already established <type>."  It does
;; not imply :APPENDING.
(defmacro with-accept-help (options &body body)
  #+Genera (declare (zwei:indentation 0 3 1 1))
  (check-type options list)
  (assert (every #'listp options))
  (dolist (option options)
    (let* ((option-name-spec (if (symbolp (first option))
				`(,(first option) :normal)
			        (first option)))
	   (option-name (first option-name-spec))
	   (option-type (second option-name-spec))
	   (option-args (rest option)))
      (check-type option-name (member :top-level-help :subhelp))
      (check-type option-type (member :normal :append :override :establish-unless-overridden))
      (setq body
	    `((with-stack-list* (*accept-help*
				  (list ',option-name-spec ,@option-args) *accept-help*)
		,@(cond ((eql option-type :override)
			 `((if (assoc (caar *accept-help*) (rest *accept-help*)
				      :test #'(lambda (a b)
						(and (eq (first a) (first b))
						     (member :override (rest b)))))
			       (pop *accept-help*)
			       (setq *accept-help*
				     (cons (first *accept-help*)
					   (delete ,option-name (rest *accept-help*)
						   :test #'(lambda (a b)
							     (eq (caar b) a))))))))
			((eql option-type :append)
			 )
			((eql option-type :establish-unless-overridden)
			 `((when (assoc (caaar *accept-help*) (rest *accept-help*)
					:key #'first)
			     (pop *accept-help*))))
			(t
			 `((when (assoc (caar *accept-help*) (rest *accept-help*)
					:test #'(lambda (a b)
						  (and (eq (first a) (first b))
						       (member :override (rest b)))))
			     (pop *accept-help*)))))
		,@body)))))
  `(progn ,@body))
