;;; -*- Mode: LISP; Syntax: Common-lisp; Package: CLIM-INTERNALS; Base: 10; Lowercase: Yes -*-

;; $fiHeader: input-editor-commands.lisp,v 1.7 91/03/29 18:00:55 cer Exp $

(in-package "CLIM-INTERNALS")

"Copyright (c) 1990, 1991 Symbolics, Inc.  All rights reserved."

;;; Define some useful input editor commands.  For now, all these are defined
;;; on INTERACTIVE-STREAM-MIXIN rather than on our specific implementation of
;;; an input editor.  This may prove to be a foolish decision down the pike.

(eval-when (compile load eval)
(defvar *ie-command-arglist* '(stream input-buffer gesture numeric-argument))
)

;; GESTURES is either a single gesture-name (as :RUBOUT), or a list of gestures
;; (as ((:CONTROL :X) :RUBOUT).  
;; FUNCTION is the function that implements an input editor command.  This associates
;; the gesture(s) with the command.
(defun add-input-editor-command (gestures function)
  (flet ((add-aarray-entry (gesture thing aarray)
	   (let ((old (find gesture aarray :key #'first :test #'equal)))
	     (if old
		 (setf (second old) thing)
	         (vector-push-extend (list gesture thing) aarray)))))
    (declare (dynamic-extent #'add-aarray-entry))
    (cond ((gesture-spec-p gestures)
	   ;; --- no equivalent for this test in the Silica world
	   ;(assert (not (ordinary-char-p characters)))
	   (add-aarray-entry gestures function *input-editor-command-aarray*))
	  (t
	   (assert (> (length gestures) 1))
	   ;; --- no equivalent for this test in the Silica world
	   ;(assert (not (ordinary-char-p (first characters))))
	   ;; We've got a command that wil be bound to a sequence of characters,
	   ;; so set up the prefix tables.
	   (let ((aarray *input-editor-command-aarray*))
	     (dorest (rest gestures)
	       (let* ((prefix (first rest))
		      (rest (rest rest)))
		 (if (null rest)
		     (add-aarray-entry prefix function aarray)
		     (let ((subaarray
			     (second (find prefix aarray :key #'first :test #'equal))))
		       (when (null subaarray)
			 (setq subaarray (make-array 20 :fill-pointer 0))
			 (add-aarray-entry prefix subaarray aarray))
		       (setq aarray subaarray))))))))))

(defmacro assign-input-editor-key-bindings (&body functions-and-keystrokes)
  (let ((forms nil))
    (loop
      (when (null functions-and-keystrokes) (return))
      (let* ((function (pop functions-and-keystrokes))
	     (keystrokes (pop functions-and-keystrokes)))
	(when keystrokes
	  (push `(add-input-editor-command ',keystrokes ',function)
		forms))))
    `(progn ,@(nreverse forms))))

;; These need to be defined before being used, otherwise they would
;; live in CLOE-IMPLMENTATION
#+Cloe-Runtime
(eval-when (compile load)
	   
;; Define the new key chars for Cloe CLIM.  Regular Cloe defines 0-127, we define
;; 128-139 as the F-keys (F1 thru F12), 140 for c-sh-A, and 141 as c-sh-V
(sys::define-character-name "F1" 128)
(sys::define-character-name "F2" 129)
(sys::define-character-name "F3" 130)
(sys::define-character-name "F4" 131)
(sys::define-character-name "F5" 132)
(sys::define-character-name "F6" 133)
(sys::define-character-name "F7" 134)
(sys::define-character-name "F8" 135)
(sys::define-character-name "F9" 136)
;; Note windows traps F10 as alt-space. Why?
(sys::define-character-name "F10" 137)
(sys::define-character-name "F11" 138)
(sys::define-character-name "F12" 139)
(sys::define-character-name "Arglist" 140)
(sys::define-character-name "ShowValue" 141)

)	;eval-when

;; When T, the input editor should handle help and completion.  Otherwise,
;; something like COMPLETE-INPUT will do it for us.
(defvar *ie-help-enabled* t)

;; These need to be on a per-implementation basis, naturally
;;--- If you change these, change *MAGIC-COMPLETION-CHARACTERS* too
(defvar *complete-gestures* #+Genera '(:complete :tab)
			    #-Genera '(:tab))
(defvar *help-gestures* '(:help))
(defvar *possibilities-gestures* '((:control :?)))

(defun lookup-input-editor-command (keysym aarray &optional (shift-mask 0) port)
  ;; Need to handle the help and possibilities commands specially so
  ;; that they work correctly inside of COMPLETE-INPUT
  (cond ((and *ie-help-enabled*
	      (keysym-and-shift-mask-member keysym shift-mask *help-gestures* port))
	 'com-ie-help)
	((and *ie-help-enabled*
	      (keysym-and-shift-mask-member keysym shift-mask *possibilities-gestures* port))
	 'com-ie-possibilities)
	((and *ie-help-enabled*
	      (keysym-and-shift-mask-member keysym shift-mask *complete-gestures* port))
	 'com-ie-complete)
	((and (eql aarray *input-editor-command-aarray*)
	      (keysym-and-shift-mask-matches-gesture-spec
		keysym shift-mask :- port))
	 ;; The "minus" keysym returns -1
	 -1)
	((and (eql aarray *input-editor-command-aarray*)
	      (not (zerop shift-mask))
	      (typep keysym 'numeric-keysym))
	 (numeric-keysym-number keysym))
	(t (flet ((gesture-match-p (spec)
		    (keysym-and-shift-mask-matches-gesture-spec
		      keysym shift-mask spec port)))
	     (declare (dynamic-extent #'gesture-match-p))
	     (second (find-if #'gesture-match-p aarray :key #'car))))))

(defmacro define-input-editor-command ((name &key (rescan T) (type 'motion) history)
				       arglist &body body)
  (multiple-value-bind (arglist ignores)
      (canonicalize-and-match-lambda-lists *ie-command-arglist* arglist)
    (let ((stream (first arglist)))
      `(define-group ,name define-input-editor-command
	 (defun ,name ,arglist
	   ,@(and ignores `((declare (ignore ,@ignores))))
	   ,@body
	   (setf (slot-value ,stream 'last-command-type) ',type)
	   ,@(unless history `((setf (slot-value ,stream 'previous-history) nil)))
	   ,@(when rescan `((immediate-rescan ,stream)))
	   (values))))))

#+Genera
(scl:defprop define-input-editor-command "CLIM Input Editor Command" si:definition-type-name)
#+Genera
(scl:defprop define-input-editor-command zwei:defselect-function-spec-finder
	     zwei:definition-function-spec-finder)

(defmethod interactive-stream-process-gesture ((istream interactive-stream-mixin)
						 gesture type)
  (values gesture type))

(defmethod interactive-stream-process-gesture ((istream interactive-stream-mixin)
					       (gesture character)
					       type)
  ;; a character gesture has no shifts.  Note that this will almost always return NIL.
  ;; We might want to just declare that you can't define input-editor-commands on
  ;; STANDARD-CHARs, if we think that performance is suspect.
  #+++Ignore ; 12/28/91 doughty 
  (let ((command (lookup-input-editor-command 
		   gesture
		   0
		   (port (slot-value istream 'stream)))))
    (when command
      (return-from interactive-stream-process-gesture
	(funcall command istream (slot-value istream 'input-buffer) gesture))))
  (with-slots (last-command-type command-state) istream
    (setq last-command-type 'character
	  command-state *input-editor-command-aarray*))
  (values gesture type))

(defmethod interactive-stream-process-gesture ((istream interactive-stream-mixin)
					       (gesture key-press-event)
					       type)
  (with-slots (numeric-argument last-command-type command-state) istream
    (let ((command (lookup-input-editor-command
		     (event-keysym gesture)
		     command-state
		     (event-input-state gesture)
		     (port (event-sheet gesture)))))
      (cond ((numberp command)
	     (cond ((null numeric-argument)
		    (setq numeric-argument command))
		   ((= command -1)
		    (setq numeric-argument (- numeric-argument)))
		   (t
		    (setq numeric-argument (+ (* numeric-argument 10) command))))
	     ;; Numeric arguments don't affect LAST-COMMAND-TYPE
	     (return-from interactive-stream-process-gesture nil))
	    ((arrayp command)
	     ;; A prefix character, update the state and return
	     (setq command-state command)
	     (return-from interactive-stream-process-gesture nil))
	    (command
	     (let ((argument (or numeric-argument 1)))
	       (setq numeric-argument nil
		     command-state *input-editor-command-aarray*)
	       (funcall command
			istream (slot-value istream 'input-buffer) gesture argument))
	     (return-from interactive-stream-process-gesture nil))
	    ((not (eq command-state *input-editor-command-aarray*))
	     (beep istream)
	     (setq numeric-argument nil
		   command-state *input-editor-command-aarray*)
	     (return-from interactive-stream-process-gesture nil))
	    (t
	     (setq last-command-type 'character
		   command-state *input-editor-command-aarray*)))))
  (values gesture type))


;;; Help commands, handled by special magic

(define-input-editor-command (com-ie-help :rescan nil)
			     (stream)
  (display-accept-help stream :help ""))

(define-input-editor-command (com-ie-possibilities :rescan nil)
			     (stream)
  (display-accept-help stream :possibilities ""))

(define-input-editor-command (com-ie-complete :rescan nil)
			     (stream input-buffer)
  (multiple-value-bind (string ambiguous word-start)
      (complete-symbol-name stream input-buffer)
    (when string
      (replace-input stream string :buffer-start word-start))
    (when (or ambiguous (null string))
      (beep stream))
    (when string
      (queue-rescan stream ':activation))))

(defun complete-symbol-name (stream input-buffer &aux (colon-string ":"))
  (declare (values string ambiguous word-start))
  (multiple-value-bind (word-start word-end colon)
      (word-start-and-end input-buffer '(#\space #\( #\) #\") (insertion-pointer stream))
    (when word-end
      (with-temp-substring (package-name input-buffer word-start (or colon word-start))
	(when (and colon
		   (< colon word-end)
		   (char-equal (aref input-buffer (1+ colon)) #\:))
	  (incf colon)
	  (setq colon-string "::"))
	(with-temp-substring (symbol-name 
			      input-buffer (if colon (1+ colon) word-start) word-end)
	  (multiple-value-bind (new-symbol-name success object nmatches)
	      (complete-symbol-name-1 symbol-name)
	    (declare (ignore success object))
	    (when (and new-symbol-name (not (zerop nmatches)))
	      (return-from complete-symbol-name
		(values
		  (if (and colon (> colon word-start))
		      (format nil "~A~A~A" package-name colon-string new-symbol-name)
		      new-symbol-name)
		  (/= nmatches 1)
		  word-start)))))))))

#+Genera
(defun complete-symbol-name-1 (string)
  (complete-from-possibilities string (scl:g-l-p zwei:*zmacs-completion-aarray*) '(#\-)
			       :action :complete-maximal))

#-Genera
(defun complete-symbol-name-1 (string)
  nil)

(define-input-editor-command (com-ie-scroll-forward :rescan nil)
			     (stream numeric-argument)
  (ie-scroll-window numeric-argument +1))

(define-input-editor-command (com-ie-scroll-backward :rescan nil)
			     (stream numeric-argument)
  (ie-scroll-window numeric-argument -1))

;; Scroll the frame's standard output stream vertically by some amount, 
;; one screenful being the default.
(defun ie-scroll-window (distance direction)
  (let* ((window (frame-standard-output *application-frame*))
	 (history (and window
		       (output-recording-stream-p window)
		       (output-recording-stream-output-record window))))
    (when window
      (multiple-value-bind (x y) (window-viewport-position* window)
	(incf y (* (if (= distance 1)
		       (bounding-rectangle-height (window-viewport window))
		       (* distance (stream-line-height window)))
		   direction))
	(with-bounding-rectangle* (hleft htop hright hbottom) history
	  (declare (ignore hleft hright))
	  (setq y (min (max htop y) hbottom)))
	(window-set-viewport-position* window x y)))))


;;; Some macrology for talking about the input-buffer

(defmacro ie-line-start (buffer pointer)
  `(1+ (or (position #\Newline ,buffer :end ,pointer :from-end t)
	   -1)))

(defmacro ie-line-end (buffer pointer)
  `(or (position #\Newline ,buffer :start ,pointer)
       (fill-pointer ,buffer)))

;; Things which move over words must move over whitespace until they see
;; alphanumerics, then alphanumerics until they see whitespace.
(defun move-over-word (buffer start-position reverse-p)
  (flet ((word-break-character-p (thing)
	   (when (and (characterp thing)
		      (not (alphanumericp thing)))
	     (values t t)))
	 (word-character-p (thing)
	   (when (and (characterp thing)
		      (alphanumericp thing))
	     (values t t))))
    (declare (dynamic-extent #'word-break-character-p #'word-character-p))
    (setq start-position
	  (forward-or-backward buffer start-position reverse-p #'word-character-p))
    (when start-position
      (forward-or-backward buffer start-position reverse-p #'word-break-character-p))))


;;; The basic input editing commands...

;; Don't do anything
(define-input-editor-command (com-ie-ctrl-g :rescan nil)
			     (stream)
  (with-slots (numeric-argument command-state) stream
    (setq numeric-argument nil
	  command-state *input-editor-command-aarray*))
  (beep stream))

(define-input-editor-command (com-ie-universal-argument :rescan nil :type nil)
			     (stream numeric-argument)
  (setf (slot-value stream 'numeric-argument)
	(* (or numeric-argument 1) 4)))

(define-input-editor-command (com-ie-forward-character :rescan nil)
			     (stream input-buffer numeric-argument)
  (dotimes (i numeric-argument) 
    #-excl (declare (ignore i))
    (let ((p (forward-or-backward input-buffer (insertion-pointer stream) nil #'true)))
      (if p
	  (setf (insertion-pointer stream) p)
	  (return (beep stream))))))

(define-input-editor-command (com-ie-forward-word :rescan nil)
			     (stream input-buffer numeric-argument)
  (dotimes (i numeric-argument)
    #-excl (declare (ignore i))
    (let ((p (move-over-word input-buffer (insertion-pointer stream) nil)))
      (if p
	  (setf (insertion-pointer stream) p)
	  (return (beep stream))))))

(define-input-editor-command (com-ie-backward-character :rescan nil)
			     (stream input-buffer numeric-argument)
  (dotimes (i numeric-argument)
    #-excl (declare (ignore i))
    (let ((p (forward-or-backward input-buffer (insertion-pointer stream) t #'true)))
      (if p
	  (setf (insertion-pointer stream) p)
	  (return (beep stream))))))

(define-input-editor-command (com-ie-backward-word :rescan nil)
			     (stream input-buffer numeric-argument)
  (dotimes (i numeric-argument)
    #-excl (declare (ignore i))
    (let ((p (move-over-word input-buffer (insertion-pointer stream) t)))
      (if p
	  (setf (insertion-pointer stream) p)
	  (return (beep stream))))))

(define-input-editor-command (com-ie-beginning-of-buffer :rescan nil :type nil)
			     (stream)
  (setf (insertion-pointer stream) 0))

(define-input-editor-command (com-ie-end-of-buffer :rescan nil)
			     (stream input-buffer)
  (setf (insertion-pointer stream) (fill-pointer input-buffer)))

(define-input-editor-command (com-ie-beginning-of-line :rescan nil)
			     (stream input-buffer)
  (setf (insertion-pointer stream)
	(ie-line-start input-buffer (insertion-pointer stream))))

(define-input-editor-command (com-ie-end-of-line :rescan nil)
			     (stream input-buffer)
  (setf (insertion-pointer stream)
	(ie-line-end input-buffer (insertion-pointer stream))))

;; Positions to the nearest column in the next line
(define-input-editor-command (com-ie-next-line :rescan nil)
			     (stream input-buffer numeric-argument)
  (unless (= (insertion-pointer stream) (fill-pointer input-buffer))
    (ie-next-previous stream input-buffer numeric-argument)))

(define-input-editor-command (com-ie-previous-line :rescan nil)
			     (stream input-buffer numeric-argument)
  (unless (zerop (insertion-pointer stream))
    (ie-next-previous stream input-buffer (- numeric-argument))))

;;; +ve = next, -ve = previous
(defun ie-next-previous (stream input-buffer numeric-argument)
  (unless (zerop numeric-argument)
    (let* ((pointer (insertion-pointer stream))
	   (this-line (ie-line-start input-buffer pointer))
	   (target-line this-line))
      (if (plusp numeric-argument)
	  (let (next-line-1)
	    (dotimes (i numeric-argument)
	      #-excl (declare (ignore i))
	      (setq next-line-1 (position #\Newline input-buffer :start target-line))
	      (unless next-line-1 (return))
	      (setq target-line (1+ next-line-1)))
	    (setf (insertion-pointer stream)
		  (let ((position-in-line (- pointer this-line)))
		    (min (+ target-line position-in-line)
			 (ie-line-end input-buffer target-line)))))
	  (let (prev-line-end)
	    (dotimes (i (- numeric-argument))
	      #-excl (declare (ignore i))
	      (setq prev-line-end (position #\Newline input-buffer
					    :end target-line :from-end t))
	      (unless prev-line-end (return))
	      (setq target-line prev-line-end))
	    (setf (insertion-pointer stream)
		  (let ((position-in-line (- pointer this-line)))
		    (min (+ (ie-line-start input-buffer target-line)
			    position-in-line)
			 (ie-line-end input-buffer target-line)))))))))

(define-input-editor-command (com-ie-rubout :type delete)
			     (stream input-buffer numeric-argument)
  (ie-rub-del stream input-buffer (- numeric-argument)))

(define-input-editor-command (com-ie-delete-character :type delete)
			     (stream input-buffer numeric-argument)
  (ie-rub-del stream input-buffer numeric-argument))

;;; +ve = delete, -ve = rubout
(defun ie-rub-del (stream input-buffer numeric-argument)
  (let* ((p1 (insertion-pointer stream))
	 (p2 p1)
	 (reverse-p (minusp numeric-argument))) 
    (dotimes (i (abs numeric-argument))
      #-excl (declare (ignore i))
      (let ((p3 (forward-or-backward input-buffer p2 reverse-p #'true)))
	(if p3 (setq p2 p3) (return))))
    (if (/= p1 p2)
	(ie-kill stream input-buffer
		 (cond ((eql (slot-value stream 'last-command-type) 'kill) :merge)
		       ((> (abs numeric-argument) 1) t)
		       (t nil))
		 p2 p1 reverse-p)
        (beep stream))))

(define-input-editor-command (com-ie-rubout-word :type kill)
			     (stream input-buffer numeric-argument)
  (ie-rub-del-word stream input-buffer (- numeric-argument)))

(define-input-editor-command (com-ie-delete-word :type kill)
			     (stream input-buffer numeric-argument)
  (ie-rub-del-word stream input-buffer numeric-argument))

;;; +ve = delete, -ve = rubout
(defun ie-rub-del-word (stream input-buffer numeric-argument)
  (let* ((p1 (insertion-pointer stream))
	 (p2 p1)
	 (reverse-p (minusp numeric-argument))) 
    (dotimes (i (abs numeric-argument))
      #-excl (declare (ignore i))
      (let ((p3 (move-over-word input-buffer p2 reverse-p)))
	(if p3 (setq p2 p3) (return))))
    (if (/= p1 p2)
	(ie-kill stream input-buffer
		 (if (eql (slot-value stream 'last-command-type) 'kill) :merge t)
		 p2 p1 reverse-p)
        (beep stream))))

(define-input-editor-command (com-ie-clear-input :type kill)
			     (stream input-buffer)
  ;; Just push a copy of the input buffer onto the kill ring, no merging
  (ie-kill stream input-buffer t 0 (fill-pointer input-buffer)))

(define-input-editor-command (com-ie-kill-line :type kill)
			     (stream input-buffer numeric-argument)
  (let* ((reverse-p (minusp numeric-argument))
	 (point (insertion-pointer stream))
	 (other-point (if reverse-p
			  (ie-line-start input-buffer point)
			  (ie-line-end input-buffer point))))
    (ie-kill stream input-buffer
	     (if (eql (slot-value stream 'last-command-type) 'kill) :merge t)
	     point
	     other-point
	     reverse-p)))

(define-input-editor-command (com-ie-make-room)
			     (stream input-buffer)
  (let ((point (insertion-pointer stream))
	(end (fill-pointer input-buffer)))
    (cond ((= point end)
	   (incf (fill-pointer input-buffer)))
	  (t
	   (erase-input-buffer stream point)
	   (shift-buffer-portion input-buffer point (1+ point))))
    (setf (aref input-buffer point) #\Newline)
    (redraw-input-buffer stream point)))

(define-input-editor-command (com-ie-transpose-characters)
			     (stream input-buffer)
  (let* ((start-position (min (1+ (insertion-pointer stream))
			      (fill-pointer input-buffer)))
	 (this-position (forward-or-backward input-buffer start-position t #'true))
	 (prev-position (forward-or-backward input-buffer this-position t #'true)))
    (cond ((and this-position prev-position (/= this-position prev-position))
	   (let ((this-char (aref input-buffer this-position))
		 (prev-char (aref input-buffer prev-position)))
	     (erase-input-buffer stream prev-position)
	     (setf (aref input-buffer prev-position) this-char)
	     (setf (aref input-buffer this-position) prev-char)
	     (redraw-input-buffer stream prev-position)))
	  (t (beep stream)))))


;;; Lispy input editing commands

(defun function-arglist (function)
  (declare (values arglist found-p))
  #+Genera (values (sys:arglist function) T)
  #+cloe-runtime (values (sys::arglist function) t)
  #+excl (values (excl::arglist function) t)
  #+Lucid (values (lucid-common-lisp:arglist function) t))

#+cloe-runtime
(defun sys::arglist (symbol)
  (let ((fsanda (si::sys%get symbol 'arglist))
	(argl nil)
	(fun nil))
    (unless fsanda (setq fsanda (get symbol 'arglist)))
    (if fsanda
	(progn
	  (setq fun (car fsanda) argl (cadr fsanda))
	  (return-from sys::arglist (values argl fun)))
	(return-from  sys::arglist (values nil nil)))))

(defun word-start-and-end (string start-chars &optional (start 0))
  (declare (values word-start word-end colon))
  (flet ((open-paren-p (thing)
	   (or (not (characterp thing))		;noise strings and blips are delimiters
	       (member thing start-chars)))
	 (atom-break-char-p (thing)
	   (or (not (characterp thing))		;ditto
	       (not (graphic-char-p thing))
	       (multiple-value-bind (mac nt)
		   (get-macro-character thing)
		 (and mac (not nt)))
	       (member thing '(#\space #\( #\) #\")))))
    (declare (dynamic-extent #'open-paren-p #'atom-break-char-p))
    (let* ((word-start
	     (forward-or-backward string start t #'open-paren-p))
	   (word-end
	     (and word-start
		  (or (forward-or-backward string (1+ word-start) nil
					   #'atom-break-char-p)
		      (fill-pointer string))))
	   (colon
	     (and word-start word-end
		  (position #\: string
			    :start (1+ word-start) :end (1- word-end)))))
      (values (and word-start
		   (if (atom-break-char-p (aref string word-start)) 
		       (1+ word-start)
		       word-start))
	      (and word-end (1- word-end))
	      colon))))

(define-input-editor-command (com-ie-show-arglist :rescan nil)
			     (stream input-buffer)
  (multiple-value-bind (word-start word-end colon)
      (word-start-and-end input-buffer '(#\( ) (insertion-pointer stream))
    (block doit
      (when word-end
	(with-temp-substring (package-name input-buffer word-start (or colon word-start))
	  (when (and colon
		     (< colon word-end)
		     (char-equal (aref input-buffer (1+ colon)) #\:))
	    (incf colon))
	  (with-temp-substring (symbol-name
				input-buffer (if colon (1+ colon) word-start) word-end)
	    (let* ((symbol (find-symbol (string-upcase symbol-name)
					(if colon (find-package package-name) *package*)))
		   (function (and symbol (fboundp symbol) (symbol-function symbol))))
	      (when function
		(multiple-value-bind (arglist found-p)
		    (function-arglist function)
		  (when found-p
		    (return-from doit
		      (with-input-editor-typeout (stream)
			#-Cloe-Runtime
			(format stream "~S: ~A" symbol arglist)
			#+Cloe-Runtime
			(format stream "~S (~A): ~:A" symbol found-p arglist))))))))))
      (beep stream))))

(define-input-editor-command (com-ie-show-value :rescan nil)
			     (stream input-buffer)
  (multiple-value-bind (word-start word-end colon)
      (word-start-and-end input-buffer '(#\space #\( #\) #\") (insertion-pointer stream))
    (block doit
      (when word-end
	(with-temp-substring (package-name input-buffer word-start (or colon word-start))
	  (when (and colon
		     (< colon word-end)
		     (char-equal (aref input-buffer (1+ colon)) #\:))
	    (incf colon))
	  (with-temp-substring (symbol-name 
				input-buffer (if colon (1+ colon) word-start) word-end)
	    (let* ((symbol (find-symbol (string-upcase symbol-name)
					(if colon (find-package package-name) *package*)))
		   (value (and symbol (boundp symbol) (symbol-value symbol))))
	      (when value
		(return-from doit
		  (with-input-editor-typeout (stream)
		    (format stream "~S: ~S" symbol value))))))))
      (beep stream))))


;;; Yanking commands

(defun ie-yank-from-history (history function istream numeric-argument
			     &key test replace-previous)
  (setf (slot-value istream 'previous-history) history)
  (cond ((zerop numeric-argument)
	 (display-history-contents history istream))
	((= numeric-argument 1)
	 (let ((element (funcall function history :test test)))
	   (cond (element
		  (history-replace-input history istream element
					 :replace-previous replace-previous)
		  ;; The yanking commands don't do an immediate rescan
		  ;; because that can cause premature activation.  What
		  ;; we do is queue a rescan for later, and when the user
		  ;; hits <End> the rescan is done if necessary.
		  ;;--- I don't like this, see comment on RESCAN-FOR-ACTIVATION
		  (queue-rescan istream ':activation))
		 (t (beep)))))
	(t
	 (let ((element (funcall function history :index numeric-argument :test test)))
	   (cond (element
		  (history-replace-input history istream element
					 :replace-previous replace-previous)
		  (queue-rescan istream ':activation))
		 (t (beep)))))))

(define-input-editor-command (com-ie-kill-ring-yank :history t :type yank :rescan nil)
			     (stream numeric-argument)
  (ie-yank-from-history *kill-ring* #'yank-from-history stream numeric-argument))

(define-input-editor-command (com-ie-history-yank :history t :type yank :rescan nil)
			     (stream numeric-argument)
  (let ((history
	  (and *presentation-type-for-yanking*
	       (presentation-type-history *presentation-type-for-yanking*))))
    (if history
	(ie-yank-from-history history #'yank-from-history stream numeric-argument)
        (beep))))

(define-input-editor-command (com-ie-yank-next :history t :type yank :rescan nil)
			     (stream numeric-argument)
  (let ((history (slot-value stream 'previous-history)))
    (if history
	(ie-yank-from-history history #'yank-next-from-history stream numeric-argument
			      :replace-previous t)
        (beep))))


;;; Key bindings

;;; These can be overridden on a per port level if necessary.
(define-logical-gestures
  '(
    (:ie-abort :g :control)
    (:ie-universal-argument :u :control)
    (:ie-forward-character :f :control)
    (:ie-forward-word :f :meta)
    (:ie-backward-character :b :control)
    (:ie-backward-word :b :meta)
    (:ie-beginning-of-buffer :\< :meta)
    (:ie-end-of-buffer :\> :meta)
    (:ie-beginning-of-line :a :control)
    (:ie-end-of-line :e :control)
    (:ie-next-line :n :control)
    (:ie-previous-line :p :control)
    (:ie-delete-character :d :control)
    (:ie-rubout-word :rubout :meta)
    (:ie-delete-word :d :meta)
    (:ie-kill-line :k :control)
    (:ie-make-room :o :control)
    (:ie-transpose-characters :t :control)
    (:ie-show-arglist :a :control :shift)
    (:ie-show-value :v :control :shift)
    (:ie-kill-ring-yank :y :control)
    (:ie-history-yank :y :control :meta)
    (:ie-yank-next :y :meta)
    (:ie-scroll-forward :scroll)
    (:ie-scroll-backward :scroll :meta)
    (:ie-scroll-forward :v :control)
    (:ie-scroll-backward :v :meta))
  :unique nil)

(assign-input-editor-key-bindings
  com-ie-ctrl-g		       :ie-abort
  com-ie-universal-argument    :ie-universal-argument
  com-ie-forward-character     :ie-forward-character
  com-ie-forward-word	       :ie-forward-word
  com-ie-backward-character    :ie-backward-character
  com-ie-backward-word	       :ie-backward-word
  com-ie-beginning-of-buffer   :ie-beginning-of-buffer
  com-ie-end-of-buffer	       :ie-end-of-buffer
  com-ie-beginning-of-line     :ie-beginning-of-line
  com-ie-end-of-line	       :ie-end-of-line
  com-ie-next-line	       :ie-next-line
  com-ie-previous-line	       :ie-previous-line
  com-ie-rubout		       :rubout
  com-ie-delete-character      :ie-delete-character
  com-ie-rubout-word	       :ie-rubout-word
  com-ie-delete-word	       :ie-delete-word
  com-ie-clear-input	       :clear-input
  com-ie-kill-line	       :ie-kill-line
  com-ie-make-room	       :ie-make-room
  com-ie-transpose-characters  :ie-transpose-characters
  com-ie-show-arglist	       :ie-show-arglist
  com-ie-show-value	       :ie-show-value
  com-ie-kill-ring-yank	       :ie-kill-ring-yank
  com-ie-history-yank	       :ie-history-yank
  com-ie-yank-next	       :ie-yank-next
  com-ie-scroll-forward	       :ie-scroll-forward
  com-ie-scroll-backward       :ie-scroll-backward)
