;;; -*- Mode: LISP; Syntax: Common-lisp; Package: CLIM-INTERNALS; Base: 10; Lowercase: Yes -*-

(in-package "CLIM-INTERNALS")

"Copyright (c) 1990, 1991 International Lisp Associates.  All rights reserved."

(define-slot-accessing-macro calculating-bounding-box
			     ((&optional (transformation '+identity-transformation+)
					 sx sy line-thickness)
			      &body body)
  #+Genera (declare (values min-x min-y max-x max-y))
  "Macro for calculating bounding box edges.  Use (INCLUDE-POINT X Y) to include it.
Transforms all points before calculating their effect on the bounding box."
  (when (eql transformation '+identity-transformation+)
    (setf transformation nil))
  (let ((min-x (gensymbol 'min-x))
	(min-y (gensymbol 'min-y))
	(max-x (gensymbol 'max-x))
	(max-y (gensymbol 'max-y))
	(new-x (gensymbol 'new-x))
	(new-y (gensymbol 'new-y))
	(start-x (gensymbol 'start-x))
	(start-y (gensymbol 'start-y))
	(xform (when transformation (gensymbol 'transformation)))
	(lt (when line-thickness (gensymbol 'line-thickness))))
    `(let ((,min-x nil) (,max-x nil) (,min-y nil) (,max-y nil)
	   ,@(when sx `((,start-x ,sx)))
	   ,@(when sy `((,start-y ,sy)))
	   ;; ONCE-ONLY:
	   ,@(when transformation `((,xform ,transformation)))
	   ,@(when line-thickness `((,lt ,line-thickness))))
       (macrolet ((include-point (x y)
		    ,(let ((body `((minf-or ,min-x ,new-x)
				   (maxf-or ,max-x ,new-x)
				   (minf-or ,min-y ,new-y)
				   (maxf-or ,max-y ,new-y))))
		       (if transformation
			   ``(multiple-value-bind (,',new-x ,',new-y)
				 (transform-point* ,',xform ,x ,y)
			       ,@',body)
			   ``(let ((,',new-x ,x) (,',new-y ,y))	;Poor man's once-only
			       ,@',body)))))
	 ,@body)
       (unless ,min-x
	 (setf ,min-x 0 ,min-y 0 ,max-x 0 ,max-y 0))
       ,@(when sx `((incf ,min-x ,start-x) (incf ,max-x ,start-x)))
       ,@(when sy `((incf ,min-y ,start-y) (incf ,max-y ,start-y)))
       ,@(when line-thickness
	   `((unless (zerop ,lt)
	       (decf ,min-x ,lt)
	       (decf ,min-y ,lt)
	       (incf ,max-x ,lt)
	       (incf ,max-y ,lt))))
       (values ,min-x ,min-y ,max-x ,max-y))))

;;; "It is an error to use INCLUDE-POINT outside a CALCULATING-BOUNDING-BOX."
(defmacro include-point (x y)
  "Include this point in the bounding box."
  (declare (ignore x y))
  #+Genera (declare (zwei:indentation 1 1))	;Stupid indenter
  (warn "You must use ~S only inside ~S" 'include-point 'calculating-bounding-box)
  `(error "~S called from outside ~S" 'include-point 'calculating-bounding-box))

(define-slot-accessing-macro ls-thickness ()
  `(or (line-style-thickness line-style) 1))

(define-slot-accessing-macro ls-thickness/2 ()
  `(1+ (ceiling (ls-thickness) 2)))

#||
;;; (defmacro define-graphics-operator (name) ...) turns approximately into the following: 

(define-graphics-function-method draw-rectangle* :around
       ((stream output-recording-stream) x1 y1 x2 y2 &key filled)
  (when (stream-record-p stream)
    (let ((record (make-rectangle-output-record medium x1 y2 x2 y2 filled)))
      (graphics-operation-around-method-helper stream record)))
  (when (stream-draw-p stream)
    (call-next-method)))

(defclass rectangle-output-record (graphic-output-record-element)
     ((x1 :initarg :x1)
      (y1 :initarg :y1)
      (x2 :initarg :x2)
      (y2 :initarg :y2)
      (filled :initarg :filled)))

(defconstructor make-rectangle-output-record rectangle-output-record
  (ink transformation clipping-region line-style x1 y1 x2 y2 filled)
  :ink ink :transformation transformation :clipping-region clipping-region
  :line-style line-style :x1 x1 :y1 y1 :x2 x2 :y2 y2 :filled filled)

(defmethod bounding-box-edges* ((record rectangle-output-record))
  (with-slots-bound-as-needed (transformation start-x start-y x1 y1 x2 y2 filled) record
    (calculating-bounding-box (transformation start-x start-y)
      (include-point x1 y1)
      (include-point x1 y2)
      (include-point x2 y1)
      (include-point x2 y2))))

;;; Make sure this function works on encapsulating streams.
(defoperation draw-rectangle*-internal basic-output-recording
  ((stream basic-output-recording) x1 y1 x2 y2 &key filled)
  ;:defgenericp nil
  )

;;; Have to kludge REPLAY-1 into database so this will work (easy to do!):
(define-graphics-function-method replay-1 ((record rectangle-output-record) medium
					 &optional extent offset-x offset-y)
  (with-slots (ink transformation clipping-region line-style x1 y1 x2 y2 filled) record
    (with-drawing-options (medium :ink ink :transformation transformation
				  :clipping-region clipping-region :line-style line-style)
      (funcall #'(:graphics-internal draw-rectangle*) medium x1 y1 x2 y2)
      :filled filled)))

||#

;;; Kludge REPLAY-1 into database.  We want to be able to define a REPLAY-1 method as a
;;; graphics function method because we want to call a #'(:graphics-internal ...)
;;; generic function from inside its body.  Rather than defining the latter as a
;;; (non-portable) function spec mechanism, we have localized the knowledge of that
;;; kind of "function spec" to DEFINE-GRAPHICS-FUNCTION-METHOD.

(eval-when (compile load eval)
  (define-graphics-function-compile-time
    'replay-1 :spread-function-name 'replay-1 :method-function-name 'replay-1
    :unspread-lambda-list '() :spread-lambda-list '() :method-lambda-list '()
    :spread-arguments '()
    :type :kludge))

(defclass graphic-output-record-element
	  (displayed-output-record-element-mixin)
     ((transformation :initarg :transformation)
      (clipping-region :initarg :clipping-region)
      (ink :initarg :ink)
      (min-x :initform nil)
      (min-y)
      (max-x)
      (max-y)))

(defmethod bounding-box-edges* :around ((record graphic-output-record-element))
  (with-slots (min-x min-y max-x max-y) record
    (when (not (null min-x))
      (return-from bounding-box-edges* (values min-x min-y max-x max-y)))
    (multiple-value-bind (x1 y1 x2 y2) (call-next-method)
      (let ((cr (slot-value record 'clipping-region)))
	(cond ((eql cr +everywhere+))
	      ((eql cr +nowhere+)
	       (setf x1 x2 y1 y2))		;Empty clipping region
	      (t
	       (with-bounding-rectangle* (cx1 cy1 cx2 cy2) cr
		 (maxf x1 cx1)
		 (maxf y1 cy1)
		 (minf x2 cx2)
		 (minf y2 cy2)))))
      (setf min-x x1 min-y y1 max-x x2 max-y y2)
      (values x1 y1 x2 y2))))

(defmethod bounding-box-edges* ((record graphic-output-record-element))
  (error "The bounding-box calculation method for ~Ss has not been defined."
	 (class-name (class-of record))))

(defmethod decache-bounding-box-edges* ((record graphic-output-record-element))
  (with-slots (min-x) record
    (setf min-x nil)))

;;; --- This can't be right, can it? ---
(defmethod bounding-rectangle* ((record graphic-output-record-element))
  (bounding-box-edges* record))

;;; I'm not sure about this one either.
(defmethod* (setf bounding-rectangle*) (minx miny maxx maxy (record graphic-output-record-element))
  (declare (ignore minx miny maxx maxy))
  (decache-bounding-box-edges* record))

;;; The default method -- we have already tested to see that the
;;; bounding box contains this element.
(defmethod output-record-refined-sensitivity-test ((record graphic-output-record-element) x y)
  (declare (ignore x y))
  t)

(defun graphics-operation-around-method-helper (stream record)
  (multiple-value-bind (dx dy)
      (point-position* (output-recording-stream-output-record-absolute-position stream))
    (with-slots (start-x start-y) record
      (setf start-x (- dx) start-y (- dy))))
  (add-output-record stream record))

(defclass line-graphic-output-record-element (graphic-output-record-element)
    ((line-style :initarg :line-style)))

(defclass text-graphic-output-record-element (graphic-output-record-element)
    ((text-style :initarg :text-style)))

#+Genera
(scl:defprop define-graphics-operator "CLIM Graphics Operator" si:definition-type-name)

;;; The macro which defines the interesting CLIM parts of a Silica graphics
;;; operator.  These include the output record class, its REPLAY-1 method, and the
;;; methods which know how to do the graphics operator on CLIM sheets (including
;;; the output-recording behaviour).  Yet to be defined: highlighting behaviour
;;; (will crib from older version!).

(defmacro define-graphics-operator (name &key
				    bounding-box-points bounding-box
				    highlighting-test highlighting-function
				    output-recording-before-hook output-recording-after-hook
				    output-record-slots)
  (declare #+Genera (zwei:indentation 1 3))
  (macrolet ((check-required-arg (&rest versions)
	       `(let ((count 0)
		      #+Genera (compiler:default-warning-function name)
		      #+Genera (compiler:default-warning-definition-type
				 'define-graphics-operator))
		  ,@(with-collection (dolist (version versions)
				       (collect `(when ,version (incf count)))))
		  (unless (= count 1)
		    (warn ,(if (cdr versions)
			       "You must define ~:[at least~;no more than~] one of ~
			        ~{~A~^, ~}"
			       "~S: must define a method for ~*~{~A~}")
			  (plusp count) ',versions)))))
    (check-required-arg bounding-box-points bounding-box)
    #+Ignore (check-required-arg highlighting-function))

  ;; A couple of coercion functions
  (labels ((make-keyword (symbol)
	     (intern (string symbol) *keyword-package*))
	   (make-local (thing)
	     (typecase thing
	       (keyword thing)
	       (symbol (intern (string thing) *clim-internals-package*))
	       (list (mapcar #'make-local thing))
	       (otherwise thing))))
    (let* ((function (find-graphics-function name))
	   ;; Call them RECTANGLE-OUTPUT-RECORDs instead of DRAW-RECTANGLE-OUTPUT-RECORDs:
	   (subname (if (string-equal name 'draw- :end1 5)
			(subseq (string name) 5)
			(string name)))
	   (record-element-name (fintern "~A-~A" subname 'output-record))
	   (gf-type (gf-type function))
	   (base-record-type (ecase gf-type
			       ((:point :line :area) ':line)
			       (:text ':text)))
	   (style-argument (fintern "~A-~A" base-record-type 'style))
	   (style-keyword (make-keyword style-argument))
	   (base-class (fintern "~A-~A" base-record-type 'graphic-output-record-element))
	   (style-data (case base-record-type
			 (:line '(copy-line-style (medium-line-style stream)))
			 (:text '(clim-stream::stream-merged-text-style stream))))
	   (record-slot-names (mapcar #'make-local (gf-method-argument-list function)))
	   (slot-descriptions
	     (with-collection (dolist (slot-name record-slot-names)
				(collect `(,slot-name :initarg ,(make-keyword slot-name))))))
	   (spread-function (gf-spread-function-name function))
;	   (spread-lambda-list (mapcar #'make-local (gf-spread-lambda-list function)))
;	   (spread-rest-argument (second (member '&rest spread-lambda-list)))
	   (constructor-name (fintern "~A-~A-~A" '%make subname 'output-record))
	   (method-lambda-list (mapcar #'make-local (gf-method-lambda-list function)))
	   (method-argument-list (mapcar #'make-local (gf-method-argument-list function)))
	   (method-arguments
	     (let ((keywords? nil))
	       (with-collection
		 (dolist (arg method-lambda-list)
		   (cond ((eql arg '&key) (setf keywords? t))
			 (keywords? (when (listp arg)
				      (setf arg (first arg))
				      (when (listp arg) (setf arg (second arg))))
				    (collect (make-keyword arg))
				    (collect arg))
			 (t (collect arg)))))))
	   (highlighting-test-args (and highlighting-test (first highlighting-test)))
	   (highlighting-function-medium (and highlighting-function
					      (first (first highlighting-function)))))
      ;; See comment above for what this macro should be generating.
      `(progn
	 ;;Probably have to do DEFCLASS outside of DEFINE-GROUP in Genera, just like flavors.
	 (defclass ,record-element-name (,base-class) (,@slot-descriptions
						       ,@output-record-slots))
	 (define-group ,name define-graphics-operator
	   (define-constructor ,constructor-name ,record-element-name
	     (transformation clipping-region ink ,style-argument ,@method-argument-list)
	     :transformation transformation :clipping-region clipping-region
	     :ink ink ,style-keyword ,style-argument
	     ,@(with-collection (dolist (arg method-argument-list)
				  (collect (make-keyword arg)) (collect arg))))
	   (define-graphics-function-method ,spread-function :around
	       ((stream graphics-output-recording) ,@method-lambda-list)
	     (when (stream-record-p stream)
	       ,output-recording-before-hook
	       (let* ((transformation (medium-transformation stream))
		      (clipping-region (medium-clipping-region stream))
		      (record  ;; (%MAKE-xxx-OUTPUT-RECORD XF CR INK LS ...)
			(,constructor-name
			       transformation clipping-region
			       (copy-ink (medium-ink stream))
			       ,style-data
			       ,@method-argument-list)))
		 ,output-recording-after-hook
		 (graphics-operation-around-method-helper stream record)))
	     (when (stream-draw-p stream)
	       (call-next-method)))
	   (defoperation ,(gf-method-function-name function) basic-output-recording
	     ((stream basic-output-recording) ,@method-lambda-list)
	     ;:defgenericp nil
	     )
	   (defmethod bounding-box-edges* ((record ,record-element-name))
	     #+Genera (declare (sys:function-parent ,name define-graphics-operator))
	     (with-slots-bound-as-needed (transformation start-x start-y line-style
							 ,@method-argument-list) record
	       (calculating-bounding-box (transformation
					   start-x start-y
					   ,@(ecase gf-type
					       ((:point :line) '((ls-thickness/2)))
					       ;; --- Does line-thickness matter when filled?
					       ;; It doesn't under Genera.
					       (:area `((if filled 0
							  (ls-thickness/2))))
					       (:text nil)))
		 ;; include these slot-names lest the code-walker not think they're used
;		 start-x start-y
		 ,@(if bounding-box-points
		       (mapcar #'(lambda (point)
				   `(include-point ,(first point) ,(second point)))
			       bounding-box-points)
		       (if bounding-box (list bounding-box)
			   `((error "No bounding box method defined for ~A output records"
				    ',name)
			     ;; "Use" the TRANSFORMATION gensym from CALCULATING-BOUNDING-BOX:
			     (include-point nil nil)))))))
	   ,@(when highlighting-test
	       `((defmethod output-record-refined-sensitivity-test
		     ((record ,record-element-name) ,@highlighting-test-args)
		   #+Genera (declare (sys:function-parent ,name define-graphics-operator))
		   (with-slots-bound-as-needed (transformation ink clipping-region line-style
							       start-x start-y
							       ,@method-argument-list)
					       record
		     ,@(mapcar #'(lambda (lambda-var offset-var)
				   `(decf ,lambda-var ,offset-var))
			       highlighting-test-args '(start-x start-y))
		     nil			;Make sure bogus test returns NIL
		     ,@(rest highlighting-test)))))
	   ,@(when highlighting-function
	       `((defmethod highlight-output-record-1
		     ((record ,record-element-name) ,highlighting-function-medium ignore)
		   #+Genera (declare (sys:function-parent ,name define-graphics-operator))
		   (multiple-value-bind (offset-x offset-y)
		       (convert-from-relative-to-absolute-coordinates
			,highlighting-function-medium (output-record-parent record))
		     (with-slots-bound-as-needed (transformation line-style start-x start-y
								 ,@method-argument-list)
		       record
		       (with-drawing-options (,highlighting-function-medium
					      :transformation (translate-transformation
							       transformation
							       (+ offset-x start-x)
							       (+ offset-y start-y))
					      :ink +flipping-ink+)
			 (with-output-recording-options (,highlighting-function-medium
							 :record-p nil :draw-p t)
			   ,@(rest highlighting-function))))))))
	   (define-graphics-function-method replay-1 ((record ,record-element-name) medium
						    &optional extent offset-x offset-y)
	     #+Genera (declare (sys:function-parent ,name define-graphics-operator))
	     (with-slots-bound-as-needed (transformation clipping-region ink ,style-argument
							 start-x start-y
							 ,@method-argument-list)
					 record
	       (let ((transformation
		       (let ((offset-x (if offset-x (+ offset-x start-x) start-x))
			     (offset-y (if offset-y (+ offset-y start-y) start-y)))
			 (if (and (zerop offset-x) (zerop offset-y))
			     transformation
			     (translate-transformation transformation offset-x  offset-y))))
		     (clipping-region (if extent
					  (region-intersection extent clipping-region)
					  clipping-region)))
		 (with-drawing-options (medium :transformation transformation
					       :ink ink ,style-keyword ,style-argument
					       :clipping-region clipping-region)
		   (funcall #'(:graphics-internal ,spread-function) medium
			    ,@method-arguments))))))))))

(define-slot-accessing-macro bounding-box-from-coordinate-sequence (coordinate-sequence)
  `(let ((coordinates ,coordinate-sequence))
     (loop (when (null coordinates) (return))
	   (include-point (pop coordinates) (pop coordinates)))))

(define-slot-accessing-macro do-coordinates
			     ((coordinate-sequence x y &optional transformation) &body body)
  (let ((coordinates (gensymbol 'coordinates))
	(transform (gensymbol 'transformation)))
    `(let ((,coordinates ,coordinate-sequence)
	   ,x ,y
	   ,@(when transformation `((,transform ,transformation))))
       (loop (when (null ,coordinates) (return))
	     (setf ,x (pop ,coordinates) ,y (pop ,coordinates))
	     ,(when transformation
		`(multiple-value-setq (,x ,y) (transform-point* ,transform ,x ,y)))
	     ,@body))))

(defun point-close-to-line-p (x y from-x from-y to-x to-y &optional (thickness 1))
  (let ((distance (1+ thickness))
	(dx (- to-x from-x))
	(dy (- to-y from-y)))
    (cond ((and (zerop dx) (zerop dy))
	   (and (<= (abs (- x from-x)) distance)
		(<= (abs (- y from-y)) distance)))
	  ((> (abs dx) (abs dy))
	   (let ((correct-y (- to-y (floor (* (- to-x x) dy) dx))))
	     (<= (abs (- y correct-y)) distance)))
	  (t
	   (let ((correct-x (- to-x (floor (* (- to-y y) dx) dy))))
	     (<= (abs (- x correct-x)) distance))))))

(defun outline-line-with-hexagon (stream from-x from-y to-x to-y &optional (thickness 1))
  (let ((distance (1+ (round thickness 2))))
    (multiple-value-bind (x1 y1 x2 y2 x3 y3 x4 y4 x5 y5 x6 y6)
	(cond ((eq (minusp (- to-x from-x)) (minusp (- to-y from-y)))
	       (values (- from-x distance) (- from-y distance)
		       (- from-x distance) (+ from-y distance)
		       (- to-x distance) (+ to-y distance)
		       (+ to-x distance) (+ to-y distance)
		       (+ to-x distance) (- to-y distance)
		       (+ from-x distance) (- from-y distance)))
	      (t
	       (when (> to-y from-y)
		 ;; Make line go down to right.
		 (rotatef to-x from-x)
		 (rotatef to-y from-y))
	       (values (- from-x distance) (+ from-y distance)
		       (- from-x distance) (- from-y distance)
		       (- to-x distance) (- to-y distance)
		       (+ to-x distance) (- to-y distance)
		       (+ to-x distance) (+ to-y distance)
		       (+ from-x distance) (+ from-y distance))))
      (draw-line* stream x1 y1 x2 y2)
      (draw-line* stream x2 y2 x3 y3)
      (draw-line* stream x3 y3 x4 y4)
      (draw-line* stream x4 y4 x5 y5)
      (draw-line* stream x5 y5 x6 y6)
      (draw-line* stream x6 y6 x1 y1))))

(define-graphics-operator draw-point
    :bounding-box-points ((point-x point-y))
    ;; --- This could benefit from a highlighting test when thickness is big.
    :highlighting-function
    ((medium)
     line-style
     (let ((radius (1+ (ls-thickness))))
       (draw-circle* medium point-x point-y radius :filled nil))))

(define-graphics-operator draw-points
    :bounding-box (bounding-box-from-coordinate-sequence coordinate-sequence)
    ;; --- This desparately needs a highlighting test.
    :highlighting-function
    ((medium)
     (let ((radius (1+ (ls-thickness))))
       (do-coordinates (coordinate-sequence point-x point-y transformation)
	 (draw-circle* medium point-x point-y radius :filled nil)))))

(define-graphics-operator draw-line
    :bounding-box-points ((from-x from-y) (to-x to-y))
    :highlighting-test
    ((x y)
     (multiple-value-bind (x y)
    	 (untransform-point* transformation x y)
        (point-close-to-line-p x y from-x from-y to-x to-y (ls-thickness))))
    :highlighting-function
    ((medium)
     (outline-line-with-hexagon medium from-x from-y to-x to-y
    				(ls-thickness))))

(define-graphics-operator draw-lines
    :bounding-box (bounding-box-from-coordinate-sequence coordinate-sequence)
    :highlighting-test
    ((x y)
     (multiple-value-bind (x y)
	 (untransform-point* transformation x y)
       (let ((thickness (ls-thickness)))
	 (loop (when (null coordinate-sequence) (return nil))
	       (let ((x1 (pop coordinate-sequence))	;By the time it's in an output record,
		     (y1 (pop coordinate-sequence))	;we assume the coordinate sequence
		     (x2 (pop coordinate-sequence))	;has a reasonable number of values,
		     (y2 (pop coordinate-sequence)))	;so we don't use ENDP.
		 (when (point-close-to-line-p x y x1 y1 x2 y2 thickness)
		   (return t)))))))
    :highlighting-function
    ((medium)
     (let ((thickness (ls-thickness)))
       (loop (when (null coordinate-sequence) (return nil))
	     (let ((x1 (pop coordinate-sequence))
		   (y1 (pop coordinate-sequence))
		   (x2 (pop coordinate-sequence))
		   (y2 (pop coordinate-sequence)))
	       (outline-line-with-hexagon medium x1 y1 x2 y2 thickness))))))

(define-graphics-operator draw-polygon
    :bounding-box (bounding-box-from-coordinate-sequence coordinate-sequence)
    ;; --- This needs a highlighting test.
    :highlighting-function
    ;; Not great.  We really want to make this draw just outside of the actual polygon...
    ((medium)
     (draw-polygon* medium coordinate-sequence :filled nil)))

(define-graphics-operator draw-rectangle
   :bounding-box-points ((x1 y1) (x1 y2) (x2 y1) (x2 y2))
   :highlighting-test
   ((x y)
    (multiple-value-bind (x y) (untransform-point* transformation x y)
      (let ((thickness/2 (ls-thickness/2)))
      (macrolet ((in-bounds (r r1 r2)
		   `(if (< ,r1 ,r2)
			;; --- This isn't really right: thickness might not scale.
			(<= (- ,r1 thickness/2) ,r (+ ,r2 thickness/2))
			(>= (+ ,r1 thickness/2) ,r (- ,r2 thickness/2)))))
	(and (in-bounds x x1 x2)
	     (in-bounds y y1 y2))))))
   :highlighting-function
   ((medium)
    (let ((x1 x1) (y1 y1) (x2 x2) (y2 y2)
	  (thickness/2 (ls-thickness/2)))
      (macrolet ((incf-decf (r1 r2) `(progn (incf ,r1 thickness/2) (decf ,r2 thickness/2))))
	(if (< x1 x2) (incf-decf x2 x1) (incf-decf x1 x2))
	(if (< y1 y2) (incf-decf y2 y1) (incf-decf y1 y2)))
      (draw-rectangle* medium x1 y1 x2 y2 :filled nil))))

;;; Have to deal with START-ANGLE, END-ANGLE
(define-graphics-operator draw-ellipse
    :bounding-box (let ((cx center-x) (cy center-y)
			(max-dx (+ (abs radius-1-dx) (abs radius-2-dx)))
			(max-dy (+ (abs radius-1-dy) (abs radius-2-dy))))
		    (macrolet ((include (x-sign y-sign)
				 `(include-point (,x-sign cx max-dx) (,y-sign cy max-dy))))
		      (include + +) (include + -) (include - +) (include - -)))    
    :output-record-slots ((X^2-coeff :initform nil)
			  (XY-coeff  :initform nil)
			  (Y^2-coeff :initform nil)
			  (RHS :initform nil))
    :highlighting-test				;; Only works for full/filled ellipses??
    ((x y)					;; Genera doesn't do much better.  See below.
;;; (dy1^2 + dy2^2)*X^2 - 2*(dx1*dy1 + dx2*dy2)*X*Y + (dx1^2 + dx2^2)*Y^2
;;;	       (dx1*dy2 - dx2*dy1)^2		-- JGA
     ;; --- Need to take line thickness into account here.
     (multiple-value-bind (x y) (untransform-point* transformation x y)
       (with-slots (X^2-coeff XY-coeff Y^2-coeff RHS) record
	 (unless X^2-coeff
	   (setf X^2-coeff (+ (expt radius-1-dy 2) (expt radius-2-dy 2))
		 XY-coeff  (* -2 (+ (* radius-1-dx radius-1-dy)
				    (* radius-2-dx radius-2-dy)))
		 Y^2-coeff (+ (expt radius-1-dx 2) (expt radius-2-dx 2))
		 RHS (expt (+ (* radius-1-dx radius-2-dy)
			      (* radius-2-dx radius-1-dy))
			   2)))
	 (let ((x (- x center-x))
	       (y (- y center-y)))
	   (<= (+ (* x x X^2-coeff) (* x y XY-coeff) (* y y Y^2-coeff))
	       RHS)))))
    :highlighting-function
    ((medium)
     (let ((radius-1-dx radius-1-dx)
	   (radius-1-dy radius-1-dy)
	   (radius-2-dx radius-2-dx)
	   (radius-2-dy radius-2-dy)
	   (thickness/2 (ls-thickness/2)))
       (macrolet ((update (&rest things)
		    `(progn ,@(mapcar #'(lambda (thing)
					  `(incf ,thing (* thickness/2 (signum ,thing))))
				      things))))
	 (update radius-1-dx radius-1-dy radius-2-dx radius-2-dy))
       (draw-ellipse* medium center-x center-y
		      radius-1-dx radius-1-dy radius-2-dx radius-2-dy
		      :filled nil :start-angle start-angle :end-angle end-angle))))

#|| Test for Genera's highlighting/mouse-sensitivity-testing code:
(dw:with-output-as-presentation (:object 42 :type 'sys:expression )
  (graphics:with-room-for-graphics ()
    (graphics:draw-circle 30 30 8 :filled nil :thickness 4
			  :transform '(8 -2 4 12 0 0) :start-angle .2 :end-angle 1.9)))
||#

;;; Have to deal with START-ANGLE, END-ANGLE
(define-graphics-operator draw-circle
    :bounding-box (let ((cx center-x) (cy center-y)
			(r (abs radius)))
		    (include-point (+ cx r) cy)
		    (include-point (- cx r) cy)
		    (include-point cx (+ cy r))
		    (include-point cx (- cy r)))
    :output-record-slots ((r^2 :initform nil))
    :highlighting-test ((x y)
			(multiple-value-bind (x y) (untransform-point* transformation x y)
			  (with-slots (r^2) record
			    (when (null r^2)
			      (let* ((thickness/2 (if filled 0 (ls-thickness/2)))
				     (r (+ radius (abs thickness/2))))
				(setf r^2 (* r r))))
			    (let ((x (- x center-x))
				  (y (- y center-y)))
			      (<= (+ (* x x) (* y y)) r^2)))))
    :highlighting-function
    ((medium)
     (let ((thickness/2 (if filled 1 (ls-thickness/2))))
       (draw-circle* medium center-x center-y (+ radius thickness/2)
		     :filled nil :start-angle start-angle :end-angle end-angle))))


;;; DRAW-TEXT has to work differently because the size of the bounding box cannot be calculated
;;; in the abstract; it only has meaning with respect to a medium.  Thus, we have to calculate
;;; it at the time the output record is created, rather than the first time anyone asks for it.
;;; This really isn't too terrible, despite it looking like an enormous kludge...

;;;  This still doesn't interact very well with the STREAM-MERGED-TEXT-STYLE stuff, for reasons
;;;  which are too complicated to fit into this comment.  Talk to rsl for more information.
(define-graphics-operator draw-text
    :bounding-box (error "Internal error: ~S output records cannot be calculated this way.~%~
			  Please report this error to the CLIM maintainers." 'draw-text
			 (include-point 0 0))			;Kludge -- suppress warning
    :output-recording-before-hook (when (temporary-string-p string-or-char)
				    (setf string-or-char (subseq string-or-char start end)
					  start 0 end (length string-or-char)))
    :output-recording-after-hook (compute-draw-text-bounding-box record stream))

(defmethod compute-draw-text-bounding-box ((record text-output-record) stream)
  ;; Copy this out of graphics-operation-around-method-helper because
  ;; it gets called in the wrong order for text and it's just not worth
  ;; figuring out a more complicated way of fixing this now.
  ;; --- Doughty 2/15/91
  (multiple-value-bind (dx dy)
      (point-position* (output-recording-stream-output-record-absolute-position stream))
    (with-slots (start-x start-y) record
      (setf start-x (- dx) start-y (- dy))))
  (with-slots-bound-as-needed (string-or-char start end text-style point-x point-y
					      align-x align-y transformation
					      toward-point transform-glyphs
					      start-x start-y)
			      record
    (multiple-value-bind (left top right bottom)
	(calculating-bounding-box (nil start-x start-y)
	  (multiple-value-bind (off-x off-y)
	      (transform-point* transformation point-x point-y)
	    (etypecase string-or-char
	      (string
		(flet ((handle-string-box (left top right bottom baseline)
			 (ecase align-x
			   (:left)
			   (:right (let ((width (- right left)))
				     (decf left width) (decf right width)))
			   (:center (let ((width/2 (floor (- right left) 2)))
				      (decf left width/2) (decf right width/2))))
			 (ecase align-y
			   (:baseline (decf top baseline) (decf bottom baseline))
			   (:top)
                           (:center (let ((height/2 (floor (- bottom top) 2)))
                                      (decf top height/2) (decf bottom height/2)))
			   (:bottom (let ((height (- bottom top)))
				      (decf bottom height) (decf top height))))
			 (include-point (+ off-x left) (+ off-y top))
			 (include-point (+ off-x left) (+ off-y bottom))
			 (include-point (+ off-x right) (+ off-y top))
			 (include-point (+ off-x right) (+ off-y bottom))))
		  (declare (dynamic-extent #'handle-string-box))
		  (clim-stream::do-text-screen-real-estate
		    stream #'handle-string-box string-or-char
		    (or start 0) (or end (length string-or-char)) 0 0 0 0
		    text-style most-positive-fixnum)))
	      (character
		(multiple-value-bind (index font escapement-x escapement-y
				      origin-x origin-y bb-x bb-y)
		    (clim-stream::stream-glyph-for-character stream string-or-char text-style)
		  (declare (ignore index font origin-x escapement-y))
		  (let ((left point-x)
			(top point-y)
			(right (+ point-x escapement-x))
			(bottom (+ point-y bb-y)))
		    (ecase align-x
		      (:left)
		      (:right (decf left bb-x) (decf right bb-x))
		      (:center (let ((width/2 (floor bb-x 2)))
				 (decf left width/2) (decf right width/2))))
		    (ecase align-y
		      (:top)
		      (:bottom (decf top bb-y) (decf bottom bb-y))
		      (:baseline (decf top origin-y) (decf bottom origin-y)))
		    (include-point (+ off-x left) (+ off-y top))
		    (include-point (+ off-x left) (+ off-y bottom))
		    (include-point (+ off-x right) (+ off-y top))
		    (include-point (+ off-x right) (+ off-y bottom))))))))
      (with-slots (min-x min-y max-x max-y) record
	(setf min-x left min-y top max-x right max-y bottom)))))
