;;; -*- Base: 10; Mode: LISP; Package: CLIM-INTERNALS; Syntax: Common-Lisp -*-

(in-package "CLIM-INTERNALS")

"Copyright (c) 1989, 1990 International Lisp Associates.  All rights reserved."

(defmethod read-frame-command ((frame frame) stream
			       ;; should the rest of the *command-parser*
			       ;; etc. variables be passed as keywords or bound?
			       )
  (read-command (frame-command-table frame) :stream stream))

;;; --- We obviously aren't done with the design of this part.  Why the special reference to
;;; --- *standard-input*, why the (class-name (class-of frame)), why the big suit?
(defmethod frame-read-remaining-arguments-for-partial-command ((frame frame) partial-command)
  (menu-only-read-remaining-arguments-for-partial-command
    partial-command
    (find-command-table (class-name (class-of frame)))
    *standard-input*
    0))

;;; --- This is a temporary, perhaps, function to fix up the Silica tree to be more like
;;; --- CLIM expects it.  In particular, the frame and its stream panes must share a queue, 
;;; --- and all the CLIM streams must share the same pointer object.
(defmethod initialize-stream-queues ((frame frame))
  (with-slots (queue) frame
    (unless queue
      ;; --- do we really want to use the queue rather than some other stack?
      (setf queue (make-queue)))
    (flet ((initialize-queue (pane depth nth)
	     (declare (ignore depth nth))
	     (when (typep pane 'extended-stream-pane)
	       (setf (stream-input-buffer pane) queue))))
      (declare (dynamic-extent #'initialize-queue))
      (walk-tree #'initialize-queue (frame-pane frame)))))

(defun clim-top-level (frame &key (prompt "Command:  "))
  (initialize-stream-queues frame)
  (panes-need-redisplay frame)
  ;; initialize panes somehow.
  (let ((*standard-output*
	  (block find-it
	    (walk-tree #'(lambda (pane depth nth)
			   (declare (ignore depth nth))
			   (when (typep pane 'extended-stream-pane)
			     (return-from find-it pane)))
		       (frame-pane frame))))
	(interactor (frame-interactor frame)))
    (loop
      ;; --- bind standard-output, query-io, etc.
      ;; --- redisplay the redisplayable panes.
      (redisplay-frame-panes frame)
      (catch 'command-executed
	(catch-abort-gestures ("Return to ~A command level" (class-name (class-of frame)))
	  (let ((*command-parser* (if interactor *command-parser* 'menu-only-command-parser))
		(*partial-command-parser*
		  (if interactor
		      *partial-command-parser*
		      'menu-only-read-remaining-arguments-for-partial-command)))
	    (let ((*standard-input* (or interactor *standard-output*)))
	      (when interactor
		(if (stringp prompt)
		    (write-string prompt *standard-input*)
		    (funcall prompt *standard-input* frame)))
	      (progn
		#+ignore
		;; --- This was losing REPAINT events
		(stream-clear-input *standard-input*)		;kludge
		(let ((command (read-frame-command frame *standard-input*)))
		  (when interactor
		    (fresh-line *standard-input*))
		  (when command
		    (execute-frame-command frame command))))))))
      (when (frame-prop frame :resynchronize-after-command)
	(setf (frame-prop frame :resynchronize-after-command) nil)
	(throw 'resynchronize t)))))

;;; This is the function that the application programmer uses, as in
;;; :display-function '(display-command-menu foo-menu-group)
(defun display-command-menu-group (frame pane menu-group 
				   &key
				   n-columns n-rows
				   (inter-column-spacing 10)
				   inter-row-spacing)
  (window-clear pane)
  (let ((menu-group (frame-find-menu-group frame menu-group)))
    (display-menu-group menu-group pane
			:n-columns n-columns
			:n-rows n-rows
			:inter-column-spacing inter-column-spacing
			:inter-row-spacing inter-row-spacing
			:move-cursor nil)))

(defun incremental-redisplay-display-function
       (frame pane display-function &rest display-function-args)
  (declare (dynamic-extent display-function-args))
  (let ((redisplay-record (pane-prop pane 'incremental-redisplay-record)))
    (cond (redisplay-record
	   (redisplay redisplay-record pane))
	  (t
	   (window-clear pane)					;--- Seems like we have to do this.
	   (setf (pane-prop pane 'incremental-redisplay-record)
		 (updating-output (pane)
		   (apply display-function frame pane display-function-args)))))))

;;; The contract of this is to replay the contents of STREAM within
;;; the bounding-rectangle.
(defmethod frame-replay ((frame frame) stream &optional region)
  (output-recording-stream-replay stream region)
  (force-output stream))

(defmethod frame-input-context-button-press-handler
	   ((frame frame) stream button-press-event)
  (declare (ignore stream))
  (let* ((window (event-sheet button-press-event))
	 (x (event-x button-press-event))
	 (y (event-y button-press-event))
	 (highlighted-presentation (highlighted-presentation window nil))
	 (input-context *input-context*))
    (when highlighted-presentation
      ;; Unhighlight on the way out.
      ;; But only unhighlight the window that the click is from. 
      (unhighlight-highlighted-presentation window nil))
    (throw-highlighted-presentation 
      (or (and (output-recording-stream-p window)
	       (frame-find-innermost-applicable-presentation
		 frame input-context window x y))
	  *null-presentation*)
      input-context
      button-press-event)))

(defmethod frame-find-innermost-applicable-presentation
	   ((frame frame) input-context stream x y)
  (find-innermost-applicable-presentation 
    input-context stream x y
    (window-shift-mask stream) frame))

(defmethod frame-interactor ((frame frame))
  (flet ((find-it (pane &rest ignore)
           (declare (dynamic-extent ignore)
		    (ignore ignore))
	   (when (typep pane 'clim-interactor)
	     (return-from frame-interactor pane))))
    (declare (dynamic-extent #'find-it))
    (walk-sheets (frame-pane frame) #'find-it #'false nil)))

(defmethod frame-pointer-documentation-output ((frame frame))
  ;; If it's a Genera frame, use the mouse-doc line, otherwise find
  ;; the pointer documentation pane.
  #+Ignore ;;--- ok, don't do this right now.  doughty 1/4/92
  (or (pointer-documentation-handler (port frame))
      (flet ((find-it (pane &rest ignore)
	       (declare (dynamic-extent ignore)
			(ignore ignore))
	       (when (typep pane 'clim-pointer-documentation)
		 (return-from frame-pointer-documentation-output pane))))
	(declare (dynamic-extent #'find-it))
	(walk-sheets (frame-pane frame) #'find-it #'false nil))))

(defvar *pointer-documentation-interval*
	(max (floor (* 1/10 internal-time-units-per-second)) 1))
(defvar *last-pointer-documentation-time* 0)

#+Genera
(defvar *pointer-documentation-buffer*
	(make-array 80 :element-type 'cl:string-char :fill-pointer 0 :adjustable t))

;;; Produce pointer documentation
(defmethod frame-document-highlighted-presentation
	   ((frame frame) presentation input-context window x y stream)
  (let (#+Genera (documentation-window (mouse-documentation-window window)))
    ;; The documentation should never say anything if we're not over a presentation
    (when (null presentation) 
      #+Genera (if documentation-window
		   (scl:send documentation-window :clear-window)
		   (window-clear stream))
      #-Genera (window-clear stream))
    ;; Cheap test to not do this work too often
    (let ((old-shift-mask *last-pointer-documentation-shift-mask*)
	  (shift-mask (window-shift-mask window))
	  (last-time *last-pointer-documentation-time*)
	  (time (get-internal-real-time)))
      (setq *last-pointer-documentation-shift-mask* shift-mask)
      (when (and (< time (+ last-time *pointer-documentation-interval*))
		 (= shift-mask old-shift-mask))
	(return-from frame-document-highlighted-presentation nil))
      (setq *last-pointer-documentation-time* time))
    (when presentation
      #+Genera
      (when documentation-window
	(setf (fill-pointer *pointer-documentation-buffer*) 0)
	(with-output-to-string (stream *pointer-documentation-buffer*)
	  (scl:send documentation-window :clear-window)
	  (when (null (frame-document-highlighted-presentation-internal
			frame presentation input-context window x y stream))
	    (setq *last-pointer-documentation-time* 0))
	  (scl:send documentation-window :string-out *pointer-documentation-buffer*))
	(return-from frame-document-highlighted-presentation nil))
      (with-output-recording-options (stream :record-p nil)
	(with-end-of-line-action (:allow stream)
	  (with-end-of-page-action (:allow stream)
	    (window-clear stream)
	    (when (null (frame-document-highlighted-presentation-internal
			  frame presentation input-context window x y stream))
	      (setq *last-pointer-documentation-time* 0))
	    (force-output stream)))))))

(defun frame-document-highlighted-presentation-internal
       (frame presentation input-context window x y stream)
  (let ((shift-mask (window-shift-mask window)))
    (declare (fixnum shift-mask))
    (multiple-value-bind (left left-context middle middle-context right right-context)
	(find-applicable-translators-for-documentation presentation input-context
						       frame window x y shift-mask)
      (let* ((*print-length* 3)
	     (*print-level* 2)
	     (*print-circle* nil)
	     (*print-array* nil)
	     (*print-readably* nil)
	     (*print-pretty* nil))
	(flet ((document-translator (translator context-type button-name separator)
		 ;; Assumes 5 shifts and the reverse ordering of *POINTER-SHIFTS*
		 (let ((bit #o20)
		       (shift-name '("h-" "s-" "m-" "c-" "sh-")))
		   (declare (fixnum bit))
		   (dotimes (i 5)
		     #-excl (declare (ignore i))
		     (unless (zerop (logand bit shift-mask))
		       (write-string (car shift-name) stream))
		     (pop shift-name)
		     (setq bit (the fixnum (ash bit -1)))))
		 (write-string button-name stream)
		 (document-presentation-translator translator presentation context-type
						   frame nil window x y
						   :stream stream
						   :documentation-type :pointer)
		 (write-string separator stream)))
	  (declare (dynamic-extent #'document-translator))
	  (when left
	    (let ((button-name (cond ((and (eql left middle)
					   (eql left right))
				      (setq middle nil
					    right nil)
				      "L,M,R: ")
				     ((eql left middle) 
				      (setq middle nil)
				      "L,M: ")
				     (t "L: "))))
	      (document-translator left left-context button-name
				   (if (or middle right) "; " "."))))
	  (when middle
	    (let ((button-name (cond ((eql middle right)
				      (setq right nil)
				      "M,R: ")
				     (t "M: "))))
	      (document-translator middle middle-context button-name
				   (if right "; " "."))))
	  (when right
	    (document-translator right right-context "R: " "."))
	  ;; Return non-NIL if any pointer documentation was produced
	  (or left middle right))))))

;; This is derived directly from FIND-APPLICABLE-TRANSLATORS
(defun find-applicable-translators-for-documentation
       (presentation input-context frame window x y shift-mask)
  ;; Assume a maximum of three pointer buttons
  (let ((left nil)   (left-context nil)
	(middle nil) (middle-context nil)
	(right nil)  (right-context nil))
    (macrolet ((match (translator context-type button)
		 (let ((button-translator (intern (symbol-name button)))
		       (button-context (fintern "~A-~A" (symbol-name button) 'context)))
		   `(when (and (or (null ,button-translator)
				   (> (presentation-translator-priority ,translator)
				      (presentation-translator-priority ,button-translator)))
			       (keysym-and-shift-mask-matches-gesture-spec
				 ,button shift-mask
				 (presentation-translator-gesture-name ,translator)))
		      (setq ,button-translator ,translator
			    ,button-context ,context-type)))))
      (do ((presentation presentation
			 (parent-presentation-with-shared-box presentation window)))
	  ((null presentation))
	(let ((from-type (presentation-type presentation)))
	  (dolist (context input-context)
	    (let ((context-type (pop context)))	;input-context-type = first
	      (let ((translators (find-presentation-translators from-type context-type frame)))
		(when translators
		  (dolist (translator translators)
		    (when (test-presentation-translator translator
							presentation context-type
							frame window x y
							:shift-mask shift-mask)
		      (match translator context-type :left)
		      (match translator context-type :middle)
		      (match translator context-type :right)))))
	      (when (and (or left middle right)
			 *presentation-menu-translator*
			 (test-presentation-translator *presentation-menu-translator*
						       presentation context-type
						       frame window x y
						       :shift-mask shift-mask))
		(match *presentation-menu-translator* context-type :left)
		(match *presentation-menu-translator* context-type :middle)
		(match *presentation-menu-translator* context-type :right)))))))
    (values left   left-context
	    middle middle-context
	    right  right-context)))

#+Genera
(defmethod mouse-documentation-window ((window extended-stream-pane)) nil)

;;--- This causes direct-manipulation and menu-driven applications not to
;;--- maintain histories.  Is there a better heuristic?
(defmethod frame-maintain-presentation-histories ((frame frame))
  (not (null (frame-interactor frame))))

(defmethod run-frame-top-level :around ((frame frame))
  (with-simple-restart (frame-exit "Exit ~A" (class-name (class-of frame)))
    (let (
	  ;; Reset the state of the input editor and the presentation
	  ;; type system, etc., in case there is an entry into another
	  ;; application from inside the input editor, such as a Debugger
	  ;; written using CLIM.
	  ;;--- This should be done in a more modular way
	  ;;--- If you change this, change MENU-CHOOSE-FROM-DRAWER
	  (*outer-self* nil)
	  (*input-wait-test* nil)
	  (*input-wait-handler* nil)
	  (*pointer-button-press-handler* nil)
	  (*numeric-argument* nil)
	  (*blip-characters* nil)
	  (*activation-characters* nil)
	  (*accelerator-gestures* nil)
	  (*input-context* nil)
	  (*accept-help* nil)
	  (*assume-all-commands-enabled* nil)
	  (*command-parser* 'command-line-command-parser)
	  (*command-unparser* 'command-line-command-unparser)
	  (*partial-command-parser*
	    'command-line-read-remaining-arguments-for-partial-command))
      (loop
	(with-simple-restart (nil "~A top level" (class-name (class-of frame)))
	  (loop
	    (catch 'resynchronize
	      (let ((*application-frame* frame)
		    (*pointer-documentation-output*
		      (frame-pointer-documentation-output frame)))
		;; We must return the values from CALL-NEXT-METHOD,
		;; or else ACCEPTING-VALUES will return NIL
		#-CCL-2
		(return-from run-frame-top-level (call-next-method))
		;; The (RETURN-FROM FOO (CALL-NEXT-METHOD)) form above
		;; doesn't work in Coral.  If the "top level" restart
		;; above is taken, the CALL-NEXT-METHOD form blows out
		;; the second time through this code, claiming that it
		;; can't find the next method.  Hoisting the
		;; CALL-NEXT-METHOD out of the RETURN-FROM form seems
		;; to fix it...  So it conses, big deal.
		#+CCL-2
		(let ((results (multiple-value-list (call-next-method))))
		  (return-from run-frame-top-level (values-list results)))))))))))
