;;; -*- Base: 10; Mode: LISP; Package: CLIM-INTERNALS; Syntax: Common-Lisp -*-

;;; Also needed, a mechanism to pop up a menu that consists of a menu group display, perhaps
;;; in such a context as to cause a subsequent click to go to the next menu
;;; level.

(in-package "CLIM-INTERNALS")

"Copyright (c) 1990 International Lisp Associates.  All rights reserved."

;;; This should never be called with acceptably T
(define-presentation-type menu-group-element ())

(define-presentation-method present (object (type menu-group-element) stream
				     (view textual-view) &key acceptably)
  (when acceptably
    (error "We don't know how to print menu-group-elements acceptably."))
  (display-menu-group-element object stream))

(defun display-menu-group-element (menu-group-element stream)
  (format stream "~A" (menu-group-element-string menu-group-element)))

(defun display-menu-group (menu-group stream
				      &key max-width max-height
				      inter-column-spacing inter-row-spacing n-rows n-columns
				      move-cursor)
  (unless (and max-width max-height)
    (let* ((region (sheet-region stream))
	   (width  (bounding-rectangle-width region))
	   (height (bounding-rectangle-height region)))
      (unless max-width (setf max-width width))
      (unless max-height (setf max-height height))))
  (display-menu-group-internal 
   stream menu-group
   :max-width max-width :max-height max-height
   :n-rows n-rows :n-columns n-columns
   :inter-column-spacing inter-column-spacing
   :inter-row-spacing inter-row-spacing
   :move-cursor move-cursor))

(defun display-menu-group-internal (stream menu-group &key max-width max-height
					   inter-column-spacing inter-row-spacing
					   n-rows n-columns
					   move-cursor)
  (let ((elements (menu-group-elements menu-group)))
    (if (zerop (length elements))
	(with-text-face (:italic stream)
	  (write-string "[No Menu Items]" stream))
	(formatting-item-list (stream :inter-column-spacing inter-column-spacing
				      :inter-row-spacing inter-row-spacing
				      :n-rows n-rows :n-columns n-columns
				      :max-width max-width :max-height max-height
				      :move-cursor move-cursor)
	  (dovector (element elements)
	    (formatting-cell (stream)
	      (present element 'menu-group-element :stream stream)))))))

(define-presentation-translator menu-group-element->command
				(menu-group-element command global-command-table)
				(object)
  (let ((type (menu-group-element-type object)))
    (when (eql type ':command)
      (menu-group-element-value object))))

#||

(define-application-frame test-list ()
  ((list :initform nil)
   (menu) (display))
  (:pane
    (with-frame-slots (menu display)
      (vertically ()
	(make-pane 'xform-pane :contents
	   (setq menu
		 (make-pane 'extended-stream-pane
		    :display-function '(new-display-command-menu test-list-menu-group)
		    :display-time :always!
		    :hs 400 :vs 40)))
	(make-pane 'xform-pane :contents
	     (setq display
		   (make-pane 'extended-stream-pane :display-function '(display-test-list)
		      :display-time :command-loop
		      :hs 400 :vs 200))))))
  (:command-definer t)
  (:menu-group
    (("Add One" :command '(com-add-one))
     ("Clear" :command '(com-clear))

     ("Delete" :command (build-command 'com-delete-item))
     ))
  (:top-level (clim-top-level)))

(defun test-list (&key (host *default-host*) (port-type *default-port-type*)
			       (display-id *default-display-id*))
  (let ((framem (find-frame-manager :host host :display-id display-id
				    :port-type port-type))
        (frame (make-frame 'test-list)))
    (adopt-frame framem frame)
    (enable-frame frame)
    frame))

(define-presentation-type test-list-element () )

(defmethod display-test-list ((frame test-list) pane)
  (with-slots (list) frame
    (window-clear pane)
    (when list
      (formatting-table (pane)
	(let ((count 0))
	  (dolist (item list)
	    (incf count)
	    (with-output-as-presentation (:stream pane
					  :type 'test-list-element
					  :object item)
	      (formatting-row (pane)
		(formatting-cell (pane)
		  (format pane "~D" count))
		(formatting-cell (pane)
		  (format pane "~A" item))))))))))

(define-test-list-command (com-add-one :command-name "Add One")
    ()
   (with-slots (list) frame
     (push (gensym) list)))

(define-test-list-command (com-clear :command-name "Clear")
    ()
   (with-slots (list) frame
     (setf list nil)))

(define-test-list-command (com-delete-item :command-name "Delete")
    ((item 'test-list-element :translator-gesture :left))
   (with-slots (list) frame
     (setf list (delete item list))))
||#

#||

()

(define-menu-group menu-group-test
  (("Quit" :command '(com-quit))
   ("File" :menu-group 'menu-group-test-file-menu-group)
   ("Hardcopy" :command (build-command 'com-hardcopy-file))
   ("Printer" :menu-group 'menu-group-test-printer-menu-group)))

(define-menu-group menu-group-test-file-menu-group
  (("Copy" :command '(com-copy-file))
   ("Print Name" :command '(com-print-file-name))))

(define-menu-group menu-group-test-printer-menu-group
  (("Hardcopy File" :command (build-command 'com-hardcopy-file))))

(defun another-cp-test (frame)
  (enable-frame frame)
  (let ((pane (sheet-child (frame-pane frame))))
    (window-clear pane)
    (let ((*command-table* 'test-cp))
      (new-display-command-menu frame pane 'menu-group-test)
      (stream-set-cursor-position* pane 0 50)
      ;; initially
      (stream-clear-input pane)
      (unwind-protect
	  (catch 'exit-cp
	    (progn						; with-input-focus (stream)
	      (loop
		(catch 'abort-command-read
		  (format pane "~%Command: ")
		  (let ((command (read-command :stream pane)))
		    (terpri pane)
		    (let ((command-function (pop command))
			  (command-args command))
		      (let ((*standard-output* pane))
			(apply command-function command-args))))))))
	))))


||#
