;;; -*- Mode: Lisp; Package: CLIM-INTERNALS; Base: 10.; Syntax: Common-Lisp -*-
;;;
;;; Copyright (c) 1990 by Xerox Corporations.  All rights reserved.
;;;

(in-package "CLIM-INTERNALS")

"Improvements Copyright (c) 1990 by International Lisp Associates.  All rights reserved."

;;;
;;; New Stream Pane
;;;

(defclass clim-pane-stream-mixin (window-stream)
    ()
  (:default-initargs :medium t))

;;; Do we still need this
(defmethod pane-stream ((pane clim-pane-stream-mixin))
  pane)

;;; Communicate with output-protocol what the new text constraints are.
(defmethod sheet-region-changed ((pane clim-pane-stream-mixin)
				 &key &allow-other-keys)
  (let ((viewport (pane-viewport pane)))
    (when viewport
      (setf (stream-default-text-margin pane)
	    (bounding-rectangle-width (sheet-region (pane-viewport pane)))))))

;;; Communicate with output-protocol what the new text constraints are.
(defmethod ws::viewport-region-changed ((pane clim-pane-stream-mixin) viewport)
  (setf (stream-default-text-margin pane)
	(bounding-rectangle-width (sheet-region viewport))))

(define-sheet-class extended-stream-sheet (mute-input-mixin
					    ;; ---These can't be here, 'cause the
					    ;; define-sheet-class macro puts
					    ;; these superclasses after the
					    ;; "standard" superclasses it
					    ;; inserts, rather than before, so
					    ;; b-r-m gets shadowed by
					    ;; mute-repainter.  So, I put them on
					    ;; extended-stream-pane below -York
					    ;;ws::pane-background-mixin
					    ;;w::providing-output-contract
					    ws::display-function-mixin
					    pane
					    leaf-mixin
					    space-req-mixin)
    ()
  (:youth-contract-class dashboard-windowing-contract)
  (:input-contract-class standard-input-contract)
  (:output-contract-class standard-output-contract)
  (:default-initargs :medium t 
    :hs+ +fill+ :hs- +fill+ :vs+ +fill+ :vs- +fill+))

;;; We should straighten this out so that this can be
;;; subclass of LEAF-PANE.  Then it would get
;;; providing-output-contract.
(defclass extended-stream-pane (clim-pane-stream-mixin
				 ;; see --- comment above
				 ;; stream panes should fill background
				 ws::pane-background-mixin
				 ;; see --- comment above
				 ;; We want a medium provided to us
				 w::providing-output-contract
				 extended-stream-sheet)
    (#-clim-uses-lisp-streams (%gray-stream :initform nil)))
	   
(defmethod initialize-instance :after 
	   ((pane extended-stream-pane) &key &allow-other-keys)
  (setf (sheet-transformation pane) +identity-transformation+))

(defmethod sheet-grafted :after ((pane extended-stream-pane))
  (let ((xform (sheet-transformation pane)))
    (setq xform (compose-with-scaling
		  +identity-transformation+ 1 -1
		  :reuse xform))
    (setf (sheet-transformation pane) xform)))

;;; This is a soon-to-be-obsolete method, but we need it for now when the
;;; extended-stream-pane is a child of the old-style viewport.  It shouldn't
;;; get called under the new viewport scheme.
(defmethod allocate-space :after ((pane extended-stream-pane) width height)
  (declare (ignore width height))
  (ecase (graft-origin (graft pane))
    (:nw)
    (:sw
     (let ((xform (sheet-transformation pane)))
       (setq xform (compose-with-scaling
		     +identity-transformation+ 1 -1
		     :reuse xform))
       ;; --- stream-panes ALWAYS have to have a parent to manage the
       ;; viewport clipping, etc.
       (setq xform (compose-with-translation
		     xform
		     0 (1- (bounding-rectangle-height
			     (sheet-parent pane)))
		     :reuse xform))
       (setf (sheet-transformation pane) xform)))))

(defmethod pane-stream ((pane extended-stream-pane))
  (unless (port pane) 
    (error "Can't call pane-stream on ~a until it's been grafted!"
	   pane))
  #-clim-uses-lisp-streams
  (with-slots (%gray-stream) pane
    (or %gray-stream
	(setf %gray-stream (clim-stream::make-%gray-stream pane))))
  #+clim-uses-lisp-streams pane)

(defmethod update-region ((pane extended-stream-pane) width height
			  &key no-repaint &allow-other-keys)
  (when (pane-scroller pane)
    (update-extent (pane-viewport pane) width height
		       :no-repaint no-repaint)))

;;; ---This assumes that the stream-pane is always inside a viewport, which
;;; actually defines its visible size.  The stream pane's size is supposed
;;; to represent the size of the contents, but may be stretched to fill the
;;; available viewport space.
(defmethod change-space-req :around ((pane extended-stream-pane) &rest keys &key hs vs)
  (declare (dynamic-extent keys))
  ;; Assume always called with hs vs
  (multiple-value-bind (history-width history-height)
      (bounding-rectangle-size (output-recording-stream-output-record pane))
    ;; Don't ever shrink down smaller than our contents.
    (apply #'call-next-method pane :hs (max hs history-width) :vs (max vs history-height) keys)))

(defclass clim-interactor (extended-stream-pane) ())

(defmacro make-clim-pane ((&optional slot-name
				     &rest parent-options
				     &key (type ''extended-stream-pane) 
				     label (scroll-bars ':vertical)
				     &allow-other-keys)
			  &rest pane-options)
  (with-rem-keywords (parent-options parent-options '(:type :label :scroll-bars))
    (let ((default '#:default)
	  (pane (gensymbol "SCROLLABLE-PANE")))
      (macrolet ((setf-unless (slot-keyword value)
		   `(when (eq (getf parent-options ',slot-keyword default) default)
		      (setf (getf parent-options ',slot-keyword) ,value))))
	(setf-unless :hs 100)
	(setf-unless :vs 100)
	(setf-unless :hs+ +fill+)
	(setf-unless :hs- +fill+)
	(setf-unless :vs+ +fill+)
	(setf-unless :vs- +fill+))
      `(bordering (:thickness 1)
		  (let ((,pane (make-pane ,type ,@pane-options)))
		    ,@(when slot-name
			`((setq ,slot-name ,pane)))
		    (vertically ()
				(,(ecase scroll-bars
				    (:both 'scrolling)
				    (:vertical 'vscrolling)
				    (:horizontal 'hscrolling)
				    ((nil) 'ws::viewing))
				 (:subtransformationp t
				  ,@(unless scroll-bars
				      `(:controller ,pane))
				  ,@(copy-list parent-options))
				 ,pane)
				,@(when label
				    `((make-pane 'ws:label-pane :text ,label
						 :hs+ +fill+)))))))))
  
(defmacro make-clim-interactor ((&optional slot-name &rest clim-pane-options)
				&rest pane-options)
  (declare #+Genera
	   (arglist (&optional slot-name
			       &key (hs 100) (vs 100) label (scroll-bars ':vertical))
		    &rest pane-options))
  `(make-clim-pane (,slot-name :type 'clim-interactor ,@clim-pane-options)
		   ,@pane-options))
