;;; -*- Mode: Lisp; Package: CLIM-INTERNALS; Base: 10.; Syntax: Common-Lisp -*-
;;;
;;; Copyright (c) 1990 by Xerox Corporations.  All rights reserved.

"Copyright (c) 1988, 1989, 1990 International Lisp Associates.  All rights
reserved."

;;;

(in-package "CLIM-INTERNALS")

;;;
;;; New Stream Pane
;;;
;;; This mixin provides all the stream/pane functionality.  It must be mixed
;;; into a pane and probably a basic-clim-pane.
;;;

(defclass pane-extended-stream-mixin  
	 ;; I've screwed up ordering I'm sure --- RR
	 ;; The ordering of these two groups of classes matters if the
	 ;; method combinations are going to come out right.  However
	 ;; the ordering of the classes withing the two levels should not
	 ;; matter as each class defines its own stand-alone protocol.

	 (graphics-output-recording
	  sheet-output-recording
	  basic-output-recording	;better name?
	  clim-stream::input-protocol-mixin
	  ws::pane-basic-stream-mixin)
    ())

(defmethod clim-stream::stream-ensure-cursor-visible
	   ((stream pane-extended-stream-mixin) &optional cx cy)
  (declare (ignore cx cy))
  (when (or (not (output-recording-stream-p stream))
	    (stream-draw-p stream))
    (call-next-method)))

;;; ---This assumes that the stream-pane is always inside a viewport, which
;;; actually defines its visible size.  The stream pane's size is supposed
;;; to represent the size of the contents, but may be stretched to fill the
;;; available viewport space.
(defmethod change-space-req :around ((pane pane-extended-stream-mixin) 
				     &rest keys &key hs vs)
  (declare (dynamic-extent keys))
  ;; Assume always called with hs vs
  (multiple-value-bind (history-width history-height)
      (bounding-rectangle-size (output-recording-stream-output-record pane))
    ;; Don't ever shrink down smaller than our contents.
    (apply #'call-next-method pane 
	   :hs (max hs history-width) :vs (max vs history-height) keys)))

;;; --- maybe move this somewhere?
(defun erase-viewport (stream)
  ;; Just repaint the viewport, which has been set up with a
  ;; background color.
  (let ((viewport (pane-viewport stream)))
    ;; --- what if no viewport?  Just revert to draw-rectangle?  No,
    ;; there has to be a viewport to insulate the ancestors from the
    ;; size changes of the stream pane's output history.
    (when viewport
      (repaint-sheet viewport (sheet-region viewport))))
  ;; It is bogus to just draw a rectangle in some random color.  Where
  ;; did we expect the medium's background to get set?
  #+ignore
  (with-output-recording-options (stream :record-p nil)
    (with-bounding-rectangle* (minx miny maxx maxy) stream
      (draw-rectangle* stream minx miny maxx maxy
		       :filled t :ink +background+))))

;;; Temporary replacement for "window-clear".
(defmethod window-clear ((stream pane-extended-stream-mixin))
  (using-clim-medium (medium stream)
    (letf-globally (((medium-transformation medium) +identity-transformation+))
      (clear-output-history stream)
      (repaint-sheet stream +everywhere+)
      ;;(erase-viewport stream)
      (clim-stream::stream-set-cursor-position* stream 0 0)
      ;; Flush the old mouse position relative to this window
      ;; so that we don't get bogus highlighted presentations
      ;; when menus first pop up.
      (let ((pointer (clim-stream::stream-primary-pointer stream)))
	(when pointer
	  (setf (clim-stream::pointer-window pointer) nil)))
      ;;; doesn't really need to do force-output.
      (force-output stream)
      (values))))

(defmethod window-shift-mask ((window pane-extended-stream-mixin))
  (let ((pointer (clim-stream::stream-primary-pointer window)))
    (clim-stream::pointer-button-state pointer)))

;;;
;;;
;;;

(ws::define-pane-class extended-stream-pane (pane-extended-stream-mixin
					     ;; Shadow invoking-input-contract
					     standard-input-contract
					     ws::display-function-mixin
					     basic-clim-pane)
    ()
  (:default-initargs :medium t 
    :hs+ +fill+ :hs- +fill+ :vs+ +fill+ :vs- +fill+))

(defmethod initialize-instance :after 
	   ((pane extended-stream-pane) &key &allow-other-keys)
  (setf (sheet-transformation pane) +identity-transformation+))

(defclass clim-interactor (extended-stream-pane) ())
(defclass clim-pointer-documentation (extended-stream-pane) ())

;;;
;;;
;;;

;;; New stream pane class for new scrolling mechanism.  This is
;;; intended to replace extended-stream-pane once the transition
;;; period is over.
(defclass new-extended-stream-pane (extended-stream-pane)
    ()
  ;; this is now initialized to a history output record, whose stream
  ;; slot will point to us.  In a gross kludge, the stream slot gets
  ;; initialized in OUR after init method.  This may cause problems
  ;; if the history record eventually needs the stream for its own
  ;; initialization purposes.
  (:default-initargs :output-record (make-instance 'ci::coordinate-sorted-set-history)))

;;; This gets called only when the viewport has had its size changed
;;; (or the frame is re-laid-out), not whenever the scrolled region
;;; changes (like the old scrolling mechanism), so this is a 
;;; convenient point to compute the text margin.
(defmethod ws::allocate-space :around
	   ((pane new-extended-stream-pane) width height)
  (declare (ignore width height))
  (let ((viewport (ws::new-pane-viewport pane)))
    (when viewport
      (setf (clim-stream::stream-default-text-margin pane)
	    (bounding-rectangle-width (sheet-region viewport))))))

;;; When the scrolled region changes size, the viewport needs to know
;;; so that it can change the scroll bars.
(defmethod ws::sheet-region-changed ((pane new-extended-stream-pane) &key)
  (let ((viewport (ws::new-pane-viewport pane)))
    (when viewport
      (ws::scrollee-region-changed viewport pane))))

;;; Create with a special kind of history that sets the SHEET-REGION
;;; of its stream sheet. --- Maybe, instead, SHEET-REGION on this kind
;;; of stream should return the history object, which should support
;;; the region protocol.
(defmethod initialize-instance :after
	   ((stream new-extended-stream-pane) &key output-record)
  ;; this is not 
  (when (typep output-record 'ci::coordinate-sorted-set-history)
    (setf (output-history-stream output-record) stream))
  ;; no longer manufacture a record because it's initialized
  ;; via default-initargs.
  #+Ignore
  (unless output-record
    (setq output-record
	  (setf (slot-value stream 'ci::output-record)
		(make-instance 'ci::coordinate-sorted-set-history :stream stream))))
  ;; --- our basic-output-recording expects extended output...
  (multiple-value-bind (x y) (clim-stream::stream-cursor-position* stream)
    ;; I don't understand why the output record's initial position was set to
    ;; some untransformed "viewport" coordinate.  The cursor position is the
    ;; right place, no?
    (ci::output-record-set-position* output-record x y)))

#+Debugging
(defmethod replay :before (record stream &optional extent x-offset y-offset)
  (ci::highlight-output-record-1 record stream t)
  (sleep 1)
  (ci::highlight-output-record-1 record stream nil)
  )



;;;
;;; Soon to be replaced by making-application-pane
;;;
(defmacro make-clim-pane ((&optional slot-name
				     &rest parent-options
				     &key (type ''ci::new-extended-stream-pane) 
				     label (scroll-bars ':vertical)
				     &allow-other-keys)
			  &rest pane-options)
  (with-rem-keywords (parent-options parent-options '(:type :label :scroll-bars))
    (let ((default '#:default)
	  (pane (gensymbol "SCROLLABLE-PANE")))
      (macrolet ((setf-unless (slot-keyword value)
		   `(when (eq (getf parent-options ',slot-keyword default) default)
		      (setf (getf parent-options ',slot-keyword) ,value))))
	(setf-unless :hs 100)
	(setf-unless :vs 100)
	(setf-unless :hs+ +fill+)
	(setf-unless :hs- +fill+)
	(setf-unless :vs+ +fill+)
	(setf-unless :vs- +fill+))
      `(bordering (:thickness 1)
		  (let ((,pane (make-pane ,type ,@pane-options)))
		    ,@(when slot-name
			`((setq ,slot-name ,pane)))
		    (vertically ()
		      (make-pane 'scroller-pane
				 :hs+ +fill+ :vs+ +fill+
				 :hs- +fill+ :vs- +fill+
				 ,@(copy-list parent-options)
				 :contents ,pane
				 :scroll-bars ,scroll-bars)
		      #+Ignore
		      (,(ecase scroll-bars
			  (:both 'scrolling)
			  (:vertical 'vscrolling)
			  (:horizontal 'hscrolling)
			  ((nil) 'ws::viewing))
		       (:subtransformationp t
			,@(unless scroll-bars
			    `(:controller ,pane))
			,@(copy-list parent-options))
		       ,pane)
		      ,@(when label
			  `((make-pane 'ws::label-pane :text ,label
				       :hs+ +fill+)))))))))
  
(defmacro make-clim-interactor ((&optional slot-name &rest clim-pane-options)
				&rest pane-options)
  (declare #+Genera
	   (arglist (&optional slot-name
			       &key (hs 100) (vs 100) label (scroll-bars ':vertical))
		    &rest pane-options))
  `(make-clim-pane (,slot-name :type 'clim-interactor ,@clim-pane-options)
		   ,@pane-options))


