;;; -*- Mode: LISP; Syntax: Common-lisp; Package: CLIM-INTERNALS; Base: 10; Lowercase: Yes -*-

;; $fiHeader: coordinate-sorted-set.lisp,v 1.4 91/03/26 12:47:48 cer Exp $

(in-package "CLIM-INTERNALS")

"Copyright (c) 1990, 1991 Symbolics, Inc.  All rights reserved.
 Portions copyright (c) 1988, 1989, 1990 International Lisp Associates."

;;; The output record generics:
;;;  output-record-elements: (record)
;;;  add-output-record-element: (record element)
;;;  delete-output-record-element: (record element)
;;;  clear-output-record: (record)
;;;  replay-1: (record stream)
;;;  recompute-extent: (record)
;;;  recompute-extent-for-new-element: (there's a default): (record element)
;;;  recompute-extent-for-changed-element: (record element)
;;;  map-over-output-record-elements-overlapping-region:
;;;   (record region continuation
;;;    &optional (x-offset 0) (y-offset 0) &rest continuation-args)
;;;  map-over-output-record-elements-containing-point*:
;;;   (record x y continuation
;;;    &optional (x-offset 0) (y-offset 0) &rest continuation-args)

(defclass coordinate-sorted-set-output-record
	  (output-record-mixin output-record-element-mixin)
    ((coordinate-sorted-set)			;a simple vector, by gawd
     (fill-pointer :initform 0 :type fixnum)
     (tallest-box-height :initform 0)))

(defclass coordinate-sorted-set-history (coordinate-sorted-set-output-record 
					  stream-output-history-mixin)
    ()
  )

(defmethod initialize-instance :after ((record coordinate-sorted-set-output-record)
				       &key (size 200))
  (with-slots (coordinate-sorted-set) record
    (setf coordinate-sorted-set (make-array size))))


(defclass top-level-coordinate-sorted-set (coordinate-sorted-set-output-record) ())

(defmethod bounding-rectangle-set-edges ((record top-level-coordinate-sorted-set)
					 left top right bottom)
  #+ignore (assert (<= left right))
  #+ignore (assert (<= top bottom))
  (with-slots ((bl left) (bt top) (br right) (bb bottom) parent) record
    ;; Top-level output records must not have their upper left corner any
    ;; "later" than (0,0), or else scroll bars and scrolling will not do
    ;; the right thing.
    (setq bl (min left 0)
	  bt (min top 0)
	  br right
	  bb bottom))
  record)


;;; For debugging.
(defmethod output-record-elements ((record coordinate-sorted-set-output-record))
  (with-slots (coordinate-sorted-set fill-pointer) record
    (let ((result (make-list fill-pointer)))
      (replace result coordinate-sorted-set :end1 fill-pointer :end2 fill-pointer)
      result)))

(defmethod output-record-element ((record coordinate-sorted-set-output-record) index)
  (with-slots (coordinate-sorted-set) record
    (svref coordinate-sorted-set index)))

(defmethod output-record-element-count ((record coordinate-sorted-set-output-record))
  (slot-value record 'fill-pointer))

(defmethod clear-output-record ((record coordinate-sorted-set-output-record))
  (with-slots (coordinate-sorted-set fill-pointer tallest-box-height) record
    (setf tallest-box-height 0)
    ;; Release pointers to objects
    (fill coordinate-sorted-set nil :start 0 :end fill-pointer)
    (setf fill-pointer 0)))

(defmethod add-output-record-element ((record coordinate-sorted-set-output-record) element)
  (with-slots (coordinate-sorted-set fill-pointer tallest-box-height) record
    (let ((vector coordinate-sorted-set)
	  (fp fill-pointer))
      (declare (type simple-vector vector) #+Genera (sys:array-register vector))
      (declare (fixnum fp))
      (maxf tallest-box-height (bounding-rectangle-height element))
      (with-bounding-rectangle* (left top right bottom) element
        (declare (ignore left top))
	;; Quick check for doing output at the bottom of the window
	(if (or (zerop fp)
		(let ((other-element (svref vector (1- fp))))
		  (when (eq other-element element)
		    (return-from add-output-record-element nil))
		  (with-bounding-rectangle* (other-left other-top other-right other-bottom)
					    other-element
		    (declare (ignore other-left other-top))
		    (or (> bottom other-bottom)
			(and (= bottom other-bottom) 
			     (>= right other-right))))))
	    (multiple-value-setq (coordinate-sorted-set fill-pointer)
	      (simple-vector-push-extend element vector fp 200))
	  (let ((index (coordinate-sorted-set-index-for-position
			 vector right bottom 0 fp)))
	    (declare (fixnum index))
	    ;; Make sure that the new element comes after any element it overlaps
	    ;; so that replaying happens in the right order.
	    (loop
	      (if (and (< index fp)
		       (region-intersects-region-p element (svref vector index)))
		  (incf index)
		  (return)))
	    (multiple-value-setq (coordinate-sorted-set fill-pointer)
	      (simple-vector-insert-element element index vector fp 200))))))))

(defmethod delete-output-record-element
	   ((record coordinate-sorted-set-output-record) element &optional (errorp t))
  (with-slots (coordinate-sorted-set fill-pointer tallest-box-height) record
    (let ((index (coordinate-sorted-set-position element coordinate-sorted-set fill-pointer)))
      (cond (index
	     (let ((new-fp (the fixnum (1- fill-pointer)))
		   (vector coordinate-sorted-set))
	       (declare (type simple-vector vector) (fixnum new-fp)
			#+Genera (sys:array-register vector))
	       (unless (= (the fixnum index) new-fp)
		 ;; Shift the whole vector downward
		 (do ((i (the fixnum index) (1+ i)))
		     ((= i new-fp))
		   (declare (fixnum i) (optimize (speed 3) (safety 0)))
		   (setf (svref vector i) (svref vector (1+ i)))))
	       (setf fill-pointer new-fp)
	       t))
	    (errorp
	     (error "The element ~S was not found in ~S" element record))))))

(defmethod map-over-output-record-elements-overlapping-region
	   ((record coordinate-sorted-set-output-record) region continuation
	    &optional (x-offset 0) (y-offset 0) &rest continuation-args)
  (declare (dynamic-extent continuation continuation-args))
  (declare (optimize (safety 0)))
  (let ((vector (slot-value record 'coordinate-sorted-set))
	(length (slot-value record 'fill-pointer)))
    (declare (type simple-vector vector) (fixnum length)
	     #+Genera (sys:array-register vector))
    (if (or (null region) (eql region +everywhere+))
	(dovector ((element index) vector :start 0 :end length :simple-p t)
	  (apply continuation element continuation-args))
      (with-bounding-rectangle* (left1 top1 right1 bottom1) region
	(translate-positions x-offset y-offset left1 top1 right1 bottom1)
	(let ((start (coordinate-sorted-set-index-for-position vector 0 top1 0 length))
	      (limit (+ bottom1 (slot-value record 'tallest-box-height))))
	  (declare (fixnum start limit))
	  ;; Subtract out the record offset from the region, to make comparison fair
	  (multiple-value-bind (xoff yoff)
	      (output-record-position* record)
	    (translate-positions (- xoff) (- yoff) left1 top1 right1 bottom1))
	  (do ((index start (the fixnum (1+ (the fixnum index)))))
	      ((= (the fixnum index) length))
	    (declare (fixnum index))
	    (let ((element (svref vector index)))
	      (with-bounding-rectangle* (left2 top2 right2 bottom2) element
		(when (ltrb-overlaps-ltrb-p left1 top1 right1 bottom1
					    left2 top2 right2 bottom2)
		  (apply continuation element continuation-args))
		(when (> bottom2 limit)
		  (return nil))))))))))

(defmethod map-over-output-record-elements-containing-point*
	   ((record coordinate-sorted-set-output-record) x y continuation
	    &optional (x-offset 0) (y-offset 0) &rest continuation-args)
  (declare (dynamic-extent continuation continuation-args))
  (declare (optimize (safety 0)))
  (translate-positions x-offset y-offset x y)
  (let ((vector (slot-value record 'coordinate-sorted-set))
	(length (slot-value record 'fill-pointer))
	(bound (slot-value record 'tallest-box-height)))
    (declare (type simple-vector vector) (fixnum length bound)
	     #+Genera (sys:array-register vector))
    (let ((end (coordinate-sorted-set-index-for-position vector 0 (+ y bound 1) 0 length))
	  (limit (- y bound)))
      (declare (fixnum end limit))
      (multiple-value-bind (xoff yoff)
	  (output-record-position* record)
	(translate-positions (- xoff) (- yoff) x y))
      (do ((index (min (1- length) end) (the fixnum (1- (the fixnum index)))))
	  ((< (the fixnum index) 0))
	(declare (fixnum index))
	(let ((element (svref vector index)))
	  (with-bounding-rectangle* (left top right bottom) element
	    (when (ltrb-contains-point*-p left top right bottom x y)
	      (apply continuation element continuation-args))
	    (when (< bottom limit)
	      (return nil))))))))

;; Like POSITION, but searches coordinate sorted sets
(defun coordinate-sorted-set-position (object vector fill-pointer)
  (declare (type simple-vector vector) (fixnum fill-pointer))
  (declare (optimize (speed 3) (safety 0)))
  (with-bounding-rectangle* (left top right bottom) object
    (declare (ignore left top))
    ;; Binary search to find where this one goes.
    (let ((search-index (coordinate-sorted-set-index-for-position
			  vector right bottom 0 fill-pointer)))
      (declare (fixnum search-index))
      ;; Search back over things in the same place.
      (when (< search-index fill-pointer)
	(dovector ((element index) vector :start 0 :end (1+ search-index)
					  :from-end t :simple-p t)
	  (when (eq element object)
	    (return-from coordinate-sorted-set-position index))
	  (with-bounding-rectangle* (other-left other-top other-right other-bottom)
				    element
	    (declare (ignore other-left other-top))
	    (unless (and (= right other-right) (= bottom other-bottom))
	      (return)))))
      ;; Search forward too.
      (dovector ((element index) vector
		 :start (if (< search-index fill-pointer) (1+ search-index) 0)
		 :end fill-pointer :simple-p t)
	(when (eq element object)
	  (return index)
	  (when (> (bounding-rectangle-bottom element) bottom)
	    (return nil)))))))

;; Binary search; dictionary order Y, X.
(defun coordinate-sorted-set-index-for-position (vector right bottom start end)
  (declare (type simple-vector vector) (fixnum start end))
  (declare (optimize (speed 3) (safety 0)))
  (let ((below start)
	(above end))
    (declare (fixnum below above))
    (assert (<= below above))			;Binary search will loop otherwise.
    (let (#+Genera (vector vector))
      #+Genera (declare (sys:array-register vector))
      (loop
	(when (= above below)
	  (return above))
	(let* ((index (the fixnum (ash (the fixnum (+ above below)) -1)))
	       (other-box (svref vector index)))
	  (with-bounding-rectangle* (other-left other-top other-right other-bottom)
				    other-box
	    (declare (ignore other-left other-top))
	    (cond ((or (< bottom other-bottom)
		       (and (= bottom other-bottom) (< right other-right)))
		   (setq above index))
		  ((or (> bottom other-bottom)
		       (and (= bottom other-bottom) (> right other-right)))
		   (if (= below index)
		       (return above)
		       (setq below index)))
		  (t
		   (return index)))))))))
