;;; -*- Mode: LISP; Syntax: Common-lisp; Package: CLIM-INTERNALS; Base: 10; Lowercase: Yes -*-

(in-package "CLIM-INTERNALS")

"Copyright (c) 1990, 1991 Symbolics, Inc.  All rights reserved."

;;; menu/command

(define-presentation-type command-menu-element ())

;; This should never be called with :ACCEPTABLY T
(define-presentation-method present
			    (element (type command-menu-element) stream (view textual-view)
			     &key acceptably)
  (when acceptably
    (error "There is no way to print command-menu elements acceptably."))
  (let* ((menu (pop element))
	 (keystroke (pop element))
	 (item (pop element))
	 (type (command-menu-item-type item))
	 (command (and (or (eql type ':command)
			   (eql type ':function))
		       (extract-command-menu-item-value item t))))
    (flet ((body (stream)
	     (if keystroke
		 (format stream "~A (~C)" menu keystroke)
	         (format stream "~A" menu))
	     (when (eql type ':menu)
	       (write-string " >" stream))))
      (declare (dynamic-extent #'body))
      (if (and command
	       (not (command-enabled-p (command-name command) *application-frame*)))
	  (with-text-style ('(nil :italic :smaller) stream) (body stream))
	  (body stream)))))

(defun display-command-table-menu (command-table stream
				   &key max-width max-height
					n-rows n-columns
					inter-column-spacing inter-row-spacing 
					(cell-align-x ':left) (cell-align-y ':top)
					no-initial-spacing move-cursor)
  (unless (or max-width max-height)
    (multiple-value-bind (width height)
	#+Silica (let ((region (sheet-region stream)))
		   (values (bounding-rectangle-width region)
			   (bounding-rectangle-height region)))
	#-Silica (window-inside-size stream)
	(unless max-width (setf max-width width))
	(unless max-height (setf max-height height))))
  (let ((menu (slot-value (find-command-table command-table) 'menu)))
    (if (zerop (count-if #'(lambda (x) (not (null (first x)))) menu))
	(with-text-face (:italic stream)
	  (write-string "[No menu items]" stream))
        (formatting-item-list (stream :max-width max-width :max-height max-height
				      :n-rows n-rows :n-columns n-columns
				      :inter-column-spacing inter-column-spacing
				      :inter-row-spacing inter-row-spacing
				      :no-initial-spacing no-initial-spacing
				      :move-cursor move-cursor)
	  (dovector (element menu)
	    (cond ((eql (command-menu-item-type (third element)) :divider)
		   ;;--- Draw a dividing line
		   )
		  ((first element)
		   (formatting-cell (stream :align-x cell-align-x :align-y cell-align-y)
		     (present element 'command-menu-element
			      :stream stream :single-box t)))))))))

;; This doesn't actually need to execute the command itself, because it
;; presents each of the menu items as COMMAND-MENU-ELEMENTs and establish
;; an input context of COMMAND.  Thus the COMMAND-MENU-ELEMENT-TO-COMMAND
;; and COMMAND-MENU-ELEMENT-TO-SUB-MENU translators will take care of
;; executing the command.
(defun menu-execute-command-from-command-table
       (command-table
	&key (associated-window (frame-top-level-window *application-frame*))
	     (default-style *command-table-menu-text-style*) label
	     cache (unique-id command-table) (id-test #'eql) cache-value (cache-test #'eql))
  (setq command-table (find-command-table command-table))
  (unless cache-value
    (setq cache-value (slot-value command-table 'menu-tick)))
  (let ((menu-items (slot-value command-table 'menu)))
    (with-menu (menu associated-window)
      #-silica (setf (window-label menu) label)
      (with-text-style (default-style menu)
	(flet ((menu-choose-body (stream type)
		 (declare (ignore type))
		 (menu-choose-command-drawer stream menu-items nil)))
	  (declare (dynamic-extent #'menu-choose-body))
	  ;; The translators referred to above will ensure that we get only
	  ;; valid, enabled commands
	  (menu-choose-from-drawer
	    menu `(command :command-table ,command-table) #'menu-choose-body
	    :cache cache
	    :unique-id unique-id :id-test id-test
	    :cache-value cache-value :cache-test cache-test))))))

(defun menu-choose-command-from-command-table
       (command-table
	&key (associated-window (frame-top-level-window *application-frame*))
	     (default-style *command-table-menu-text-style*) label
	     cache (unique-id command-table) (id-test #'eql) cache-value (cache-test #'eql))
  (setq command-table (find-command-table command-table))
  (unless cache-value
    (setq cache-value (slot-value command-table 'menu-tick)))
  (let ((menu-items (slot-value command-table 'menu)))
    (with-menu (menu associated-window)
      #-silica (setf (window-label menu) label)
      (with-text-style (default-style menu)
	(flet ((menu-choose-body (stream type)
		 (menu-choose-command-drawer stream menu-items type)))
	  (declare (dynamic-extent #'menu-choose-body))
	  (multiple-value-bind (item gesture)
	      (menu-choose-from-drawer
		menu 'menu-item #'menu-choose-body
		:cache cache
		:unique-id unique-id :id-test id-test
		:cache-value cache-value :cache-test cache-test)
	    (when item
	      (extract-command-menu-item-value (third item) gesture))))))))

(defun menu-choose-command-drawer (stream items type)
  (formatting-item-list (stream :move-cursor nil)
    (dovector (item items)
      (cond ((eql (command-menu-item-type (third item)) :divider)
	     ;;--- Draw a dividing line
	     )
	    ((first item)
	     (formatting-cell (stream)
	       (if type
		   (with-output-as-presentation (:stream stream
						 :object item
						 :type type
						 :single-box T)
		     (present item 'command-menu-element :stream stream))
		   (present item 'command-menu-element :stream stream)))))))
  nil)

(defun extract-command-menu-item-value (menu-item gesture 
					&optional (numeric-argument *numeric-argument*))
  (let ((type (command-menu-item-type menu-item))
	(value (command-menu-item-value menu-item)))
    (if (eql type ':function)
	(funcall value gesture numeric-argument)
        value)))


;;; Presentation translator utilities

(defun add-presentation-translator-to-command-table (command-table translator
						     &key (errorp t))
  (setq command-table (find-command-table command-table))
  (with-slots (translators translators-cache) command-table
    (let* ((translator-name (presentation-translator-name translator))
	   (place (position translator-name translators
			    :key #'presentation-translator-name)))
      (cond (place
	     (when errorp
	       (cerror "Remove the translator and proceed"
		       'command-already-present
		       :format-string "Translator ~S already present in ~S"
		       :format-args (list (presentation-translator-name translator)
					  command-table)))
	     (setf (aref translators place) translator))
	    (t
	     (when (null translators)
	       (setq translators (make-array *command-table-size*
					     :adjustable t :fill-pointer 0)))
	     (vector-push-extend translator translators)))))
  ;; Maybe one day we'll do incremental updating of the cache.  That day
  ;; is not today.
  (incf *translators-cache-tick*))

(defun remove-presentation-translator-from-command-table (command-table translator-name
							  &key (errorp t))
  (setq command-table (find-command-table command-table))
  (with-slots (translators) command-table
    (let ((translator
	    (if (symbolp translator-name)
		(find-presentation-translator translator-name command-table :errorp nil)
		translator-name)))
      (cond ((find translator translators)
	     (setq translators (delete translator translators))
	     (incf *translators-cache-tick*))
	    (t
	     (when errorp
	       (cerror "Proceed without any special action"
		       'command-not-present
		       :format-string "Translator ~S not present in ~S"
		       :format-args (list translator-name command-table))))))))

(defun map-over-command-table-translators (function command-table &key (inherited t))
  (declare (dynamic-extent function))
  (if inherited
      (do-command-table-inheritance (comtab command-table)
	(let ((translators (slot-value comtab 'translators)))
	  (when translators
	    (dovector (translator translators)
	      (funcall function translator)))))
    (let ((translators (slot-value (find-command-table command-table) 'translators)))
      (when translators
	(dovector (translator translators)
	  (funcall function translator))))))

(defun find-presentation-translator (translator-name command-table &key (errorp t))
  (declare (values command command-table))
  (when (typep translator-name 'presentation-translator)
    (setq translator-name (presentation-translator-name translator-name)))
  (do-command-table-inheritance (comtab command-table)
    (let ((translator (find translator-name (slot-value comtab 'translators)
			    :key #'presentation-translator-name)))
      (when translator
	(return-from find-presentation-translator
	  (values translator comtab)))))
  (when errorp
    (error 'command-not-accessible
	   :format-string "Translator ~S is not accessible in ~S"
	   :format-args (list translator-name command-table))))

(defun remove-presentation-translator (name)
  (maphash #'(lambda (key comtab)
	       (declare (ignore key))
	       (remove-presentation-translator-from-command-table comtab name :errorp nil))
	   *all-command-tables*))

(defun clear-presentation-translator-caches ()
  (maphash #'(lambda (key comtab)
	       (declare (ignore key))
	       (with-slots (translators-cache) comtab
		 (when translators-cache
		   (clrhash translators-cache))))
	   *all-command-tables*)
  (incf *translators-cache-tick*)
  (values))

(defun write-command-argument-translator
       (arg-name arg-type gesture 
	command-name command-table pointer-documentation
	count n-required found-key)
  (let ((translator-options nil))
    (when (consp gesture)
      (dolist (indicator '(:tester :menu :priority :echo
				   :documentation :pointer-documentation))
	(let ((option (getf (rest gesture) indicator)))
	  (when option
	    (setq translator-options
		  (append translator-options `(,indicator ,option))))))
      (setq gesture (first gesture)))
    (unless (constantp arg-type)
      (error "It is illegal to define a translator on a non-constant presentation type"))
    (setq arg-type (eval arg-type))
    (let ((translator-name (fintern "~A-~A-~A-~A"
				    arg-type 'to command-name 'translator)))
      (flet ((unsupplied (n)
	       (make-list n :initial-element '*unsupplied-argument*)))
	`(define-presentation-to-command-translator
		 ,translator-name
		 (,arg-type ,command-name ,command-table
		  :gesture ,gesture
		  :pointer-documentation ,pointer-documentation
		  ,@translator-options)
		 (object)
		 ,(if found-key
		      `(list ,@(unsupplied n-required)
			     ,(intern (string arg-name) *keyword-package*)
			     object)
		      `(list ,@(unsupplied count)
			     object
			     ,@(unsupplied (- n-required count 1)))))))))
