;;; -*- Mode: LISP; Syntax: Common-lisp; Package: CLIM-INTERNALS; Base: 10 -*-

(in-package "CLIM-INTERNALS")

"Copyright (c) 1991 International Lisp Associates.  All rights reserved."

;;; Character-drawing methods for Postscript implementation

(defmacro with-postscript-stream-glyph-for-character ((medium) &body body)
  (let ((port (gensymbol 'port))
	(ppi (gensymbol 'ppi)))
  `(let* ((,port (port ,medium)))
     (macrolet ((stream-glyph-for-character (medium character style &optional our-font)
		  (unless (eql medium ',medium)
		    (warn "You are calling ~S on the wrong medium."
			  'stream-glyph-for-character))
		  `(multiple-value-bind (character-set index)
		       (char-character-set-and-index ,character)
		     (let (;; For now we are asserting that each string
			   ;; passed to WRITE-STRING will have no style
			   ;; changes within it.  This is what our-font
			   ;; is all about.
			   (pfd (or ,our-font
				    (text-style-mapping ,',port character-set ,style))))
		       (when (null pfd)
			 (error "Text style mapping not found for ~S~*" ,style))
		       (let* ((pface (on-postscript::pfd-face pfd))
			      (point-size (on-postscript::pfd-point-size pfd))
			      (height (* point-size (on-postscript::pface-height pface)))
			      (ascent (* (on-postscript::pface-ascent pface) height))
			      (CWT (on-postscript::pface-width-table pface))
			      (relwidth (if (numberp CWT) 
					    CWT
					    (aref CWT index)))
			      (escapement-x (* height relwidth))
			      (escapement-y 0)
			      (origin-x 0)
			      (origin-y ascent)
			      (bb-x ;; really ought know real dope, but not avl yet.
				escapement-x)
			      (bb-y height))
			 (values index pfd escapement-x escapement-y origin-x origin-y
				 bb-x bb-y (numberp cwt)))))))
     ,@body))))

(defmethod stream-glyph-for-character ((medium on-postscript::basic-postscript-medium)
				       character appearance &optional our-font)
  #+Genera
  (declare (values index font escapement-x escapement-y origin-x origin-y bb-x bb-y))
  (with-postscript-stream-glyph-for-character (medium)
    (stream-glyph-for-character medium character appearance our-font)))

;;; --- All the other methods on this specialize the stream as being for
;;; OUTPUT-PROTOCOL.  The reason we don't do this here is because:
;;; a) We don't actually use the stream argument for anything, and
;;; b) One of the important callers of this, STRING-WIDTH for Postscript
;;;    media, doesn't know what the stream should be.
;;; This generic function (and STREAM-GLYPH-FOR-CHAR) should probably be
;;; booted down into Silica and made into honest-to-goodness methods on
;;; media instead of on streams.  That would remove these mixed-level
;;; methods from the code; we could get rid of all the CLIM-xxx-STUFF
;;; files that way.

(defmethod stream-scan-string-for-writing  (stream
					    (medium on-postscript::basic-postscript-medium)
					    string start end
					    style cursor-x max-x
					    &optional glyph-buffer)
  #+Genera (declare
	     (values write-char next-char-index new-cursor-x new-baseline new-height font))
  (with-postscript-stream-glyph-for-character (medium)
    (stream-scan-string-for-writing-body)))
