;;; -*- Mode: LISP; Syntax: Common-lisp; Package: CLIM-INTERNALS; Base: 10; Lowercase: Yes -*-

;; $fiHeader: clim-defs.lisp,v 1.4 91/03/26 12:47:12 cer Exp $

(in-package "CLIM-INTERNALS")

"Copyright (c) 1990, 1991 Symbolics, Inc.  All rights reserved.
 Portions copyright (c) 1988, 1989, 1990 International Lisp Associates."

;;; Moved the default-xxx-stream macros into lisp-utilities, because
;;; they're needed in the stream layer.


(defun-inline position-translate (point dx dy)
  (values (+ (point-x point) dx)
	  (+ (point-y point) dy)))

(defun-inline position-translate* (x y dx dy)
  (values (+ x dx)
	  (+ y dy)))

(defmacro translate-positions (x-delta y-delta &body points)
  (once-only (x-delta y-delta)
    `(progn
       ,@(let ((forms nil))
	   (dorest (pts points cddr)
	     (push `(incf ,(first pts)  ,x-delta) forms)
	     (push `(incf ,(second pts) ,y-delta) forms))
	   (nreverse forms)))))

(defmacro translate-fixnum-positions (x-delta y-delta &body points)
  (once-only (x-delta y-delta)
    `(progn
       ,@(let ((forms nil))
	   (dorest (pts points cddr)
	     (push `(setf ,(first pts)  (the fixnum (+ ,(first pts)  ,x-delta))) forms)
	     (push `(setf ,(second pts) (the fixnum (+ ,(second pts) ,y-delta))) forms))
	   (nreverse forms)))))

(defun translate-point-sequence (x-delta y-delta points)
  (let ((new-points (make-list (length points))))
    (do ((coords points (cddr coords))
	 (new-coords new-points (cddr new-coords)))
	((null coords))
      (let ((x (first coords))
	    (y (second coords)))
	(translate-positions x-delta y-delta x y)
	(setf (first new-coords) x)
	(setf (second new-coords) y)))
    new-points))

;; Destructive version of the above
(defun ntranslate-point-sequence (x-delta y-delta points)
  (do ((coords points (cddr coords)))
      ((null coords))
    (translate-positions x-delta y-delta (first coords) (second coords)))
  points)

(defclass io-buffer
	  ()
     ((size :accessor io-buffer-size :initarg :size)
      (buffer :accessor io-buffer-buffer)
      (input-pointer :accessor io-buffer-input-pointer)
      (output-pointer :accessor io-buffer-output-pointer)))

(defmacro define-accessors (flavor &body instance-variables)
  (let ((forms nil))
    (dolist (iv instance-variables)
      (let ((name (fintern "~A-~A" flavor iv)))
	(push `(defmethod ,name ((,flavor ,flavor))
		 (slot-value ,flavor ',iv))
	      forms)
	(push `(defmethod (setf ,name) (new-val (,flavor ,flavor))
		 (setf (slot-value ,flavor ',iv) new-val))
	      forms)))
    `(define-group ,flavor define-accessors 
       ,@(nreverse forms))))

;;; Implementation defining and managing tools

(defvar *implementations* nil)

(defun define-implementation (name creation-function)
  (pushnew name *implementations*)
  (setf (get name 'creation-function) creation-function))

(defun window-type-creation-function (window-stream-type)
  (let ((cf (get window-stream-type 'creation-function)))
    (unless cf
      (error "No window creation function is defined for type ~S" window-stream-type))
    cf))


;;; Event definitions

#-Silica
(progn ; --- these commented out on 12/23/91 by doughty because they're in Silica.

(defclass device-event ()
    (#-Silica (window :reader event-window :initarg :window)
     #+Silica (sheet :reader event-sheet :initarg :sheet)
     #+Silica (type :reader event-type :initarg :type)
     #+Silica (time :reader event-time :initarg :time)))
 
;;; --- this is KEY-EVENT in Silica, but who's counting?
(defclass keyboard-event (device-event)
    ((char :reader keyboard-event-char :initarg :char)))
 
(defclass key-press-event (keyboard-event) ())
(defclass key-release-event (keyboard-event) ())
 
(defclass pointer-event (device-event)
    ((x :reader pointer-event-x :initarg :x :type fixnum)
     (y :reader pointer-event-y :initarg :y :type fixnum)
     #+Silica (native-x :reader pointer-event-native-x :initarg :native-x)
     #+Silica (native-y :reader pointer-event-native-y :initarg :native-y)
     #+ignore (pointer :reader pointer-event-pointer :initarg :pointer)))
 
;;; --- button-xxx-event in Silica
(defclass pointer-button-event (pointer-event)
    ((button :reader pointer-event-button :initarg :button)
     (shift-mask :reader pointer-event-shift-mask :initarg :shift-mask)))
 
(defclass pointer-button-press-event (pointer-button-event) ()) 
(defclass pointer-button-release-event (pointer-button-event) ()) 
;;--- How is this different from a button press event?
(defclass pointer-click-event (pointer-button-event) ())
(defclass pointer-click-hold-event (pointer-button-event) ()) 
(defclass pointer-double-click-event (pointer-button-event) ())
 
(defclass pointer-motion-event (pointer-event) ())
 
(defclass pointer-enter-event (pointer-motion-event) ()) 
(defclass pointer-exit-event (pointer-motion-event) ())

;;; --- Unnecessary in Silica?
(defclass window-event (device-event) ())

(defclass window-size-or-position-change-event (window-event)
    ((left   :initarg :left)
     (top    :initarg :top)
     (right  :initarg :right)
     (bottom :initarg :bottom)))
    
;; X and Y had better be fixnums
(define-constructor make-button-press-event pointer-button-press-event
		    (window x y button shift-mask)
  :window window :x x :y y :button button :shift-mask shift-mask)

;; Right now, we don't defaultly generate button release events
(defvar *generate-button-release-events* nil)
(define-constructor make-button-release-event pointer-button-release-event
		    (window x y button shift-mask)
  :window window :x x :y y :button button :shift-mask shift-mask)

(define-constructor make-window-size-or-position-change-event window-size-or-position-change-event
		    (window left top right bottom)
  :window window :left left :top top :right right :bottom bottom)

)

(defmacro with-stream-cursor-position-saved ((stream) &body body)
  (let ((x '#:x)
	(y '#:y))
    `(multiple-value-bind (,x ,y) (stream-cursor-position* ,stream)
       (unwind-protect
	   (progn ,@body)
	 (stream-set-cursor-position* ,stream ,x ,y)))))

;;; --- Moved with-output-recording-options down to stream/stream-defs.lisp for the nonce.
;;; --- doughty 9/29/91. (and again 12/23/91)

(eval-when (compile eval load)
(defvar *output-record-constructor-cache* (make-hash-table))

(defmacro construct-output-record (type &rest init-args #+Genera &environment #+Genera env)
  (let ((constructor nil))
    (cond ((and (constantp type #+Genera env)
		(setq constructor (gethash (eval type) *output-record-constructor-cache*)))
	   `(,constructor ,@init-args))
	  (t `(construct-output-record-1 ,type ,@init-args)))))

(defmacro define-output-record-constructor (record-type arglist &rest initialization-arguments)
  (let ((constructor-name (fintern "~A-~A" record-type 'constructor)))
    `(progn
       (define-constructor ,constructor-name ,record-type ,arglist ,@initialization-arguments)
       (setf (gethash ',record-type *output-record-constructor-cache*)
	     ',constructor-name))))
)	;eval-when

(defmacro with-new-output-record ((stream &optional record-type record &rest init-args)
				  &body body #+Genera &environment #+Genera env)
  #+Genera (declare (zwei:indentation 0 3 1 1))
  (unless record-type
    (setq record-type `'linear-output-record))
  (let ((constructor nil)
	(ignore-record nil))
    (when (and (constantp record-type #+Genera env)
	       (setq constructor
		     (gethash (eval record-type) *output-record-constructor-cache*))))
    (unless record
      (setq record '#:record
	    ignore-record t))
    `(flet ((with-new-output-record-body (,record)
	      ,@(when ignore-record `((declare (ignore ,record))))
	      ,@body))
       (declare (dynamic-extent #'with-new-output-record-body))
       (with-new-output-record-internal
	 #'with-new-output-record-body
	 ,stream ,record-type ',constructor ,@init-args))))

(defmacro with-output-to-output-record ((stream &optional record-type record &rest init-args)
					&body body)
  #+Genera (declare (zwei:indentation 0 3 1 1))
  (default-output-stream stream)
  ;; --- validate protocol here.
  `(with-output-recording-options (,stream :draw-p nil :record-p t)
     (letf-globally (((output-recording-stream-current-output-record-stack ,stream) nil)
		     ((output-recording-stream-output-record ,stream) nil)
		     ((output-recording-stream-text-output-record ,stream) nil))
       (with-new-output-record (,stream ,record-type ,record ,@init-args)
	 (with-stream-cursor-position-saved (,stream)
	   ,@body)))))

(defmacro with-output-as-presentation ((&key object type stream modifier
					     single-box (allow-sensitive-inferiors t)
					     (record-type `'standard-presentation)
					     parent)
				       &body body)
  #+Genera (declare (zwei:indentation 0 3 1 1))
  (default-output-stream stream)
  ;; Maybe with-new-output-record should turn record-p on?
  (let ((nobject '#:object)			;(once-only (object type) ...)
	(ntype '#:type))
    `(with-output-recording-options (,stream :record-p t)
       (let ((,nobject ,object)
	     (,ntype ,type))
	 (with-new-output-record (,stream ,record-type nil
				  :object ,nobject
				  :type (if ,ntype
					    (expand-presentation-type-abbreviation ,ntype)
					    (presentation-type-of ,nobject))
				  :single-box ,single-box
				  :allow-sensitive-inferiors ,allow-sensitive-inferiors
				  ,@(when modifier `(:modifier ,modifier))
				  ,@(when parent `(:parent ,parent)))
	   ,@body)))))


;;; Presentation type stuff

;; The current input context consists of a list of context entries.  Each entry
;; is a list of the form (CONTEXT-TYPE CATCH-TAG)
(defvar *input-context* nil)

(defun-inline input-context-type (context-entry)
  (first context-entry))

(defun-inline input-context-tag (context-entry)
  (second context-entry))

;; This is the presentation you get if you click while not over anything...
(defvar *null-presentation*)

(defun make-input-context-clauses (pt-var clauses)
  (let ((new-clauses nil))
    (dolist (clause clauses (nreverse new-clauses))
      (let ((type (first clause))
	    (body (rest clause)))
	(push `((presentation-subtypep ,pt-var ',type)
		,@body)
	      new-clauses)))))	;eval-when

(defmacro with-input-context ((type &key override) 
			      (&optional object-var type-var event-var options-var)
			      form
			      &body clauses)
  #+Genera (declare (zwei:indentation 0 2 2 4 3 2))
  (let ((ignores nil))
    (when (null object-var)
      (setq object-var '#:object)
      (push object-var ignores))
    (when (null type-var)
      (setq type-var '#:presentation-type)
      (unless clauses (push type-var ignores)))
    (when (null event-var)
      (setq event-var '#:event)
      (push event-var ignores))
    (when (null options-var)
      (setq options-var '#:options)
      (push options-var ignores))
    `(flet ((body-continuation () ,form)
	    (context-continuation (,object-var ,type-var ,event-var ,options-var)
	      ,@(and ignores `((declare (ignore ,@ignores))))
	      (cond ,@(make-input-context-clauses type-var clauses))))
       (declare (dynamic-extent #'body-continuation #'context-continuation))
       (with-input-context-1
	 (expand-presentation-type-abbreviation ,type) ,override
	 #'body-continuation #'context-continuation))))

(defmacro with-input-focus ((stream) &body body)
  (let ((old-input-focus '#:old-input-focus))
    `(let ((,old-input-focus nil))
       (unwind-protect
	   (progn (setq ,old-input-focus (stream-set-input-focus ,stream))
		  ,@body)
	 (when ,old-input-focus
	   (stream-restore-input-focus ,stream ,old-input-focus))))))


;;; From MENUS.LISP
;;; For now, MENU-CHOOSE requires that you pass in a parent.
(defmacro with-menu ((menu &optional (associated-window nil aw-p)) &body body)
  (let ((window '#:associated-window))
    `(let ((,window ,(if aw-p
			 associated-window
			 `(frame-top-level-window *application-frame*))))	;once-only
       (using-resource (,menu menu (window-top-level-window ,window) (window-root ,window))
	 (letf-globally (((stream-default-view ,menu) +menu-view+))
	   ,@body)))))

;;; From ACCEPTING-VALUES.LISP
(defmacro accepting-values ((&optional stream &rest args) &body body)
  (declare (arglist (&optional stream
		     &key frame-class own-window exit-boxes
			  initially-select-query-identifier
			  resynchronize-every-pass
			  label x-position y-position)
		    &body body))
  #+Genera (declare (zwei:indentation 0 3 1 1))
  (default-query-stream stream accepting-values)
  `(flet ((accepting-values-body (,stream) ,@body))
     (declare (dynamic-extent #'accepting-values-body))
     (accept-values-1 ,stream #'accepting-values-body ,@args)))


;;; Drawing state macros

#||
;;; --- Commented out 12/23/91 until new graphics stuff is incorporated
(defmacro with-clipping-region ((stream region) &body body)
  (default-output-stream stream with-clipping-region)
  `(flet ((with-clipping-region-body (,stream) ,@body))
     (declare (dynamic-extent #'with-clipping-region-body))
     (with-clipping-region-1 ,stream ,region #'with-clipping-region-body)))

(defgeneric with-drawing-options-1 (medium continuation &key)
  (declare (dynamic-extent continuation)))

(defmacro with-drawing-options ((medium &rest options) &body body)
  (declare (arglist (medium
		     &key ink clipping-region transformation
			  line-style line-unit line-thickness line-dashes
			  line-joint-shape line-cap-shape
			  text-style text-family text-face text-size)))
  #+Genera (declare (zwei:indentation 0 3 1 1))
  (default-output-stream medium)
  `(flet ((with-drawing-options-body () ,@body))
     (declare (dynamic-extent #'with-drawing-options-body))
     (with-drawing-options-1 ,medium #'with-drawing-options-body ,@options)))

#|| ;; --- These are all in silica;graphics-definitions 12/28/91 doughty
(defmacro with-identity-transformation ((medium) &body body)
  `(with-drawing-options (,medium :transformation +identity-transformation+)
     ,@body))

(defmacro with-translation ((medium dx dy) &body body)
  `(with-drawing-options (,medium
			  :transformation (make-translation-transformation ,dx ,dy))
     ,@body))
 
(defmacro with-scaling ((medium sx &optional (sy nil sy-p)) &body body)
  `(with-drawing-options (,medium
			  :transformation (let* ((scale-x ,sx)
						 (scale-y ,(if sy-p sy 'scale-x)))
					    (make-scaling-transformation scale-x scale-y)))
     ,@body))
 
(defmacro with-rotation ((medium angle &optional (origin nil origin-p)) &body body)
  `(with-drawing-options (,medium
			  :transformation (make-rotation-transformation ,angle
									,@(if origin-p `(,origin) nil)))
     ,@body))

||#

;;; --- end of stuff commented out 12/23/91 by doughty
||#


;; Establish a local +Y-downward coordinate system at the current cursor position,
;; and execute the body
(defmacro with-local-coordinates ((&optional stream) &body body)
  (default-output-stream stream with-local-coordinates)
  (let (( x '#:x)  ( y '#:y)
	(tx '#:tx) (ty '#:ty))
    `(multiple-value-bind (,x ,y)
	 (stream-cursor-position* ,stream)
       (multiple-value-bind (,tx ,ty)
	   (transform-point* (medium-transformation ,stream) 0 0)
	 (with-drawing-options
	     (,stream :transformation (make-translation-transformation
					(- ,x ,tx) (- ,y ,ty)))
	   ,@body)))))

;; Establish a local +Y-upward coordinate system at the current cursor position,
;; and execute the body
(defmacro with-first-quadrant-coordinates ((&optional stream) &body body)
  (default-output-stream stream with-first-quadrant-coordinates)
  (let (( x '#:x)  ( y '#:y)
	(tx '#:tx) (ty '#:ty))
    `(multiple-value-bind (,x ,y)
	 (stream-cursor-position* ,stream)
       (multiple-value-bind (,tx ,ty)
	   (transform-point* (medium-transformation ,stream) 0 0)
	 (with-drawing-options
	     ;; Don't flip the stream over if we already have
	     (,stream :transformation (if (medium-+Y-upward-p ,stream)
					  +identity-transformation+
					  (make-transformation 1 0 0 -1
							       (- ,x ,tx) (- ,y ,ty))))
	   (letf-globally (((medium-+Y-upward-p ,stream) t))
	     ,@body))))))

;; Establish a first quadrant coordinate system, execute the body, and then
;; place the output in such a way that the upper left corner of it is where
;; the cursor was.  Finally, leave the cursor at the end of the output.
;; HEIGHT is useful when you are doing this inside of incremental redisplay,
;; and the graphics are going to change size from pass to pass.
(defmacro with-room-for-graphics ((&optional stream &key record-type height (move-cursor t))
				  &body body)
  (default-output-stream stream with-room-for-graphics)
  (unless record-type
    (setq record-type `'linear-output-record))
  `(flet ((with-room-for-graphics-body (,stream) ,@body))
     (declare (dynamic-extent #'with-room-for-graphics-body))
     (with-room-for-graphics-1 ,stream ,record-type ,move-cursor
			       #'with-room-for-graphics-body
			       ,@(and height `(:height ,height)))))



;;; Application frame variables
(defvar *default-application*)
(defvar *pointer-documentation-output* nil)
(defvar *sizing-application-frame* nil)


;;; Command processor variables
(defvar *command-parser* 'command-line-command-parser)
(defvar *command-unparser* 'command-line-command-unparser)
(defvar *partial-command-parser* 'command-line-read-remaining-arguments-for-partial-command)


;;; Conditions

(define-condition abort-gesture (error)
  ((event :initform nil :initarg :event :reader abort-gesture-event))
  (:report
    (lambda (condition stream)
      (format stream "Abort event ~S seen" (abort-gesture-event condition)))))

(defun handle-abort-gesture (condition)
  (declare (ignore condition))
  (abort))

(defmacro catch-abort-gestures ((format-string &rest format-args) &body body)
  `(with-simple-restart (abort ,format-string ,@format-args)
     (handler-bind ((abort-gesture #'handle-abort-gesture))
       ,@body)))

(define-condition accelerator-gesture (error)
  ((event :initform nil :initarg :event
	  :reader accelerator-gesture-event)
   (numeric-argument :initform 1 :initarg :numeric-argument
		     :reader accelerator-gesture-numeric-argument))
  (:report
    (lambda (condition stream)
      (format stream "Accelerator event ~S seen" (accelerator-gesture-event condition)))))


;;; Useful functions for a few things

(defun true (&rest args)
  (declare (ignore args))
  T)

(defun false (&rest args)
  (declare (ignore args))
  nil)

;;; Allegro complains about setting the function slot of IGNORE,
;;; a symbol in the COMMON-LISP package.
#-(or Minima excl) 
(defun ignore (&rest args)
  (declare (ignore args))
  nil)


#+(and pcl genera)
(progn

(walker:define-walker-template scl:letf walker::walk-let)
(walker:define-walker-template scl:letf* walker::walk-let*)
)

;;; --- from Silica's CLIM-DEFS, 12/23/91 (doughty)

(defvar *whitespace* '(#\Space #\Tab))

(defun whitespace-character-p (character)
  (member character *whitespace* :test #'char-equal))

(defun word-break-character-p (thing)
  (and (characterp thing)
       (not (alpha-char-p thing))))

(defun newline-p (thing)
  (and (characterp thing)
       (char-equal thing #\Newline)))

(defun writable-character-p (character)
  ;; do we need this extra level of safety?
  (and (characterp character)
       (or (graphic-char-p character)
	   (not (null (find character '#(#\Newline #\Return #\Tab) :test #'char-equal))))))

;;; Macros used by graphics-output-recording.

;;; A bit of scaffolding for some macroexpansion kludgery below.
;;; All of this is necessary only because we don't have a portable code walker.
(eval-when (compile load eval)
  (defvar *slot-accessing-macros* nil))

;;; Define a macro as needing macroexpansion in WITH-SLOTS-BOUND-AS-NEEDED.
;;; If LAMBDA-LIST is omitted, we assume the macro is defined elsewhere.
(defmacro define-slot-accessing-macro (name &optional (lambda-list nil lambda-list-p)
				       &body body)
  `(define-group ,name define-slot-accessing-macro
     (eval-when (compile load eval) (pushnew ',name *slot-accessing-macros*))
     ,(when lambda-list-p `(defmacro ,name ,lambda-list ,@body))))

;;; => (LET ((FOO (SLOT-VALUE OBJECT 'FOO))) <body>), if <body> uses the variable FOO.
;;; This allows us to fetch the value from the object only once, and only as needed.
;;; Sort of a kludge, but allows for elegant code.
(defmacro with-slots-bound-as-needed (slot-names object &body body &environment environment)
  (labels ((body-uses-variable-p (variable body)
	     (some #'(lambda (x) (form-uses-variable-p variable x)) body))
	   (form-uses-variable-p (variable form)	;stupid (simple) code walker
	     (when (atom form)			;Take care of symbols now.
	       ;; Symbol macros?  I don't gotta show you no stinkin' symbol macros.
	       (return-from form-uses-variable-p
		 (eql variable form)))		;The business end of this code walker.
	     (when (member (first form) *slot-accessing-macros*)  ;; Ugly Kludge #42...
	       (setf form (macroexpand-1 form environment)))	  ;; for LS-THICKNESS
	     ;; We check the form of non-quoted lisp code for variable uses.  We don't
	     ;; handle special operators, but that doesn't seem to be a problem.
	     (when (member (first form) '(let let*)) ; We handle these special operators
	       (return-from form-uses-variable-p
		 (or (some #'(lambda (x)
			       (and (listp x) (form-uses-variable-p variable (second x))))
			   (second form))
		     (body-uses-variable-p variable (cddr form)))))
	     (when (member (first form) '(multiple-value-bind multiple-value-setq))
	       (return-from form-uses-variable-p
		 (body-uses-variable-p variable (cddr form))))
	     (when (member (first form) '(flet labels))
	       (dolist (fn (second form))
		 (let ((lambda-state 'nil))
		   (dolist (ll (second fn)) ;; The lambda-list
		     (cond
		       ((member ll '(&optional &key &aux)) (setf lambda-state ll))
		       ((and lambda-state (consp ll))
			(when (form-uses-variable-p variable (second ll))
			  (return-from form-uses-variable-p t))))))
		 (when (body-uses-variable-p variable (cddr fn))
		   (return-from form-uses-variable-p t)))
	       (return-from form-uses-variable-p
		 (body-uses-variable-p variable (cddr form))))
	     (and (not (member (car form) '(quote function)))
		  (body-uses-variable-p variable (cdr form)))))
    `(let (,@(with-collection
	       (dolist (var slot-names)
		 (multiple-value-bind (variable-name slot-name)
		     (if (symbolp var) (values var var)
			 (values (first var) (second var)))
		   (when (body-uses-variable-p variable-name body)
		     (collect `(,variable-name (slot-value ,object ',slot-name))))))))
       ,@body)))
