;;; -*- Mode: LISP; Syntax: Common-lisp; Package: CLIM-INTERNALS; Base: 10 -*-

(in-package "CLIM-INTERNALS")

"Copyright (c) 1989, 1990 International Lisp Associates.  All rights reserved."

(defclass font-base
  ()
  ((name :type 'string :initarg :name)
   (style :type '(or symbol list) :initarg :style)
   (size :type 'fixnum :initarg :size)
   (id :type 'fixnum :initarg :id)
   (number :type 'fixnum :reader font-number)
   (qd-spec :type 'list :reader font-qd-spec)
   (derived :reader font-derived-p)))

(defmethod font-number :before ((font font-base))
  (with-slots (name number) font
      (unless (boundp 'number)
        (ccl:%stack-block ((name-string 256)
                           (num 2))
          (ccl:%put-string name-string name)
          (#_GetFNum :ptr name-string :ptr num)
          (setf number (ccl:%get-word num))
          ;
          ; GetFNum returns 0 if name is not found.  This conflicts with
          ; the system font number which requires that we explicitly check
          ; if that is what we are asking for.  See Mac Tech Note #191.
          ;
          (when (zerop number)
            (#_GetFontName :word 0 :ptr name-string)
            (when (string-not-equal (ccl:%get-string name-string) name)
              (setf number nil)))))))

(defmethod font-qd-spec :before ((font font-base))
  (with-slots (style size number qd-spec) font
    (unless (boundp 'number)
      (font-number font))
    (unless (boundp 'qd-spec)
      (when number
        (setf qd-spec (make-qd-font-spec :number number
                                         :face (font-style-qd-face style)
                                         :size size))))))

(defun font-style-qd-face (style)
  (etypecase style
    (symbol (if (eq style :plain)
              0
              (let ((font-face (assoc style on-mcl::*mac-font-style-alist*)))
                (unless font-face
                  (error "Unknown font style keyword ~S" style))
                (dpb 1 (byte 1 (cdr font-face)) 0))))
    (list (let ((face 0))
            (dolist (key style)
              (setf face (logior face (font-style-qd-face key))))
            face))))

(defmethod font-derived-p :before ((font font-base))
  (with-slots (number size derived) font
    (unless (boundp 'number)
      (font-number font))
    (unless (boundp 'derived)
      (when number
        (setf derived (zerop (#_RealFont :word number :word size :word)))))))

(defmethod font-spec ((font font-base) &key (value :multiple))
  (with-slots (name style size) font
    (ecase value
           (:multiple (values name style size))
           (:list (values (list name style size))))))
     

(defun valid-font-style-p (style)
  (or (eq style :plain)
      (assoc style on-mcl::*mac-font-style-alist*)))

(defmethod initialize-instance :before ((font font-base) &key name style &allow-other-keys)
  (unless name
    (error "Font name must be specified"))
  (unless (valid-font-style-p style)
    (error "Font style ~S invalid" style)))

(defmethod initialize-instance :after ((font font-base) &key &allow-other-keys)
  (font-number font)
  (font-qd-spec font)
  (font-derived-p font))

;
; Font Metrics
;
(defclass font-metrics-mixin
  ()
  ((ascent :type 'flonum :reader font-ascent)
   (descent :type 'flonum :reader font-descent)
   (leading :type 'flonum :reader font-leading)
   (max-width :type 'flonum :reader font-max-width)
   (widths :type '(simple-array ccl::short) :initform nil)))

(defclass font
  (font-metrics-mixin font-base)
  ())

(defun make-font (name style size)
  (make-instance 'font :name name :style style :size size))

(defun make-font-from-style (style)
  (let ((spec (on-mcl::generate-font style)))
    (make-instance 'font :name (pop spec) :style (pop spec) :size (pop spec))))

(defvar *font-table* (make-hash-table :test #'eq))

(defun get-font-from-style (style)
  (etypecase style
    (text-style)
    (list (setq style (parse-text-style style))))
  (or (gethash style *font-table*)
      (setf (gethash style *font-table*) (make-font-from-style style))))

(defmethod stream-glyph-for-character ((stream on-mcl::basic-mcl-medium)
				       character style
				       &optional our-font)
  (declare #+Genera (values index font escapement-x escapement-y origin-x origin-y bb-x bb-y))
  (multiple-value-bind (character-set index)
      (char-character-set-and-index character)
    (unless (eql character-set *standard-character-set*)
      (error "Can't handle non-standard character sets yet!"))
    (let* ((font (or our-font
                     (get-font-from-style style)))
           (escapement-x (font-glyph-width font index))
           (escapement-y 0)
           (origin-x 0)
           (origin-y (font-ascent font))
           (bb-x escapement-x)
           (bb-y (+ origin-y (font-descent font))))
      (values index font escapement-x escapement-y origin-x origin-y bb-x bb-y))))

(defstruct (qd-font-spec (:type list))
  number
  face
  size)

(defun port-qd-font-spec (port)
  (let ((fs (make-qd-font-spec)))
    (setf (qd-font-spec-number fs) (ccl:rref port grafport.txfont)
          (qd-font-spec-face   fs) (ccl:rref port grafport.txface)
          (qd-font-spec-size   fs) (ccl:rref port grafport.txsize))
    fs))

(defun port-set-qd-font-spec (qd-font-spec)
  (#_TextFont :word (qd-font-spec-number qd-font-spec))
  (#_TextFace :word (qd-font-spec-face qd-font-spec))
  (#_TextSize :word (qd-font-spec-size qd-font-spec)))

(eval-when (eval compile)
(defmacro with-temporary-port (port &body body)
  (let ((old-port (gensymbol "OLD-PORT")))
    `(ccl::%stack-block ((,old-port 4)
                         (,port #.(ccl:record-length :grafport)))
       (unwind-protect
         (progn
           (#_GetPort :ptr ,old-port)
           (#_OpenPort :ptr ,port)
           ,@body)
         (#_SetPort :ptr (ccl::%get-safe-ptr ,old-port))
         (#_ClosePort :ptr ,port)))))

(defmacro with-font-in-port ((font port) &body body)
    (let ((old-font-spec (gensymbol "OLD-FONT-SPEC"))
          (old-port (gensymbol "OLD-PORT")))
      `(let ((,old-font-spec (port-qd-font-spec ,port)))
         (ccl::%stack-block ((,old-port 4))
           (#_GetPort :ptr ,old-port)
           (unwind-protect
             (progn
               (#_SetPort :ptr ,port)
               (port-set-qd-font-spec (font-qd-spec ,font))
               ,@body)
             (port-set-qd-font-spec ,old-font-spec)
             (#_SetPort :ptr (ccl::%get-safe-ptr ,old-port)))))))

#+ignore ;; I don't think we need to restore the pen position.
(defmacro with-pen-pos ((x y) &body body)
  (let ((old-position (gensymbol "OLD-POSITION")))
    `(ccl:with-port ccl:wptr
       (ccl:%stack-block ((,old-position 4))
         (#_GetPen :ptr ,old-position)
         (unwind-protect
             (progn (#_MoveTo (round ,x) (round ,y))
                    ,@body)
           (#_MoveTo :word (ccl:%get-word ,old-position 2)
                        :word (ccl:%get-word ,old-position 0)))))))
)



(defmethod stream-write-char-internal ((stream on-mcl::basic-mcl-medium)
				       index font color x y)
  (with-slots (on-mcl::grafport) stream
    (ccl:with-port on-mcl::grafport
      (on-mcl::xupdate-graphics-ink color stream)
      (#_MoveTo :word (round x) :word (round y))
      (with-font-in-port (font on-mcl::grafport)
        (#_DrawChar :word index)))))

(eval-when (eval compile)
(defmacro qd-draw-string (string &optional len)
  (let ((string-buffer (gensymbol "STRING-BUFFER")))
    `(ccl:%stack-block ((,string-buffer 256))
       (ccl:%put-string ,string-buffer ,string)
       ,@(when len
           `((ccl:%put-byte ,string-buffer ,len)))
       (#_DrawString :ptr ,string-buffer))))
)

(defmethod stream-write-string-internal ((stream on-mcl::basic-mcl-medium)
					 glyph-buffer start end font color x y)
  (setq x (round x) y (round y))
  ; QD draws from the baseline up:
  (incf y (round (+ (font-ascent font)
                    ;(font-descent font)
                    (font-leading font)
                    )))
  (when (<= end start) (return-from stream-write-string-internal))
  (let ((length (- end start)))
    (unless (and (zerop start)
                 (= length (length glyph-buffer)))
      (setf glyph-buffer (replace (make-array length) glyph-buffer
                                  :start2 start
                                  :end2 end)))
    (with-slots (on-mcl::grafport) stream
      (#_MoveTo :word x :word y)
      (on-mcl::xupdate-graphics-ink color stream)
      (with-font-in-port (font on-mcl::grafport)
        ; (with-temporary-string (span-buffer :length 255 :adjustable nil)
        (let ((span-buffer (make-string 255)))
          (do* ((remaining length (- remaining span))
                (span (min remaining 255) (min remaining 255))
                (index 0 (+ index span)))
               ((<= remaining 0))
            (dotimes (i span)
              (setf (aref span-buffer i) (code-char (aref glyph-buffer (+ index i)))))
            ;(replace span-buffer glyph-buffer :start2 index :end2 (+ index span))
            (qd-draw-string span-buffer span)))))))

(eval-when (eval compile)
(defmacro extract-fixed (pointer offset)
  (let ((2_16 (expt 2 16)))
    `(let ((whole (ccl:%get-word ,pointer ,offset))
           (frac (ccl:%get-word ,pointer (+ ,offset 2))))
       (when (ldb-test (byte 1 15) whole)
         (decf whole ,2_16))
       (+ (float whole) (float (/ frac ,2_16))))))

(defmacro extract-fixed-word-immed (word)
  (let ((2_8 (expt 2 8)))
    `(let ((whole (ldb (byte 8 8) ,word))
           (frac (ldb (byte 8 0) ,word)))
       (when (ldb-test (byte 1 7) whole)
         (decf whole ,2_8))
       (+ (float whole) (float (/ frac ,2_8))))))

(defmacro fixed-to-fixed-word (pointer offset)
  `(dpb (ccl:%get-signed-word ,pointer ,offset) (byte 8 8)
        (dpb (ldb (byte 8 8) (ccl:%get-word ,pointer (+ ,offset 2))) (byte 8 0)
             0)))
)

(defmethod load-metrics ((font font))
  (with-slots (ascent descent leading max-width widths) font
    (unless widths
      (with-temporary-port port
        (with-font-in-port (font port)
          (ccl:%stack-block ((metrics 20))
            (#_FontMetrics :ptr metrics)
            (setf ascent (extract-fixed metrics 0)
                  descent (extract-fixed metrics 4)
                  leading (extract-fixed metrics 8)
                  max-width (extract-fixed metrics 12))
                  ; slh: was creating special variable width-table-handle!
            (let ((widths-array (make-array 256 :element-type '(unsigned-byte 16))))
              (ccl:with-dereferenced-handles ((width-table (ccl::%get-safe-ptr metrics 16)))
                (dotimes (index 256)
                  (setf (aref widths-array index)
                        (fixed-to-fixed-word width-table (* index 4)))))
              (setf widths widths-array)))))))
  font)

(defmethod font-glyph-width ((font font) glyph)
  (with-slots (widths) font
    (unless widths
      (load-metrics font))
    (extract-fixed-word-immed (aref widths glyph))))
