/*
 *
 *  Kurt Konolige, February 1992
 *
 *  Low-level functions to gather up a group of side-looking
 *  sonar readings, convert them to depth buffer
 * 
 */


#include <stdio.h>

#include "constants.h"
#include "structures.h"
#include "parameters.h"
#include "constructs.h"
#include "dispall.h"

/* Main function: distributes left or right sonar readings into
 * buckets, counts them up and returns them as a linear array.
 */


static int xnum,ynum,thnum;	/* global stepping params */
static float xoff,yoff,dx,dy;
static int fxnum,fynum,fthnum;	/* global stepping params */
static float fxoff,fyoff,fdx,fdy;

/* set up the quantization bucket,
 *  setting global offset and delta variables
 */


void
side_setup(int xmin,int xmax,int xn,int ymin,int ymax,int yn) 
{
  int i, j;

  xoff = xmin;
  yoff = ymin;
  xnum = xn;
  ynum = yn;
  dx = (xmax - xmin) / xn;
  dy = (ymax - ymin) / yn;
}


void
side_interp(side which,unsigned char *ans) 
     /* ans is a depth buffer; which is 1 for left, 0 for right */
{
  float *x_buf, *y_buf;
  int start, end, mx;
  float xdif;
  int i, xi, yi;
  float x,y;

  if (which == LEFT)
    { start = sl_buf->start;	/* left buffer */
      end   = sl_buf->end;
      mx    = sl_buf->limit;
      x_buf = sl_buf->xbuf;
      y_buf = sl_buf->ybuf;
      which = 1;
     }
     else
     {
      start = sr_buf->start;	/* right buffer */
      end = sr_buf->end;
      mx = sr_buf->limit;
      x_buf = sr_buf->xbuf;
      y_buf = sr_buf->ybuf;
      which = -1;
     }


  /* empty depth buffer */

  for (i = 0; i <= ynum; i++) ans[i] = SLBUF_NULL;

  /* put sonar points into depth buffer */

  for (i = start; i != end; i++) /* put sonar readings into buckets */
    { if (i > mx) {if (end == 0) { break; } 
		   else {i = -1; continue; }}	/* reset if beyond the end */
      x = (which * y_buf[i]) - xoff;  /* let x-coord be side of robot, y front */
      y = x_buf[i] - yoff;
      xi = (int)(x / dx);
      yi = (int)(y / dy);
/*  fprintf(stderr,"x: %f  y: %f  xi: %d  yi: %d\n",x,y,xi,yi); */
      if (xi >= 0 && yi >=0 && yi < ynum && xi < ans[yi])
	{ ans[yi] = xi;
/*	  if (yi > 0) ans[--yi] = xi;   
	  if (yi < ynum) ans[yi + 2] = xi; */
	}
    }
}


void
trans_ss_arr(side which,unsigned char *arr,float *xarr,float *yarr) 
     /* translate to rw x,y; which is LEFT, RIGHT, or FRONT */
{
  int i;  float f;

  if (which == FRONT)		/* front buffer */
    { for (i = 0; i < fynum; i++)
	{
	  if (arr[i] == SLBUF_NULL) continue;
	  yarr[i] = (i * fdy) + fyoff;
	  xarr[i] =  (arr[i] * fdx) + fxoff;
	}
    }
  else
    { if (which == RIGHT) f = -1.0; else f = 1.0;
      for (i = 0; i < ynum; i++)
	{
	  if (arr[i] == SLBUF_NULL) continue;
	  xarr[i] = (i * dy) + yoff;
	  yarr[i] = f * ((arr[i] * dx) + xoff);
	}
    }
}

front_setup(xmin,xmax,xn,ymin,ymax,yn) 
{
  int i, j;

  fxoff = xmin;
  fyoff = ymin;
  fxnum = xn;
  fynum = yn;
  fdx = (xmax - xmin) / xn;
  fdy = (ymax - ymin) / yn;
}

front_interp(ans)		/* ans is a depth buffer */
int ans[];
{
  float *x_buf, *y_buf;
  int start, end, mx;
  float xdif;
  int i, xi, yi;
  float x,y;

  start = sraw_buf->start;	/* front buffer */
  end = sraw_buf->end;
  mx = sraw_buf->limit;
  x_buf = sraw_buf->xbuf;
  y_buf = sraw_buf->ybuf;

  /* empty depth buffer */

  for (i = 0; i <= fynum; i++) ans[i] = SLBUF_NULL;

  /* put sonar points into depth buffer */

  for (i = start; i != end; i++) /* put sonar readings into buckets */
    { if (i > mx) {if (end == 0) { break; } 
		   else {i = -1; continue; }}	/* reset if beyond the end */
      x = x_buf[i] - fxoff;  /* let x-coord be front of robot, y side */
      y = y_buf[i] - fyoff;
      xi = (int)(x / fdx);
      yi = (int)(y / fdy);
/*  fprintf(stderr,"xo: %f  yo: %f  x: %f  y: %f  xi: %d  yi: %d\n",x_buf[i],y_buf[i],x,y,xi,yi);  */
      if (xi >= 0 && yi >=0 && yi < fynum && xi < fxnum && xi < ans[yi])
	{ ans[yi] = xi;
/*	  if (yi > 0) ans[--yi] = xi;   
	  if (yi < fynum) ans[yi + 2] = xi; */
	}
    }
}

front_interp_with_rot(ans,ang)	/* ans is a depth buffer */
     double ang;		/* ang is angle of rotation for points */
     int ans[];			/* this one returns the correct buffer order */
{
  float *x_buf, *y_buf;
  int start, end, mx;
  float xdif;
  int i, xi, yi;
  float x,y;
  float cth, sth;
  float tx, ty;

  cth = cos(ang * deg_to_rad);	/* we should only do this when dth > 0... */
  sth = sin(ang * deg_to_rad); 

  start = sraw_buf->start;	/* front buffer */
  end = sraw_buf->end;
  mx = sraw_buf->limit;
  x_buf = sraw_buf->xbuf;
  y_buf = sraw_buf->ybuf;

  /* empty depth buffer */

  for (i = 0; i <= fynum; i++) ans[i] = SLBUF_NULL;

  /* put sonar points into depth buffer */

  for (i = start; i != end; i++) /* put sonar readings into buckets */
    { if (i > mx) {if (end == 0) { break; } 
		   else {i = -1; continue; }}	/* reset if beyond the end */
      tx = x_buf[i];
      ty = y_buf[i];
      x = (tx * cth) - (ty * sth);
      y = (ty * cth) + (tx * sth);

      x -= fxoff;  /* let x-coord be front of robot, y side */
      y -= fyoff;
      xi = (int)(x / fdx);
      yi = (int)(y / fdy);
/*  fprintf(stderr,"xo: %f  yo: %f  x: %f  y: %f  xi: %d  yi: %d\n",x_buf[i],y_buf[i],x,y,xi,yi);  */
      if (xi >= 0 && yi >=0 && yi < fynum && xi < fxnum && xi < ans[yi])
	{ ans[yi] = xi;
/*	  if (yi > 0) ans[--yi] = xi;   
	  if (yi < fynum) ans[yi + 2] = xi; */
	}
    }
}




/* 
 * Side-sonar occupancy functions, a little smarter than the old ones
 * 
 * 
 */

side_occ(d,s1,s2,xy,ang)	/* Same as occupied2, but uses side sonars too */
     double ang;
{
  float *x_buf, *y_buf;
  int start, end, mx;
  float df, s1f, s2f;
  int i;
  float cth, sth, ret;
  float tx, ty, x, y;

  cth = cos(ang * deg_to_rad);	/* we should only do this when dth > 0... */
  sth = sin(ang * deg_to_rad); 


  if (xy < 0)
    { df = -d; }
  else { df = d; }
  if ( s1 > s2)
    { s1f = s2; s2f = s1; }
  else 
    { s1f = s1; s2f = s2; }

  ret = 5000.0;

  /* check sonar points for containment in rectangle */

 if (xy == -1)		/* -Y direction */
    { 
      start = sr_buf->start;	/* right buffer */
      end = sr_buf->end;
      mx = sr_buf->limit;
      x_buf = sr_buf->xbuf;
      y_buf = sr_buf->ybuf;

      for (i = start; i != end; ++i) /* put sonar readings into buckets */
	{ if (i > mx) {i = -1; continue; }    /* reset if beyond the end */
	  tx = x_buf[i];
	  ty = y_buf[i];
	  x = (tx * cth) - (ty * sth);
	  y = (ty * cth) + (tx * sth);

	  if (y <= df && x >= s1f && x <= s2f)
	    { if ((- y) < ret) ret = - y; }
	}
    }
  else if (xy == 1)		/* +Y direction */
    {
      start = sl_buf->start;	/* left buffer */
      end   = sl_buf->end;
      mx    = sl_buf->limit;
      x_buf = sl_buf->xbuf;
      y_buf = sl_buf->ybuf;

      for (i = start; i != end; ++i) /* put sonar readings into buckets */
	{ if (i > mx) {i = -1; continue; }    /* reset if beyond the end */
	  tx = x_buf[i];
	  ty = y_buf[i];
	  x = (tx * cth) - (ty * sth);
	  y = (ty * cth) + (tx * sth);

	  if (y >= df && x >= s1f && x <= s2f)
	    { if (y < ret) ret = y; }
	}
    }
  return((int)ret);
}

/*
 * side buffer interpretations
 *
 */

dbuf left_side, right_side;	/* buffers for left/right sonar readings */

int ss_infl_thresh = 100;
int ss_medval_gap = 5;
float ss_medval_slope = 1.1;

int ss_gap_fill = 5;
int ss_gap_fill_brk = 6;

int ss_bkpt_gap = 6;
int ss_bkpt_num = 9;

void				/* interprets side sonar buffer */
conv_ss_buf(dbuf *buf,side side) /* LEFT or RIGHT */
{
  side_interp(side,buf->buf);
  medval(buf->buf,buf->num,ss_medval_gap,ss_medval_slope,SLBUF_NULL);
/*  interpolate_gaps(buf->buf,buf->num, ss_gap_fill, ss_gap_fill_brk); */
  trans_ss_arr(side,buf->buf,buf->xbuf,buf->ybuf);
  find_breaks(buf->buf,buf->edge,buf->num,ss_bkpt_gap,ss_bkpt_num);
  conv(lap_conv_arr,lap_conv_arr_n,buf->buf,buf->lap,buf->num);
  find_inflections(buf->edge,buf->lap,buf->num,ss_infl_thresh);
}

#define SS_XMIN -2000
#define SS_XMAX   500
#define SS_XNUM    70
#define SS_YMIN   100
#define SS_YMAX  2500
#define SS_YNUM   200

void
init_side_interps()
{
  side_setup(SS_YMIN,SS_YMAX,SS_YNUM,SS_XMIN,SS_XMAX,SS_XNUM);
  left_side.num = SS_XNUM;
  right_side.num = SS_XNUM;
}
