/*#########################################
 * servo.c
 *
 * movement servoing for
 * the erratic simulator
 *#########################################
 */

#include <malloc.h>
#include <stdio.h>
#include "structures.h"
#include "constants.h"
#include "parameters.h"

servo_robot errservo;		/* holds the current robot servoing state */

reset_servo_state()		/* reset total state */
{
  reset_servo_trans();
  reset_servo_rot();
}

reset_servo_trans()		/* reset translation servo */
{
  errservo.tr_state = STOPPED;
  errservo.tr_pos = 0.0;
  errservo.tr_accum = 0.0;
}

extern float head_control;	/* current heading to achieve */

reset_servo_rot()		/* reset rotation servo */
{
  errservo.rot_state = STOPPED;
  errservo.rot_pos = 0.0;
  errservo.rot_accum = 0.0;
  head_control = -1;
}


int
in_tr_bounds(v,a,d,ac)		/* returns 1 if robot can halt in distance d-ac, */
				/* given current velocity v and accel a */
     float v,a,d,ac;
{
  ac = d - ac;
  if (a == 0.0) return(0);	/* no accel, can't do it */
  else if ((ac > 0.0 && d < 0.0) || (ac < 0.0 && d > 0.0)) return(0); /* overshoot */
  else
    {
      v = (v*v) / (2.0*a);
      if (v <= ABS(ac)) return(1); else return(0);
    }
}


void
ramp_up_tv(v,a,d)		/* increases velocity in current dir, if possible */
     float v,a,d;
{
  float s;
  s = FSIGNUM(v);
  if (s == 0) s = FSIGNUM(d);
  v += inc_time * a * s;
  if (ABS(v) > erratic.mtv) v = s * erratic.mtv;
  erratic.tv = v;
}

int
ramp_down_tv(v,a)		/* decreases velocity to zero */
     float v,a;
{
  a = v - inc_time * a * FSIGNUM(v);
  if (SIGNUM(v) != SIGNUM(a))	/* we've hit bottom */
    { v = 0.0; }
  else 
    { v = a; }
  erratic.tv = v;
  if (v == 0.0) return(1); else return(0);
}

void
reverse_tv(v,a)			/* reverse our velocity */
     float v,a;
{
  v -= inc_time * a * FSIGNUM(v);
  erratic.tv = v;
}


void 
achieve_tvel(v,a,tv)		/* changes v towrads tv, if possible */
     float v,a,tv;
{
  float s, inc;
  s = tv - v;
  if (s != 0.0)
    {
      inc = inc_time * a * FSIGNUM(s);
      if (ABS(inc) > ABS(s)) v = tv;
      else v += inc;
      if (ABS(v) > erratic.mtv)	/* velocity is too great here */
	v = FSIGNUM(v) * erratic.mtv;
      erratic.tv = v;
    }
  if (v == 0.0 && tv == 0.0)
    errservo.tr_state = STOPPED;
}

void
servo_trans()			/* check for translation servoing */
{
  float v, tv, a, d, ac;
  switch(errservo.tr_state)
    {
    case STOPPED:		/* do nothing */
      break;

    case POS:			/* position servoing */
      errservo.tr_accum += inc_time * erratic.tv; /* increment position accumulation */
      v = erratic.tv;
      a = erratic.mta;
      d = errservo.tr_pos;
      ac = errservo.tr_accum;
      if (SIGNUM(v) == 0 || SIGNUM(v) == SIGNUM(d)) /* going in the same direction */
	{
	  if (in_tr_bounds(ABS(v),a,d,ac)) ramp_up_tv(v,a,d); /* keep going faster */
	  else if (ramp_down_tv(v,a)) reset_servo_trans(); /* we're stopped */
	}
      else reverse_tv(v,a);	/* reverse direction */
      break;

    case VEL:			/* velocity servoing */
      v = erratic.tv;
      a = erratic.mta;
      tv = errservo.tr_pos;
      achieve_tvel(v,a,tv);
      break;
    }
}


int
in_rot_bounds(v,a,d,ac)		/* returns 1 if robot can halt at servo pos d-ac, */
				/* given current velocity v and accel a */
     float v,a,d,ac;
{
  ac = d - ac;
  if (a == 0.0) return(0);	/* no accel, can't do it */
  else if ((ac > 0.0 && d < 0.0) || (ac < 0.0 && d > 0.0)) return(0); /* overshoot */
  else
    {
      v = (v*v) / (2.0*a);
      if (v <= ABS(ac)) return(1); else return(0);
    }
}

void
ramp_up_rv(v,a,d)		/* increases velocity in current dir, if possible */
     float v,a,d;
{
  float s;
  s = FSIGNUM(v);
  if (s == 0) s = FSIGNUM(d);
  v += inc_time * a * s;
  if (ABS(v) > erratic.mrv) v = s * erratic.mrv;
  erratic.rv = v;
}

int
ramp_down_rv(v,a)		/* decreases velocity to zero */
     float v,a;
{
  a = v - inc_time * a * FSIGNUM(v);
  if (SIGNUM(v) != SIGNUM(a))	/* we've hit bottom */
    { v = 0.0; }
  else 
    { v = a; }
  erratic.rv = v;
  if (v == 0.0) return(1); else return(0);
}

void
reverse_rv(v,a)			/* reverse our velocity */
     float v,a;
{
  v -= inc_time * a * FSIGNUM(v);
  erratic.rv = v;
}


void 
achieve_rvel(v,a,tv)		/* changes v towrads tv, if possible */
     float v,a,tv;
{
  float s, inc;
  s = tv - v;
  if (s != 0.0)
    {
      inc = inc_time * a * FSIGNUM(s);
      if (ABS(inc) > ABS(s)) v = tv;
      else v += inc;
      if (ABS(v) > erratic.mrv)	/* velocity is too great here */
	v = FSIGNUM(v) * erratic.mrv;
      erratic.rv = v;
    }
  if (v == 0.0 && tv == 0.0)
    errservo.rot_state = STOPPED;
}

void
servo_rot()			/* check for translation servoing */
{
  float v, tv, a, d, ac;
  switch(errservo.rot_state)
    {
    case STOPPED:		/* do nothing */
      break;

    case POS:			/* position servoing */
      errservo.rot_accum += inc_time * erratic.rv; /* increment position accumulation */
      v = erratic.rv;
      a = erratic.mra;
      d = errservo.rot_pos;
      ac = errservo.rot_accum;
      if (SIGNUM(v) == 0 || SIGNUM(v) == SIGNUM(d)) /* going in the same direction */
	{
	  if (in_rot_bounds(ABS(v),a,d,ac)) ramp_up_rv(v,a,d); /* keep going faster */
	  else if (ramp_down_rv(v,a)) reset_servo_rot(); /* we're stopped */
	}
      else reverse_rv(v,a);	/* reverse direction */
      break;

    case VEL:			/* velocity servoing */
      v = erratic.rv;
      a = erratic.mra;
      tv = errservo.rot_pos;
      achieve_rvel(v,a,tv);
      break;
    }
}


void 
servo()				/* do trans and rot servos */
{
  servo_trans();
  servo_rot();
}


void 
c_move(d)			/* just translate this amount, in mm */
     float d;
{
  last_move = 0;
  errservo.tr_state = POS;
  errservo.tr_pos = d;
  errservo.tr_accum = 0.0;
}

void
c_rotate(d)			/* rotate this amount in angular units (radians) */
     float d;
{
  last_move = 0;
  errservo.rot_state = POS;
  errservo.rot_pos = d;
  errservo.rot_accum = 0.0;
}

void
c_rotate_deg(d)			/* rotate this amount in degrees */
     float d;
{
  c_rotate(deg_to_ang(d));
}

void 
c_trvel(d)			/* translate at this speed, in mm/sec */
     float d;
{
  last_move = 0;
  errservo.tr_state = VEL;
  errservo.tr_pos = d;
  errservo.tr_accum = 0.0;
}

void
c_rotvel(d)			/* rotate at this speed, in ang units/sec */
     float d;
{
  last_move = 0;
  errservo.rot_state = VEL;
  errservo.rot_pos = d;
  errservo.rot_accum = 0.0;
}

void
c_rotvel_deg(d)			/* rotate at this speed, in deg/sec */
     float d;
{
  c_rotvel(deg_to_ang(d));
}

