/* 
 * C functions to read and write the IO buffers
 *
 *   Changed to reflect packet CRC check Dec 1993
 */

#include <stdio.h>
#include <sys/time.h>
#include <sys/types.h>
#include <sgtty.h>

#define MAX_IN_SIZE 256
#define MAX_OUT_SIZE 256

int comm_in_line, comm_out_line; /* file descriptors */

char out_buf[MAX_OUT_SIZE];
char  in_buf[MAX_IN_SIZE];
char temp_buf[MAX_OUT_SIZE];
int in_buf_p, in_buf_max;	/* current pointer, max chars in buffer */
#define OUT_BUF_BEG 3		/* start of data */
int out_buf_p = OUT_BUF_BEG;


/*------------------------ Output of Packets ------------------
 *
 * Format for packets:
 *
 *   SYNC1
 *   SYNC2
 *   data count
 *   data[0..data count]
 *   CRC  (addition of all data and count, restricted to one byte)
 *   END
 *
 */

#define SYNC1_CHAR   	0xfa	/* first sync byte of packet */
#define SYNC2_CHAR   	0xfb	/* second sync byte of packet */
#define END_CHAR        0xfe	/* byte at end of packet */
#define CRC_STATE       0x1	/* states of the input packet reader */
#define COUNT_STATE     0x2
#define DATA_STATE      0x3

void
put_buf_byte(n)			/* puts a byte, forces output if buffer full */
 
{ if (out_buf_p < MAX_OUT_SIZE) /* check overflow */
    out_buf[out_buf_p++] = n;
}


int
output_packet(void)		/* send out packet, including byte count */
{
  int s;
  out_buf[0] = SYNC1_CHAR;
  out_buf[1] = SYNC2_CHAR;
  out_buf[2] = out_buf_p - OUT_BUF_BEG;
  put_buf_byte(calc_crc(&out_buf[2]));
  put_buf_byte(END_CHAR);
  s = write(comm_out_line,out_buf,out_buf_p);
/*  fflush(tty_f); */
  out_buf_p = OUT_BUF_BEG;
  return(s);
}

int
calc_crc(u_char *ptr)		/* ptr is array of bytes, first is data count */
{
  int n;
  int c = 0;
  n = *(ptr++);
  while (n-- > 0) {
    c += *(ptr++);
    c = c & 0xff;
  }
  return(c);
}
    

write_out_buf()			/* writes the buffer out */

{  write(comm_out_line,out_buf,out_buf_p);
   out_buf_p = 0;
 }

/*-------- read buffer routines for getting chars from the stream -------*/

int debug_stream = 0;		/* for printing out received chars */

get_buf_byte()			/* returns one char from buffer, does */
				/* a READ if necessary */
{ 
  int c;
  if (in_buf_p >= in_buf_max)	/* need to do a READ */
    { 
      in_buf_max = read(comm_in_line,in_buf,MAX_IN_SIZE); 
      in_buf_p = 0;
      if (in_buf_max < 0) {perror("read fault"); 
			   in_buf_max = 0;
			   return(-1);}; /* we failed to read anything */
    }
  
  c = 0xff & in_buf[in_buf_p++];
  if (debug_stream) printf(" %d ",c);
  return(c);
}


char *form_arg_string()		/* returns a pointer to a string */
{
  int i;
  char *p;
  p = temp_buf;
  i = get_buf_byte();		/* string length */
  if (i <= 0 || i > 100) i = 100;
  while (i--)
    { *p++ = get_buf_byte(); }
  *p = 0;
  p = temp_buf;
  return(p);
}


clear_in_buf()			/* clear out input buffer */

{ 
  in_buf_p = 0;
  in_buf_max = 0;
}


int
check_in_buf(int wait)		/* returns 0 if no input waiting, 1 if yes */
     
{ int f;
  struct timeval tp;		/* time interval structure for timeout */
  fd_set fdset;			/* fd set ??? */
  
  if ( in_buf_p < in_buf_max )	/* have some chars in the buffer */
    { return(in_buf_max - in_buf_p); }
  else
    { tp.tv_sec = wait / 1000;		/* we're polling */
      tp.tv_usec = (wait % 1000) * 1000;

      FD_ZERO(&fdset);
      FD_SET(comm_in_line,&fdset);
      f = comm_in_line + 1;
      f = select(f,&fdset,NULL,NULL,&tp) ;
      if (f >= 1)		/* we have something, could be a refused connect */
	{ in_buf_max = read(comm_in_line,in_buf,MAX_IN_SIZE); 
	  in_buf_p = 0;
	  if (in_buf_max < 0) {perror("read fault"); 
			       in_buf_max = 0;
			       return(-1);} /* we failed to read anything */
	  else
	    { return(in_buf_max - in_buf_p); }
	}
      else return(0);
     }
}


wait_in_buf()			/* wait until there's input */
{ if (in_buf_p == in_buf_max)	/* need to do a READ */
    { in_buf_max = read(comm_in_line,in_buf,MAX_IN_SIZE); 
      in_buf_p = 0;
      if (in_buf_max < 0) {in_buf_max = 0;}; /* we failed to read anything */
    }
}


/* -------------- Input packet processing ----------------------- */


u_char data_buf[MAX_IN_SIZE]; /* for holding current packet */
int data_buf_p;
int accept_count = 0; 
int reject_count = 0;

int
process_input(void)
{
  static int state = SYNC1_CHAR;
  static int count, pc;
  int c;
  while (check_in_buf(0) > 0) { /* have a byte */

    c = get_buf_byte();

    pc = accept_count;
    switch(state) {

    case SYNC1_CHAR:		/* waiting for packet header */
      if (c == SYNC1_CHAR)	/* yes, go on */
	state = SYNC2_CHAR;
      break;
      
    case SYNC2_CHAR:
      if (c == SYNC2_CHAR) { 	/* yes, go on */
	state = COUNT_STATE;
      }
      else state = SYNC1_CHAR;	/* failed, go back to waiting for header */
      break;

    case COUNT_STATE:
      count = c;
      data_buf_p = 1;
      data_buf[0] = c;
      state = DATA_STATE;		/* read in the data */
      break;

    case DATA_STATE:
      if (count-- <= 0) {		/* check CRC */
	if (c == calc_crc(data_buf)) /* got it ok */
	  accept_count++;
	else 
	  reject_count++;
	state = SYNC1_CHAR;	/* back to the beginning */
      }
      else {
	data_buf[data_buf_p++] = c; /* read in data */
      }
      break;
    }
    if (accept_count > pc) 
      {
	data_buf_p = 1;
	return(0);
      }
  }
  return(-1);
}

int
get_data_byte()			/* return the next data byte from buffer */
{
  return(data_buf[data_buf_p++]);
}

void
echo_packet()			/* send the packet back */
{
  int i = data_buf[0];
  u_char *p;
  p = &data_buf[1];
  while (i--) put_buf_byte(*(p++));
  output_packet();
}


/* ------------------------- Communication channels ----------------- */

input_comm_line(fd)		/* set up comm fd's */
int fd;
{
  if (fd < 0) return(comm_in_line);
  else
    { comm_in_line = fd;
      in_buf_p = 0;
      in_buf_max = 0;	
    }
}

output_comm_line(fd)		/* set up comm fd's */
int fd;
{
  if (fd < 0) return(comm_in_line);
  else
    { comm_out_line = fd; 
      out_buf_p = OUT_BUF_BEG;
    }
}


/* -------------- Check a file for read available ---------------- */

listen_in_buf(fd,wait)		/* returns 0 if no input waiting, 1 if yes */
				/* wait is time in ms to wait */
{ int f;
  struct timeval tp;		/* time interval structure for timeout */
  fd_set fdset;			/* fd set ??? */
  
  tp.tv_sec = 0;		/* we're polling */
  tp.tv_usec = 0;
  if (wait > 1000) tp.tv_sec = (wait / 1000);
  else tp.tv_usec = wait * 1000;

  FD_ZERO(&fdset);
  FD_SET(fd,&fdset);
  f = fd + 1;
  f = select(f,&fdset,NULL,NULL,&tp) ;
  return(f);
}

