/* 
 * eval-rules.c -- evaluate fuzzy control rules
 * 
 */

#include <stdio.h>

#include "constants.h"
#include "structures.h"
#include "parameters.h"


float c_goal, c_progress, c_active;

c_ret *
eval_rules(beh_rule *rules, int n, beh_params params)
				/* assumes that the parameters have been updated */
{
  float accv = 0.0, turnv = 0.0;
  int i;
  float ant, cv;
  int cp;
  float totacc = 0.0, maxacc = 0.0, totturn = 0.0, maxturn = 0.0;
  c_ret ret = {0.0, 0.0, 0.0, 0.0};
  for (i=0; i<n; i++)
    {
      ant = params[rules->antecedent].f; /* get antecedent value */
      cv  = 0.0;
      if (ant >= 0.0)
	{
	  cv = params[rules->parameter].f;
	  cp = rules->consequent;
	  if (cv < 0.0) cv = 0.0;
	  else cv = cv * ant;
	}
      switch(cp)
	{
	case ACCEL:
	  totacc += ant;
	  if (ant > maxacc) maxacc = ant;
	  accv += cv;
	  break;
	case DECEL:
	  totacc += ant;
	  if (ant > maxacc) maxacc = ant;
	  accv -= cv;
	  break;
	case TURN_RIGHT:
	  totturn += ant;
	  if (ant > maxturn) maxturn = ant;
	  turnv -= cv;
	  break;
	case TURN_LEFT:
	  totturn += ant;
	  if (ant > maxturn) maxturn = ant;
	  turnv += cv;
	  break;
	}
      rules++;
    }
  
  if (maxacc > 0.0) 
    {
      ret.accv = accv / totacc;
      ret.acc = maxacc;
    }

  if (maxturn > 0.0) 
    {
      ret.turnv = turnv / totturn;
      ret.turn = maxturn;
    }

  return(&ret);
}



void
eval_bcl(beh_closure *b)	/* evaluates one behavior closure */
{
  (b->beh->update_fn)(b->params); /* evaluate update function */
  b->vals = *eval_rules(b->beh->rules,b->beh->n,b->params);
  b->activity = c_active;
  b->goal = c_goal;
}


beh_closure *			/* sets up a behavior closure */
				/* usage: init_bcl(&bhvr,&beh_params,pri) */
init_bcl(behavior *b, beh_params p, int priority) 
{
  beh_closure *bcl = (beh_closure *)malloc(sizeof(beh_closure));
  bcl->priority = priority;
  bcl->beh = b;
  bcl->params = (b->setup_fn)(p); /* evaluate setup function */
  bcl->running = 1;
  return(bcl);
}


float exec_acc, exec_turn;	/* for reference */

execute(float acc, float turn)	/* execute these commands */
{
  exec_acc = acc; exec_turn = turn;
  if (ABS(acc) > 10.0) set_control(VEL,acc);
  if (turn > 0.0)
    { if (target_head() - turn > 6.0) turn = 0.0; }
  else 
    { if (target_head() - turn < -6.0) turn = 0.0; }
  if (ABS(turn) > 1.0) set_control(HEAD,turn*DEG_TO_RAD);
}


/*
 * Behavior list
 *
 * List of behavior closures, sorted by priority
 *
 */


#define MAX_BCS 20
beh_closure *behaviors[MAX_BCS]; 
int num_bcs = 0;		/* current number of behaviors */

void
add_behavior(beh_closure *b)	/* add this to the behavior list */
{
  int p = b->priority;
  int i, j;
  for (i=0; i<num_bcs; i++)	/* sort by priority */
    {
      if (behaviors[i]->priority > p) break;
    }
  for (j = num_bcs; j > i; j--)
    {
      behaviors[j] = behaviors[j-1];
    }
  behaviors[j] = b;
  num_bcs++;
}

void 
remove_behavior(beh_closure *b)	/* remove it from the list */
{
  int i,j;
  for (i=0; i<num_bcs; i++)
    { if (behaviors[i] == b) break; }
  for (; i<num_bcs-1; i++)
    { behaviors[i] = behaviors[i+1]; }
  num_bcs--;
}


void
execute_current_behaviors()	/* execute those on the list */
{
  int p = 0, pri = 0, i;
  float maxact = 0.0, act = 0.0, cxt = 0.0;
  float acc = 0.0, turn = 0.0;
  beh_closure *b;

  for (i=0; i < num_bcs; i++)
    {
      b = behaviors[i];
      if (b->running)
	{
	  eval_bcl(b);
	  if (p < b->priority)	/* change of priority */
	    {
	      maxact = act;
	      p = b->priority;
	    }
	  cxt = 1.0;		/* context, to be used later */
	  b->activity = MIN(cxt*b->activity, 1.0-maxact);
	  b->vals.turn  = MIN(b->activity,b->vals.turn);
	  b->vals.acc   = MIN(b->activity,b->vals.acc);
	  act = MAX(act,b->activity);
	}
    }

  for (i=0; i < num_bcs; i++)	/* defuzzification */
    {
      b = behaviors[i];
      if (b->running)
	{
	  acc += b->vals.acc * b->vals.accv;
	  turn += b->vals.turn * b->vals.turnv;
	}
    }
  
  execute(acc,turn);
}


/*
 * test function
 */


extern behavior constant_velocity;
extern behavior avoid_collision;
extern behavior go_to_pos;

void
test_control_proc(int reset)
{
  static int init = 1;
  static int cp = 0;
  static beh_closure *gp;
  beh_closure *p;
  beh_params par = beh_alloc(3);
  c_ret *ret;

  if (reset)
    {
      init = 1;
      cp = 0;
      return;
    }

  if (init)
    {
      par[0].f = 300.0;
      p = init_bcl(&constant_velocity, par, 2);
      add_behavior(p);
      par[0].f = 1.5; par[1].f = 0.8; par[2].f = 4.0;
      p = init_bcl(&avoid_collision, par, 0);
      add_behavior(p);

      init = 0;
      behavior_buttons();
    }
  else
    {
      if (cp == 0)
	{
	  if (pointlist_p > 0)	/* have a control point */
	    {
	      par[0].f = 300.0; /* speed */
	      par[1].p = pointlist[0]; /* control point */
	      par[2].f = 250.0;	/* success radius */
	      gp = init_bcl(&go_to_pos, par, 1);
	      add_behavior(gp);
	      behavior_buttons();
	      cp = 1;
	    }
	}
      else if (cp == 1)
	{
	  if (gp->goal > 0.9)	/* achieved it */
	    {
	      rem_point(pointlist[0]); /* pop previous point */
	      if (pointlist_p == 0)
		{
		  remove_behavior(gp);
		  cp = 0;
		  behavior_buttons();
		}
	      else
		{
		  (gp->params)[6].p = pointlist[0];
		}
	    }
	}
      execute_current_behaviors();
    }
}


