/* 
 * 
 * C functions to read and write the IO buffers
 *
 */

#include <stdio.h>
#include <sys/time.h>
#include <sys/types.h>
#include <sgtty.h>
#include <fcntl.h>

#define MAX_IN_SIZE 256
#define MAX_OUT_SIZE 256
#define MAX_REC_SIZE 256

/* #include "/homedir/flakey/src4/comm/protocol.h" */

int comm_in_line, comm_out_line, comm_rec_line; /* file descriptors */
int read_fault_flag = 0;	/* read faults */

char out_buf[MAX_OUT_SIZE];
char  in_buf[MAX_IN_SIZE];
char rec_buf[MAX_REC_SIZE];
char temp_buf[MAX_OUT_SIZE];
int in_buf_p, in_buf_max;	/* current pointer, max chars in buffer */
#define OUT_BUF_BEG 3		/* start of data */
int out_buf_p = OUT_BUF_BEG;
int rec_buf_p;

/*------------ output routines for Flakey commands --------------*/

/*------------------------ Output of Packets ------------------
 *
 * Format for packets:
 *
 *   SYNC1
 *   SYNC2
 *   data count
 *   data[0..data count]
 *   CRC  (addition of all data and count, restricted to one byte)
 *   END
 *
 */

#define SYNC1_CHAR   	0xfa	/* first sync byte of packet */
#define SYNC2_CHAR   	0xfb	/* second sync byte of packet */
#define END_CHAR        0xfe	/* byte at end of packet */
#define CRC_STATE       0x1	/* states of the input packet reader */
#define COUNT_STATE     0x2
#define DATA_STATE      0x3

void
put_buf_byte(n)			/* puts a byte, forces output if buffer full */
 
{ if (out_buf_p < MAX_OUT_SIZE) /* check overflow */
    out_buf[out_buf_p++] = n;
}


int
output_packet(void)		/* send out packet, including byte count */
{
  int s;
  out_buf[0] = SYNC1_CHAR;
  out_buf[1] = SYNC2_CHAR;
  out_buf[2] = out_buf_p - OUT_BUF_BEG;
  put_buf_byte(calc_crc(&out_buf[2]));
  put_buf_byte(END_CHAR);
  if (read_fault_flag == 0)
    s = write(comm_out_line,out_buf,out_buf_p);
/*  fflush(tty_f); */
  out_buf_p = OUT_BUF_BEG;
  return(s);
}

int
calc_crc(u_char *ptr)		/* ptr is array of bytes, first is data count */
{
  int n;
  int c = 0;
  n = *(ptr++);
  while (n-- > 0) {
    c += *(ptr++);
    c = c & 0xff;
  }
  return(c);
}
    

write_out_buf()			/* writes the buffer out */

{  write(comm_out_line,out_buf,out_buf_p);
   out_buf_p = 0;
 }

/*------------ output routines for recording -------------------- */

put_rec_byte(n)			/* puts a byte, forces output if buffer full */
 
{ if (rec_buf_p >= MAX_REC_SIZE) /* need to write here */
    { if (comm_rec_line > 0) write(comm_rec_line,rec_buf,MAX_REC_SIZE);
      rec_buf_p = 0;
    }
  
  rec_buf[rec_buf_p++] = n;
}

close_record_file()			/* writes the buffer out, closes it */

{  write(comm_rec_line,rec_buf,rec_buf_p);
   rec_buf_p = 0;
   close(comm_rec_line);
   comm_rec_line = 0;
 }


/*-------- read buffer routines for getting chars from the stream -------*/

int debug_stream = 0;

get_buf_byte()			/* returns one char from buffer, does */
				/* a READ if necessary */

{ int c;
  if (in_buf_p >= in_buf_max)	/* need to do a READ */
    { in_buf_max = read(comm_in_line,in_buf,MAX_IN_SIZE); 
      in_buf_p = 0;
      if (in_buf_max < 0) {perror("read fault"); 
			   in_buf_max = 0;
			   return(-1);}; /* we failed to read anything */
    }
  
  c = 0xff & in_buf[in_buf_p++];
  if (debug_stream) { printf(" %d ",c); fflush(stdout); }
  if (comm_rec_line > 0) put_rec_byte(c); /* need to save it */
  return(c);
}


clear_in_buf()			/* clear out input buffer */

{ in_buf_p = 0;
  in_buf_max = 0;
}

int
check_in_buf(int wait)		/* returns 0 if no input waiting, # chars if yes */
     
{ int f;
  struct timeval tp;		/* time interval structure for timeout */
  fd_set fdset;			/* fd set ??? */
  
  if ( in_buf_p < in_buf_max )	/* have some chars in the buffer */
    { return(in_buf_max - in_buf_p); }
  else
    { 
      tp.tv_sec = wait / 1000;	/* we're polling */
      tp.tv_usec = (wait % 1000) * 1000;

      FD_ZERO(&fdset);
      FD_SET(comm_in_line,&fdset);
      f = comm_in_line + 1;
      f = select(f,&fdset,NULL,NULL,&tp) ;
      if (f >= 1)		/* we have something, could be a refused connect */
	{ in_buf_max = read(comm_in_line,in_buf,MAX_IN_SIZE); 
	  in_buf_p = 0;
	  if (in_buf_max < 0) {perror("read fault"); 
			       read_fault_flag = 1;
			       in_buf_max = 0;
			       return(-1);} /* we failed to read anything */
	  else
	    { return(in_buf_max - in_buf_p); }
	}
      else return(0);
     }
}


wait_in_buf()			/* wait until there's input */
{ if (in_buf_p == in_buf_max)	/* need to do a READ */
    { in_buf_max = read(comm_in_line,in_buf,MAX_IN_SIZE); 
      in_buf_p = 0;
      if (in_buf_max < 0) {in_buf_max = 0;}; /* we failed to read anything */
    }
}

write_in_buf_to_file(fd,n)	/* write out N chars from the in buffer to FD */
int n,fd;			
{ in_buf[in_buf_p + n] = 0;	/* zero-terminate this line */
  return(write(fd,in_buf+in_buf_p,n+1)); 
}

write_array_to_file(fd,n,arr)	/* write out N chars from the array to FD */
int n,fd;
int arr[];
{ int i;
  char *op;
  op = temp_buf;
  for (i = 0; i < n; i++)
    *op++ = arr[i];
  *op++ = 0;	/* zero-terminate this line */
  return(write(fd,temp_buf,n+1)); 
}


write_in_buf_to_array(arr,n,rev)	/* write out N chars from in buffer to LISP arr */
int n, rev;			/* rev is 0 for normal order, 1 for reverse */
int arr[];
{ int i,j;
  if (rev)
    { i = in_buf_p;
      for (j=n-1; j >= 0; j--)
	{ arr[j] = 0xff & in_buf[i++]; }
    }
  else
    { i = in_buf_p;
      for (j=0; j < n; j++)
	{ arr[j] = 0xff & in_buf[i++]; }
    }
}


/* -------------- Input packet processing ----------------------- */


u_char data_buf[MAX_IN_SIZE]; /* for holding current packet */
int data_buf_p;
int accept_count = 0; 
int reject_count = 0;

int
process_input(void)
{
  static int state = SYNC1_CHAR;
  static int count, pc;
  int c;
  while (check_in_buf(0) > 0) { /* have a byte */

    c = get_buf_byte();

    pc = accept_count;
    switch(state) {

    case SYNC1_CHAR:		/* waiting for packet header */
      if (c == SYNC1_CHAR)	/* yes, go on */
	state = SYNC2_CHAR;
      break;
      
    case SYNC2_CHAR:
      if (c == SYNC2_CHAR) { 	/* yes, go on */
	state = COUNT_STATE;
      }
      else state = SYNC1_CHAR;	/* failed, go back to waiting for header */
      break;

    case COUNT_STATE:
      count = c;
      data_buf_p = 1;
      data_buf[0] = c;
      state = DATA_STATE;		/* read in the data */
      break;

    case DATA_STATE:
      if (count-- <= 0) {		/* check CRC */
	if (c == calc_crc(data_buf)) /* got it ok */
	  accept_count++;
	else 
	  reject_count++;
	state = SYNC1_CHAR;	/* back to the beginning */
      }
      else {
	data_buf[data_buf_p++] = c; /* read in data */
      }
      break;
    }
    if (accept_count > pc) 
      {
	data_buf_p = 1;
	return(0);
      }
  }
  return(-1);
}

int
get_data_byte()			/* return the next data byte from buffer */
{
  return(data_buf[data_buf_p++]);
}

void
echo_packet()			/* send the packet back */
{
  int i = data_buf[0];
  u_char *p;
  p = &data_buf[1];
  while (i--) put_buf_byte(*(p++));
  output_packet();
}


/* ------------------------- Communication channels ----------------- */



input_comm_line(fd)		/* set up comm fd's */
int fd;
{
  if (fd < 0) return(comm_in_line);
  else
    { 
      read_fault_flag = 0;
      comm_in_line = fd;
      in_buf_p = 0;
      in_buf_max = 0;	
    }
}

output_comm_line(fd)		/* set up comm fd's */
int fd;
{
  if (fd < 0) return(comm_in_line);
  else
    { comm_out_line = fd; 
      out_buf_p = OUT_BUF_BEG;
    }
}

record_comm_line(fd)		/* set up comm fd's */
int fd;
{
  if (fd < 0) return(comm_rec_line);
  else
    { comm_rec_line = fd; 
      rec_buf_p = 0;
    }
}


/* -------------- Check a file for read available ---------------- */

listen_in_buf(fd,wait)		/* returns 0 if no input waiting, 1 if yes */
				/* wait is time in ms to wait */
{ int f;
  struct timeval tp;		/* time interval structure for timeout */
  fd_set fdset;			/* fd set ??? */
  
  tp.tv_sec = 0;		/* we're polling */
  tp.tv_usec = 0;
  if (wait > 1000) tp.tv_sec = (wait / 1000);
  else tp.tv_usec = wait * 1000;

  FD_ZERO(&fdset);
  FD_SET(fd,&fdset);
  f = fd + 1;
  f = select(f,&fdset,NULL,NULL,&tp) ;
  return(f);
}




/* -------------- Decompress a depth packet ---------------- 
 *
 *  The compression strategy is in ~flakey/src/comm/prot-vision.c
 *
 */

decompress_in_buf_to_array(arr,n)	
     /* Decompress N chars from in buffer to LISP arr;
      * Does it in reverse order of ARR(!).
      * FD /= 0 if we are writing to file FD
      * Always skips over inbuf bytes.
      */

int n;
int arr[];
{ 
  int *dp;
  unsigned char *ip;
  int cur;
  int diff;
  int save, j, k;
  int cc;			

  ip = in_buf + in_buf_p;
  dp = arr + n - 1;
  cur = get_buf_byte();
  *dp-- = cur;
  save = 0;
  j = 0;

  while (--n)			/* decompression loop */
    {
      while (j < 16)		/* fill the save buffer */
	{ save = (save << 8) | get_buf_byte();
	  j += 8;
	}
      j -= 2;
      k = (save >> j) & 0x3;

/*      printf("s: %x k: %d c: %d ",save,k,cur); */

      if (k != 0x2)		/* small change here */
	{ if (k == 0x3) cur -= 1;
	  else cur += k; }
      else
	{
	  j -= 2;
	  k = (save >> j) & 0x3;
	  if (k != 0x3)		/* mod change here */
	    {
	      if (k == 0) { cur = 0xff; }
	      else
		{
		  j -= 2;
		  if (k == 0x1)	/* mod positive */
		    { cur += ((save >> j) & 0x3) + 2; }
		  else		/* mod negative */
		    { cur += ((save >> j) & 0x3) - 5; }
		}
	    }
	  else			/* big change here */
	    {
	      j -= 8;
	      cur = (save >> j) & 0xff;
	    }
	}
      *dp-- = cur;		/* save that puppy */
    }

}


