#include "xpnet.h"
#include <ctype.h>


/* checks if the formulas represented by nodes node1 & node2 are
   inverses.  returns 1 if they are, 0 if they aren't */
int
  inverses (node1, node2)
node *node1, *node2;
{
  if (node1->type == tensor && node2->type == par ||
      node2->type == tensor && node1->type == par)
    /* we need to check if the corresponding subformulea are inverses */
    return inverses (get_parent (node1, 0), get_parent (node2, 0))
      && inverses (get_parent (node1, 1), get_parent (node2, 1));
  else if (node1->type == axiom || node2->type == axiom)
    /* both formulea must be literals; see if they are inverses */
    return inv_literal (node1->name, node2->name);
  else if (node1->type == pax || node2->type == pax)
    /* use the strings indicating the name to check if they are inverses */
    return str_inverse (node1->name, 0, strlen (node1->name) - 1,
			node2->name, 0, strlen (node2->name) - 1);
  else return 0;
}

/* compares 2 strings, returns 1 if they are equal up to the length of the
   shorter string, 0 ow */
str_eq (s1, s2)
char *s1, *s2;
{
  while (*s1 == *s2 && *s1 != '\0')
    {
      ++s1;
      ++s2;
    }
  if (*s1 == '\0' || *s2 == '\0')
    return 1;
  else
    return 0;
}

/* increment index_p until it is no longer the index of a whitespace char */
skip_whitespace (str, index_p)
     char *str;
     int *index_p;
{
  int index = *index_p;

  while (isspace (str[index]))
    ++index;
  *index_p = index;
}

/* skip words (a word is a literal or connective, a bunch of chars 
   containing anything but parentheses, whitespace, or null char */
skip_word (str, index_p)
char *str;
int *index_p;
{
  int index = *index_p;

  /* a word is terminated by whitespace, null, or paren */
  while (!isspace (str[index]) && str[index] != '\0' &&
	 str[index] != ')' && str[index] != '(')
    ++index;
  *index_p = index;
}

skip_formula (str, index_p)
char *str;
int *index_p;
{
  int index, num_parens;

  if (str[*index_p] == '(')
    {
      /* we need to skip stuff, counting parentheses, until we find the
	 matching paren */
      index = *index_p + 1;
      num_parens = 1;
      while (num_parens > 0)
	{
	  /* correct the number of parens found */
	  if (str[index] == '(')
	    ++num_parens;
	  else if (str[index] == ')')
	    --num_parens;
	  ++index;
	}
      *index_p = index;
    }
  else
    /* skip the literal */
    skip_word (str, index_p);
}	  

int
inv_literal (s1, s2)
char *s1, *s2;
{
  while (*s1 == *s2)
    {
      ++s1;
      ++s2;
    }
  /* the strings are inverses of one another if one ends in perp_str
     while the other just ends */
  return str_eq (s1, perp_str) && (isspace (*s2) || *s2 == ')' || *s2 == '\0')
    || str_eq (s2, perp_str) && (isspace (*s1) || *s1 == ')' || *s1 == '\0');
}

/* basically, strips off surrounding (), if any, and calls appropriate
   functions to see if to see if resulting formulae are inverses */
int
  subfmla_inv (str1, str1begin, str1end, str2, str2begin, str2end)
char *str1, *str2;
int str1begin, str1end, str2begin, str2end;
{
  if (str1[str1begin] == '(' && str2[str2begin] == '(')
    {
      /* both subformulae are more than just literals. Add 1 to begin &
	 subtract 1 from end to eliminate the parentheses */
      return str_inverse (str1, str1begin + 1, str1end - 1,
			  str2, str2begin + 1, str2end - 1);
    }
  else if (str1[str1begin] == '(' || str2[str2begin] == '(')
    /* one subformula is more than just a literal & the other isn't */
    return 0;
  else
    /* both subformulea are literals */
    return inv_literal (str1 + str1begin, str2 + str2begin);
}

int
  str_inverse (str1, str1begin, str1end, str2, str2begin, str2end)
char *str1, *str2;
int str1begin, str1end, str2begin, str2end;
{
  int i, j, fmla1begin, fmla2begin, fmla1end, fmla2end;

  /* see if the two formulea are just literals */
  i = str1begin;
  j = str2begin;
  skip_whitespace (str1, &i);
  skip_whitespace (str2, &j);
  fmla1begin = i;
  fmla2begin = j;
  skip_formula (str1, &i);
  skip_formula (str2, &j);
  fmla1end = i - 1;
  fmla2end = j - 1;
  skip_whitespace (str1, &i);
  skip_whitespace (str2, &j);

  if (i == str1end && j == str2end)
    /* neither has subformulas, so they are literals, check if inverses */
    return inv_literal (str1 + fmla1begin, str2 + fmla2begin);
  else if (i == str1end || j == str2end)
    /* one fmla is just a literal, but the other isn't */
    return 0;

  /* if we get here we know that both formulae have subformulae -- check
     to see that one connective is par and the other tensor */
  if (!(str_eq (str1 + i, tensor_str) && str_eq (str2 + j, par_str) ||
	str_eq (str1 + i, par_str) && str_eq (str2 + j, tensor_str)))
    return 0;

  /* if the two subformulae don't match, then the formulae don't match */
  if (!subfmla_inv (str1, fmla1begin, fmla1end, str2, fmla2begin, fmla2end))
    return 0;

  /* if we get here, we know first subformulae match -- now check 2nd ones */
  skip_word (str1, &i);   /* skip connective in both formulae */
  skip_word (str2, &j);
  skip_whitespace (str1, &i);
  skip_whitespace (str2, &j);
  fmla1begin = i;
  fmla2begin = j;
  skip_formula (str1, &i);
  skip_formula (str2, &j);
  fmla1end = i - 1;
  fmla2end = j - 1;

  /* if the second subformulae match, then the formulae match */
  return subfmla_inv (str1, fmla1begin, fmla1end, str2, fmla2begin, fmla2end);
}
  


#if 0

typedef struct
{
  int begin, end;	        /* begin & end indecies of subfmla in str */
  int num_literals;             /* number of literals in subformula */
  int used;                     /* shows if this subfmla was used, that is
				   matched against another subfmla */
} subfmla;

int
  count_subfmlas (str, begin, end, type_str_p, num_fmlas_p)
char *str, **type_str_p;
int *num_fmlas_p, begin, end;
{
  char *type_str;
  int num_fmlas, i, type_len;
  
  /* skip the first formula & following whitespace */
  i = begin;
  skip_formula (str, &i);
  if (i > end)
    {
      /* there's only one subformula in here */
      *num_fmlas_p = 1;
      return 1;
    }
  skip_whitespace (str, &i);
  /* find the connective */
  if (str_eq (str + i, tensor_str))
      type_str = tensor_str;
  else if (str_eq (str + i, par_str))
      type_str = par_str;
  else
    return 0;
  type_len = strlen (type_str);
  i += type_len;

  /* skip second formula and surrounding whitespace */
  skip_whitespace (str, &i);
  skip_formula (str, &i);
  skip_whitespace (str, &i);

  /* now count the rest of the formulas in str */
  num_fmlas = 2;      /* found 2 subformulas so far */
  while (i <= end)
    {
      /* check to make sure the connective is the one we need it to be */
      if (str_eq (str + i, type_str))
	return 0;
      i += type_len;
      /* now skip the formula and surrounding whitespace */
      skip_whitespace (str, &i);
      skip_formula (str, &i);
      ++num_fmlas;
      skip_whitespace (str, &i);
    }

  /* now assign number of formulas results */
  *type_str_p = type_str;
  *num_fmlas_p = num_fmlas;
  /* everything was OK */
  return 1;
}

get_subfmla_info (str, begin, num_subfmlas, info_p)
char *str;
int begin, num_subfmlas;
subfmla **info_p;
{
  subfmla *info;
  int i, index, num_parens;

  info = (subfmla *) malloc (num_subfmlas * sizeof (subfmla));
  index = begin;
  for (i = 0; i < num_subfmlas; ++i)
    {
      /* get info about one subfmla of str */
      info[i].used = 0;
      if (str[index] == '(')
	{
	  /* we need to find the end of the subformula (by keeping track
	     of parentheses) and at the same time count number of literals */
	  ++index;
	  /* set begin and end so they do not include the () arond fmla */
	  info[i].begin = index;
	  num_parens = 1;
	  info[i].num_literals = 0;
	  while (num_parens > 0)
	    {
	      /* correct the number of parentheses found */
	      if (str[index] == '(' || str[index] == ')')
		{
		  num_parens += (str[index] == '(') ? 1 : -1;
		  ++index;
		}
	      else if (isspace (str[index]))
		skip_whitespace (str, &index);
	      else /* it is a literal or connective */
		{
		  skip_word (str, &index);
		  /* count literals and connectives as literals for now */
		  ++(info[i].num_literals);
		}
	    }
	  /* index is now one char beyond last paren in subfmla, set end
	     to char before last paren in subfmla */
	  info[i].end = index - 2;
	  /* now correct the number of literals */
	  info[i].num_literals = info[i].num_literals / 2 + 1;
	}
      else
	{
	  /* there's only one literal */
	  info[i].begin = index;
	  info[i].num_literals = 1;
	  skip_word (str, &index);
	  info[i].end = index - 1;
	}
      /* get ready to go to next subformula by skipping over the whitespace
	 and connective */
      skip_whitespace (str, &index);
      skip_word (str, &index);
      skip_whitespace (str, &index);
    }
  /* assign return value */
  *info_p = info;  
}

int
  str_inverse (str1, str1begin, str1end, str2, str2begin, str2end)
char *str1, *str2;
int str1begin, str1end, str2begin, str2end;
{
  int i, j, num_subfmlas1, num_subfmlas2, found;
  char *type1, *type2;
  subfmla *info1, *info2;

  /* get connective and subformula count for both strings */
  if (count_subfmlas (str1, str1begin, str1end, &type1, &num_subfmlas1) == 0)
    return 0;
  if (count_subfmlas (str2, str2begin, str2end, &type2, &num_subfmlas2) == 0)
    return 0;

  /* each should have the same number of subformulas */
  if (num_subfmlas1 != num_subfmlas2)
    return 0;

  /* base case */
  if (num_subfmlas1 == 1)
    {
      /* see if the literals are inverses of one another */
      if (str1end - str1begin < str2end - str2begin)
	{
	  /* str1 is shorter, see if str2 is str1 with perp_str at the end */
	  if (strncmp (str1 + str1begin, str2 + str2begin,
		       str1end - str1begin + 1) == 0 &&
	      strncmp (str2 + str2begin + str1end - str1begin + 1,
		       perp_str, strlen (perp_str)) == 0)
	    return 1;
	  else
	    return 0;
	}
      else
	{
	  /* str2 is shorter, see if str1 is str2 with perp_str at the end */
	  if (strncmp (str2 + str2begin, str1 + str1begin,
		       str2end - str2begin + 1) == 0 &&
	      strncmp (str1 + str1begin + str2end - str2begin + 1,
		       perp_str, strlen (perp_str)) == 0)
	    return 1;
	  else
	    return 0;
	}
    }

  /* inverses have to have the opposite connective (each connective
     can only be a tensor or a par, so strcmp is sufficient) */
  if (strcmp (type1, type2) == 0)
    return 0;

  /* get info on subformulas of strings and put in arrays */
  get_subfmla_info (str1, str1begin, num_subfmlas1, &info1);
  get_subfmla_info (str2, str2begin, num_subfmlas2, &info2);

  /* now try to match up the subformulas */
  for (j = 0; j < num_subfmlas1; ++j)
    {
      /* try to match subfmla j in str1 to some subfmla in str2 */
      found = 0;
      for (i = 0; i < num_subfmlas2 && !found; ++i)
	{
	  /* see if it matches against subfmla i in str2 */
	  if (info1[j].num_literals == info2[i].num_literals &&
	      !info2[i].used &&
	      str_inverse (str1, info1[j].begin, info1[j].end,
			   str2, info2[i].begin, info2[i].end))
	      found = info2[i].used = 1;
	}
      if (!found)
	return 0;
    }
  /* free stuff */
  free ((char *) info1);
  free ((char *) info2);
  /* we've matched (as inverses) all subfmlas in str1 against one in str2,
     so the strings are inverses */
  return 1;
}

#endif
