#include "xpnet.h"

extern char *strstr ();
/* max length of pax formula in chars */

static node *tail = (node *) 0;
static node **copy_array;
     
node *
     make_new_node (node_type)
op_type node_type;
{
  node *new;
  
  new = (node *) malloc (sizeof (node));
  new->type = node_type;
  new->next = (node *) 0;
  new->x = new->y = UNINITIALIZED;
  new->mark = 0;
  new->name = (char *) 0;
  switch (node_type)
    {
    case axiom:
      new->op.axiom.sibling = (node *) 0;
      new->op.axiom.bar_pos = UNINITIALIZED;
      new->op.axiom.child = (node *) 0;
      break;
    case pax:
      new->op.pax.num_siblings = UNINITIALIZED;
      new->op.pax.child = (node *) 0;
      break;
    case tensor:
      new->op.tensor.parents[0] = new->op.tensor.parents[1] =
	new->op.tensor.child = (node *) 0;
      break;
    case par:
      new->op.par.parents[0] = new->op.par.parents[1] =
	new->op.par.child = (node *) 0;
      break;
    case cut:
      new->op.cut.parents[0] = new->op.cut.parents[0] = (node *) 0;
      break;
    }
  return new;
}

/* assumes we already know that this node is in the list */
delete_node (node_to_delete)
     node *node_to_delete;
{
  node *temp;

  if (node_to_delete == head)
    {
      if (node_to_delete == tail)
	tail = (node *) 0;
      temp = head;
      head = head->next;
      free_node (temp);
    }
  else
    {
      /* increment temp until the next node after temp is the one we want
	 to delete */
      for (temp = head; temp->next != node_to_delete; temp = temp->next);
      if (node_to_delete == tail)
	tail = temp;
      temp->next = node_to_delete->next;
      free_node (node_to_delete);
    }
  /* we have one fewer node in the graph */
  --num_nodes;
}

add_node (cnode, head_p, tail_p)
     node *cnode, **head_p, **tail_p;
{
  /* add to net 1 */
  if (*head_p == (node *) 0)
    *head_p = cnode;
  else
    (*tail_p)->next = cnode;
  *tail_p = cnode;
}

add_to_graph (new)
     node *new;
{
  /* add new nodes to list */
  add_node (new, &head, &tail);
  /* fix other pointers and number of nodes */
  switch (new->type)
    {
    case axiom:
      /* axiom is really two nodes added */
      num_nodes += 2;
      tail = new->next;
      break;
    case pax:
      /* pax is many nodes added; increment tail 'til it's the last in list */
      for ( ; tail->next != (node *) 0; tail = tail->next);
      num_nodes += new->op.pax.num_siblings + 1;
      break;
    case tensor:
    case par:
    case cut:
      ++num_nodes;
      /* fix the pointers of the parent */
      assign_child (get_parent (new, 0), new);
      assign_child (get_parent (new, 1), new);
      break;
    }
}

free_node (cnode)
     node *cnode;
{
  if (cnode->name != (char *) 0)
    free (cnode->name);
  if (cnode->type == pax)
    free ((char *) cnode->op.pax.siblings);
  free ((char *) cnode);
}

free_graph ()
{
  node *temp;
  
  while (head != (node *) 0)
    {
      temp = head->next;
      free_node (head);
      head = temp;
    }
  head = (node *) 0;
  num_nodes = 0;
}

node *
  get_selected_node (x, y)
int x, y;
{
  node *cnode;
  int found = 0;
  
  for (cnode = head; cnode != (node *) 0 && !found; )
    {
      if (x >= cnode->x && x < cnode->x + cnode->width &&
	  y >= cnode->y && y < cnode->y + font_height)
	found = 1;
      if (!found)
	cnode = cnode->next;
    }
  return cnode;
}

put_parent (child, parent, which)
     node *child, *parent;
     int which;
{
  if (child->type == tensor)
    child->op.tensor.parents[which] = parent;
  else if (child->type == par)
    child->op.par.parents[which] = parent;
  else if (child->type == cut)
    child->op.cut.parents[which] = parent;
}

assign_child (parent, child)
     node *parent, *child;
{
  switch (parent->type)
    {
    case axiom:
      parent->op.axiom.child = child;
      break;
    case pax:
      parent->op.pax.child = child;
      break;
    case tensor:
      parent->op.tensor.child = child;
      break;
    case par:
      parent->op.par.child = child;
      break;
    }
}

node *
  get_parent (child, which)
node *child;
int which;
{
  if (child->type == tensor)
    return child->op.tensor.parents[which];
  else if (child->type == par)
    return child->op.par.parents[which];
  else if (child->type == cut)
    return child->op.cut.parents[which];
  else 
    return (node *) 0;
}

node *
  get_child (parent)
node *parent;
{
  if (parent->type == axiom)
    return parent->op.axiom.child;
  else if (parent->type == pax)
    return parent->op.pax.child;
  else if (parent->type == tensor)
    return parent->op.tensor.child;
  else if (parent->type == par)
    return parent->op.par.child;
  else 
    return (node *) 0;
}

int
get_bar_pos (gnode)
node *gnode;
{
  if (gnode->type == axiom)
    return gnode->op.axiom.bar_pos;
  else if (gnode->type == pax)
    return gnode->op.pax.bar_pos;
  else
    return 0;
}

add_to_locations (mark, x_diff, y_diff)
     int mark, x_diff, y_diff;
{
  node *cnode;
  
  for (cnode = head; cnode != (node *) 0; cnode = cnode->next)
    if (mark == ALL || cnode->mark == mark)
      {
	cnode->x += x_diff;
	cnode->y += y_diff;
	if (cnode->type == axiom)
	  cnode->op.axiom.bar_pos += y_diff;
	else if (cnode->type == pax)
	  cnode->op.pax.bar_pos += y_diff;
      }
}

/* returns 1 if the formula is just a literal; 0 ow */
int
is_literal (name)
char *name;
{
  int i;
  
  i = 0;
  skip_whitespace (name, &i);
  /* if the first char is a paren it has subformulae so it's not a literal */
  if (name[i] == '(')
    return 0;
  /* skip over the first literal and the whitespace after it */
  skip_word (name, &i);
  skip_whitespace (name, &i);
  /* it is just a literal if we are now at the end of the string */
  return (i == strlen (name));
}

/* return formula of node. parentheses will be added unless node is an
   axiom */
char *
  add_parens (cnode)
node *cnode;
{
  if (cnode->type == axiom || cnode->type == pax && is_literal (cnode->name))
    return copystring (cnode->name);
  else
    return catstrings ("(", cnode->name, ")", (char *) 0);
}

char *
get_connective (cnode)
node *cnode;
{
  switch (cnode->type)
    {
    case tensor:
      return tensor_str;
      break;
    case par:
      return par_str;
      break;
    case cut:
      return cut_str;
      break;
    }
}

char *
  get_label_displayed (cnode)
node *cnode;
{
  /* the text written below a cut is always the cut string */
  if (cnode->type == cut)
    return cut_str;
  /* for axioms and paxs we want to write out the formula at the node,
     and if the full_formula option is selected we want to write out
     the entire formula at the node */
  else if (cnode->type == axiom || cnode->type == pax || full_formula)
    return cnode->name;
  /* otherwise we just write out the connective at the node */
  else
    return get_connective (cnode);
}

char *
  create_node_name (cnode)
node *cnode;
{
  String parent0_name, parent1_name, name;
  
  if (cnode->type == axiom)
    return cnode->name;
  else
    {
      parent0_name = add_parens (get_parent (cnode, 0));
      parent1_name = add_parens (get_parent (cnode, 1));
      name = catstrings (parent0_name, " ", get_connective (cnode),
			 " ", parent1_name, (char *) 0);
      free (parent0_name);
      free (parent1_name);
      return name;
    }
}

number_nodes (graph)
     node *graph;
{
  node *cnode;
  int count;
  
  for (cnode = graph, count = 0; cnode != (node *) 0;
       cnode = cnode->next, ++count)
    cnode->mark = count;
}

reset_numbers (graph)
     node *graph;
{
  node *cnode;
  
  for (cnode = graph; cnode != (node *) 0; cnode = cnode->next)
    cnode->mark = 0;
}



write_graph_to_file (filename)
     char *filename;
{
  node *cnode;
  int count;
  FILE *file;
  char str[MAX_FORMULA];
  
  /* check if the file exists and open it */
  if ((file = fopen (filename, "w")) == (FILE *) 0)
    {
      cat_message ("Cannot open file \"", filename, "\".", (char *) 0);
      reset_operation();
      return;
    }
  /* write out the number of nodes */
  fprintf (file, "%d\n", num_nodes);
  /* number all the nodes */
  number_nodes (head);
  /* write out each node */
  for (cnode = head, count = 0; cnode != (node *) 0;
       cnode = cnode->next, ++count)
    {
      /* write out number */
      fprintf (file, "%d\t", count);
      /* write out type */
      if (cnode->type == axiom)
	fprintf (file, "axiom\t");
      else if (cnode->type == pax)
	fprintf (file, "pax\t");
      else if (cnode->type == tensor)
	fprintf (file, "tensor\t");
      else if (cnode->type == par)
	fprintf (file, "par\t");
      else if (cnode->type == cut)
	fprintf (file, "cut\t");
      /* write out position of middle of text */
      fprintf (file, "%d %d\t", cnode->x + cnode->width / 2, 
	       cnode->y + font_height / 2);
      /* write out specifics */
      if (cnode->type == axiom)
	fprintf (file, "%d\t%s", cnode->op.axiom.bar_pos,
		 unparse_formula (cnode->name, str));
      else if (cnode->type == pax)
	fprintf (file, "%d %d\t%s", cnode->op.pax.bar_pos,
		 cnode->op.pax.num_siblings,
		 unparse_formula (cnode->name, str));
      else
	/* print out info on parents */
	fprintf (file, "%d %d", (get_parent (cnode, 0))->mark,
		 (get_parent (cnode, 1))->mark);
      /* finish off the line */
      fprintf (file, "\n");
    }
  /* now set all the marks back to 0 */
  reset_numbers (head);
  /* close things down */
  fclose (file);
  reset_operation();
  set_label (widgets->message, "Done.");
}


read_graph_from_file (filename)
     char *filename;
{
  node **node_array, *cnode, *prev_axiom = (node *) 0;
  int count, total_count, temp, parent0, parent1;
  int num_siblings, pax_nodes = 0, i;
  FILE *file;
  char str_array[MAX_FORMULA], *ptr;
  op_type type;
  
  /* check if the file exists and open it */
  if ((file = fopen (filename, "r")) == (FILE *) 0)
    {
      cat_message ("Cannot open file \"", filename, "\".", (char *) 0);
      reset_operation();
      return;
    }
  
  /* prepare for the new graph */
  free_graph ();
  
  /* find out how many nodes there are */
  fscanf (file, "%d", &total_count);
  /* allocate an array of pointers to point to all these critters
     (this array makes finding parents alot easier) */
  node_array  = (node **) malloc (total_count * sizeof (node *));
  
  for (count = 0; count < total_count; ++count)
    {
      /* read the node number & make sure we haven't screwed up */
      fscanf (file, "%d", &temp);
      if (temp != count)
	{
	  set_label (widgets->message, "Net file has incorrect numbering.");
	  free_graph ();
	  reset_operation();
	  return;
	}
      /* get type of node */
      fscanf (file, "%s", str_array);
      if (strcmp (str_array, "axiom") == 0)
	type = axiom;
      else if (strcmp (str_array, "pax") == 0)
	type = pax;
      else if (strcmp (str_array, "tensor") == 0)
	type = tensor;
      else if (strcmp (str_array, "par") == 0)
	type = par;
      else if (strcmp (str_array, "cut") == 0)
	type = cut;
      else
	{
	  cat_message ("Unknown node type \"", str_array, "\"in net file.",
		       (char *) 0);
	  free_graph ();
	  reset_operation(); 
	  return;
	}
      /* allocate the node */
      cnode = make_new_node (type);
      /* get postition of middle of text of object */
      fscanf (file, "%hd %hd", &cnode->x, &cnode->y);
      /* assign specifics for each node & add to graph */
      if (type == axiom)
	{
	  /* read bar position */
	  fscanf (file, "%hd", &cnode->op.axiom.bar_pos);
	  /* get name */
	  fscanf (file, "%s", str_array);
	  /* replace the "~" in the file, if it is there, with perp_str */
	  if ((ptr = strstr(str_array, "~")) != (char *) 0)
	    {
	      str_array[ptr - str_array] = '\0';
	      strcat (str_array, perp_str);
	    }
	  cnode->name = copystring (str_array);
	  /* compute width */
	  cnode->width = get_label_width (cnode);
	  /* deal with the problem that axiom nodes come in pairs */
	  if (prev_axiom == (node *) 0)
	    prev_axiom = cnode;
	  else
	    {
	      prev_axiom->next = cnode;
	      prev_axiom->op.axiom.sibling = cnode;
	      cnode->op.axiom.sibling = prev_axiom;
	      add_to_graph (prev_axiom);
	      prev_axiom = (node *) 0;
	    }
	}
      else if (type == pax)
	{
	  /* read bar position */
	  fscanf (file, "%hd", &cnode->op.pax.bar_pos);
	  /* get number of siblings */
	  fscanf (file, "%d", &num_siblings);
	  /* get name */
	  fgets (str_array, MAX_FORMULA, file);
	  /* parse the formula -- replace the perp_str, par_str, tensor_str
	     with "~", "par", "tensor" */
	  (void) parse_formula (str_array);
	  /* copy it over */
	  cnode->name = copystring (str_array);
	  /* compute width */
	  cnode->width = get_label_width (cnode);
	  /* deal with the problem that pax nodes come in bunches */
	  ++pax_nodes;
	  /* fix up next pointer of previous node */
	  if (pax_nodes > 1)
	    node_array[count - 1]->next = cnode;
	  if (pax_nodes == num_siblings + 1)
	    {
	      /* this is last node in pax; have to assign sibling pointers */
	      assign_siblings (node_array[count - pax_nodes + 1], pax_nodes);
	      add_to_graph (node_array[count - pax_nodes + 1]);
	      pax_nodes = 0;
	    }
	}
      else
	{
	  /* get the indecies of parents & fix parent/child pointers */
	  fscanf (file, "%d %d", &parent0, &parent1);
	  put_parent (cnode, node_array[parent0], 0);
	  put_parent (cnode, node_array[parent1], 1);
	  /* fix name */
	  cnode->name = create_node_name (cnode);
	  /* compute the width */
	  cnode->width = get_label_width (cnode);
	  /* put in graph */
	  add_to_graph (cnode);
	}
      /* adjust the x and y so that they are the upper left corner of
	 the text, not the middle of the text */
      cnode->x = cnode->x - cnode->width / 2;
      cnode->y = cnode->y - font_height / 2;
      /* put this node in the array */
      node_array[count] = cnode;
    }
  
  /* clean up */
  fclose (file);
  reset_operation(); 
  free ((char *) node_array);
  clear_message ();
  redisplay_canvas ();
}

/* returns 1 if string entered is a real or integer number, ow returns 0 */
Digits(s)
     char *s;
{
  int decimalpt = 0;
  if ((*s == '+') || (*s == '-'))
    s++;
  while (((*s) != '\0') && (isdigit(*s) || (*s == '.')))
    {
      if (*s == '.')
	decimalpt++;
      s++;
    }
  return (((*s == '\0') || (*s == '\n')) && (decimalpt < 2));
}



get_factor_and_scale_graph (scale_str)
char *scale_str;
/* 
  By Hay
  Desc: 
  get_factor_and_scale_graph check for validity of the value of 
  the scale factor, scale_str, before calling on the the scale
  function with apropriate X and Y scale factor to do the scaling 
  of the proof net.

  scale_str must be a number, i.e. int or float value, > 0.0
  else error msg will be displayed in label widget.
*/

{
  char ErrMsg[100];
  float Ratio;

  if (!strcmp(scale_str, ""))
    set_label(widgets->message, "No scale factor given!");
  else
    if (Digits(scale_str))
      {
	sscanf(scale_str, "%f", &Ratio);
	if (Ratio <= 0.0)
	  set_label(widgets->message, "Scale factor must be > 0.0");
	else if (Ratio >= 50)
	  set_label(widgets->message, "Scale factor must be < 50.0");
	else if (current_operation == scale_x)
	  scale_graph(Ratio, 1.0);
	else if (current_operation == scale_y)
	  scale_graph(1.0, Ratio);
	else   /* is scale_u */
	  scale_graph(Ratio, Ratio);
      }
    else
      {
	strcat(ErrMsg, "Invalid scale factor: ");
	strcat(ErrMsg, scale_str);
	set_label(widgets->message, ErrMsg);
      }
  reset_operation();
}

scale_graph (XRatio, YRatio)
     float XRatio;
     float YRatio;
/*
  By Hay
  Desc:   
  scale_graph scale the graph, head, by XRatio in X, and YRatio in Y
  This function  will be called by get_factor_and_scale_graph()
*/

{
  node *cnode;
  
  cnode = head;
  while (cnode != (node *) 0)
    {
      if (cnode->type == axiom)
	cnode->op.axiom.bar_pos = (short) 
	  (YRatio * ((float)cnode->op.axiom.bar_pos) + 0.5);
      else if (cnode->type == pax)
	cnode->op.pax.bar_pos = (short) 
	  (YRatio * ((float)cnode->op.pax.bar_pos) + 0.5);
      cnode->y = (short)(YRatio * ((float)cnode->y) + 0.5);
      cnode->x = (short)(XRatio * ((float)cnode->x) + 0.5);
      cnode = cnode->next;
    }
  redisplay_canvas();
  set_label(widgets->message, "Thanks!");
}

/* This function makes a copy especially for checking graph. Thus it
   assumes that all the MARK fields are 0 and should be left as such
   after we are done, and it doesn't mess with the NAME, X, Y, or WIDTH
   fields. Also, leave pointers to the new graph in copy_array. */
node *
  copy_graph ()
{
  int count, node_num, pax_nodes = 0;
  node *old_node, *new_node;
  
  copy_array = (node **) malloc (num_nodes * sizeof (node *));
  
  number_nodes (head);
  
  /* copy nodes -- old node is node in old graph, new_node is in new graph*/
  for (old_node = head, count = 0; old_node != (node *) 0;
       old_node = old_node->next, ++count)
    {
      new_node = make_new_node (old_node->type);
      /* put this node in copy_array and update next pointer of last node */
      copy_array[count] = new_node;
      if (count > 0)
	copy_array[count - 1]->next = new_node;
      if (old_node->type == axiom)
	{
	  if ((node_num = old_node->op.axiom.sibling->mark) < count)
	    {
	      /* this is the second sibling, assign sibling values for both */
	      copy_array[node_num]->op.axiom.sibling = new_node;
	      new_node->op.axiom.sibling = copy_array[node_num];
	    }
	}
      else if (old_node->type == pax)
	{
	  ++pax_nodes;
	  if (pax_nodes == old_node->op.pax.num_siblings + 1)
	    {
	      /* this is last node in pax; have to assign sibling pointers */
	      assign_siblings (copy_array[count - pax_nodes + 1], pax_nodes);
	      pax_nodes = 0;
	    }
	}
      else
	{
	  /* fix parent/child pointers between this node and its parents */
	  node_num = (get_parent (old_node, 0))->mark;
	  assign_child (copy_array[node_num], new_node);
	  put_parent (new_node, copy_array[node_num], 0);
	  node_num = (get_parent (old_node, 1))->mark;
	  assign_child (copy_array[node_num], new_node);
	  put_parent (new_node, copy_array[node_num], 1);
	}
    }
  
  reset_numbers (head);
  
  /* return the copy */
  return *copy_array;
}

free_copy ()
{
  int i;
  
  /* in order to free the nodes we allocated in copy_graph, we use the
     array copy_array, which still holds pointers to all the nodes */
  for (i = 0; i < num_nodes; ++i)
    free_node (copy_array[i]);
  
  /* now we free the array */
  free ((char *) copy_array);
  copy_array = (node **) 0;
}

from_type
  get_from_parent (parent, child)
node *parent, *child;
{
  if (parent == (get_parent (child, 0)))
    return from_parent0;
  else
    return from_parent1;
}

/* Mark all the nodes we can get to from here with num. Return 1 if all
   was OK, and 0 if there is a loop somewhere. */
mark_connected_nodes (from, start, num)
     from_type from;
     node *start;
     int num;
{
  int check_parent0, check_parent1, check_child, check_siblings, i;
  node *child_node;
  
  /* if we've reached the end of the graph or we've already marked this
     node, we're OK */
  if (start == (node *) 0 || start->mark == num)
    return 1;
  
  /* there shouldn't be a loop, that is, we shouldn't go back to where
     we started from */
  if (start->mark == 3)
    return 0;
  
  /* otherwise mark all connected nodes */
  start->mark = num;
  check_parent0 = (start->type != axiom && from != from_parent0);
  check_parent1 = (start->type != axiom && from != from_parent1);
  check_child = (start->type != cut && from != from_child);
  check_siblings = ((start->type == axiom || start->type == pax) 
		    && from != from_sibling);
  if (check_parent0 &&
      mark_connected_nodes (from_child, get_parent (start, 0), num) == 0)
    return 0;
  if (check_parent1 &&
      mark_connected_nodes (from_child, get_parent (start, 1), num) == 0)
    return 0;
  if (check_child)
    {
      child_node = get_child (start);
      if (child_node != (node *) 0 && mark_connected_nodes
	  (get_from_parent (start, child_node), child_node, num) == 0)
	return 0;
    }
  if (check_siblings)
    {
      if (start->type == axiom &&
	  mark_connected_nodes (from_sibling,
				start->op.axiom.sibling, num) == 0)
	return 0;
      else if (start->type == pax)
	for (i = 0; i < start->op.pax.num_siblings; ++i)
	  if (mark_connected_nodes (from_sibling,
				    start->op.pax.siblings[i], num) == 0)
	    return 0;
    }
  
  /* if all tests have passed, the graph is OK */
  return 1;
}

split (net, cnode, net1_p, net2_p)
     node *net, *cnode, **net1_p, **net2_p;
{
  node *head1, *head2, *tail1, *tail2, *anode, *parent1, *parent2;
  
  /* mark the nodes you can get to from each parent */
  cnode->mark = 3;
  parent1 = get_parent (cnode, 0);
  parent2 = get_parent (cnode, 1);
  if (mark_connected_nodes (from_child, parent1, 1) == 0 ||
      mark_connected_nodes (from_child, parent2, 2) == 0)
    {
      /* there was a loop */
      reset_numbers (net);
      return 0;
    }
  for (anode = net; anode != (node *) 0; anode = anode->next)
    {
      if (anode->mark == 0)
	{
	  /* there were nodes not connected to either parent */
	  reset_numbers (net);
	  return 0;
	}
    }
  /* ok, now we have to turn the net into two separate nets, one with
     all the nodes marked 1, the other with all the nodes marked 2 */
  head1 = head2 = (node *) 0;
  for (anode = net; anode != (node *) 0; anode = anode->next)
    {
      if (anode->mark == 1)
	add_node (anode, &head1, &tail1);
      else if (anode->mark == 2)
	add_node (anode, &head2, &tail2);
      /* ignore it if the mark is 3 */
    }
  /* set the NEXT field of the tails to null to end the lists */
  if (tail1 != (node *) 0)
    tail1->next = (node *) 0;
  if (tail2 != (node *) 0)
    tail2->next = (node *) 0;
  /* eliminate node from graph by assigning its parents' kids to be nil */
  assign_child (parent1, (node *) 0);
  assign_child (parent2, (node *) 0);
  /* reset all the marks to be 0 again */
  reset_numbers (head1);
  reset_numbers (head2);
  /* assign return values */
  *net1_p = head1;
  *net2_p = head2;
  
  /* tell caller that we were able to split the graph */
  return 1;
}

/* removes a par without kids from the graph - yes, it loses all pointers
   in the graph to cnode, but these lost nodes will be garbage collected
   later in free_copy */
remove_par (prev, cnode)
     node *prev, *cnode;
{
  /* make the list skip over this node */
  prev->next = cnode->next;
  /* tell the parents that their child has gone away */
  assign_child (get_parent (cnode, 0), (node *) 0);
  assign_child (get_parent (cnode, 1), (node *) 0);
}

checkit (net)
     node *net;
{
  node *prev, *cnode, *net1, *net2;
  int found = 0, count;
  
  /* count the number of nodes in the net */
  for (cnode = net, count = 0; cnode != (node *) 0;
       ++count, cnode = cnode->next);

  /* check if our net has at least 2 nodes (for an axiom) */
  if (count < 2)
    return 0;

  /* check if our net is just an axiom or a proper axiom */
  if (net->type == axiom && count == 2 ||
      net->type == pax && count == net->op.pax.num_siblings + 1)
    return 1;
  
  prev = (node *) 0;
  for (cnode = net; cnode != (node *) 0; prev = cnode, cnode = cnode->next)
    {
      /* look for PARs with no children & remove them */
      if (cnode->type == par && cnode->op.par.child == (node *) 0)
	{
	  /* cnode could not be the first node since axioms are always
	     the first at least 2 nodes */
	  remove_par (prev, cnode);
	  found = 1;
	}
    }
  if (found)
    /* check out the rest of the incrementnet */
    return checkit (net);
  
  for (cnode = net; cnode != (node *) 0; cnode = cnode->next)
    {
      /* look for TENSORs with no kids and CUTs */
      if (cnode->type == tensor && cnode->op.tensor.child == (node *) 0 ||
	  cnode->type == cut)
	if (split (net, cnode, &net1, &net2))
	  /* if we can split the graph, check the leftover parts */
	  return (checkit (net1) && checkit (net2));
    }
  
  /* if the net is not just an axiom, there are no childless PARs, and we
     can't split on any of the TENSORs or CUTs, it is not a proof net */
  return 0;
}

/* check to see if the graph is a legitimate proof net, return 1 if so, 0 OW */
graph_is_proof_net ()
{
  node *new_net;
  int ok;
  
  if (head == (node *) 0)
    return 0;

  new_net = copy_graph ();
  ok = checkit (new_net);
  /* checkit is a majorly destructive procedure - after its execution,
     basically nothing is left of new_net */
  free_copy ();
  
  /* return the results! */
  return ok;
}


