#include "xpnet.h"
#include <string.h>

#define min(a,b) (((a) < (b)) ? (a) : (b))
#define max(a,b) (((a) > (b)) ? (a) : (b))


static node **copy_array;

/* local globals */
static int (*move_proc) (), (*button_press_proc) ();
static XFontStruct *font;
static int places_picked, cuts_done;
static int moved;
static node *curr_node;

/* used for moving stuff around interactively */
static node *cut_elm_node;
static node_pairs_list *cut_prs_head, *cut_prs_tail;
static node_pairs_list *ax_prs_head, *ax_prs_tail;
static node_pairs_list *del_prs_head, *del_prs_tail;
static nodelist *nodes_to_add_hd, *nodes_to_add_tl;

static int cp_num_nodes;
static node *cp_head, *cp_tail;
static node_pairs_list *cp_cut_prs_head, *cp_cut_prs_tail;
static node_pairs_list *cp_ax_prs_head, *cp_ax_prs_tail;
static node_pairs_list *cp_del_prs_head, *cp_del_prs_tail;

static char *text;
static int num_pax_nodes, max_pax_nodes = 0;
static int total_ax_prs, total_cut_prs, total_del_prs;
static int text_x, text_y, text_width, old_x, old_y;
static int x1, y1, x2, y_bound, offset;
static Display *dpy;
static Window canvas_win;
static GC gc;

/* When resetting operation, the main part is the following
           current_operation = none;
   We need to reverse-video the panel button that is just in operation.
   The following function does just that.  In addition, it also sets
           previous_operation = none;
   for axiom and pax.  This is because without that, axiom & pax -- when
   cancelled - reverses the panel button for "file" or "scale".
*/
void reset_operation()
{
  switch (current_operation)
    {
    case axiom:  
      previous_operation = none;
      reverse_colors(widgets->axiom);
      break;
    case pax:  
      previous_operation = none;
      reverse_colors(widgets->pax);
      break;
    case expand:
      reverse_colors(widgets->expand);
      break;
    case move:
      reverse_colors(widgets->move);
      break;
    case abort:
      reverse_colors(widgets->abort);
      break;
    case tensor:
      reverse_colors(widgets->tensor);
      break;
    case par:
      reverse_colors(widgets->par);
      break;
    case cut:
      reverse_colors(widgets->cut);
      break;
    case cut_elm:
      reverse_colors(widgets->cut_elm);
      break;
    case delete:
      reverse_colors(widgets->delete);
      break;
    case scale:
      reverse_colors(widgets->scale);
      break;
    case scale_x:
      reverse_colors(widgets->scale_x);
      break;
    case scale_y:
      reverse_colors(widgets->scale_y);
      break;
    case scale_u:
      reverse_colors(widgets->scale_u);
      break;
    case file:
      reverse_colors(widgets->file);
      break;
    case save:
      reverse_colors(widgets->save);
      break;
    case read:
      reverse_colors(widgets->read);
      break;
    case psprn:
      reverse_colors(widgets->psprn);
      break;
    case xpn2ps:
      reverse_colors(widgets->xpn2ps);
      break;
    case quit:
      /* This shouldn't be needed, since quitting is quitting... BUT...	 it 
	 might be useful later, especially if we want "QUIT" to be confirmed.*/
      reverse_colors(widgets->quit);
      break;
    }
  current_operation = none;
}

null_proc ()
{ }

realloc_segments ()
{
  int num_segments;

  if (segments != (XSegment *) 0)
    free ((char *) segments);
  num_segments = max (max_pax_nodes + 1, 3);
  segments = (XSegment *) malloc (num_segments * sizeof (XSegment));
}

/*  This sets the line in preparation for drawing boxes around the labels
    of the nodes.  this is because set_thicklines() sets it too thick for
    the box, it didn't look good.  To reset, use unset_thicklines().
*/
set_boxlines()
{
  XGCValues gcv;
  int gcmask;

  gcv.line_width = 2;
  gcmask = GCLineWidth;
  XChangeGC (dpy, gc, gcmask, &gcv);
}

/*  This sets the line in preparation for highlighting, as in case of
    doing a delete.  To reset, use unset_thicklines().
*/
set_thicklines()
{
  XGCValues gcv;
  int gcmask;

  gcv.line_width = 3;
  gcmask = GCLineWidth;
  XChangeGC (dpy, gc, gcmask, &gcv);
}

/* change the line style in back to thin lines for other line drawing */
unset_thicklines()
{
  XGCValues gcv;
  
  gcv.line_width = 0;
  XChangeGC (dpy, gc, GCLineWidth, &gcv);
}

/* change the line style in preparation for drawing PAR lines */
set_par()
{
  XGCValues gcv;
  int gcmask;

  gcv.line_style = LineOnOffDash;
  gcmask = GCLineStyle;
  XChangeGC (dpy, gc, gcmask, &gcv);
}

/* change the line style in back to solid lines for other line drawing */
unset_par()
{
  XGCValues gcv;
  
  gcv.line_style = LineSolid;
  XChangeGC (dpy, gc, GCLineStyle, &gcv);
}

/* draw string with UL corner of string at x, y */ 
draw_string (x, y, str)
     int x, y;
     String str;
{
  XSetForeground(dpy, gc, app_data.NodeColor);
  XDrawString (dpy, canvas_win, gc, x, y + font->ascent,
	       str, strlen (str));
  XSetForeground(dpy, gc, app_data.LinkColor);
}

int_switch (a_p, b_p)
     int *a_p, *b_p;
{
  int temp;
  
  temp = *a_p;
  *a_p = *b_p;
  *b_p = temp;
}

redisplay_canvas ()
{
  clear_canvas ();
  draw_graph (ALL);
}

put_vert_seg_info (x, y, width, index)
     int x, y, width;
{
  /* assign both x components to be the same */
  segments[index].x1 = segments[index].x2 = x + width / 2;
  /* assign y2 (the one further away from connecting bar) to be y */
  segments[index].y2 = y;
}

put_bar_pos_info (bar_pos, num_nodes)
int bar_pos, num_nodes;
{
  int i, xmin, xmax;

  /* fill in info at end of vert segments near bar */
  xmin = xmax = segments[0].x1;
  for (i = 0; i < num_nodes; ++i)
    {
      segments[i].y1 = bar_pos;
      xmin = min (segments[i].x1, xmin);
      xmax = max (segments[i].x1, xmax);
    }
  /* fill in horizontal segment info */
  segments[num_nodes].y1 = segments[num_nodes].y2 = bar_pos;
  segments[num_nodes].x1 = xmin;
  segments[num_nodes].x2 = xmax;
}

draw_graph (mark)
     int mark;
{
  int x, i, num_nodes, bar_pos;
  node *cnode, *parent, *nd;
  
  cnode = head;
  while (cnode != (node *) 0)
    {
      if (mark == ALL || cnode->mark == mark)
	{
	  /* display node and its connections */
	  switch (cnode->type)
	    {
	    case axiom:
	    case pax:
	      bar_pos = get_bar_pos (cnode);
	      num_nodes = (cnode->type == axiom) ? 2 :
		cnode->op.pax.num_siblings + 1;
	      /* fill in vertical segment info */
	      for (nd = cnode, i = 0; i < num_nodes; ++i, nd = nd->next)
		put_vert_seg_info (nd->x, nd->y, nd->width, i);
	      put_bar_pos_info (bar_pos, num_nodes);
	      /* draw formulas and connecting bar */
	      for (nd = cnode, i = 0; i < num_nodes; ++i, nd = nd->next)
		draw_string (nd->x, nd->y, nd->name);
	      draw_stuff (0, 1, num_nodes + 1);
	      /* skip to the last node in this bunch */
	      for (i = 1; i < num_nodes; ++i, cnode = cnode->next);
	      break;
	    case tensor:
	    case par:
	      /* fill in segment data */
	      parent = get_parent (cnode, 0);
	      segments[0].x1 = parent->x + parent->width / 2;
	      segments[0].y1 = parent->y + font_height;
	      parent = get_parent (cnode, 1);
	      segments[1].x1 = parent->x + parent->width / 2;
	      segments[1].y1 = parent->y + font_height;
	      segments[0].y2 = segments[1].y2 = cnode->y;
	      x = cnode->width / 4;
	      segments[0].x2 = cnode->x + x;
	      segments[1].x2 = cnode->x + cnode->width - x;
	      if (segments[0].x1 > segments[1].x1)
		int_switch (&(segments[0].x2), &(segments[1].x2));
	      /* draw node */
	      if (cnode->type == par)
		set_par();
	      draw_stuff (0, 1, 2);
	      if (cnode->type == par)
		unset_par();
	      draw_string (cnode->x, cnode->y, get_label_displayed (cnode));
	      break;
	    case cut:
	      /* fill in drawing data */
	      parent = get_parent (cnode, 0);
	      put_vert_seg_info (parent->x, parent->y + font_height,
				 parent->width, 0);
	      parent = get_parent (cnode, 1);
	      put_vert_seg_info (parent->x, parent->y + font_height,
				 parent->width, 1);
	      put_bar_pos_info (cnode->y, 2);
	      /* draw in node */
	      draw_stuff (0, 1, 3);
	      draw_string (cnode->x, cnode->y, cut_str);
	      break;
	    }
	}
      /* go to next node */
      cnode = cnode->next;
    }
}


cancel_current_canvas_op ()
{
  node *temp;

  clear_action_functions ();
  redisplay_canvas ();
  /* free all current nodes */
  while (curr_node != (node *) 0)
    {
      temp = curr_node->next;
      free_node (curr_node);
      curr_node = temp;
    }
  curr_node = (node *) 0;
}

void
  handle_move (widget, event)
Widget widget;
XPointerMovedEvent *event;
{
  if (move_proc != null_proc)
    (*move_proc) (event->x, event->y);
}

void
  handle_expose (widget, event)
Widget widget;
XExposeEvent *event;
{
  if (event->count == 0)
    redisplay_canvas ();
}

void
  handle_button_press (widget, event)
Widget widget;
XButtonPressedEvent *event;
{
  if (button_press_proc != null_proc)
    (*button_press_proc) (event->x, event->y);
}

static XtActionsRec canvas_actions[] =
{
  { "Handle_move", (XtActionProc) handle_move },
  { "Handle_expose", (XtActionProc) handle_expose },
  { "Handle_button_press", (XtActionProc) handle_button_press }
};

init_canvas (widgets)
     widgets_list widgets;
{
  Arg arg[8];
  int i;
  String canvas_translations =
    "<Motion>:	Handle_move()\n\
	<Expose>:	Handle_expose()\n\
	<Btn1Down>:	Handle_button_press()\n\
	<Btn2Down>:	Handle_button_press()\n\
	<Btn3Down>:	Handle_button_press()";
  
  /* create canvas */
  i = 0;
  set_and_inc (arg, XtNheight, (XtArgVal) 400, i);
  set_and_inc (arg, XtNwidth, (XtArgVal) 400, i);
  set_and_inc (arg, XtNfromHoriz, (XtArgVal) widgets->panel, i);
  set_and_inc (arg, XtNleft, (XtArgVal) XtChainLeft, i);
  set_and_inc (arg, XtNright, (XtArgVal) XtChainRight, i);
  set_and_inc (arg, XtNbottom, (XtArgVal) XtChainBottom, i);
  set_and_inc (arg, XtNtop, (XtArgVal) XtChainTop, i);
  set_and_inc (arg, XtNlabel, (XtArgVal) "", i);
  widgets->canvas = XtCreateManagedWidget ("canvas", labelWidgetClass,
					   widgets->form, arg, i);
  
  /* deal with event handling */
  clear_action_functions ();
  XtAddActions (canvas_actions, XtNumber (canvas_actions));
  XtOverrideTranslations (widgets->canvas, 
			  XtParseTranslationTable (canvas_translations));
  
  /* mess with font stuff */
  XtSetArg (arg[0], XtNfont, (XtArgVal) &font);
  XtGetValues (widgets->canvas, arg, 1);
  font_height = font->ascent + font->descent;

  /* allocate segments */
  realloc_segments ();
}

/* make the graphics context */
init_canvas_gc ()
{
  XGCValues gcv;
  int gcmask;
  Pixel fg, bg;
  Arg arg[2];
  XWindowAttributes win_atribs;

  /* get X (as opposed to Xt) info */
  dpy = XtDisplay (widgets->canvas);
  canvas_win = XtWindow (widgets->canvas);
  /* get the foreground and background colors for the window */
  XtSetArg (arg[0], XtNforeground, (XtArgVal) &fg);
  XtSetArg (arg[1], XtNbackground, (XtArgVal) &bg);
  XtGetValues (widgets->canvas, arg, 2);
  /* get the attributes of this window -- we need to know the depth */
  XGetWindowAttributes (dpy, canvas_win, &win_atribs);
  /* set the graphics context */
  gcmask = GCForeground | GCBackground | GCFunction | GCFont;
  if (win_atribs.depth == 1)
    {
      /* have monochrome monitor; ignore NodeColor and LinkColor options */
      app_data.NodeColor = app_data.LinkColor = gcv.foreground = bg ^ fg;
    }
  else
    {
      /* set the link & node colors to be XORed with bg */
      app_data.NodeColor = bg ^ app_data.NodeColor;
      app_data.LinkColor = bg ^ app_data.LinkColor;
      /* set foreground to link color (we change it when we draw nodes) */
      gcv.foreground = app_data.LinkColor;
    }
  gcv.background = bg;
  gcv.function = GXxor;
  gcv.font     = font->fid;
  gc = XCreateGC (dpy, canvas_win, gcmask, &gcv);
}

/* resets the action responses back to be nothing */
clear_action_functions ()
{
  move_proc = button_press_proc = null_proc;
}

clear_canvas ()
{
  XClearWindow (dpy, canvas_win);
}

char *
  copystring (str)
char *str;
{
  char *result;
  
  result = (char *) malloc (sizeof (char) * (strlen (str) + 1));
  strcpy (result, str);
  return result;
}

char *
  catstrings (va_alist)
int va_alist;
{
  va_list args;
  int strsize = 0;
  char *result, *str, *strptr;
  
  va_start (args);
  while ((str = va_arg (args, char *)) != (char *) 0)
    strsize += strlen (str);
  result = (char *) malloc (strsize + 1);
  strptr = result;
  va_start (args);
  while ((str = va_arg (args, char *)) != (char *) 0)
    {
      strcpy (strptr, str);
      strptr += strlen (str);
    }
  return result;
}

char *
  vcatstrings (args)
va_list args;
{
  va_list temp_args;
  int strsize = 0;
  char *result, *str, *strptr;
  
  temp_args = args;
  while ((str = va_arg (args, char *)) != (char *) 0)
    strsize += strlen (str);
  result = (char *) malloc (strsize + 1);
  strptr = result;
  args = temp_args;
  while ((str = va_arg (args, char *)) != (char *) 0)
    {
      strcpy (strptr, str);
      strptr += strlen (str);
    }
  return result;
}

draw_stuff (draw_text, draw_segments, num_segments)
     int draw_text, draw_segments, num_segments;
{
  if (draw_text)
    draw_string (text_x, text_y, text);
  if (draw_segments)
    XDrawSegments (dpy, canvas_win, gc, segments, num_segments,
		   CoordModeOrigin);
}

move_things (x, y)
     int x, y;
{
  int draw_text, draw_segments, num_segments;
  int connected_segments;
  
  /* check if selection is in bounds: if cursor is too low when 
     positioning axiom link or pax or too high when positioning tensor,
     par, or cut links, ignore this position */
  if ((curr_node->type == pax && places_picked == num_pax_nodes ||
       curr_node->type == axiom && places_picked == 2) && y > y_bound ||
      (curr_node->type == tensor || curr_node->type == par ||
       curr_node->type == cut) && y < y_bound && places_picked == 2)
    return;
  
  /* draw text if we are moving axiom literals or selecting final
     positions of tensor, par, or cut links */
  draw_text = (curr_node->type == axiom && places_picked < 2 ||
	       curr_node->type == pax && places_picked < num_pax_nodes ||
	       (curr_node->type == tensor || curr_node->type == par ||
		curr_node->type == cut) && places_picked == 2);
  /* draw segments to connect together nodes */
  draw_segments = (curr_node->type == pax && places_picked == num_pax_nodes ||
		   curr_node->type != pax && places_picked == 2);
  connected_segments = (curr_node->type == axiom || curr_node->type == pax ||
			curr_node->type == cut) && draw_segments;
  if (draw_segments)
    {
      if (curr_node->type == pax)
	num_segments = num_pax_nodes + 1;
      else if (curr_node->type == axiom || curr_node->type == cut)
	num_segments = 3;
      else
	num_segments = 2;
    }
  
  /* erase old thing */
  if (moved)
    draw_stuff (draw_text, draw_segments, num_segments);
  
  /* assign new coords */
  if (draw_text)
    {
      if (curr_node->type != cut)
	/* x position of cut text is determined by parents */
	text_x = x;
      /* draw text a bit below cursor */
      text_y = y + offset;
    }
  if (draw_segments)
    {
      if (connected_segments)
	/* we've selected a new height for connecting bar */
	put_bar_pos_info (y, num_segments - 1);
      else /* pars and tensors */
	{
	  /* set segment ends near text to reflect new text place */
	  segments[0].x2 = x + x1;
	  segments[1].x2 = x + x2;
	  segments[0].y2 = segments[1].y2 = y;
	}
    }
  
  /* draw new thing */
  draw_stuff (draw_text, draw_segments, num_segments);
  /* now that we've drawn something, we can erase it */
  moved = 1;
}

locate_link (x, y)
     int x, y;
{
  node *cnode;

  /* we have gotten all the info we need */
  if (curr_node->type == axiom)
    /* set final position */
    curr_node->op.axiom.bar_pos = curr_node->next->op.axiom.bar_pos =
      segments[0].y1;
  else if (curr_node->type == pax)
    /* set final pos of bar */
    for (cnode = curr_node; cnode != (node *) 0; cnode = cnode->next)
      cnode->op.pax.bar_pos = segments[0].y1;
  else      /* par, tensor, cut */
    {
      /* set final position */
      curr_node->x = text_x;
      curr_node->y = text_y;
    }
  if (curr_node->type == par)
    unset_par ();
  set_label (widgets->message, "Thanks!");
  add_to_graph (curr_node);
  clear_action_functions ();
  /*  Setting "current_operation" to none, and reverse the box! */
  reset_operation();
  curr_node = (node *) 0;
}

locate_formula (x, y)
int x, y;
{
  node *node_just_located, *cnode;
  int i, tot_num_formulas;
  char *item;

  /* find which node we have located */
  for (i = 0, node_just_located = curr_node; i < places_picked;
       ++i, node_just_located = node_just_located->next);
  /* one more formula has been located */
  ++places_picked;
  /* put in location */
  node_just_located->x = x;
  node_just_located->y = y;
  /* assign the data for the vertical segment */
  put_vert_seg_info (x, y, text_width, places_picked - 1);
  /* find the total number of formulas to be placed */
  tot_num_formulas = (curr_node->type == axiom) ? 2 : num_pax_nodes;

  /* assign vars for next graphics bit */
  if (places_picked < tot_num_formulas)
    {
      /* set up for placing of next formula */
      text_width = node_just_located->next->width;
      text = node_just_located->next->name;
      item = (curr_node->type == axiom) ? "literal." : "formula.";
      cat_message ("Please choose place for next ", item, (char *) 0);
      /* don't want to draw next form until cursor has moved a bit */
      moved = 0;
    }
  else       /* set up for drawing connecting lines */
    {
      /* y_bound is lowest we will allow horizontal line to be draw at */
      y_bound = curr_node->y;
      for (cnode = curr_node->next; cnode != (node *) 0; cnode = cnode->next)
	y_bound = min (y_bound, cnode->y);
      /* initialize the height of bar to be something reasonable */
      put_bar_pos_info (y_bound - font_height, tot_num_formulas);
      /* draw the segments */
      draw_stuff (0, 1, tot_num_formulas + 1);
      /* give instructions in case user wants to change height */
      set_label (widgets->message, "Please choose place for connecting bar.");
      /* initial place for lines has been selected */
      moved = 1;
      /* next thing to be located is position of bar */
      button_press_proc = locate_link;
    }
}

/*  This gives receive_nd the child of give_nd, and resetting all corresponding
    pointers.
*/
void
  re_assign_child (receive_nd, give_nd)
node *receive_nd, *give_nd;
{
  node *child;
  int parent_num;

  child = get_child (give_nd);

  assign_child (receive_nd, child);
  if (child != (node *) 0)
    {
      parent_num = (get_parent(child, 0) == give_nd) ? 0 : 1;
      put_parent (child, receive_nd, parent_num);
    }
}
  

/*  This takes 2 axioms, "nd1" and "nd2", and does a cut-elimination on
    those 2.
*/
void 
  cut_axiom (nd1, nd2)
node *nd1, *nd2;
{
  node *temp;
  if (nd1->op.axiom.sibling == nd2)
    {
      /* "nd1" and "nd2" are axiom siblings, just delete them both. */
      delete_node (nd1);
      delete_node (nd2);
    }
  else
    {
      /*  This loops ensures that we delete the axiom that comes later.
	  we will delete nd1 and its sibling, so if nd1 comes "before" nd2
	  in the list of nodes, then switch the two.  Iterate this loop
	  until we hit either node.  If we hit nd1, we do the switch.
	  If we hit nd2, we don't do the switch.  However, we exit
	  as soon as we hit either one.  And upon exiting, nd1 is the
	  node "prior" to nd1, in the list of nodes started by "head."*/
      temp = head;
      while ((temp != nd2) && (temp != nd1))
	temp = temp->next;
      if (temp==nd1)
	{
	  nd1 = nd2;
	  nd2 = temp;
	}

      /*  we use "nd2" as "nd1->op.axiom.sibling".  So, we reassign
	  children as well as coordinates accordingly.  Once done,
	  we delete the appropriate node, i.e. "nd1" and its sibling.
      */
      nd2->op.axiom.bar_pos = 
	nd2->op.axiom.sibling->op.axiom.bar_pos =
	  min (nd2->op.axiom.bar_pos, nd1->op.axiom.bar_pos);
      nd2->x = nd1->op.axiom.sibling->x;
      nd2->y = nd1->op.axiom.sibling->y;
      
      re_assign_child (nd2, nd1->op.axiom.sibling);
      /* NOTE: if we delete "nd1" first, "nd1->op.axiom.sibling" would probably
	 give a core dump!!!
	*/
      delete_node (nd1->op.axiom.sibling);
      delete_node (nd1);
    }
}

/* This function sets up everything needed to draw a cut from nd1 to nd2.
   One function using the values set by "put_cut" is move_things (i.e. the
   move_proc).
*/
void
  put_cut (nd1, nd2)
node *nd1, *nd2;
{
  
  /*  making a new node for the cut, parented by "nd1" and "nd2".
    */
  curr_node = make_new_node(cut);
  put_parent(curr_node, nd1, 0);
  put_parent(curr_node, nd2, 1);
  /* get the expanded name of the connective */
  curr_node->name = create_node_name (curr_node);
  curr_node->width = text_width = get_label_width (curr_node);

  /*  Values needed by function "move_things".
    */
  offset = (curr_node->type == cut);
  segments[0].x1 = segments[0].x2 = nd1->x + nd1->width/2;
  segments[1].x1 = segments[1].x2 = nd2->x + nd2->width/2;
  segments[0].y2 = nd1->y + font_height;
  segments[1].y2 = nd2->y + font_height;
  set_label (widgets->message, "Please choose place for connective.");

  /* y_bound is bound that we will not allow lines to be draw above */
  y_bound = max (nd1->y, nd2->y) + font_height;

  /* needed by "move_things." */
  places_picked = 2;  
  /* initial place for lines has been selected, this allows erasing
     drawn lines.
     */
  moved = 1;
  
  /* get x position of text */
  text_x = (segments[0].x1 + segments[1].x1 - text_width) / 2;
  /* select a good initial position for link */
  text_y = y_bound + offset + font_height;
  text = get_label_displayed (curr_node);
  
  put_bar_pos_info (y_bound + font_height, 2);
  draw_stuff (1, 1, 3);
}

/* Resetting all variables for the pairs used by "cut_elm".
*/
void reset_prs_vars()
{
  total_cut_prs = total_ax_prs = total_del_prs = 0;
  cut_prs_tail = cut_prs_head = (node_pairs_list *) 0;
  ax_prs_tail = ax_prs_head = (node_pairs_list *) 0;
  del_prs_tail = del_prs_head = (node_pairs_list *) 0;
  nodes_to_add_hd = nodes_to_add_tl = (nodelist *) 0;
}

/* Freeing the list of pairs of nodes, used by "cut_elm."
*/
void
  free_prs (prs_type)
cut_type prs_type;
{
  node_pairs_list *temp, *temp2;

  switch (prs_type) {
  case ax_prs:
    temp = ax_prs_head;
    while (temp != (node_pairs_list *) 0)
      {
	temp2 = temp->next;
	free((char *) temp);
	temp = temp2;
      }
    break;
  case cut_prs:
    temp = cut_prs_head;
    while (temp != (node_pairs_list *) 0)
      {
	temp2 = temp->next;
	free((char *) temp);
	temp = temp2;
      }
    break;
  case del_prs:
    temp = del_prs_head;
    while (temp != (node_pairs_list *) 0)
      {
	temp2 = temp->next;
	free((char *) temp);
	temp = temp2;
      }
    break;
  }
}

/*  This function applies cuts to the list of pairs of nodes already
    in memory.
*/
void
  apply_cuts(prs_type)
cut_type prs_type;
{
  node_pairs_list *temp, *temp2;

  switch (prs_type) {
  case ax_prs:
    /*  For "ax_prs", i.e. axioms, a cut is to apply the function
	"cut_axiom."
     */
    for (temp = ax_prs_head; temp != (node_pairs_list *) 0; 
	 temp=temp->next)
      cut_axiom (temp->nd_pair[0],temp->nd_pair[1]);
    break;
  case del_prs:
    /*  For "del_prs", i.e. delete-pairs, a cut is merely to delete
	the nodes.
     */
    for (temp = del_prs_head; temp!= (node_pairs_list *) 0; 
	 temp=temp->next)
      {
	delete_node(temp->nd_pair[0]);
	delete_node(temp->nd_pair[1]);
      }
    break;
  }
}
  
/*  Since we cannot add the new "cut" nodes, created by "cut elimination",
    until the very end, (in case the user does an "abort.") we just add
    the node that is to be added to the graph to this list.  If the user
    chooses to abort, we'll just free the entries for this list.
*/
void
  add_nodes_to_add ()
{
  nodelist *temp;

  temp = (nodelist *) malloc (sizeof (nodelist));
  temp->entry = curr_node;
  temp->next = (nodelist *) 0;
  if (nodes_to_add_hd == (nodelist *) 0)
    nodes_to_add_hd = temp;
  else
    nodes_to_add_tl->next = temp;
  nodes_to_add_tl = temp;
  curr_node = (node *) 0;
}

/*  This function finishes what function "add_nodes_to_add()" has started.
    It takes the list started by "nodes_to_add_hd" and add them all to
    the graph.
*/
void
  add_nodes()
{
  nodelist *temp, *temp2;
  node *parent;
  for (temp = nodes_to_add_hd; temp != (nodelist *) 0;
       temp = temp->next)
      add_to_graph(temp->entry);

  /*  Now that we added them, we'll free this list. 
    */
  temp = nodes_to_add_hd;
  while (temp != (nodelist *) 0)
    {  temp2 = temp->next;
       free ((char *) temp);
       temp = temp2;
     }
}

/****************************************************************************
   The set of functions starting with "cp_" are to operate on a copy of the
   data structure, i.e. graph.  It does a fake-application of the
   "cut-elimination" on the cp_graph so that we can draw the updated
   graph on the screen.  
*****************************************************************************/
cp_redisplay_canvas ()
{
  node *tmphead, *tmp_ret_val;
  
  tmphead = head;
  head = cp_head;
  redisplay_canvas();
  head = tmphead;
}

cp_free_graph ()
{
  node *temp;
  
  while (cp_head != (node *) 0)
    {
      temp = cp_head->next;
      free_node ((char *)cp_head);
      cp_head = temp;
    }
  cp_head = (node *) 0;
  cp_num_nodes = 0;
}

/*  This resembles the function "copy_graph" in file "graph.c".  However,
    since we need all the info on screen-display, we need to copy all that
    as well...
*/
node *
  cp_copy_graph ()
{
  int count, node_num, pax_nodes = 0;
  node *old_node, *new_node;
  
  copy_array = (node **) malloc (num_nodes * sizeof (node *));
  
  number_nodes (head);
  
  /* copy nodes -- old node is node in old graph, new_node is in new graph*/
  for (old_node = head, count = 0; old_node != (node *) 0;
       old_node = old_node->next, ++count)
    {
      new_node = make_new_node (old_node->type);
      /* put this node in copy_array and update next pointer of last node */
      copy_array[count] = new_node;
      new_node->x = old_node->x;
      new_node->y = old_node->y;
      new_node->width = old_node->width;
      new_node->name = strdup(old_node->name);
      new_node->mark = old_node->mark;

      if (count > 0)
	copy_array[count - 1]->next = new_node;
      if (old_node->type == axiom)
	{
	  if ((node_num = old_node->op.axiom.sibling->mark) < count)
	    {
	      /* this is the second sibling, assign sibling values for both */
	      copy_array[node_num]->op.axiom.sibling = new_node;
	      new_node->op.axiom.sibling = copy_array[node_num];
	    }
	  new_node->op.axiom.bar_pos = old_node->op.axiom.bar_pos;
	}
      else if (old_node->type == pax)
	{
	  ++pax_nodes;
	  new_node->op.pax.bar_pos = old_node->op.pax.bar_pos;
	  if (pax_nodes == old_node->op.pax.num_siblings + 1)
	    {
	      /* this is last node in pax; have to assign sibling pointers */
	      assign_siblings (copy_array[count - pax_nodes + 1], pax_nodes);
	      pax_nodes = 0;
	    }
	}
      else
	{
	  /* fix parent/child pointers between this node and its parents */
	  node_num = (get_parent (old_node, 0))->mark;
	  assign_child (copy_array[node_num], new_node);
	  put_parent (new_node, copy_array[node_num], 0);
	  node_num = (get_parent (old_node, 1))->mark;
	  assign_child (copy_array[node_num], new_node);
	  put_parent (new_node, copy_array[node_num], 1);
	}
    }
  reset_numbers (head);
  
  /* return the copy */
  return *copy_array;
}

node *
  cp_get_selected_node (x, y)
int x, y;
{
  node *tmphead, *tmp_ret_val;

  tmphead = head;
  head = cp_head;
  tmp_ret_val = get_selected_node (x, y);
  head = tmphead;
  return tmp_ret_val;
}

cp_delete_node (node_to_delete)
     node *node_to_delete;
{
  node *temp;

  if (node_to_delete == cp_head)
    {
      if (node_to_delete == cp_tail)
	cp_tail = (node *) 0;
      temp = cp_head;
      cp_head = cp_head->next;
      free_node (temp);
    }
  else
    {
      /* increment temp until the next node after temp is the one we want
	 to delete */
      for (temp = cp_head; temp->next != node_to_delete; temp = temp->next);
      if (node_to_delete == cp_tail)
	cp_tail = temp;
      temp->next = node_to_delete->next;
      free_node (node_to_delete);
    }
  /* we have one fewer node in the graph */
  --cp_num_nodes;
}

void 
  cp_cut_axiom (nd1, nd2)
node *nd1, *nd2;
{
  node *temp;
/* cut out node 1!  Replace node1's sibling by nd2 */
  if (nd1->op.axiom.sibling == nd2)
    {
      cp_delete_node (nd1);
      cp_delete_node (nd2);
    }
  else
    {
      temp = cp_head;
      while ((temp != nd2) && (temp != nd1))
	temp = temp->next;
      if (temp==nd1)
	{
	  nd1 = nd2;
	  nd2 = temp;
	}

      nd2->op.axiom.bar_pos = 
	nd2->op.axiom.sibling->op.axiom.bar_pos =
	  min (nd2->op.axiom.bar_pos, nd1->op.axiom.bar_pos);
      nd2->x = nd1->op.axiom.sibling->x;
      nd2->y = nd1->op.axiom.sibling->y;
      
      re_assign_child (nd2, nd1->op.axiom.sibling);
      cp_delete_node (nd1->op.axiom.sibling);
      cp_delete_node (nd1);
    }
}

/*  This gets the list of "cp_..." to hold the corresponding nodes of
    the pairs list (ax_prs_head and del_prs_head).
    This is the list used when the corresponding fake-cut-elimination
    functions are applied on the cp_graph.
    We don't need the list "cut_prs_head" because we don't do the
    fake cut-elimination thing for cut pairs.
*/
void 
  cp_pairs()
{
  node_pairs_list *temp, *cp_temp;

  cp_ax_prs_tail = cp_ax_prs_head = (node_pairs_list *) 0;
  cp_del_prs_tail = cp_del_prs_head = (node_pairs_list *) 0;
  
  if (total_del_prs)
    {
      for (temp=del_prs_head; temp != (node_pairs_list *) 0;
	   temp = temp->next)
	{
	  cp_temp = (node_pairs_list *) malloc (sizeof (node_pairs_list));
	  cp_temp->next = (node_pairs_list *) 0;
	  cp_temp->nd_pair[0] = 
	    cp_get_selected_node(temp->nd_pair[0]->x,
				 temp->nd_pair[0]->y);
	  cp_temp->nd_pair[1] = 
	    cp_get_selected_node(temp->nd_pair[1]->x,
				 temp->nd_pair[1]->y);
	  if (cp_del_prs_head == (node_pairs_list *) 0)
	    cp_del_prs_head = cp_temp;
	  else
	    cp_del_prs_tail->next = cp_temp;
	  cp_del_prs_tail = cp_temp;
	}
    }

  if (total_ax_prs)
    {
      for (temp=ax_prs_head; temp != (node_pairs_list *) 0;
	   temp = temp->next)
	{
	  cp_temp = (node_pairs_list *) malloc (sizeof (node_pairs_list));
	  cp_temp->next = (node_pairs_list *) 0;
	  cp_temp->nd_pair[0] = 
	    cp_get_selected_node(temp->nd_pair[0]->x,
				 temp->nd_pair[0]->y);
	  cp_temp->nd_pair[1] = 
	    cp_get_selected_node(temp->nd_pair[1]->x,
				 temp->nd_pair[1]->y);
	  if (cp_ax_prs_head == (node_pairs_list *) 0)
	    cp_ax_prs_head = cp_temp;
	  else
	    cp_ax_prs_tail->next = cp_temp;
	  cp_ax_prs_tail = cp_temp;
	}
    }
}	        
      
/*  This uses the "cp_" functions to draw the updated graph, while the
    user is putting the location for the connective bars for the "cuts"
    resulted on cut_elimination.
*/
void
  draw_cut_graph()
{
  node_pairs_list *temp;
  node *index, *cp_cut_elm_node;

  cp_head = cp_copy_graph();
  cp_redisplay_canvas();
  for (index = cp_head;
       index->next != (node *) 0;
       index = index->next);
  cp_tail = index;
  cp_pairs();
  /* delete all the nodes that is to be deleted. */
  cp_cut_elm_node = cp_get_selected_node(cut_elm_node->x,
					 cut_elm_node->y);
  cp_delete_node(cp_cut_elm_node);
  if (total_del_prs)
      for (temp = cp_del_prs_head; temp != (node_pairs_list *) 0; 
	   temp=temp->next)
	{
	  cp_delete_node (temp->nd_pair[0]);
	  cp_delete_node (temp->nd_pair[1]);
	}
  if (total_ax_prs)
      for (temp = cp_ax_prs_head; temp != (node_pairs_list *) 0; 
	   temp=temp->next)
	cp_cut_axiom (temp->nd_pair[0],temp->nd_pair[1]);
	
  cp_redisplay_canvas();
  /* draw the graph as it will be, with axioms and nodes deleted,
     but "cuts" not added yet.
    */

  /* free the cp_graph used just for this function!
   */
  cp_free_graph();
}

/*  This procedure is the "button_press_proc" while doing cut_elimination,
    if there are nodes that need to be "cut".  It draws as many connectives
    for as many cuts as needed.
*/
draw_cuts(x,y)
int x,y;
{
  node_pairs_list *temp;
  int i;

  cuts_done++;  /*  counting the number of connectives drawn. */
  curr_node->x = text_x;
  curr_node->y = text_y;

  /*  Add this node to the list of nodes to be added to the graph,
      if the user doesn't decide to "abort."
      */
  add_nodes_to_add();
  if (cuts_done == total_cut_prs)  /* done with ALL "cuts" */
    {
      delete_node(cut_elm_node);
      if (total_del_prs)
	{
	  apply_cuts(del_prs);
	  free_prs(del_prs);
	}
      add_nodes();
      if (total_ax_prs)
	{
	  apply_cuts(ax_prs);
	  free_prs(ax_prs);
	}
      free_prs(cut_prs);
      set_label(widgets->message, "Thanks.");
      cancel_current_canvas_op();
      reset_operation();
      cut_elm_node = (node *) 0;
      reset_prs_vars();
    }
  else
    {
      /*  There are more nodes to draw connectives for... 
          The loop finds which node is having its location determined,
	  then set up all necessary information for "move_things" and
	  next execution of "draw_cuts", if the user doesn't abort.
	  */
      for (i=0, temp = cut_prs_head; i < cuts_done;
	   temp = temp->next, i++);
      put_cut(temp->nd_pair[0], temp->nd_pair[1]);
    }
}

/*  This adds the pair of nodes "nd1" and "nd2" as the next pair of nodes
    on the corresponding list of pairs, indicated by "prs_type".
*/
void
  add_pairs (nd1, nd2, prs_type)
node *nd1, *nd2;
cut_type prs_type;
{
  node_pairs_list *temp;

  temp = (node_pairs_list *) malloc(sizeof(node_pairs_list));
  temp->nd_pair[0] = nd1;
  temp->nd_pair[1] = nd2;
  temp->next = (node_pairs_list *) 0;
  switch (prs_type) 
    {
    case ax_prs:  
      total_ax_prs++;
      if (ax_prs_head == (node_pairs_list *) 0)
	ax_prs_head = temp;
      else
	ax_prs_tail->next = temp;
      ax_prs_tail = temp;
      break;
    case cut_prs:
      total_cut_prs++;
      if (cut_prs_head == (node_pairs_list *) 0)
	cut_prs_head = temp;
      else
	cut_prs_tail->next = temp;
      cut_prs_tail = temp;
      break;
    case del_prs:
      total_del_prs++;
      if (del_prs_head == (node_pairs_list *) 0)
	del_prs_head = temp;
      else
	del_prs_tail->next = temp;
      del_prs_tail = temp;
      break;
    }
}
  
/*  "cut_elim" on 2 nodes is really a recursive process.
    This function is mostly self-explanatory.
*/
void 
  recursive_cut (nd1, nd2)
node *nd1, *nd2;
{
  if ((nd1->type == pax) || (nd2->type == pax))
      add_pairs(nd1, nd2, cut_prs);
  else 
    if ((nd1->type == axiom) && (nd2->type == axiom))
      add_pairs(nd1, nd2, ax_prs);
    else
      {
	add_pairs (nd1,nd2, del_prs);
	recursive_cut ( get_parent(nd1,0),
		       get_parent(nd2,0));
	recursive_cut ( get_parent(nd1,1),
		       get_parent(nd2,1));
      }
}

/*  This procedure locates the node that is to be "cut-eliminated".
    the node has to be a "cut" node.
*/
locate_cut_elm (x,y)
     int x, y;
{
  node *selected_node;
  selected_node = get_selected_node (x, y);
  if (selected_node != (node *) 0)
    if (selected_node->type == cut)
      {
	/*  selected node is a "cut" node.
	    "recursive_cut" will set up appropriate information on the
	    lists of pairs.  All other operations can follow...
	  */
	cut_elm_node = selected_node;
	recursive_cut (get_parent(cut_elm_node,0),
		       get_parent(cut_elm_node,1));
	redisplay_canvas();
	
	if (total_cut_prs)
	  {  /* There are nodes that need to have "cut" drawn for.  */
	    draw_cut_graph();
	    move_proc = move_things;
	    button_press_proc = draw_cuts;
	    /* Set up info for the first "cut". */
	    cuts_done = 0;
	    moved = 0;
	    put_cut(cut_prs_head->nd_pair[0], cut_prs_head->nd_pair[1]);
	  }
	else
	  {
	    /* user cannot "abort" from here on...   apply all 
	       cut operations, using the lists of pairs...
	     */
	    delete_node(cut_elm_node);
	    if (total_del_prs)
	      {
		apply_cuts(del_prs);
		free_prs(del_prs);
	      }
	    if (total_ax_prs)
	      {
		apply_cuts(ax_prs);
		free_prs(ax_prs);
	      }
	    set_label(widgets->message, "Thanks.");
	    cancel_current_canvas_op();
	    reset_operation();
	    cut_elm_node = (node *) 0;
	    reset_prs_vars();
	  }
	return;
      }
    else
      {
	/* user picked a node that wasn't a "cut" node.  Repeat,
	   have the user re-pick a node or "abort."
	 */
	set_label (widgets->message, "You have to pick a <cut> node.");
	return;
      }
  else
    /* user didn't pick a node, repeat. */
    return;
}
locate_parent (x, y)
     int x, y;
{
  node *parent;
  int inv;

  /* assign parent (premise) */
  parent = get_selected_node (x, y);

  /* if no node was chosen, or chosen node is a cut, or chosen node already
     has already been used another parent must be chosen */
  if (parent == (node *) 0 || parent->type == cut ||
      get_child (parent) != (node *) 0 ||
      places_picked == 1 && parent == get_parent (curr_node, 0))
    return;
  /* If the user has selected the second premise to cut, and these
     premises are not inverses of each other, cancel operation. */
  if (curr_node->type == cut && places_picked == 1 
      && !inverses (get_parent (curr_node, 0), parent))
    {
      set_label (widgets->message, "Can't cut: premises are not inverses.");
      /*  Setting "current_operation" to none, and reverse the box! */
      reset_operation();
      cancel_current_canvas_op ();
      return;
    }
  /* have located one more place (in this case, a parent) */
  ++places_picked;
  /* assign chosen node to be the parent of this node */
  put_parent (curr_node, parent, places_picked - 1);

  /* assign graphics and other info in fields */
  if (curr_node->type == cut)
    put_vert_seg_info (parent->x, parent->y + font_height, parent->width,
		       places_picked - 1);
  else /* tensor, par */
    {
      /* assign data for endpoints near parents */
      segments[places_picked - 1].x1 = parent->x + parent->width / 2;
      segments[places_picked - 1].y1 = parent->y + font_height;
    }
  if (places_picked == 1)
    set_label (widgets->message, "Please choose second premise.");
  else /* places_picked == 2, set up for drawing connecting lines */
    {
      set_label (widgets->message, "Please choose place for connective.");
      /* y_bound is bound that we will not allow lines to be draw above */
      y_bound = max ((get_parent (curr_node, 0))->y,
		     (get_parent (curr_node, 1))->y) + font_height;
      /* get the expanded name of the connective */
      curr_node->name = create_node_name (curr_node);
      curr_node->width = text_width = get_label_width (curr_node);
      /* set up to move around node representation */
      move_proc = move_things;
      text = get_label_displayed (curr_node);
      if (curr_node->type == cut)
	{
	  /* get x position of text */
	  text_x = (segments[0].x1 + segments[1].x1 - text_width) / 2;
	  /* select a good initial position for link */
	  text_y = y_bound + offset + font_height;
	  put_bar_pos_info (y_bound + font_height, 2);
	  draw_stuff (1, 1, 3);
	}
      else /* par & tensor */
	{
	  /* save offsets from left of text where segments attach */
	  x1 = text_width / 4;
	  x2 = text_width - x1;
	  /* if the top section of segment[0] is more to left, switch the
	     bottoms around */
	  if (segments[0].x1 > segments[1].x1)
	    int_switch (&x1, &x2);
	  /* select a good initial position for link */
	  text_x = (segments[0].x1 + segments[1].x1 - text_width) / 2;
	  text_y = y_bound + offset + font_height;
	  segments[0].x2 = text_x + x1;
	  segments[1].x2 = text_x + x2;
	  segments[0].y2 = segments[1].y2 = y_bound + font_height;
	  if (curr_node->type == par)
	    set_par ();   /* draw dotted lines */
	  draw_stuff (1, 1, 2);
	}
      button_press_proc = locate_link;
      /* initial place for lines has been selected */
      moved = 1;
    }
  
}

/* This draws a box around the label located at x, y, with width "width."
   Since this is used just to "hilite", the function "set_boxlines()"
   is used to widen the width of the lines.
*/
draw_circling_box (x, y, width)
     int x, y, width;
{
  XPoint points[5];
  
  set_boxlines();
  points[0].x = points[4].x = points[3].x = x - 1;
  points[0].y = points[4].y = points[1].y = y - 1;
  points[1].x = points[2].x = x + width + 1;
  points[2].y = points[3].y = y + font_height + 1;
  XDrawLines(dpy, canvas_win, gc, points, 5, CoordModeOrigin);
  unset_thicklines();
} 

/* This is used to hilite both the lines and the nodes involving
   "cnode".  It hilites the lines from "cnode" to its parents (if
   there are any), and it also draws a little box surrounding the label
   of "cnode."
*/
hilite_lines(cnode)
     node *cnode;
{
  int x, i, num_nodes, bar_pos;
  node *parent, *nd;

  switch (cnode->type)
    {
    case cut:
      /* fill in drawing data */
      parent = get_parent (cnode, 0);
      put_vert_seg_info (parent->x, parent->y + font_height,
			 parent->width, 0);
      parent = get_parent (cnode, 1);
      put_vert_seg_info (parent->x, parent->y + font_height,
			 parent->width, 1);
      put_bar_pos_info (cnode->y, 2);
      /* draw in node */
      draw_stuff (0, 1, 3);
      set_thicklines();
      draw_stuff (0, 1, 3);
      unset_thicklines();
      draw_circling_box (cnode->x, cnode->y, cnode->width);
      break;
    case par:
      set_par();
    case tensor:
      parent = get_parent(cnode, 0);
      segments[0].x1 = parent->x + parent->width / 2;
      segments[0].y1 = parent->y + font_height;
      parent = get_parent(cnode, 1);
      segments[1].x1 = parent->x + parent->width / 2;
      segments[1].y1 = parent->y + font_height;
      segments[0].y2 = segments[1].y2 = cnode->y;
      segments[0].x2 = cnode->x + cnode->width / 4;
      segments[1].x2 = cnode->x + 3 * cnode->width / 4;
      if (segments[0].x1 > segments[1].x1)
	int_switch (&(segments[0].x2), &(segments[1].x2));
      draw_stuff(0, 1, 2);
      set_thicklines();
      draw_stuff(0, 1, 2);
      unset_thicklines();
      if (cnode->type == par)
	unset_par();
      draw_circling_box (cnode->x, cnode->y, cnode->width);
      break;
    case axiom:
      bar_pos = get_bar_pos (cnode);
      /* fill in vertical segment info */
      put_vert_seg_info (cnode->x, cnode->y, cnode->width, 0);
      put_vert_seg_info (cnode->op.axiom.sibling->x, 
			 cnode->op.axiom.sibling->y, 
			 cnode->op.axiom.sibling->width, 1);
      put_bar_pos_info (bar_pos, 2);

      draw_circling_box(cnode->x, cnode->y, cnode->width);
      draw_circling_box(cnode->op.axiom.sibling->x, 
			cnode->op.axiom.sibling->y, 
			cnode->op.axiom.sibling->width);
      draw_stuff (0, 1, 3);
      set_thicklines();
      draw_stuff (0, 1, 3);
      unset_thicklines();
      break;
    case pax:
      bar_pos = get_bar_pos (cnode);
      num_nodes = cnode->op.pax.num_siblings + 1;
      /* fill in vertical segment info */
      for (nd = cnode, i = 0; i < num_nodes; nd = cnode->op.pax.siblings[i++])
	put_vert_seg_info (nd->x, nd->y, nd->width, i);
      put_bar_pos_info (bar_pos, num_nodes);
      /* draw formulas and connecting bar */
      for (nd = cnode, i = 0; i < num_nodes; nd = cnode->op.pax.siblings[i++])
	draw_circling_box (nd->x, nd->y, nd->width);
      draw_stuff (0, 1, num_nodes + 1);
      set_thicklines();
      draw_stuff (0, 1, num_nodes + 1);
      unset_thicklines();
      break;
    }
}
      
/* This hilites the node "cnode" if it has not already been hilited,
   and re-iterate on any children or siblings of "cnode".
*/
hilite_nodes_to_delete(cnode, check_stat, hilited_hd)
     node *cnode;
     int check_stat;
     nodelist *hilited_hd;

{
  node *tmp_node;
  nodelist *p;
  int i, match=0;

  /* This part checks to see if the node has been hilited.  */
  if ((p=hilited_hd) != (nodelist *) 0)
    { 
      /* upon exiting this loop, p is the last item of the list
	 headed by "head", or p->entry is "cnode".
	*/
      while ((p->entry != cnode) && (p->next!= (nodelist *) 0))
	p=p->next;

      /* if p is the last item of the list, and p->entry is not
	 "cnode", then "cnode" has NOT been hilited.  So, add
	 it on the list because it's being hilited now.*/
      if ((p->next==NULL) && (p->entry != cnode))
	{
	  p->next = (nodelist *) malloc (sizeof(nodelist));
	  p->next->entry = cnode;
	  p->next->next = NULL;
	}
      else
	/* p->entry is "cnode."  So set match=1, so that nothing
	   will be done. */
	match = 1;
    }
  else
    { /* list of hilited nodes is empty, so create it. */
      hilited_hd = (nodelist *) malloc (sizeof(nodelist));
      hilited_hd->entry = cnode;
      hilited_hd->next = NULL;
    }
    

  if (!match)
    { /* node has not been hilited, hilite it and its children
	 or siblings accordingly.
	*/
      if ((cnode->type!=axiom) && (cnode->type!=pax))
	hilite_lines(cnode);
      if (check_stat)
	{ /* check_stat = 1 means that axioms and paxes should
	     check for siblings.  check_stat = 0 means not to worry
	     about siblings, as only one sibling needs to check for
	     others.
	   */
	  if ((cnode->type == axiom) || (cnode->type == pax))
	    hilite_lines(cnode);
	  if (cnode->type == axiom)
	    hilite_nodes_to_delete(cnode->op.axiom.sibling, 0, hilited_hd);
	  else if (cnode->type == pax)
	    for (i = 0; i < cnode->op.pax.num_siblings; i++)
	      hilite_nodes_to_delete(cnode->op.pax.siblings[i], 0, hilited_hd);
	}

      /* if this node has a child, recurse on its child. */
      if ((tmp_node = get_child(cnode)) != (node *) 0)
	hilite_nodes_to_delete(tmp_node, 0, hilited_hd);
    }
}
  
/* This recursively deletes the nodes that are to be deleted becasue
   "cnode" is.  "check_stat" = 1 means to worry about recursing on
   siblings, "check_stat" = 0 means not to worry about it.
*/
recursive_delete (cnode, check_stat)
     node *cnode;
     int check_stat;
{
  node *tmp_node;
  int i;

  /* recurse on its child if it has one. */
  if ((tmp_node = get_child (cnode)) != (node *) 0)
    recursive_delete (tmp_node, 1);

  /* if it needs to check (recursive_delete) on siblings, do so. */
  if (check_stat)
    if (cnode->type == axiom)
      recursive_delete (cnode->op.axiom.sibling, 0);
    else if (cnode->type == pax)
      for (i = 0; i < cnode->op.pax.num_siblings; i++)
	recursive_delete(cnode->op.pax.siblings[i], 0);

  /* if the node is something other than "axiom" or "pax",
     set its parents' pointer to "child" to null, as its
     parents'child is itself, which is... dying.
   */
  if ((cnode->type != axiom) && (cnode->type != pax))
    {
      assign_child (get_parent (cnode, 0), (node *) 0);
      assign_child (get_parent (cnode, 1), (node *) 0);
    }
  /* finally, delete the node! */
  delete_node (cnode);
}

/* locate the node to do "branch delete" on.  Meaning, deleting the
   node and all its dependents.
*/
locate_node_to_delete (x, y)
     int x, y;
{ node *cnode;
  int i;

  cnode = get_selected_node (x, y);
  /* if person didn't select any node, then can't delete it */
  if (cnode == (node *) 0)
    return;
  /* first, hilite the nodes to be deleted. */
  hilite_nodes_to_delete (cnode, 1, NULL);

  /* actually do the deleting only after user confirms by
     clicking on "delete" panel button AGAIN.
    */
  clear_action_functions();
  set_label(widgets->message, "<delete> to delete; <abort> otherwise.");
  head_delete = cnode;

}  

locate_node_to_expand (x, y)
     int x, y;
{
  node *cnode;
  
  cnode = get_selected_node (x, y);
  if (cnode == (node *) 0)
    return;
  set_label (widgets->message, cnode->name);
  clear_action_functions ();
  /*  Setting "current_operation" to none, and reverse the box! */
  reset_operation();
}

move_component (x, y)
     int x, y;
{
  /* don't redraw if hasn't moved or there are more move events coming */
  if (x == x1 && y == y1)
    return;
  /* erase component in old place */
  draw_graph (MOVE_MARK);
  /* update position */
  add_to_locations (MOVE_MARK, x - x1, y - y1);
  /* save current location */
  x1 = x;
  y1 = y;
  /* draw component in new place */
  draw_graph (MOVE_MARK);
}

/* this takes care of everything when a "cut-elimination" is aborted. */
abort_cut_elm()
{
  nodelist *temp, *temp2;
  node *child;

  /*  this frees the list of nodes that were to be added.  It is sure
      to free the character strings as well as the node-entries and all...
   */
  temp = nodes_to_add_hd;
  while (temp != (nodelist *) 0)
    {
      temp2 = temp->next;
      /*
      free((char *) temp->entry->name);
      free((char *) temp->entry);
      */
      free_node(temp->entry);
      free((char *) temp);
      temp = temp2;
    }

  /*  Free the list of pairs if there's any...  */
  if (total_cut_prs)
    free_prs(cut_prs);
  if (total_ax_prs)
    free_prs(ax_prs);
  if (total_del_prs)
    free_prs(del_prs);

  /* reset all pointers. */
  reset_prs_vars();
  cut_elm_node = (node *) 0;

}


abort_move ()
{
  /* erase component in moved place */
  draw_graph (MOVE_MARK);
  /* restore old position */
  add_to_locations (MOVE_MARK, old_x - x1, old_y - y1);
  /* draw component in restored place */
  draw_graph (MOVE_MARK);
}

locate_component (x, y)
     int x, y;
{
  /* clean up */
  reset_numbers (head);
  /*  Setting "current_operation" to none, and reverse the box! */
  reset_operation();
  clear_action_functions ();
  set_label (widgets->message, "Thanks!");
}

get_graph_component (x, y)
     int x, y;
{
  node *node_picked;
  
  if ((node_picked = get_selected_node (x, y)) == (node *) 0)
    return;
  /* mark all the nodes connected to this one with MOVE_MARK */
  mark_connected_nodes (from_none, node_picked, MOVE_MARK);
  /* save the current location of the pointer */
  old_x = x1 = x;
  old_y = y1 = y;
  move_proc = move_component;
  button_press_proc = locate_component;
  /* give instrustions to user */
  set_label (widgets->message, "Please select new place for component.");
}

get_label_width (cnode)
node *cnode;
{
  char *label;

  label = get_label_displayed (cnode);
  return XTextWidth (font, label, strlen (label));
}

setup_operation (type, name)
     op_type type;
     String name;
{
  String instructions;
  node *next_node;
  
  curr_node = make_new_node (type);
  if (type == axiom)
    {
      /* create the other axiom node */
      next_node = make_new_node (axiom);
      curr_node->next = next_node;
      curr_node->op.axiom.sibling = next_node;
      next_node->op.axiom.sibling = curr_node;
      /* assign names */
      curr_node->name = copystring (name);
      curr_node->width = text_width = get_label_width (curr_node);
      next_node->name = catstrings (name, perp_str, (char *) 0);
      next_node->width = get_label_width (next_node);
      text = curr_node->name;
      move_proc = move_things;
      instructions = "Please choose place for first literal.";
    }
  else
    instructions = "Please choose first premise.";

  offset = (type == cut);
  
  set_label (widgets->message, instructions);
  button_press_proc = (type == axiom) ? locate_formula : locate_parent;
  moved = 0;
  places_picked = 0;
}

setup_pax (pax_head, num_nodes)
node *pax_head;
int num_nodes;
{
  node *cnode;

  /* first figure out the widths of all the formulas */
  for (cnode = pax_head; cnode != (node *) 0; cnode = cnode->next)
    cnode->width = get_label_width (cnode);
  /* allocate the segments for display */
  if (num_nodes > max_pax_nodes)
    {
      max_pax_nodes = num_nodes;
      realloc_segments ();
    }
  /* set up for placing the first formula */
  curr_node = pax_head;
  num_pax_nodes = num_nodes;
  text = pax_head->name;
  text_width = pax_head->width;
  move_proc = move_things;
  button_press_proc = locate_formula;
  set_label (widgets->message, "Please choose place for formula.");
  moved = 0;
  places_picked = 0;
}

setup_cut_elm ()
{
  set_label (widgets->message, "Please pick a <cut> node.");
  button_press_proc = locate_cut_elm;
  places_picked = 0;
  cut_elm_node = (node *) NULL;
  reset_prs_vars();
}

setup_delete ()
{
  set_label (widgets->message, "Please pick node (& descendents) to delete.");
  head_delete = NULL;
  button_press_proc = locate_node_to_delete;
}

setup_expand ()
{
  set_label (widgets->message, "Please pick node to expand.");
  button_press_proc = locate_node_to_expand;
}

setup_move ()
{
  if (head == (node *) 0)
    return;
  set_label (widgets->message, "Please pick component to move.");
  button_press_proc = get_graph_component;
}
