functor SCutils
  (structure Term : TERM
   structure Basic : BASIC
   structure Rename : RENAME sharing Rename.Term = Term
   structure Naming : NAMING  sharing Naming.Term = Term
   structure Sb : SB  sharing Sb.Term = Term
   structure Reduce : REDUCE sharing Reduce.Term = Term
   structure Print : PRINT  sharing Print.Term = Term
   structure IPrint : PRINT  sharing IPrint.Term = Term
             sharing Print.F = IPrint.F
             sharing Print.S = IPrint.S
   structure Lists : LISTS
  ) : SCUTILS =
 struct 

  structure F = Print.F
  structure S = Print.S
  structure Term = Term

  structure Switch =
  struct
    exception UnknownSwitch = Basic.UnknownSwitch

    (* Control *)
    fun control s = raise UnknownSwitch("SCutils.control",s)

    (* Warning *)
    fun warn s = raise UnknownSwitch("SCutils.warn",s)

    (* Tracing *)
    fun trace s = raise UnknownSwitch("SCutils.trace",s)

  end  (* structure Switch *)

  
   (**** Auxiliary functions****)


   (* Abbreviations for printing and conversion to strings *)

   local val String = F.String o S.string
   in
     fun makeformat_general_list op_paren cl_paren separator mft_el l  =
               F.HVbox([ String op_paren ]
               @ (case l of
                  nil => nil
                  | _ => (Lists.flatten
                            (map (fn e=> (mft_el e)
                                   @ [String separator, F.Break])
                                 (Lists.butlast l)))
                         @ (mft_el (Lists.last l)))
               @ [ String cl_paren ])
   end
   val makeformat_list = makeformat_general_list "[" "]" ","

          
   structure Abbrev = 
    struct
      val mft = Print.makeformat_term
      val mft' = IPrint.makeformat_term
      fun mfv vb = IPrint.makeformat_varbind vb
      val mfl = makeformat_list

      val mst = Print.makestring_term
      val mst' = IPrint.makestring_term
      fun msv (Term.Varbind(s,tm)) = (s^":"^(mst tm))
      val msl = Lists.makestring_list

      val pt = print o Print.makestring_term
      val pt' = print o IPrint.makestring_term
      fun pv (Term.Varbind(s,tm)) = (print (s^":"); pt tm)
      val pl = Lists.print_list
   end

  
   open Term
   open Abbrev

   fun is_eq(tm,tm') =
   let val clash = ref (Type,Type)
          
       fun eq(Evar(_,_,_,ref(SOME tm)),tm') = eq(tm,tm')
         | eq(tm,Evar(_,_,_,ref(SOME tm'))) = eq(tm,tm')
         | eq(tm as (Evar(vb,stamp,_,_),Evar(vb',stamp',_,_))) = 
                (clash:=tm; stamp=stamp' andalso isveq(vb,vb'))
         | eq(tm as (Evar _,_)) = (clash := tm; false)
         | eq(tm as (Uvar(vb,stmp),Uvar(vb',stmp'))) =
                (clash:=tm; stmp=stmp' andalso isveq(vb,vb'))
         | eq(tm as (Uvar _,_)) = (clash := tm; false)
         | eq(tm as (Bvar v,Bvar v')) = (clash:=tm; v=v')
         | eq(tm as (Bvar _,_)) = (clash := tm; false)
         | eq(Appl(M,N),Appl(M',N')) = eq(M,M') andalso eq (N,N')
         | eq(tm as (Appl _,_)) = (clash := tm; false)
         | eq(tm as (Const(E(ref{Bind=vb,...})),Const(E(ref{Bind=vb',...})))) =
                (clash:=tm; isveq(vb,vb'))
         | eq(tm as (Const _,_)) = (clash := tm; false)
         | eq(tm as (Pi((xofA as Varbind(x,A),B),_),
                     Pi((xofA' as Varbind(x',A'),B'),_))) = 
                 let val xofA = if Sb.free_in xofA B then xofA 
                                else Varbind(anonymous,A)
                     val xofA' = if Sb.free_in xofA' B' then xofA' 
                                else Varbind(anonymous,A')
                 in
                   (clash:=tm; isveq(xofA,xofA') andalso eq(B,B'))
                 end
         | eq(tm as (Pi _,_)) = (clash := tm; false)
         | eq(tm as (Abst(xofA,B),Abst(xofA',B'))) = 
             (clash:=tm; isveq(xofA,xofA') andalso eq(B,B'))
         | eq(tm as (Abst _,_)) = (clash := tm; false)
         | eq(HasType(M,N),HasType(M',N')) = eq(M,M') andalso eq (N,N')
         | eq(tm as (HasType _,_)) = (clash := tm; false)
         | eq(Type,Type) = true
         | eq(tm as (Type,_)) = (clash := tm; false)
         | eq(tm,tm') = raise Basic.Illegal 
                     ("SCutils.eq: Didn't provide for the case to compare\n"^
                      (mst' tm)^"\nand\n"^(mst' tm'))
       and isveq (Varbind(v,b),Varbind(v',b')) =
            (clash := (Bvar v,Bvar v'); v=v' andalso eq(b,b'))
       val res = eq(tm,tm')
   in (res) end



   val is_in = Lists.is_inEQ is_eq
   val remove = Lists.removeEQ is_eq

   (* Getting the type of variables and constants *)

   fun get_type (Uvar(Varbind(_,a),_)) = a
    |  get_type (Evar(Varbind(_,a),_,_,_)) = a
    |  get_type (Const(E(ref{Bind=Varbind(_,tp),...}))) = tp
    |  get_type tm = raise Basic.Illegal 
                        ("SCutils.get_type: "^(mst' tm)
                         ^" is not Uvar, Evar, or const")

   (* Splitting an application into head and its arguments -
      argumentlist will be empty if no arguments *)

   fun general_head_args tm =
       let fun mha (Appl(M,N)) arglist = mha M (N::arglist)
             | mha tm arglist = (tm, arglist)
       in mha tm nil end



   (* determine whether the Uvar/Evar/Const/Bvar occurs in A *)
   fun occurs_in_p p A =
   let fun occ (tm as (Uvar _)) = p tm
         | occ (tm as (Evar _)) = p tm
         | occ (tm as (Const _)) = p tm
         | occ (Appl(M1,M2)) = (occ M1) orelse (occ M2)
         | occ (Abst(xofA as Varbind(x,A),M)) = 
               if p (Bvar x) then false
                  else (vocc xofA) orelse (occ M)
         | occ (Pi((xofA as Varbind(x,A),B),_)) = 
               if p (Bvar x) then false
                  else (vocc xofA) orelse (occ B)
         | occ (HasType(M,A)) = (occ M) orelse (occ A)
         | occ (tm as (Bvar _)) = p tm
         | occ tm = false
       and vocc (Varbind(x,A)) = (occ A)
   in
      occ A
   end

   fun occurs_in (Evar(_,_,_,ref(SOME r))) = occurs_in r
     | occurs_in (Evar(Varbind(s,_),stamp,_,_)) =
       let fun match (Evar(_,_,_,ref(SOME r))) = match r
             | match (Evar(Varbind(s',_),stamp',_,_))= s=s' andalso stamp=stamp'
             | match _ = false
       in
          occurs_in_p match
       end
     | occurs_in (Uvar(Varbind(s,_),stamp)) =
       occurs_in_p (fn Uvar(Varbind(s',_),stamp') =>
                          s=s' andalso stamp=stamp'
                       | _ => false)
     | occurs_in (Const(E(ref{Bind=Varbind(s,_),...}))) =
       occurs_in_p (fn (Const(E(ref{Bind=Varbind(s',_),...}))) => s=s'
                       | _ => false)
     | occurs_in (Bvar v) =
       occurs_in_p (fn (Bvar v') => v=v' | _ => false)
     | occurs_in tm = raise Basic.Illegal 
                      ("SCutils.occurs_in: can't determine if "^(mst' tm)
                       ^" occurs")


    fun instantiate tm =
      let fun ins (Evar(vb,stamp,deps,ref(SOME instant))) = ins instant
            | ins (Evar(vb,stamp,deps,bound)) = Evar(vins vb,stamp,deps,bound)
            | ins (Uvar(vb,stamp)) = Uvar(vins vb,stamp)
            | ins (tm as (Const _)) = tm
            | ins (Appl(M1,M2)) = Appl(ins M1,ins M2)
            | ins (Abst(xofA,M)) = Abst(vins xofA,ins M)
            | ins (Pi((xofA,B),occ)) = Pi((vins xofA,ins B),occ)
            | ins (HasType(M,A)) = HasType(ins M,ins A)
            | ins (tm as (Bvar _)) = tm
            | ins tm = tm
          and vins (Varbind(x,A)) = Varbind(x,ins A)
      in
        ins tm
      end

   (* Return a list of all Uvars and/or Evars in a term, this list is sorted:
      variables that depend on others occur AFTER those they depend upon  *)


   fun get_evars tm =
   let fun add_deps tm depl =
           let val uvl = rev (get tm nil)
               fun prepend nil l = l
                 | prepend (h::t) l = if is_in h l then prepend t l
                                         else h::(prepend t l)
           in
            prepend uvl depl
           end
       and get (Evar (_,_,_,ref(SOME M))) uvl = get M uvl
         | get (tm as (Evar (Varbind(_,b),_,_,_))) uvl =
               add_deps b (if is_in tm uvl then uvl else tm::uvl)
         | get (tm as (Uvar (Varbind(_,b),_))) uvl = add_deps b uvl
         | get (tm as (Const _)) uvl = uvl
         | get (Appl(M1,M2)) uvl = get M2 (get M1 uvl)
         | get (Abst(xofA,M)) uvl = get M (vget xofA uvl)
         | get (Pi((xofA,B),occ)) uvl = get B (vget xofA uvl)
         | get (HasType(M,A)) uvl = get A (get M uvl)
         | get (tm as (Bvar _)) uvl = uvl
         | get tm uvl = uvl
       and vget (Varbind(x,A)) uvl = get A uvl
   in
     get (instantiate tm) nil
   end


   fun get_uvars tm =
   let fun add_deps tm depl =
           let val uvl = rev (get tm nil)
               fun prepend nil l = l
                 | prepend (h::t) l = if is_in h l then prepend t l
                                         else h::(prepend t l)
           in
            prepend uvl depl
           end
       and get (Evar (_,_,_,ref(SOME M))) uvl = get M uvl
         | get (tm as (Evar (Varbind(_,b),_,_,_))) uvl = get b uvl
         | get (tm as (Uvar (Varbind(_,b),_))) uvl =
               add_deps b (if is_in tm uvl then uvl else tm::uvl)
         | get (tm as (Const _)) uvl = uvl
         | get (Appl(M1,M2)) uvl = get M2 (get M1 uvl)
         | get (Abst(xofA,M)) uvl = get M (vget xofA uvl)
         | get (Pi((xofA,B),occ)) uvl = get B (vget xofA uvl)
         | get (HasType(M,A)) uvl = get A (get M uvl)
         | get (tm as (Bvar _)) uvl = uvl
         | get tm uvl = uvl
       and vget (Varbind(x,A)) uvl = get A uvl
   in
     get (instantiate tm) nil
   end

   fun get_uevars tm =
   let fun add_deps tm depl =
           let val vl = rev (get tm nil)
               fun prepend nil l = l
                 | prepend (h::t) l = if is_in h l then prepend t l
                                         else h::(prepend t l)
           in
            prepend vl depl
           end
       and get (Evar (_,_,_,ref(SOME M))) vl = get M vl
         | get (tm as (Evar (Varbind(_,b),_,_,_))) vl =
               add_deps b (if is_in tm vl then vl else tm::vl)
         | get (tm as (Uvar (Varbind(_,b),_))) vl =
               add_deps b (if is_in tm vl then vl else tm::vl)
         | get (tm as (Const _)) vl = vl
         | get (Appl(M1,M2)) vl = get M2 (get M1 vl)
         | get (Abst(xofA,M)) vl = get M (vget xofA vl)
         | get (Pi((xofA,B),occ)) vl = get B (vget xofA vl)
         | get (HasType(M,A)) vl = get A (get M vl)
         | get (tm as (Bvar _)) vl = vl
         | get tm vl = vl
       and vget (Varbind(x,A)) vl = get A vl
   in
     get (instantiate tm) nil
   end

   fun get_bvars tm =
   let fun get (tm as (Evar (_,_,_,ref(SOME M)))) = get M 
         | get (tm as (Evar (Varbind(_,b),_,_,_))) = get b 
         | get (tm as (Uvar (Varbind(_,b),_))) = get b 
         | get (tm as (Const _)) = nil
         | get (tm as (Appl(M1,M2))) = Lists.union (get M1) (get M2)
         | get (tm as (Abst(Varbind(x,A),M))) =
                Lists.union (get A) (Lists.remove x (get M))
         | get (tm as (Pi((Varbind(x,A),B),occ)))  =
                Lists.union (get A) (Lists.remove x (get B))
         | get (tm as (HasType(M,A))) =
                Lists.union (get M) (get A)
         | get (tm as (Bvar x)) = [x]
         | get Type = nil
         | get tm = raise Basic.Illegal 
                    ("SCutils.get_bvar: don't know how to get bvars from "
                     ^(mst' tm))
       val res = map (fn x => Bvar x) (get tm)
   in
     (res)
   end

(*
   fun get_bvars tm =
   let fun add_bvar tm (vl as (boundl,varl)) =
           if (is_in tm boundl) orelse (is_in tm varl) then vl
              else (boundl,tm::varl)
       and get (tm as (Evar (_,_,_,ref(SOME M)))) vl =
               get M vl
         | get (tm as (Evar (Varbind(_,b),_,_,_))) vl = 
               get b vl
         | get (tm as (Uvar (Varbind(_,b),_))) vl = 
               get b vl
         | get (tm as (Const _)) vl = 
               vl
         | get (tm as (Appl(M1,M2))) vl = 
               get M2 (get M1 vl)
         | get (tm as (Abst(xofA,M))) vl =
               get M (vget xofA vl)
         | get (tm as (Pi((xofA,B),occ))) vl =
               get B (vget xofA vl)
         | get (tm as (HasType(M,A))) vl =
               get A (get M vl)
         | get (tm as (Bvar _)) vl =
               add_bvar tm vl
         | get tm vl =
               vl
       and vget (Varbind(x,A)) (boundl,varl) =
           let val Bx=Bvar x
               val bl=if is_in Bx boundl then boundl else Bx::boundl
               val (_,varl) = get A (bl,varl)
           in (bl,varl) end
       val (_,bvars) = get tm (nil,nil)
   in
     (trace_get_bvars(fn()=>"Bvars in "^(mst' tm)^":\n"^(msl mst' bvars)^"\n");
      bvars)
   end
*)

   fun is_closed tm =
         if  (occurs_in_p (fn (Evar _) => true | (Uvar _) => true
                            | _  => false)
                          tm)
         then (get_uvars tm) @ (get_evars tm)
         else nil


   (*** Substitution/occurences of types ***)
   fun forced_sb_p M p tm =
    let fun fsbp (Evar (_,_,_,ref(SOME M))) = fsbp M
          | fsbp (tm as (Evar (Varbind(v,b),stamp,depl,inst))) =
                if p tm then M 
                else Evar(Varbind(v,fsbp b),stamp,map fsbp depl,inst)
          | fsbp (tm as (Uvar (Varbind(v,b),stamp))) =
                if p tm then M else (Uvar (Varbind(v,fsbp b),stamp))
          | fsbp (tm as (Const _)) = if p tm then M else tm 
          | fsbp (tm as Appl(M1,M2)) = if p tm then M else Appl(fsbp M1,fsbp M2)
          | fsbp (tm as Abst(xofA,B)) =
                   if p tm then M
                   else Abst(vfsbp xofA, fsbp B)
          | fsbp (tm as Pi((xofA,B),occ)) =
                   if p tm then M
                   else Pi((vfsbp xofA,fsbp B),occ)
          | fsbp (tm as HasType(A,B)) = 
                   if p tm then M
                   else HasType(fsbp A,fsbp B)
          | fsbp (tm as (Bvar _)) = if p tm then M else tm 
          | fsbp tm = tm
       and vfsbp (Varbind(x,A)) = Varbind(x,fsbp A)
   in
     fsbp tm
   end

   (* force the substitution of a term M for x, where x can be:
      - an Evar
      - a Uvar
      - a Bvar
      - a Const 
      - an Application (in this latter case we do not check not to go inside
                        of variable bindings!!!)
      throughout the term tm

    *)
  local
   fun forced_sb' M x tm =
   let val x_is_appl = case x of Appl _ => true | _ => false 
       fun conf nil v = false
         | conf ((Bvar v')::t) v = v=v' orelse conf t v
         | conf _ _ = raise Basic.Illegal
                       "SCutils.forced_sb.conflict: Shouldn't occur"
       val Mbvars = get_bvars M
       fun conflict v = (conf Mbvars v)

       fun fsb (tm as (Evar(_,_,_,ref(SOME r)))) = fsb r
         | fsb (tm as (Evar(Varbind(v,b),stamp,deps,bound))) = 
             let fun match (Evar(Varbind(v',b'),stamp',_,_)) =
                          v=v' andalso stamp=stamp' 
                   | match _ = false
             in
               if match x then M
                  else
                   Evar(fvsb (Varbind(v,b)),stamp,
                        fold (fn (deptm,deplist) =>
                                 case fsb deptm of
                                      (uv as Uvar _) => uv::deplist 
                                    | _ => deplist)
                             (rev deps)
                             nil,
                        bound)
             end
         | fsb (tm as (Uvar(vb as (Varbind(v,b)),stamp))) = 
             let fun match (Uvar(Varbind(v',b'),stamp')) =
                          v=v' andalso stamp=stamp' 
                   | match _ = false
             in
              if match x then M
                 else Uvar(fvsb vb,stamp)
             end
         | fsb (tm as (Const _)) = if is_eq(tm,x) then M else tm
         | fsb (tm as (Appl(M1,M2))) =
                if x_is_appl andalso is_eq(tm,x) then M
                   else Appl(fsb M1,fsb M2)
         | fsb (tm as (Abst(xofA as (Varbind(v,A)),B))) = 
                let fun match (Bvar v') = v=v'
                      | match _ = false
                in
                  if match x then tm
                  else if conflict v then
                    let val newv = Naming.new_vname conflict xofA
                    in
                      Abst(Varbind(newv,fsb A), 
                           fsb (forced_sb' (Bvar newv) (Bvar v) B))
                    end
                  else Abst(fvsb xofA, fsb B)
                 end 
         | fsb (tm as (Pi((xofA as (Varbind(v,A)),B),occ))) =
                let fun match (Bvar v') = v=v'
                      | match _ = false
                in
                  if match x then tm
                  else if conflict v then
                       let val newv = Naming.new_vname conflict xofA
                       in
                         make_pi(Varbind(newv,fsb A), 
                                 fsb (forced_sb' (Bvar newv) (Bvar v) B))
                       end
                  else make_pi(fvsb xofA, fsb B)
                end 
         | fsb (tm as (HasType(M,A))) = HasType(fsb M,fsb A)
         | fsb (tm as (Bvar v)) =
              (case x of Bvar v' => if v=v' then M else tm
                     | _ => tm)
         | fsb (tm as Type) = Type
         | fsb tm = tm
       and fvsb (Varbind(x,A)) = Varbind(x,fsb A)
       val res =
           case x of
             Const _ => fsb tm
           | Uvar _ =>  fsb tm
           | Evar(_,_,_,ref(SOME r)) => forced_sb' M r tm
           | Evar _ => fsb tm
           | Bvar _ => fsb tm
           | Appl _ => fsb tm
           |  _ => raise Basic.Illegal 
            ("Scutils.forced_sb: Not Evar/Uvar/Const/Bvar/Appl\n"^(mst' x))
    in
      (res)
    end
   in 
     fun forced_sb M x tm =
         (let val res = forced_sb' M x tm
          in res end)
   end

   (* A complicated form of substitution to take care of several existential
      variables
    *)

  local
   fun forced_sb' rek Mlist M x tm =
   let val _ = (Rename.push_state(); Rename.init_newvar (tm::Mlist))
       val x_is_appl = case x of Appl _ => true | _ => false 
       fun conf nil v = Rename.newvar_conflict v
         | conf ((Bvar v')::t) v = v=v' orelse conf t v
         | conf _ _ = raise Basic.Illegal
                       "SCutils.forced_sb.conflict: Shouldn't occur"
       val conflict = conf (get_bvars M)

       fun rename xofA =
           Rename.newvar (Naming.new_vname conflict xofA)

       fun fsb (tm as (Evar(_,_,_,ref(SOME r)))) = fsb r
         | fsb (tm as (Evar(Varbind(v,b),stamp,deps,bound))) = 
             let fun match (Evar(Varbind(v',b'),stamp',_,_)) =
                          v=v' andalso stamp=stamp' 
                   | match _ = false
             in
               if match x then M
                  else
                   Evar(fvsb (Varbind(v,b)),stamp,
                        fold (fn (deptm,deplist) =>
                                 case fsb deptm of
                                      (uv as Uvar _) => uv::deplist 
                                    | _ => deplist)
                             (rev deps)
                             nil,
                        bound)
             end
         | fsb (tm as (Uvar(vb as (Varbind(v,b)),stamp))) = 
             let fun match (Uvar(Varbind(v',b'),stamp')) =
                          v=v' andalso stamp=stamp' 
                   | match _ = false
             in
              if match x then M
                 else Uvar(fvsb vb,stamp)
             end
         | fsb (tm as (Const _)) = if is_eq(tm,x) then M else tm
         | fsb (tm as (Appl(M1,M2))) =
                if x_is_appl andalso is_eq(tm,x) then M
                   else Appl(fsb M1,fsb M2)
         | fsb (tm as (Abst(xofA as (Varbind(v,A)),B))) = 
                let fun match (Bvar v') = v=v'
                      | match _ = false
                in
                  if match x then tm
                  else Abst(fvsb xofA, fsb B)
                 end 
         | fsb (tm as (Pi((xofA as (Varbind(v,A)),B),occ))) =
                let fun match (Bvar v') = v=v' | match _ = false
                in if match x then tm
                  else
                  let val occurs = (occurs_in x A)
                      val (res_argl,res_body) =
                             if occurs
                             (* make different substitutions for each
                                existential variable *)
                             then (map (fn M =>
                                         let val A'= forced_sb' rek Mlist M x A
                                             val newv = rename xofA
                                         in
                                           Varbind(newv,A')
                                         end)
                                       Mlist,
                                   fsb B)
                             (* otherwise just descend recursively *)
                             else ([fvsb xofA], fsb B)
                      val res = fold make_pi res_argl res_body
                  in
                    (* Finally, we put in the recursive argument if desired *)
                    if rek andalso occurs
                       then make_pi(xofA,res)
                       else res
                  end
                end 
         | fsb (tm as (HasType(M,A))) = HasType(fsb M,fsb A)
         | fsb (tm as (Bvar v)) =
              (case x of Bvar v' => if v=v' then M else tm
                     | _ => tm)
         | fsb (tm as Type) = Type
         | fsb tm = tm
       and fvsb (Varbind(x,A)) = Varbind(x,fsb A)
       val res =
            case x of
              Const _ => fsb tm
            | Uvar _ =>  fsb tm
            | Evar(_,_,_,ref(SOME r)) => forced_sb' rek Mlist M r tm
            | Evar _ => fsb tm
            | Bvar _ => fsb tm
            | Appl _ => fsb tm
            |  _ => raise Basic.Illegal 
             ("Scutils.forced_sb_rek: Not Evar/Uvar/Const/Bvar/Appl\n"^(mst' x))
    in
      (Rename.restore_state(); res)
    end
   in 
     fun forced_sb_rec rek Mlist M x tm =
         (let val res = forced_sb' rek Mlist M x tm
          in (res) end)
   end


   fun type_sb_closed (recurse,mutual) Blist B A M =
   let (* We will disregard the number of conjuncts!!! *)
       val mutual = 1 (* irrelevant!! *)
       val (a,X_uvar_list) = general_head_args A
       val absfun = fold (fn (uv as Uvar(xofB as Varbind(x,b),stamp),tm) =>
                              Abst(xofB, forced_sb (Bvar x) uv tm)
                         | (uv,tm) => raise Basic.Illegal
                                 ("SCutils.type_sb_closed: in a !x1...!xn = "
                                  ^(mst' A)^" -- "^(mst' uv)^" is not a uvar"))
                        X_uvar_list
       val abs = absfun B
       val abslist = map absfun Blist
       val res0 = (forced_sb_rec recurse abslist abs a M)
       val res = Reduce.beta_norm res0
    in
       res
    end

   fun type_sb_open B A M =
   let val (a,X_uvar_list) = general_head_args A
       val (b,M_list) = general_head_args B
       val res = 
           if not(is_eq(a,b))
              then (*
                   raise Basic.Illegal 
                   ("SCutils.type_sb_open: [a M1...//a !X1...]M but a<>a:\n"
                     ^(mst' a)^" <> "^(mst' b)) *)
                   let val r = forced_sb B A M in
                       (r)
                   end
           else if (length X_uvar_list) <> (length M_list)
                   then raise Basic.Illegal 
                     "SCutils.type_sb_open: [a M1..Mm//a !X1..!Xn]M but m<>n\n"
                 else 
                (fold (fn ((uv as Uvar(xofB as Varbind(x,b),stamp),m),tm) =>
                              forced_sb m uv tm
                         | ((uv,m),tm) => raise Basic.Illegal
                                 ("SCutils.type_sb_open: in a !x1...!xn = "
                                  ^(mst' A)^" -- "^(mst' uv)^" is not a uvar"))
                        (Lists.zip X_uvar_list M_list)
                        M)
    in
       res
    end

 end (* struct *)
