functor Graph (structure Lists : LISTS
               structure Imperative : IMPERATIVE) : GRAPH =
  struct

   open Imperative
   open Lists
    
   datatype 'a graph = Graph of 'a list * ('a * 'a) list
   
   fun empty_graph () = Graph(nil,nil)
   fun add_edge (g as Graph(nl,el)) a b =
       if is_in (a,b) el then g
       else if a=b then Graph( if is_in a nl then nl else a::nl, (a,b)::el )
       else Graph( (if is_in a nl then nil else [a])
                   @(if is_in b nl then nil else [b]) @ nl, (a,b)::el )

   fun from_node (Graph(nl,el)) a = 
       if is_in a nl then filter (fn (a',_) => a=a') el
                     else nil
   fun to_node (Graph(nl,el)) a = 
       if is_in a nl then filter (fn (_,a') => a=a') el
                     else nil
   fun adjacent (Graph(nl,el)) a =
       if is_in a nl
          then fold (fn(a,b) => union [a] b)
                    (map (fn (_,b) => b)
                         (filter (fn (a',_) => a=a') el))
                    nil
          else nil

   fun remove_edge (g as Graph(nl,el)) a b =
       if not(is_in (a,b) el) then g
       else let val el = remove (a,b) el
                val nl=if null(from_node (Graph(nl,el)) a) then remove a nl
                          else nl
                val nl=if null(to_node (Graph(nl,el)) b) then remove b nl
                          else nl
            in
              Graph(nl,el)
            end
   fun nodes (Graph(nl,_)) = nl
   fun edges (Graph(_,el)) = el
   
   fun print_graph pel (g as Graph(nl,_)) =
       (map (fn n => (map (fn (a,b) => (pel a;
                                        print "-->";
                                        pel b; print "  ")) 
                          (from_node g n);
                      print "\n"))
            nl;
        ())
   
   
   (*** Strongly connected components ***)
   
   fun strongly_connected (g as (Graph(nl,vl))) =
      let
      (* Reference variables *)
      val count = ref 1
      val stack = init_stack nl
      val dfnumber = init_table (map (fn s => (s,0)) nl)
      val lowlink = init_table (map (fn s => (s,0)) nl)
      val marklist = init_mark nl
      val component = init_stack nl
      val components = init_stack [nl]
   
      fun searchc g v =
          (unmark v marklist;
           set dfnumber v (!count);
           inc count;
           set lowlink v (get dfnumber v);
           push v stack;
           map (fn w =>
                   if is_marked w marklist
                      then (searchc g w;
                            set lowlink v (min(get lowlink v, get lowlink w)))
                   else if (get dfnumber w) < (get dfnumber v)
                           andalso (is_on w stack)
                      then set lowlink v (min(get dfnumber w, get lowlink v))
                   else ())
               (adjacent g v);
           if (get lowlink v) = (get dfnumber v)
              then (do_until
                           (fn () => let val x = peek stack
                                     in  push x component end)
                           (fn () => let val x = pop stack in x=v end);
                    push (stack2list component) components;
                    empty_stack component)
              else ())
   in 
       (empty_table dfnumber; empty_table lowlink;
        empty_stack component; empty_stack components;
        count:=1;
        empty_mark marklist; map (fn n => mark n marklist) nl;
        empty_stack stack;
        while_do (fn () => case next_marked marklist of
                                NONE => false | _ => true)
                 (fn () => case next_marked marklist of
                              SOME v => searchc g v
                            | NONE => (print ("strongly_connected: NONE");
                                       raise Basic.Illegal ("strongly_connected: NONE")));
        stack2list components)
   end
   
   (*** Depth-first search for spanning tree ***)
   
   fun spanning_forest (g as (Graph(nl,vl))) =
       let val T = ref g
           val marklist = init_mark nl
           fun search g v =
               (unmark v marklist;
                map (fn w =>
                        if is_marked w marklist
                           then (T := add_edge (!T) v w;
                                 search g w)
                           else ())
                    (adjacent g v);
                ())
       in
         (T := empty_graph();
          empty_mark marklist; map (fn n => mark n marklist) nl;
          while_do (fn () => case next_marked marklist of
                                  NONE => false | _ => true)
                   (fn () => case next_marked marklist of
                                SOME v => search g v
                              | NONE => (print ("connected: NONE");
                                         raise Basic.Illegal ("strongly_connected: NONE")));
          !T)
       end
   
   (*** Printing out the equivalence classes of a graph ***)
   
   local
      fun conn_graph (g as Graph(_,el)) =
          let val nl = strongly_connected g
              fun find_node a nil =
                            raise Basic.Illegal ("find_node: shouldn't occur")
               |  find_node a (h::t) = if is_in a h then h 
                                          else find_node a t
          in
            fold (fn ((a,b), g) =>
                     let val al = find_node a nl
                         val bl = find_node b nl
                     in
                       add_edge g al bl
                     end )
                 el
                 (empty_graph())
          end

      datatype 'a entry = Singleton of 'a  | Class of string
      
      fun postprocess (Graph(nl,el)) =
          let val classes = filter (fn e => length e>1) nl
              fun position e =
                  let fun pos n nil = ~999999
                       |  pos n (h::t) = if e=h then n else pos (n+1) t
                  in
                    makestring(pos 1 classes)
                  end
              fun num [e] = Singleton e
               |  num e =   Class(position e)
          in 
           (map (fn c => (position c, c)) classes,
            Graph(map num nl, map (fn (a,b) => (num a,num b)) el))
          end
   in
      fun print_sconn_graph pel g =
          let val (el,g') = postprocess (conn_graph g)
          in
            (if null el then ()
                else (print "Equivalence Classes:\n";
                      map (fn (n,l) =>
                              (print ("class "^n^": ");
                               print_list pel l;
                               print "\n"))
                          el;
                      ());
            print "Vertices:\n";
            print_graph (fn (Singleton e) => (pel e;())
                          | (Class n) => print n) g')
          end
   end

end (* functor Graph *)
