functor Deps
  (structure Basic : BASIC
   structure Term : TERM
   structure Trail : TRAIL sharing Trail.Term = Term
   structure Sb : SB  sharing Sb.Term = Term
   structure Reduce : REDUCE sharing Reduce.Term = Term
   structure Sign : SIGN        sharing Sign.Term = Term
   structure SCutils : SCUTILS sharing SCutils.Term = Term
   structure Constraints : CONSTRAINTS
                    sharing Constraints.Term = Term
   structure Unify : UNIFY  sharing Unify.Term = Term
                            sharing Unify.Constraints = Constraints
   structure Progtab : PROGTAB  sharing Progtab.Term = Term
   structure TypeRecon : TYPE_RECON  sharing TypeRecon.Term = Term
   structure Store : STORE 
             sharing Progtab = Store.Progtab
             sharing Sign = Store.Sign
             sharing Term = Store.Term
   structure Formatter : FORMATTER
   structure Lists : LISTS
   structure Graph : GRAPH
  ) : DEPS =
 struct

   structure Graph = Graph
   structure Term = Term
   structure Sign = Sign
   structure Progtab = Progtab
   structure Constraints = Constraints

  structure Switch =
  struct
    val m = "Deps"
    exception UnknownSwitch = Basic.UnknownSwitch

    (* Control *)
    fun control s = raise UnknownSwitch(m^".control",s)

    (* Warning *)
    fun warn s = raise UnknownSwitch(m^".warn",s)

    (* Tracing *)
    fun trace s = raise UnknownSwitch(m^".trace",s)

  end  (* structure Switch *)



   open SCutils.Abbrev


   
  val trace_depsref = ref false
  fun trace_deps f = if (!trace_depsref) then f() else ()

  structure Switch =
  struct
    exception UnknownSwitch = Basic.UnknownSwitch

    (* Control *)
    fun control s = raise UnknownSwitch("Deps.control",s)

    (* Warning *)
    fun warn s = raise UnknownSwitch("Deps.warn",s)

    (* Tracing *)

    fun trace "deps" = trace_depsref 
      | trace s = raise UnknownSwitch("Deps.trace",s)
  end  (* structure Switch *)

   (*** Auxiliary functions ***)
  fun Consts'forall_tm () =
      (print "Module Deps: !Consts.forall_tm needed!";
       raise Basic.Illegal "Module Deps: !Consts.forall_tm needed!")

   fun is_ttype (Term.Type) = true
     | is_ttype _           = false

   fun is_kind (A) =
     let val (_,B) = Reduce.pi_vbds A
      in is_ttype B end

   (*** Determining the levels type families ***)

   datatype depend = Dep of string * depend list
  
   fun coll_sentry (Term.E (ref {Bind = Term.Varbind(c, cty),...})) = 
             Dep (c, coll_term cty) 
    |  coll_sentry _ = raise Basic.Illegal "Deps.coll_sentry: shouldn't occur"
   and coll_term (Term.Const centry) = [coll_sentry centry]
    |  coll_term (Term.Appl (t1,t2)) = Lists.union (coll_term t1) (coll_term t2)
    |  coll_term (Term.Abst (_,t)) = (coll_term t)
    |  coll_term (Term.Pi ((Term.Varbind(_,tv),tb),_)) = 
                       Lists.union (coll_term tv) (coll_term tb)
    |  coll_term (Term.HasType (t,_)) = (coll_term t)
    |  coll_term _ = []

   fun maxl l =
       let fun mx nil res = res
            |  mx [n] res = if n>res then n else res 
            |  mx (h::t) res = mx t (if h>res then h else res)
       in mx l ~999999 end

   fun process (Dep(_,l)) = processl l
   and processl [] = 0
    |  processl l = 1 + (maxl (map process l))
     
   fun level (Term.E (ref {Bind=Term.Varbind(A,B),...})) =
       if is_kind B then SOME(processl (coll_term B))
          else NONE
    |  level _ = raise Basic.Illegal "Deps.level: shouldn't occur"
           
   fun sign_levels sign =
     let fun add_fam (SOME (item as (Term.E (ref {Bind=Term.Varbind(A,_),...})),
                             sign')) fams =
               (case level item of
                     SOME res => add_fam (Sign.sig_item sign') ((A,res)::fams)
                   | NONE     => add_fam (Sign.sig_item sign') fams)
           | add_fam _ fams = fams
      in add_fam (Sign.sig_item sign) nil end

   fun print_levels sign =
       (map (fn (c,n) => print ("    "^c^" : level "^(makestring n)
                                ^" judgment.\n"))
            (sign_levels sign);
        ())


   (*** Determining the dependency information on clauses ***)

   fun coll_sentry (C as (Term.E (ref {Bind = Term.Varbind(c, cty),...}))) = 
       if is_kind cty 
          then case level C of
                 (SOME res) => [(c,res)]
               | _ => []
          else []
    |  coll_sentry _ = raise Basic.Illegal "Deps.coll_sentry: shouldn't occur"
   and coll_term (Term.Const centry) = coll_sentry centry
    |  coll_term (Term.Appl (t1,t2)) = Lists.union (coll_term t1) (coll_term t2)
    |  coll_term (Term.Abst (_,t)) = (coll_term t)
    |  coll_term (Term.Pi ((Term.Varbind(_,tv),tb),_)) = 
                       Lists.union (coll_term tv) (coll_term tb)
    |  coll_term (Term.HasType (t,_)) = (coll_term t)
    |  coll_term _ = []

   fun get_head (Term.Const
                 (C as (Term.E (ref {Bind = Term.Varbind(CN, CT),...})))) =
                 (case level C of
                    SOME n => SOME (CN,  n)
                  | NONE   => NONE)
    |  get_head (Term.Appl (t1,t2)) = get_head t1
    |  get_head  _ = NONE
 
   fun has_const (Term.E (ref {Bind=Term.Varbind(C,A),...})) =
     let val (Al,B) = Reduce.pi_vbds A
         val hconst = get_head B
      in 
       case hconst of
            NONE => NONE
         | (SOME (res as (_,lvl))) =>
             let val l = fold (fn(a,b)=> Lists.union a b)
                              (map (fn (Term.Varbind(_,A)) =>
                                       coll_term A)
                                   Al)
                              nil
             in
              case l of nil => NONE  | _ => SOME(C,res,l)
             end
      end
     |  has_const _ = raise Basic.Illegal "Deps.has_const: shouldn't occur"

    fun coll_levels sign =
      let fun add_fam (SOME (item, sign')) fams =
                (case has_const item of
                      SOME res => add_fam (Sign.sig_item sign') (res::fams)
                    | NONE     => add_fam (Sign.sig_item sign') fams)
            | add_fam _ fams = fams
       in add_fam (Sign.sig_item sign) nil end
 
    fun clause_deps sign = map (fn (c,(cHd,cHd_level),deps) =>
                                   (c,cHd, map (fn (t,lvl) => t) deps))
                               (coll_levels sign)

    fun print_clause_deps sign =
        (map (fn (c,cHd,deps) => ( print (c^" : ");
                                   print cHd;
                                   print (" <-- ");
                                   Lists.print_list print deps;
                                   print "\n"))
             (clause_deps sign);
         ())


    (*** Making the dependency graph for a signature ***)

    fun graph_deps sign =
        fold (fn ((_,(cHd,_),deps),g) => 
                  fold (fn ((c,_), gr) => Graph.add_edge gr c cHd)
                       deps
                       g)
             (coll_levels sign)
             (Graph.empty_graph())

    fun print_graph_deps s = let val G = graph_deps s
                  in Graph.print_sconn_graph (fn s=>print s) G
                  end

    val list_deps = Graph.strongly_connected o graph_deps

    local open Formatter
        fun format_list fel l =
            let fun fl nil = []
                 |  fl (h::h'::t) = [fel h, String ",", Break0 1 1]
                                    @ (fl (h'::t))
                 |  fl [h] = [fel h]
             in
               HVbox((String "[")::((fl l) @ [ String "]"]))
            end
    in
       fun print_list_deps s =
           let val dl = list_deps s
            in print( makestring_fmt( 
                (format_list (fn l => format_list String l) dl)))
            end
    end


   (*** Determining program entries to match with a given clause ***)

   fun print_progentry (Progtab.Progentry {Name=n, Vars=vl, Indx=ix,...}) =
       let val b = SCutils.get_type n
       in
        ( print "Progentry ";
          pt' n;
          map (fn (Term.Varbind(s,_))=> print (" "^s)) vl;
          print ": "; pt b; print ".";
          case ix of NONE => ()
                   | SOME(t) => (print " INDEX "; pt' t);
          print "\n") end

   local
     val res = ref NONE :
               (Term.term * Progtab.progentry * Term.varbind list) option ref
   in
   fun unify (uvars,evars,x,A,D,con) =
       let 
           val _ = trace_deps (fn ()=>
               (print "UNIFY:\nuvars ="; print (msl mst' uvars);
                print "\nevars = "; print (msl mst' evars);
                print "\nx = "; pt' x;
                print "\nA = "; pt' A;
                print "\nD = "; pt' D; print "\n";
                ()))
     
           val (piA_list, A) = Reduce.pi_vbds A
           val (Ahead,Aargs) = Reduce.head_args A
           val index = Progtab.get_index Ahead Aargs
           val _ = trace_deps (fn()=>
                   (case Ahead of
                    (Term.Const(Term.E(ref {Full = f, Prog = index_r, ...}))) =>
                       print ((mst' Ahead)^" = Const(E(ref {Full = "^
                              (mst' f)^", Prog = (ref "^
                              (case !index_r of NONE => "NONE"
                               | SOME(ri) => "SOME("^(makestring ri)^")")
                              ^"),  ...}))\n")
                    | _ => print ((mst' Ahead)^" != Const(E(ref ...))\n");
                   print ("Ahead = "^(mst' Ahead)^
                          "\nAargs = "^(msl mst' Aargs)^"\n");
                   case index of NONE => (print "No index!\n")
                    | SOME (tm) => (print "Index: "; pt' tm)
                   ))
       in
           fn (p as Progtab.Progentry {Name=n,Vars=vl,Indx=ix,...})=>
           (res := NONE;
            if Progtab.indexes_match index ix
               then let val _ =  trace_deps (fn()=>
                                              print "Index match\n")
                        val nesb = Sb.new_evar_sb (rev vl) uvars
                        val (_,b) = Reduce.pi_vbds(SCutils.get_type n)
                     in
                     (Trail.trail (fn () =>
                      let
                        val fresharg = Sb.apply_sb nesb b
                        val con'=Unify.unify1 fresharg A con
                        val _ = trace_deps (fn ()=>
                                            print "Unification success.\n" )
                        val xval = Sb.app_to_evars n nesb
                        val D' = SCutils.forced_sb xval x D
                        (* val env = vars2env uvars (nesb @ (tl evars)) *)
                        (*
                        val (env,D'') = TypeRecon.abst_over_evars nil
                                             TypeRecon.empty_env
                                             D'
                        val (result,syndef) =
                            TypeRecon.env_to_pis env D'' (Term.Type)
                        *)
                        val result = D'
                     in
                       res := SOME(result,
                                   p,
                                   piA_list)
                     end handle Constraints.Nonunifiable _ => ()
                               | exn => raise exn))
                    end
                else ();
             !res)
        end
 

   fun filter_rules (uvars,evars,x,A,D,con) =
       let val uvars = Lists.filter (fn (Term.Uvar _) => true | _ => false)
                                    uvars 
           val evars_ptr = ref evars
           val unify' = unify (uvars,evars,x,A,D,con)
           val (_, Abody) = Reduce.pi_vbds A
           val (Ahead,_) = Reduce.head_args Abody
           val prog = (Progtab.get_rules Ahead)
           val _ = trace_deps (fn ()=>
                          (print ("Found "^(makestring (length prog))
                           ^" potential rules for "^(mst' Ahead)^"\n")))
           fun match_sign nil res = res
            |  match_sign (p::t) res =
                   case unify' p of 
                        SOME(r as (tm,p,_))
                             => (match_sign t (r::res))
                      | NONE => (match_sign t res)
           val res = match_sign prog nil
       in
          res
       end
 
   end (* local *)

 end (* structure *)
