;;;
;;; Simple SML mode to go with COMINT package by Olin Shivers.
;;; Modified for Elf. -fp
;;;
;; YOUR .EMACS FILE
;;=============================================================================
;; Some suggestions for your .emacs file.
;;
;; ; If elf.el lives in some non-standard directory, you must tell emacs
;; ; where to get it. This may or may not be necessary.
;; (setq load-path (cons (expand-file-name "~jones/lib/emacs") load-path))
;;
;; ; Autoload elf and elf-mode from file elf.el
;; (autoload 'elf "elf" "Run an inferior Elf process." t)
;; (autoload 'elf-mode "elf" "Major mode for editing Elf source." t)
;;
;; ; Files ending in ".elf" are Elf source, so put their buffers in elf-mode.
;; (setq auto-mode-alist
;;       (cons '("\\.elf$" . elf-mode) (cons '("\\.quy$" . elf-mode)
;;	       auto-mode-alist)))
;;
;; ; The binary has an odd name or location:
;; (setq elf-program-name "/usr/local/beta-test/elfsml")
;; 
;; ; Define C-c t to run my favorite command in inferior Elf mode:
;; (setq elf-load-hook
;;       '((lambda () (define-key inferior-elf-mode-map "\C-ct"
;;                                'favorite-cmd))))
;; Change Log
;; Thu Jun  3 14:51:35 1993 -fp
;; Added variable display-elf-queries.  If T (default) redisplays Elf
;; buffer after a query has been sent.  Delays one second after sending
;; the query which is rather arbitrary.
;; Wed Jun 30 19:57:58 1993
;; - Error messages in the format line0.col0-line1.col1 can now be parsed.
;; - Error from std_in, either interactive or through elf-send-query
;;   can now be tracked.
;; - Added simple directory tracking and function elf-cd, bound to C-c d.
;; - improved filename completion in Elf mode under Lucid Emacs.
;; - replaced tail recursion in elf-indent-line by a while loop.
;; - changed elf-input-filter to ignore one character inputs.
;; - elf-error-marker is now updated on every interactive input or send.
;; - added commands elf-send-newline, bound to C-c RET
;;   and elf-send-semicolon, bound to C-c ;.
;;   These are useful when sending queries from a buffer with examples.

(require 'comint)
(provide 'elf)

(defun install-elf-keybindings (map)
  ;; Process commands:
  (define-key map "\C-c\C-r" 'elf-send-region)
  (define-key map "\C-c\C-e" 'elf-send-query)
  (define-key map "\C-c\C-m" 'elf-send-newline)
  (define-key map "\C-c\n" 'elf-send-newline) ; for Lucid Emacs
  (define-key map "\C-c;" 'elf-send-semicolon)
  (define-key map "\C-cd" 'elf-cd)
  (define-key map "\C-c`" 'elf-next-error)
  (define-key map "\C-c=" 'elf-goto-error)
  ;; Editing commands:
  (define-key map "\e\C-q" 'elf-indent-paragraph)
  (define-key map "\t" 'elf-indent-line)
  (define-key map "\177" 'backward-delete-char-untabify)
  )

(defvar elf-mode-map nil "The mode map used in elf-mode.")
(cond ((not elf-mode-map)
       (setq elf-mode-map (make-sparse-keymap))
       (install-elf-keybindings elf-mode-map)
       (define-key elf-mode-map "\C-c\C-z" 'switch-to-elf)))

(defvar elf-mode-syntax-table nil "The syntax table used in elf-mode.")

(defun set-elf-syntax (char entry)
  (modify-syntax-entry char entry elf-mode-syntax-table))
(defun set-word (char) (set-elf-syntax char "w   "))
(defun set-symbol (char) (set-elf-syntax char "_   "))

(defun map-string (func string)
  (if (string= "" string)
      ()
    (funcall func (string-to-char string))
    (map-string func (substring string 1))))

(if elf-mode-syntax-table
    ()
  (setq elf-mode-syntax-table (make-syntax-table))
  ;; A-Z and a-z are already word constituents
  ;; for fontification, it is better if _ and ' are word constituents
  (map-string 'set-word "!&$^+/<=>?@~|#*`;,-0123456789") ; word constituents
  (map-string 'set-symbol "_'")		; symbol constituents
  ;; Delimited comments are %{ }%, see 1234 below.
  (set-elf-syntax ?\ "    ")		; whitespace
  (set-elf-syntax ?\t "    ")		; whitespace
  (set-elf-syntax ?% "< 14")		; comment begin
  (set-elf-syntax ?\n ">   ")		; comment end
  (set-elf-syntax ?: ".   ")		; punctuation
  (set-elf-syntax ?. ".   ")		; punctuation
  (set-elf-syntax ?\( "()  ")		; open delimiter
  (set-elf-syntax ?\) ")(  ")		; close delimiter
  (set-elf-syntax ?\[ "(]  ")		; open delimiter
  (set-elf-syntax ?\] ")[  ")		; close delimiter
  (set-elf-syntax ?\{ "(}2 ")		; open delimiter
  (set-elf-syntax ?\} "){ 3")		; close delimiter
  ;; Actually, strings are illegal and \ is a symbol constituent.
  ;; However, for robustness we include:
  (set-elf-syntax ?\" "\"   ")		; string quote
  (set-elf-syntax ?\\ "/   ")		; escape
  )

(defvar elf-indent 3
  "Indent for Elf expressions.")

(defun elf-current-paragraph ()
  "Returns list (START END) for current paragraph."
  (let ((par-start nil)
	(par-end nil))
    (save-excursion
      (if (not (bobp)) (backward-paragraph 1))
      (if (looking-at "\n")
	  (forward-char 1))
      (setq par-start (point))
      (forward-paragraph 1)
      (setq par-end (point)))
    (list par-start par-end)))

(defun elf-indent-paragraph ()
  "Ident each line of the elf expression starting just after point.
This needs work."
  (interactive)
  (let* ((par (elf-current-paragraph))
	 (par-start (nth 0 par))
	 (par-end (nth 1 par)))
    (goto-char par-start)
    (elf-indent-lines (count-lines par-start par-end))))

(defun elf-indent-region (from to)
  "Indent each line of the region."
  (interactive "r")
  (cond ((< from to)
	 (goto-char from)
	 (elf-indent-lines (count-lines from to)))
	((> from to)
	 (goto-char to)
	 (elf-indent-lines (count-lines to from)))
	(t nil)))

(defun elf-indent-lines (n)
  "Indent N lines starting at point."
  (interactive "p")
  (if (= n 0)
      nil
    (elf-indent-line)
    (forward-line 1)
    (elf-indent-lines (1- n))))

(defun elf-comment-indent ()
  "Calculate the proper Elf comment column.
Currently does not deal specially with pragmas."
  (cond ((looking-at "%%%")
	 (current-column))
	((looking-at "%[%{]")
	 (car (calculate-elf-indent)))
	(t
	 (skip-chars-backward " \t")
	 (max (if (bolp) 0 (1+ (current-column))) comment-column))))

(defvar elf-infix-regexp ":\\|\\<->\\>\\|\\<<-\\>"
  "Regular expression to match Elf infix operators,
excluding surrounding whitespace.")

(defun looked-at ()
  "Returns the last string matched against.
Beware of intervening, even unsuccessful matches."
  (buffer-substring (match-beginning 0) (match-end 0)))

(defun elf-indent-line ()
  "Indent current line as Elf code."
  (interactive)
  (let ((old-point (point)))
    (beginning-of-line)
    (let* ((indent-info (calculate-elf-indent))
	   (indent-column (car indent-info))
	   (indent-type (car (cdr indent-info)))
	   (indent-string (cdr (cdr indent-info))))
      (skip-chars-forward " \t")	; skip whitespace
      (let ((fwdskip (- old-point (point))))
	(cond ((looking-at "%%%") nil)	; don't indent %%% comment
	      ((looking-at "%[^%{]")	; indent comment or pragma
	       (indent-for-comment)
	       (forward-char -1))
	      ((looking-at "%[%{]")	; delimited or %% comment
	       (indent-line-to indent-column fwdskip))
	      ((looking-at elf-infix-regexp) ; looking at infix operator
	       (if (string= indent-string (looked-at))
		   ;; indent string is the same as the one we are looking at
		   (indent-line-to indent-column fwdskip)
		 (indent-line-to (+ indent-column elf-indent) fwdskip)))
	      ((eq indent-type 'delimiter) ; indent after delimiter
	       (indent-line-to (+ indent-column elf-indent) fwdskip))
	      ((eq indent-type 'limit)	; no delimiter or infix found.
	       (indent-line-to indent-column fwdskip))
	      ((eq indent-type 'infix)
	       (indent-line-to (+ indent-column elf-indent) fwdskip)))))))

(defun indent-line-to (indent fwdskip)
  "Indent current line to INDENT then skipping to FWDSKIP if positive.
Assumes POINT is on the first non-whitespace character of the line."
  (let ((text-start (point))
	(shift-amount (- indent (current-column))))
    (if (= shift-amount 0)
	nil
      (beginning-of-line)
      (delete-region (point) text-start)
      (indent-to indent))
    (if (> fwdskip 0)
	(forward-char fwdskip))))

(defun calculate-elf-indent ()
  "Calculate the indentation and return a pair (INDENT INDENT-TYPE . STRING)
where INDENT-TYPE is 'DELIMITER, 'INFIX, or 'LIMIT and STRING is
the delimiter, infix operator, or the empty string, respectively."
  (interactive)
  (save-excursion
    (let ((old-point (point)))
      (if (not (bobp)) (backward-paragraph 1))
      (let ((limit (point)))
	(goto-char old-point)
	(let ((indent-data (elf-dsb limit)))
	  (cons (current-column) indent-data))))))

(defun elf-dsb (limit)
  "Scan backwards from POINT to find opening delimiter or infix operator.
This currently does not deal with comments or mis-matched delimiters."
  ;; The iterative style is painful, but tail-recursion does not 
  ;; work properly in Emacs Lisp
  (let ((result nil)
	(lparens 0) (lbraces 0) (lbrackets 0))
    (while (not result)
      (if (or (= lparens 1) (= lbraces 1) (= lbrackets 1))
	  (setq result (cons 'delimiter (looked-at)))
	(if (re-search-backward (concat "[][{}()]\\|" elf-infix-regexp)
				limit 'limit) ; return 'LIMIT if limit reached
	    (let ((found (looked-at)))
	      (cond
	       ((string= found "(") (setq lparens (1+ lparens)))
	       ((string= found ")") (setq lparens (1- lparens)))
	       ((string= found "{") (setq lbraces (1+ lbraces)))
	       ((string= found "}") (setq lbraces (1- lbraces)))
	       ((string= found "[") (setq lbrackets (1+ lbrackets)))
	       ((string= found "]") (setq lbrackets (1- lbrackets)))
	       (t;; otherwise, we are looking at an infix operator
		(if (and (= lparens 0) (= lbraces 0) (= lbrackets 0))
		    (setq result (cons 'infix found)) ; not embedded
		  nil))))		; embedded - skip
	  (setq result (cons 'limit ""))))) ; reached the limit
    result))

(defun elf-mode-variables ()
  (set-syntax-table elf-mode-syntax-table)
  ;; A paragraph is separated by blank lines or ^L only.
  (make-local-variable 'paragraph-start)
  (setq paragraph-start (concat "^[\t ]*$\\|" page-delimiter))
  (make-local-variable 'paragraph-separate)
  (setq paragraph-separate paragraph-start)
  (make-local-variable 'indent-line-function)
  (setq indent-line-function 'elf-indent-line)
  (make-local-variable 'comment-start)
  (setq comment-start "%")
  (make-local-variable 'comment-start-skip)
  (setq comment-start-skip "%+{?[ \t]*")
  (make-local-variable 'comment-end)
  (setq comment-end "\n")
  (make-local-variable 'comment-column)
  (setq comment-column 40)
  ;; (make-local-variable 'parse-sexp-ignore-comments)
  ;; (setq parse-sexp-ignore-comments t)
  )


(defun elf-mode ()
  "Major mode for editing Elf code.
Tab indents for Elf code.
Blank lines and form-feeds (^L's) separate paragraphs.
Delete converts tabs to spaces as it moves back.

For information on running an inferior Elf process, see the documentation
for inferior-elf-mode.

Customisation: Entry to this mode runs the hooks on elf-mode-hook.

Mode map
========
\\{elf-mode-map}"

  (interactive)
  (kill-all-local-variables)
  (elf-mode-variables)
  (use-local-map elf-mode-map)
  (setq major-mode 'elf-mode)
  (setq mode-name "Elf")
  (run-hooks 'elf-mode-hook))		; Run the hook

(defvar elf-prompt-regexp "^\\- \\|^\\?\\- ")
(defvar elf-program-name "elfsml")
(defvar inferior-elf-mode-map '())
(cond ((not inferior-elf-mode-map)
       (setq inferior-elf-mode-map (full-copy-sparse-keymap comint-mode-map))
       (install-elf-keybindings inferior-elf-mode-map)))

(defun elf-input-filter (input)
  "Function to filter strings before they are saved in input history.
We filter out all whitespace and anything shorter than two characters."
  (and (not (string-match "\\`\\s *\\'" input))
       (> (length input) 1)))

(defun expand-dir (dir)
  "Expand argument and check that it is a directory."
  (let ((expanded-dir (file-name-as-directory (expand-file-name dir))))
    (if (not (file-directory-p expanded-dir))
	(error "%s is not a directory" dir))
    expanded-dir))

(defun elf-cd (dir)
  "Make DIR become the current buffer's default directory and
furthermore issue an appropriate command to the inferior Elf process."
  (interactive "DChange default directory: ")
  (let ((expanded-dir (expand-dir dir)))
    (setq default-directory expanded-dir)
    (comint-simple-send (elf-proc) (concat "cd \"" expanded-dir "\";"))
    (pwd)))

(defvar cd-regexp "^\\s *cd\\s *\"\\([^\"]*\\)\""
  "Regular expression used to match cd commands in Elf buffer.")

(defun elf-directory-tracker (input)
  "Checks input for cd commands and changes default directory in buffer.
As a side-effect, it sets elf-last-region-sent to NIL to indicate interactive
input.  As a second side-effect, it resets the elf-error-marker.
Used as comint-input-sentinel in Elf buffer."
  (if (elf-input-filter input) (setq elf-last-region-sent nil))
  (set-marker elf-error-marker (process-mark (elf-proc)))
  (cond ((string-match cd-regexp input)
	 (let ((expanded-dir (expand-dir (substring input
						    (match-beginning 1)
						    (match-end 1)))))
	   (setq default-directory expanded-dir)
	   (pwd)))))

(defun inferior-elf-mode ()
  "Major mode for interacting with an inferior Elf process.

The following commands are available:
\\{inferior-elf-mode-map}

An Elf process can be fired up with \\[elf].

Customisation: Entry to this mode runs the hooks on comint-mode-hook and
inferior-t-mode-hook (in that order).

You can send queries to the inferior Elf process from other buffers.

Commands:
Return after the end of the process' output sends the text from the 
    end of process to point.
Return before the end of the process' output copies the current line
    to the end of the process' output, and sends it.
Delete converts tabs to spaces as it moves back.
Tab indents for Elf; with argument, shifts rest
    of expression rigidly with the current line.
C-M-q does Tab on each line starting within following expression.
Paragraphs are separated only by blank lines.  % start single comments,
delimited comments are enclosed in %{...}%.
If you accidentally suspend your process, use \\[comint-continue-subjob]
to continue it."
  (interactive)
  (kill-all-local-variables)
  (comint-mode)
  (setq comint-prompt-regexp elf-prompt-regexp)
  (setq comint-input-filter 'elf-input-filter)
  (setq comint-input-sentinel 'elf-directory-tracker)
  (elf-mode-variables)

  ;; For sequencing through error messages:
  (make-local-variable 'elf-error-marker)
  (setq elf-error-marker (point-max-marker))
  ;; Workaround for problem with Lucid Emacs version of comint.el:
  ;; must exclude double quotes " and must include $ and # in filenames.
  (make-local-variable 'comint-match-partial-pathname-chars)
  (setq comint-match-partial-pathname-chars 
	"^][<>{}()!^&*\\|?`'\" \t\n\r\b")

  (setq major-mode 'inferior-elf-mode)
  (setq mode-name "Inferior Elf")
  (setq mode-line-process '(": %s"))
  (use-local-map inferior-elf-mode-map)

  (run-hooks 'inferior-elf-mode-hook))


(defun elf (&optional cmd)
  "Run an inferior Elf process, input and output via buffer *elf*.
If there is a process already running in *elf*, just switch to that buffer.
With argument, allows you to edit the command line (default is value
of elf-program-name).  Runs the hooks from inferior-elf-mode-hook (after the
comint-mode-hook is run).

\(Type \\[describe-mode] in the process buffer for a list of commands.)"

  (interactive (list (and current-prefix-arg
			  (read-string "Run Elf: " elf-program-name))))
  (let ((cmd (or cmd elf-program-name)))
    (if (not (comint-check-proc (elf-proc-buffer)))
	(progn (set-buffer (apply 'make-comint "elf" cmd nil nil))
	       (inferior-elf-mode))))
  (switch-to-buffer (elf-proc-buffer)))

(defun switch-to-elf (eob-p)
  "Switch to the Elf process buffer.
With argument, positions cursor at end of buffer."
  (interactive "P")
  (if (elf-proc-buffer)
      (pop-to-buffer (elf-proc-buffer))
      (error "No current process buffer. "))
  (cond (eob-p
	 (push-mark)
	 (goto-char (point-max)))))


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; PARSING ERROR MESSAGES
;;; 
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;;; This is an modified version of ML error parsing due to Olin Shivers.

(defvar elf-error-regexp
  "^[^\n]+:[-0-9.]+ \\(Error\\|Warning\\):"
  "Regexp for matching error.")

(defun mtch (n)
  (let ((b (match-beginning n))
	(e (match-end n)))
    (if (or (null b) (null e)) nil
      (buffer-substring (match-beginning n) (match-end n)))))

(defun mtch-int (n)
  (let ((str (mtch n)))
    (if (null str) nil
      (string-to-int str))))

;; This function parses the error message into a 2 element list
;;     (file start-line)
(setq elf-error-parser
  (function (lambda (pt)
    (save-excursion
      (goto-char pt)
      (re-search-forward "^[-=? \t]*\\(.+\\):\
\\([0-9]+\\)\\(\\.\\([0-9]+\\)\\)?\\(-\\([0-9]+\\)\\(\\.\\([0-9]+\\)\\)?\\)?\
.+\\(Error\\|Warning\\):")
      (list (mtch 1)			; file
	    (mtch-int 2)		; start line
	    (or (mtch-int 4) 1)		; start column, if given, else 1
	    (mtch-int 6)		; end line, if given, else nil
	    (or (mtch-int 8) 1)		; end column, if given, else 1
	    )))))

(defun mark-relative (line0 col0 line1 col1)
  (if (not (= line0 1))
      (forward-line (1- line0)))
  ;; work around bug: from std_input, first line is off by one.
  (forward-char (if (not (= line0 1)) (1- col0) (1- (1- col0))))
  ;; select region, if non-empty
  (cond ((not (null line1))
	 (push-mark (point))
	 (cond ((not (= line1 line0))
		(forward-line (- line1 line0))
		(forward-char (1- col1)))
	       (t (forward-char (- col1 col0))))
	 (exchange-point-and-mark))))

(defun mark-absolute (line0 col0 line1 col1)
  (goto-line line0)
  ;; don't use move-to-column since <tab> is 1 char to lexer
  (forward-char (1- col0))
  ;; select region, if non-empty
  (cond ((not (null line1))
	 (push-mark (point))
	 (goto-line line1)
	 (forward-char (1- col1))
	 (exchange-point-and-mark))))

(defun elf-next-error ()
  "Find the next error by parsing the inferior Elf buffer.
Move the error message on the top line of the window;
put the cursor at the beginning of the error source. If the
error message specifies a range, the mark is placed at the end."
  (interactive)
  (let ((case-fold-search nil)
	(elf-buffer (elf-proc-buffer)))
    (pop-to-buffer elf-buffer)
    (goto-char elf-error-marker)	; Goto last found error
    (if (re-search-forward elf-error-regexp (point-max) t) ; go to next err
	(let* ((parse (funcall elf-error-parser (match-beginning 0)))
	       (file (nth 0 parse))
	       (line0 (nth 1 parse))
	       (col0 (nth 2 parse))
	       (line1 (nth 3 parse))
	       (col1 (nth 4 parse)))
	  (set-marker elf-error-marker (point))

	  (set-window-start (get-buffer-window elf-buffer)
			    (save-excursion (beginning-of-line) (point)))
	  (cond ((equal file "std_in")
		 ;; Error came from direct input
		 (cond ((null elf-last-region-sent)
			;; from last interactive input in the Elf buffer
			(goto-char (point-max))
			(comint-previous-input 1)
			(setq elf-last-region-sent t)
			(goto-char (process-mark (elf-proc)))
			(mark-relative line0 col0 line1 col1))
		       ((eq elf-last-region-sent t)
			;; from the waiting input in the Elf buffer
			(goto-char (process-mark (elf-proc)))
			(mark-relative line0 col0 line1 col1))
		       (t
			;; from a region sent from some buffer
			(let ((buf (car elf-last-region-sent))
			      (start (nth 1 elf-last-region-sent)))
			  (switch-to-buffer-other-window buf)
			  (goto-char start)
			  (mark-relative line0 col0 line1 col1)))))
		;; Error came from a source file
		((file-readable-p file)
		 (switch-to-buffer-other-window (find-file-noselect file))
		 (mark-absolute line0 col0 line1 col1))
		(t
		 (error (concat "Can't read file " file)))))
	(error "No error message found."))))

(defun elf-goto-error ()
  "Go to the error reported on the current line 
and set the error cursor to this line."
  (interactive)
  (pop-to-buffer (elf-proc-buffer))
  (beginning-of-line)
  (set-marker elf-error-marker (point))
  (elf-next-error))


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; Sending stuff to an inferior Elf process.
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defvar display-elf-queries t
  "*If NIL, the Elf buffer will be selected after it has been sent a query.")

(defun display-elf-buffer (&optional buffer)
  "Display the Elf buffer so that the end of output is visible."
  ;; Accept output from Elf process
  (sit-for 1)
  (let* ((elf-buffer (if (null buffer) (elf-proc-buffer)
		       buffer))
	 (_ (set-buffer elf-buffer))
	 (elf-process (elf-proc))
	 (proc-mark (process-mark elf-process))
	 (_ (display-buffer elf-buffer))
	 (elf-window (get-buffer-window elf-buffer)))
    (if (not (pos-visible-in-window-p proc-mark elf-window))
	(progn
	  (push-mark proc-mark)
	  (set-window-point elf-window proc-mark)))))

(defvar elf-last-region-sent nil
  "Contains a list (buffer start end) identifying the last region sent
to the Elf process for error tracking.
If NIL, then the last input was interactive.
If T, then the last input was interactive, but has already been copied
to the end of the Elf buffer.")

(defun elf-send-region (start end &optional and-go)
  "Send the current region to the inferior Elf process.
Prefix argument means switch-to-elf afterwards.
If the region is short, it is sent directly, via COMINT-SEND-REGION."
  (interactive "r\nP")
  (if (> start end)
      (elf-send-region end start and-go)
    (setq elf-last-region-sent (list (current-buffer) start end))
    (let ((cur-buf (current-buffer)))
      (switch-to-buffer (elf-proc-buffer))
      (set-marker elf-error-marker (process-mark (elf-proc)))
      (switch-to-buffer cur-buf))
    (comint-send-region (elf-proc) start end)
    (if (not (string= (buffer-substring (1- end) end) "\n"))
	(comint-send-string (elf-proc) "\n"))
    ;; Next two lines mess up when an Elf error occurs, since the
    ;; newline is not read and later messes up counting.
    ;; (if (not and-go)
    ;;	(comint-send-string (elf-proc) "\n"))
    (if and-go (switch-to-elf t)
      (if display-elf-queries (display-elf-buffer)))))

(defun elf-send-query (&optional and-go)
  "Send the current paragraph to the inferior Elf process.
Prefix argument means switch-to-elf afterwards."
  (interactive "P")
  (let* ((par (elf-current-paragraph))
	 (query-start (nth 0 par))
	 (query-end (nth 1 par)))
    (elf-send-region query-start query-end and-go)))

(defun elf-send-newline (&optional and-go)
  "Send a newline to the inferior Elf process.
If a prefix argument is given, switches to Elf buffer afterwards."
  (interactive "P")
  (comint-send-string (elf-proc) "\n")
  (if and-go (switch-to-elf t)
    (if display-elf-queries (display-elf-buffer))))

(defun elf-send-semicolon (&optional and-go)
  (interactive "P")
  (comint-send-string (elf-proc) ";\n")
  (if and-go (switch-to-elf t)
    (if display-elf-queries (display-elf-buffer))))

(defun elf-proc-buffer ()
  "Returns the current Elf process buffer."
  (get-buffer "*elf*"))

(defun elf-proc ()
  "Returns the current Elf process."
  (let ((proc (get-buffer-process (elf-proc-buffer))))
    (or proc
	(error "No current process."))))
