;;; -*- Mode: LISP; Syntax: Common-lisp; Package: DTP; Base: 10 -*-

;;;----------------------------------------------------------------------------
;;;
;;;	File		Prover.Lisp
;;;	System		Don's Theorem Prover
;;;
;;;	Written by	Don Geddis (Geddis@CS.Stanford.Edu)
;;;
;;;	Provides	prove, prove-next-answer, prove-all-remaining-answers

(in-package "DTP")

(eval-when (compile load eval)
  (export
   '(prove prove-next-answer prove-all-remaining-answers) ))

;;;----------------------------------------------------------------------------
;;;
;;;	Public

;;;----------------------------------------------------------------------------

(defun prove
    (query &key (theory *default-theory*) (all-answers nil) (return-form nil))
  "Returns (1) bound query(ies), (2) label(s), (3) <answer(s)>, (4) <proof>"
  (let (*proof*)
    (setq *proof*
      (make-proof :query query :theory theory :return-form return-form) )
    (setf (proof-query-conjunctions *proof*)
      (mapcar #'list-to-conjunction (dnf query)) )
    (let ((*goal-node-id-count* 0))
      (setf (proof-goal-nodes *proof*)
	(mapcar #'make-goal-node (proof-query-conjunctions *proof*)) ))
    (cond
     (all-answers
      (prove-all-remaining-answers *proof*) )
     (t
      (prove-next-answer *proof*) ))
    ))

;;;----------------------------------------------------------------------------

(defun prove-next-answer (&optional (*proof* *proof*))
  "Returns (1) bound query, (2) label, (3) <answer> or :NOT-AN-ANSWER, and
(4) <proof>"
  (let ((*proof-line-count* 0)
	(*depth* 0)
	(*subgoal-map* nil) )
    (explode-answer (prove-next-answer-internal *proof*)) ))

;;;----------------------------------------------------------------------------

(defun prove-all-remaining-answers (&optional (*proof* *proof*))
  "Returns (1) list of bound queries, (2) list of labels, (3) list of
<answers>, and (4) <proof>"
  (let ((*proof-line-count* 0)
	(*depth* 0)
	(*subgoal-map* nil)
	answers bounds label-values )
    (setq answers (prove-all-remaining-answers-internal *proof*))
    (setq bounds (mapcar #'apply-answer answers))
    (setq label-values (mapcar #'extract-label answers))
    (values bounds label-values answers *proof*) ))

;;;----------------------------------------------------------------------------
;;;
;;;	Private

;;;----------------------------------------------------------------------------

(defun prove-next-answer-internal (&optional (*proof* *proof*))
  "Return <answer> or :NOT-AN-ANSWER"
  (when (find :proofs *trace*)
    (format *debug-io* "Looking for next answer of ~A~%"
	    (proof-query *proof*) ))
  (loop
      while (or (proof-new-answers *proof*)
		(proof-subgoal-agenda *proof*)
		(proof-query-conjunctions *proof*) )
      for next-answer = (pop (proof-new-answers *proof*))
      for next-subgoal = (first (proof-subgoal-agenda *proof*))
      for next-conjunction = (first (proof-query-conjunctions *proof*))
      do
	(cond
	 (next-answer
	  (add-to-end next-answer (proof-answers *proof*))
	  (return next-answer) )
	 (next-subgoal
	  (if (exhausted-p next-subgoal)
	      (pop (proof-subgoal-agenda *proof*))
	    (expand next-subgoal) ))
	 (t
	  (if (exhausted-p next-conjunction)
	      (add-to-end
	       (pop (proof-query-conjunctions *proof*))
	       (proof-used-conjunctions *proof*) )
	    (expand next-conjunction) )))
      finally (return :not-an-answer) ))

;;;----------------------------------------------------------------------------

(defun prove-all-remaining-answers-internal (&optional (*proof* *proof*))
  "Return list of all remaining <answers>"
  (loop
      for next-answer = (prove-next-answer-internal *proof*)
      until (eq next-answer :not-an-answer)
      collect next-answer into answers
      when (find :proofs *trace*)
      do (format *debug-io* "Found answer ~A~2%" next-answer)
      finally (return answers) ))

;;;----------------------------------------------------------------------------

(defun explode-answer (answer)
  "Returns (1) bound *PROOF* query, (2) ANSWER, (3) *PROOF*"
  (declare (type answer answer))
  (if (answer-p answer)
      (values (apply-answer answer) (extract-label answer) answer *proof*)
    (values nil nil :not-an-answer *proof*) ))

;;;----------------------------------------------------------------------------

(defun apply-answer (answer)
  (declare (type answer answer))
  (let ((form (proof-return-form *proof*)))
    (unless form (setq form (proof-query *proof*)))
    (if (answer-ae-binding-lists answer)
	(cons 'or
	      (cons (plug form (answer-binding-list answer))
		    (mapcar #'(lambda (bl) (plug form bl))
			    (answer-ae-binding-lists answer) )))
      (plug form (answer-binding-list answer)) )))

;;;----------------------------------------------------------------------------

(defun extract-label (answer)
  (declare (type answer answer))
  (let ((label (answer-label answer)))
    (when label (label-value label)) ))

;;;----------------------------------------------------------------------------

(defun make-goal-node (conjunction)
  "Return clause corresponding to negated conjunction"
  (declare (type dtp-conjunction conjunction))
  (with-slots (list) conjunction
    (loop
	for conjunct in list
	for literal = (slot-value conjunct 'literal)
	for new-literal = (copy-literal-node literal)
	do (setf (literal-negated-p new-literal)
	     (not (literal-negated-p new-literal)) )
	collect new-literal into literals
	finally
	  (return
	    (make-kb-node
	     :id (make-new-id "GOAL" (incf *goal-node-id-count*))
	     :clause (make-clause-node :literals literals)) ))
    ))

;;;----------------------------------------------------------------------------
