;;; -*- Mode: LISP; Syntax: Common-lisp; Package: DTP; Base: 10 -*-

;;;----------------------------------------------------------------------------
;;;
;;;	File		Memoing.Lisp
;;;	System		Don's Theorem Prover
;;;
;;;	Written by	Don Geddis (Geddis@CS.Stanford.Edu)
;;;
;;;	Provides	find-subgoal

(in-package "DTP")

;;;----------------------------------------------------------------------------
;;;
;;;	Public

;;;----------------------------------------------------------------------------

(defun find-subgoal (conjunct)
  (declare (type dtp-conjunct conjunct))
  "Return (1) an old or new subgoal SAMEP as LITERAL, (2) binding list or nil"
  (let ((bl nil)
	literal sg )
    (setq literal
      (literal-plug
       (slot-value conjunct 'literal)
       (slot-value conjunct 'binding-list) ))
    (setq literal (nsimplify-terms literal)) ; Hook for term rewriting
    (setq sg (find-stored-subgoal literal))
    (if sg
	(progn
	  (setq bl
	    (dtp-samep
	     (literal-terms (slot-value sg 'literal))
	     (literal-terms literal)
	     t ))
	  (when (find :proofs *trace*)
	    (indent-line)
	    (format *debug-io* "Attaching to existing subgoal ~A~%"
		    (slot-value sg 'literal) )))
      (progn
	(setq sg (make-instance 'dtp-subgoal :literal literal))
	(memo-subgoal sg)
	(when (find :proofs *trace*)
	  (indent-line)
	  (format *debug-io* "Creating new subgoal ~A~%" literal) )))
    (values sg bl) ))

;;;----------------------------------------------------------------------------
;;;
;;;	Private

;;;----------------------------------------------------------------------------

(defun find-stored-subgoal (literal)
  (declare (type literal-node literal))
  (let (sg-list)
    (setq sg-list
      (gethash (literal-relation literal) (proof-subgoal-index *proof*)) )
    (find literal sg-list
	  :test #'(lambda (lit sg)
		    (literal-samep lit (slot-value sg 'literal)) ))
    ))

;;;----------------------------------------------------------------------------

(defun memo-subgoal (subgoal)
  (declare (type dtp-subgoal subgoal))
  (let ((relation (literal-relation (slot-value subgoal 'literal)))
	(table (proof-subgoal-index *proof*)) )
    (if (gethash relation table)
	(push subgoal (gethash relation table))
      (setf (gethash relation table) (list subgoal)) )
    ))

;;;----------------------------------------------------------------------------
