;;; -*- Mode: LISP; Syntax: Common-lisp; Package: DTP; Base: 10 -*-

;;;----------------------------------------------------------------------------
;;;
;;;	File		Internals.Lisp
;;;	System		Don's Theorem Prover
;;;
;;;	Written by	Don Geddis (Geddis@CS.Stanford.Edu)

(in-package "DTP")

;;;----------------------------------------------------------------------------

(defmacro add-to-end (item list)
  `(if ,list
       (rplacd (last ,list) (list ,item))
     (setf ,list (list ,item)) ))

(defmacro add-new-to-end (item list)
  `(unless (find ,item ,list)
     (add-to-end ,item ,list) ))

(defmacro add-new-to-beginning (item list)
  `(setf ,list (cons ,item (remove ,item ,list))) )

;;;----------------------------------------------------------------------------

(defun reset-dtp ()
  (mapc #'unintern *all-gensymed-variables*)
  (setq *all-gensymed-variables* nil
	*trace* *trace-defaults*
	*default-theory* 'global
	*node-id-count* 0 )
  (reset-hierarchy)
  (reset-database) )

;;;----------------------------------------------------------------------------

(defun find-vars (list)
  (cond
   ((consp list)
    (append (find-vars (car list))
	    (find-vars (cdr list)) ))
   ((varp list)
    (list list) )))

(defun binding-list-vars (binding-list)
  (remove-duplicates
   (loop
       for binding in binding-list
       collect (car binding)
       collect (cdr binding) )))

;;;----------------------------------------------------------------------------

(defun make-new-variable (var)
  (let (var-name)
    (setq var-name
      (subseq (symbol-name var) 0 (position #\_ (symbol-name var))) )
    (setq var-name (concatenate 'string var-name "_"))
    (push (gentemp var-name) *all-gensymed-variables*)
    (first *all-gensymed-variables*) ))

;;;----------------------------------------------------------------------------

(defun merge-binding-lists (binding-list &rest more-binding-lists)
  (if more-binding-lists
      (let (bl2 merge)
	(setq bl2 (reduce #'merge-binding-lists more-binding-lists))
	(when bl2
	  (setq binding-list (remove '(t . t) binding-list :test #'equal))
	  (setq bl2 (remove '(t . t) bl2 :test #'equal))
	  (if (or binding-list bl2)
	      (progn
		(setq merge (append-binding-lists binding-list bl2))
		(when merge (append merge '((t . t)))) )
	    '((t . t)) )))
    binding-list ))

;;;----------------------------------------------------------------------------

(defun unify-collection (sexp &rest more-sexps)
  (if more-sexps
      (let ((other-sexp (apply #'unify-collection more-sexps))
	    bl )
	(setq bl (dtp-unifyp sexp other-sexp))
	(when bl (plug sexp bl)) )
    sexp ))

;;;----------------------------------------------------------------------------

(defun make-new-id (str &optional (num nil))
  (if num
      (intern (format nil "~A-~D" str num) *dtp-package*)
    (intern
     (format nil "~A-~D"
	     str (incf *node-id-count*) )
     *dtp-package* )))

;;;----------------------------------------------------------------------------

(defun list-rename-variables (list)
  (let ((vars (find-vars list))
	bl )
    (setq bl (mapcar #'(lambda (x) (cons x (make-new-variable x))) vars))
    (plug list bl) ))

;;;----------------------------------------------------------------------------

(defun permutations (list-of-items)
  (if (= (length list-of-items) 1)
      (list list-of-items)
    (loop
	for item in list-of-items
	for remaining = (remove item list-of-items :test #'equal)
	appending
	  (loop
	      for perm in (permutations remaining)
	      collect (cons item perm) ))
    ))

;;;----------------------------------------------------------------------------

(defun tree-find (item tree)
  (cond
   ((or (null tree) (atom tree))
    nil )
   ((find item tree)
    t )
   (t
    (some #'(lambda (x) (tree-find item x)) tree) )))

;;;----------------------------------------------------------------------------

(defun partition (list test)
  "Return list with items satisfying TEST first, others following"
  (let (best others)
    (setq best (remove-if-not test list))
    (setq others (remove-if test list))
    (append best others) ))

;;;----------------------------------------------------------------------------
