;;; -*- Mode: LISP; Syntax: Common-lisp; Package: DTP; Base: 10 -*-

;;;----------------------------------------------------------------------------
;;;
;;;	File		Conjunctions.Lisp
;;;	System		Don's Theorem Prover
;;;
;;;	Written by	Don Geddis (Geddis@CS.Stanford.Edu)
;;;
;;;	Provides	expand (conjunction), exhausted-p (conjunction),
;;;			propogate (conjunction)

(in-package "DTP")

;;;----------------------------------------------------------------------------
;;;
;;;	Public

;;;----------------------------------------------------------------------------

(defmethod expand ((conjunction dtp-conjunction))
  (with-slots (list answers parent-subgoal stack-pointer binding-list label)
      conjunction
    (unless (< stack-pointer 0)
      (let ((conjunct (nth stack-pointer list)))
	(if conjunct
	    (expand conjunct)
	  (let ((mgu (merge-conjunction-binding-lists conjunction))
		(label (merge-conjunction-labels conjunction))
		(ae-bl (append-conjunction-ae-binding-lists conjunction))
		answer )
	    (unless (eq mgu :not-a-binding-list)
	      (if (some #'(lambda (ans) (typep ans 'me-answer)) answers)
		  (setq answer
		    (make-me-answer
		     :binding-list mgu :label label :ae-binding-lists ae-bl ))
		(setq answer
		  (make-answer
		   :binding-list mgu :label label :ae-binding-lists ae-bl )))
	      (if parent-subgoal
		  (propogate answer parent-subgoal)
		(add-to-end answer (proof-new-answers *proof*)) ))
	    (pop answers)
	    (decf stack-pointer) ))
	))))

(defmethod expand :around ((conjunction dtp-conjunction))
  "For proof tracing"
  (when (find :proofs *trace*)
    (indent-line)
    (format *debug-io* "Expanding conjunction~{ ~A~}~%"
	    (mapcar #'(lambda (c) (slot-value c 'literal))
		    (slot-value conjunction 'list) )))
  (incf *depth*)
  (prog1
      (call-next-method)
    (decf *depth*) ))

;;;----------------------------------------------------------------------------

(defmethod exhausted-p ((conjunction dtp-conjunction))
  (< (slot-value conjunction 'stack-pointer) 0) )

;;;----------------------------------------------------------------------------

(defmethod propogate (answer (conjunction dtp-conjunction))
  (with-slots (list answers stack-pointer) conjunction
    (if (eq answer :not-an-answer)
	(progn
	  (pop answers)
	  (decf stack-pointer) )
      (let ((next-conjunct (nth (incf stack-pointer) list))
	    mgu )
	(push answer answers)
	(setq mgu (merge-conjunction-binding-lists conjunction))
	(when next-conjunct
	  (if (eq mgu :not-a-binding-list)
	      (progn
		(pop answers)
		(decf stack-pointer)
		(return-from propogate) )
	    (setf (slot-value next-conjunct 'binding-list) mgu) ))
	(expand conjunction) ))
      ))

(defmethod propogate :around (answer (conjunction dtp-conjunction))
  "For proof tracing"
  (when (find :proofs *trace*)
    (indent-line)
    (format *debug-io* "Propogating ~S to conjunction~{ ~A~}~%"
	    answer
	    (mapcar #'(lambda (c) (slot-value c 'literal))
		    (slot-value conjunction 'list) )))
  (incf *depth*)
  (prog1
    (call-next-method)
    (decf *depth*) ))

;;;----------------------------------------------------------------------------
;;;
;;;	Private

;;;----------------------------------------------------------------------------

(defmethod expand ((conjunct dtp-conjunct))
  (with-slots (literal parent-conjunction binding-list transform-binding-list
	       answer-count subgoal used-subgoals )
      conjunct
    (when (eq subgoal :uninitialized)
      (multiple-value-bind (new-subgoal tbl) (find-subgoal conjunct)
        (setq subgoal new-subgoal)
        (setq transform-binding-list tbl)
	(add-new-to-beginning new-subgoal (proof-subgoal-agenda *proof*)) ))
    (with-slots (answers conjuncts-to-propogate-to) subgoal
      (let ((answer (nth answer-count answers)))
        (incf answer-count)
        (when (and transform-binding-list answer)
          (setq answer (copy-answer answer))
	  (setf (answer-binding-list answer)
	        (plug (answer-binding-list answer) transform-binding-list) ))
        (cond
         (answer
	  (propogate answer parent-conjunction) )
         ((exhausted-p subgoal)
	  (add-to-end subgoal used-subgoals)
	  (setq answer-count 0)
	  (setq subgoal :uninitialized)
	  (propogate :not-an-answer parent-conjunction) )
         (t                             ; Create a new forward-inf conjunction
	  (decf answer-count)
          (let ((new-conj (make-instance-conjunction parent-conjunction))
		(parent-subgoal
		 (slot-value parent-conjunction 'parent-subgoal) )
                head-conjunct )

            ;; Set up the new conjunction
	    (when new-conj
	      (if parent-subgoal
		  (push new-conj
			(slot-value
			 parent-subgoal 'forward-inference-conjunctions ))
		(push new-conj
		      (proof-forward-inference-conjunctions *proof*) ))
	      (setq head-conjunct (first (slot-value new-conj 'list)))
	      (add-to-end head-conjunct conjuncts-to-propogate-to) )
            
            ;; Continue with the original conjunction
            (add-to-end subgoal used-subgoals)
            (setq answer-count 0)
            (setq subgoal :uninitialized)
            (propogate :not-an-answer parent-conjunction) ))))
      )))

(defmethod expand :around ((conjunct dtp-conjunct))
  "For proof tracing"
  (if (find :conjunct-proofs *trace*)
      (progn
	(indent-line)
	(format *debug-io* "Expanding conjunct ~A~%"
		(literal-plug (slot-value conjunct 'literal)
			      (slot-value conjunct 'binding-list) ))
	(incf *depth*)
	(prog1
	    (call-next-method)
	  (decf *depth*) ))
    (call-next-method) ))

;;;----------------------------------------------------------------------------

(defmethod propogate (answer (conjunct dtp-conjunct))
  (declare (type answer answer))
  (with-slots (parent-conjunction transform-binding-list answer-count) conjunct
    (when (and (>= (slot-value parent-conjunction 'stack-pointer) 0)
	       (eq (nth (slot-value parent-conjunction 'stack-pointer)
			(slot-value parent-conjunction 'list) )
		   conjunct ))
      (if (eq answer :not-an-answer)
	  (propogate :not-an-answer parent-conjunction)
	(let ((new-answer answer))
	  (when transform-binding-list
	    (setq new-answer (copy-answer answer))
	    (setf (answer-binding-list new-answer)
	      (plug (answer-binding-list answer) transform-binding-list) ))
	  (incf answer-count)
	  (propogate new-answer parent-conjunction) )))))

(defmethod propogate :around (answer (conjunct dtp-conjunct))
  "For proof tracing"
  (declare (type answer answer))
  (if (find :conjunct-proofs *trace*)
      (progn
	(indent-line)
	(format *debug-io* "Propogating ~S to conjunct ~A~%"
		answer
		(literal-plug (slot-value conjunct 'literal)
			      (slot-value conjunct 'binding-list) ))
	(incf *depth*)
	(prog1
	    (call-next-method)
	  (decf *depth*) ))
    (call-next-method) ))

;;;----------------------------------------------------------------------------

(defun merge-conjunction-binding-lists (conjunction)
  "Return the merge of the answer binding lists, or :NOT-A-BINDING-LIST"
  (declare (type dtp-conjunction conjunction))
  (let (binding-lists final-binding-list)
    (setq binding-lists
      (mapcar #'answer-binding-list (slot-value conjunction 'answers)) )
    (setq binding-lists
      (remove nil
	      (cons (slot-value conjunction 'binding-list) binding-lists) ))
    (if (cdr binding-lists)
	(setq final-binding-list (reduce #'merge-binding-lists binding-lists))
      (setq final-binding-list (first binding-lists)) )
    (if (and binding-lists (null final-binding-list))
	:not-a-binding-list
      final-binding-list )))

;;;----------------------------------------------------------------------------

(defun append-conjunction-ae-binding-lists (conjunction)
  "Return the collection of the answer extraction binding lists"
  (declare (type dtp-conjunction conjunction))
  (let ((c-ae-bl (slot-value conjunction 'ae-binding-list))
	binding-lists )
    (setq binding-lists
      (mapcar #'answer-ae-binding-lists (slot-value conjunction 'answers)) )
    (when c-ae-bl
      (setq binding-lists (cons (list c-ae-bl) binding-lists)) )
    (reduce #'append binding-lists) ))

;;;----------------------------------------------------------------------------

(defun merge-conjunction-labels (conjunction)
  "Return a single label which is the conjunction of the answer labels"
  (declare (type dtp-conjunction conjunction))
  (let ((labels (mapcar #'answer-label (slot-value conjunction 'answers))))
    (setq labels (cons (slot-value conjunction 'label) labels))
    (setq labels (remove nil labels))
    (when labels (reduce #'label-and labels)) ))

;;;----------------------------------------------------------------------------

(defun list-to-conjunction (list)
  (declare (type list list))
  (loop
      for sublist in list
      collect (make-instance 'dtp-conjunct :literal (list-to-literal sublist))
      into conjuncts
      finally
	(let (conjunction)
	  (setq conjunction
	    (make-instance 'dtp-conjunction :list conjuncts :origin 'query) )
	  (dolist (conjunct conjuncts)
	    (setf (slot-value conjunct 'parent-conjunction) conjunction) )
	  (return conjunction) )))

;;;----------------------------------------------------------------------------

(defun make-instance-conjunction (conjunction)
  "Return a new forward-chaining conjunction using CONJUNCTION state"
  (let ((bl (merge-conjunction-binding-lists conjunction))
        (label (merge-conjunction-labels conjunction))
        new-conj conjuncts )
    (when (eq bl :not-a-binding-list)
      (return-from make-instance-conjunction) )
    (with-slots (list parent-subgoal stack-pointer origin) conjunction
      (setq new-conj
	(make-instance 'dtp-conjunction
	  :parent parent-subgoal :origin origin
	  :binding-list bl :label label ))
      (setq conjuncts
	(mapcar
	 #'(lambda (conjunct)
	     (setq conjunct (copy-conjunct conjunct))
	     (setf (slot-value conjunct 'parent-conjunction) new-conj)
	     (setf (slot-value conjunct 'used-subgoals) nil)
	     conjunct )
	 (nthcdr stack-pointer list) ))
      (setf (slot-value new-conj 'list) conjuncts)
      new-conj )))

;;;----------------------------------------------------------------------------

(defun copy-conjunct (conjunct)
  "Return a copy of the conjunct"
  (let (nc)
    (with-slots (literal parent-conjunction binding-list transform-binding-list
		 answer-count subgoal used-subgoals )
	conjunct
      (setq nc
	(make-instance 'dtp-conjunct
	  :literal literal :parent parent-conjunction ))
      (setf (slot-value nc 'binding-list) binding-list
	    (slot-value nc 'transform-binding-list) transform-binding-list
	    (slot-value nc 'answer-count) answer-count
	    (slot-value nc 'subgoal) subgoal
	    (slot-value nc 'used-subgoals) used-subgoals ))
    nc ))

;;;----------------------------------------------------------------------------
